/*************************************************************************
 * $Id: irmp3.c,v 1.17 2001/04/09 22:58:00 dpotter Exp $
 *
 * irmp3.c -- The main part of IRMP3
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/select.h>

#include "log.h"
#include "config.h"
#include "mod.h"


/*************************************************************************
 * GLOBALS
 */
#ifndef VERSION
#error "VERSION undefined - Use Makefile to compile!"
#endif
char *progname = NULL;		// progname (argv[0])
int terminate = 0;		// terminate flag
int reload = 0;			// reload flag
int daemonize = 1;		// daemon flag

/*************************************************************************
 * CATCH SIGNALS
 */
void catch_int (int signal)
{
	log_printf(LOG_NORMAL, "Got Signal %d! Terminating...\n", signal);
	terminate = 1;
}

void catch_hup (int signal)
{
	log_printf(LOG_NORMAL, "Got Signal %d! Reloading...\n", signal);
	reload = 1;
}


/*************************************************************************
 * THE MAIN LOOP
 */
int loop (void)
{
	int retval, rc, highfd,n;
	fd_set fdset;
	struct timeval tv;
	mod_t *m;

	retval = 0;
	do {
		// set the fds to watch
		FD_ZERO(&fdset);
		highfd=0;
		for (m=mod_list; m; m=m->next)
			if (m->poll)
			{
				for (n=0;n<__FD_SETSIZE;n++) {
					if(FD_ISSET(n,m->watchfd_set)) {
						FD_SET(n, &fdset);
						 if (n > highfd)
							 highfd = n;
					}
				}
			}
		tv.tv_sec = 0;
		tv.tv_usec = 500000;
		highfd++;

		// wait for data or timeout
		rc = select(highfd, &fdset, NULL, NULL, &tv);
		if (terminate || reload)
			break;

		// select reported error
		if (rc < 0) {
			if (errno == EINTR) {
				log_printf(LOG_NOISYDEBUG, "loop(): Ignoring EINTR on select()...\n");
				rc = 0;
				continue;
			}
			perror("select()");
			retval = -1;
			break;

		// input detected, look who can handle it
		// cycle through each fd.  For each one that has input, check all mods
		// to have them poll it.  (need to handle this better)
		} else if (rc > 0) {
			for (n=0;n<=highfd;n++)
				if(FD_ISSET(n,&fdset))
					for (m=mod_list;m;m=m->next)
						if (m->poll && FD_ISSET(n,m->watchfd_set))
							m->poll(n);
		// select timed out
		} 

		// disperse messages and update modules
		mod_update();

	} while (rc >= 0);

	return retval;
}


/*************************************************************************
 * COMMAND LINE HELP
 */
void usage (void)
{
	printf("IRMP3 v%s (C) by Andreas Neuhaus and David Potter\n", VERSION);
	printf("Questions, comments, ideas to David Potter <dp-irmp3@dpotter.com>\n");
	printf("Usage: %s [OPTION]...\n", progname);
	printf("	-h              this help\n");
	printf("	-q              silent operation (no logging except errors)\n");
	printf("	-qq             quiet operation (no logging)\n");
	printf("	-v              increase verbose level (more output)\n");
	printf("	-l file         logfile to use if running in background\n");
	printf(" 	                default: irmp3.log\n");
	printf("	-f              run in foreground and log to stdout\n");
	printf("	-c file         specifies the config file to use\n");
	printf("	                default: /etc/irmp3/irmp3.conf\n");
	printf("	-m message      input message to send at startup\n");
	printf("\n");
}


/*************************************************************************
 * MAIN
 */
int main (int argc, char *argv[])
{
	char c;
	int loglevel = LOG_NORMAL;
	char *logfile = "irmp3.log";
	char *configfile = "/etc/irmp3/irmp3.conf";
    char *commandLineMessage = NULL;
	char *execcmd=NULL,*execargs[48],*tmp=NULL;
	int rc,childpid,fd;

	// parse command line parameters
	(progname = strchr(argv[0], '/')) ? progname++ : (progname = argv[0]);
	while ((c = (char)getopt(argc, argv, "hqvl:fc:m:")) != EOF) {
		if (c == ':' || c == '?') {
			usage();
			return 1;
		}
		switch (c) {
			case 'h':
				usage();
				return 1;
			case 'q':
				loglevel--;
				break;
			case 'v':
				loglevel++;
				break;
			case 'l':
				logfile = optarg;
				break;
			case 'f':
				daemonize = 0;
				break;
			case 'c':
				configfile = optarg;
				break;
            case 'm':           // message passed in from command line
                commandLineMessage = optarg;
                break;
			default:
				usage();
				return 1;
		}
	}
	if (optind < argc) {
		usage();
		return 1;
	}

	// read configuration
	if (config_read(configfile)) {
		fprintf(stderr, "Unable to read config file %s\n", configfile);
		return 2;
	}

	// setup logging
	log_open(daemonize ? logfile : NULL, loglevel);

	// print copyright
	log_printf(LOG_NORMAL, "");
	log_printf(LOG_NORMAL, "IRMP3 v%s (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>\n", VERSION);
	log_printf(LOG_NORMAL, "Questions, comments, ideas to current maintainer: David Potter <dp-irmp3@dpotter.com>\n");

	// become a daemon
	if (daemonize) {
		close (STDIN_FILENO);		// close unnecessary file descriptors.
		close (STDOUT_FILENO);
		close (STDERR_FILENO);
		chdir("/");			// don't prevent sysadmin from unmounting any filesystem	
		umask(0);			// discard my parent's umask.
	
		if ((childpid = fork()) < 0) {
			log_printf(LOG_NORMAL, "Cannot fork first child!\n");
			exit (-1);
		} else if (childpid > 0)
			exit(0);		// parent process quits here
						// first child process continues on

		if (setpgrp() == -1 ) {		// change process group
			log_printf(LOG_NORMAL, "Cannot change process group ID!\n");
			exit (-1);
		}
		if ((fd = open("/dev/tty",O_RDWR)) >= 0 ) {
			ioctl(fd,TIOCNOTTY,(char*)NULL);  // lose controlling tty
			close (fd);
		}
		signal(SIGHUP, SIG_IGN);	// temporarily until we finish daemonizing
		if ((childpid = fork()) < 0) { 	// 2nd fork prevents us from re-acquiring a controlling tty
			log_printf(LOG_NORMAL, "Cannot fork second child!\n");
			exit (-1);
		} else if (childpid > 0)
			exit(0);		// first child process quits here.
						// second child process now continues as daemon.
	}

	// initialize all modules
	mod_init();

	// handle signals
	signal(SIGCHLD,mod_sigchld);
	signal(SIGINT,catch_int);
	signal(SIGTERM,catch_int);
	signal(SIGHUP,catch_hup);

    if(commandLineMessage)
        mod_sendmsg(MSGTYPE_INPUT, commandLineMessage);

	FD_ZERO(&blank_fd);

	// run the main loop
	do {

		// the main loop
		rc = loop();
		// reload data if neccessary
		if (reload) {
			config_reload();
			mod_reload();
			reload=0;
		}
	} while (!terminate && rc >= 0);

	// see if we should exec a program on exit
	tmp=config_getstr("exec_on_close",NULL);
	if(tmp) {
		execcmd=(char *)malloc(strlen(tmp)+2);
		strcpy(execcmd,tmp);
	}

	// deinitialize all modules
	mod_deinit();

	// free up configuration
	config_free();

	// shutdown logging
	log_printf(LOG_NORMAL, "IRMP3 v%s shutdown\n", VERSION);
	if(execcmd) log_printf(LOG_NORMAL,"executing on exit: %s\n",execcmd);
	log_close();

	// exit program
	if(execcmd) {
		rc=0;
		execargs[rc]=strtok(execcmd," ");
		while(execargs[rc] != NULL)
			execargs[++rc]=strtok(NULL," ");
		execvp(execargs[0],execargs);		// launch the specified program
	}
	return 0;
}


/*************************************************************************
 * EOF
 */
