/*************************************************************************
 * $Id: mod_beep.c,v 1.9 2001/03/27 02:29:07 dpotter Exp $
 *
 * mod_beep.c - audible beep support
 *
 * Copyright (C) 1999 by David E. Potter <dp-irmp3@dpotter.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "tools.h"
#include "config.h"
#include "log.h"
#include "mod.h"
#include "mod_beep.h"


/*************************************************************************
 * GLOBALS
 */

int beepflag = 1;
cmd_list beep_on_cmds;
cmd_list beep_off_cmds;



/*************************************************************************
 * MODULE INFO
 */
mod_t mod_beep = {
	mod_beep_deinit,	// deinit
	NULL,				// reload
	&blank_fd,			// watchfd
	NULL,				// poll
	NULL,				// update
	mod_beep_message,	// message
	NULL,				// SIGCHLD handler
};

/*************************************************************************
 * COMMAND LIST MESSAGE
 */

int cmdlist_append(cmd_list *cmdlist,char *command)
{
	cmd_list *newcmd;

	log_printf(LOG_NOISYDEBUG,"mod_beep(): adding beep event: %s\n",command);

	if (cmdlist->command == NULL) {  // this is the first command in the list
		cmdlist->command=(char *)malloc(strlen(command)+2);
		if (!cmdlist->command) return 0; // malloc failed
		strcpy(cmdlist->command,command);
		return 1;
				}
	
	while (cmdlist->next != NULL) {
		cmdlist=cmdlist->next;
			}
	newcmd=(cmd_list *)malloc(sizeof(cmd_list));
	if(!newcmd) return 0;
	bzero(newcmd,sizeof(cmd_list));
	newcmd->command=(char *)malloc(strlen(command)+2);
	if(!newcmd->command) return 0;	//malloc failed
	cmdlist->next=newcmd;
	strcpy(newcmd->command,command);

	return 1;
}

int cmdlist_search(cmd_list *cmdlist, char *command)
{
	while (cmdlist != NULL && cmdlist->command != NULL) {
		if (!strncasecmp(command,cmdlist->command,strlen(cmdlist->command))) {
			log_printf(LOG_NOISYDEBUG,"mod_beep(): recognized beep event: %s\n",command);
			return 1;
			}
		cmdlist=cmdlist->next;
		}
	return 0;
}
/*************************************************************************
 * BEEP PLEASE
 */

void beep(char *beepsound)
{
	char *beepargs[48];	// a reasonable number of arguments
	char beepcmd[512]; // a reasonable size for an expanded play command.
	char *beepplay,*beepdev,*beepfile;
	int childpid,c=0;

			beepplay = config_getstr("beep_play", "");
			beepdev  = config_getstr("beep_device", "");
	beepfile = config_getstr(beepsound, "");
			strcpy(beepcmd,beepplay);	// can't use the original because strtok munges it.
			beepargs[c]=strtok(beepcmd," ");
			while (beepargs[c] != NULL) {
				beepargs[++c]=strtok(NULL," ");
				if (beepargs[c] != NULL) {
					if (!strcmp(beepargs[c],"%f"))
						beepargs[c]=beepfile;
					if (!strcmp(beepargs[c],"%d"))
						beepargs[c]=beepdev;
				}
			}
			// fork off a new process

			if ((childpid = fork()) == -1 ) {
				log_printf(LOG_DEBUG,"mod_beep(): fork failed.\n");
			} else if (childpid == 0 ) {  // child process

				// close stdio to prevent child from interfering with parent
				close(STDIN_FILENO);
				close(STDOUT_FILENO);
				close(STDERR_FILENO);

				// spawn player
				execvp(beepargs[0], beepargs);
				log_printf(LOG_NORMAL,"Failed to exec beep program %s\n",beepargs[0]);
				exit(-1);
			} else {					// parent process
				log_printf(LOG_DEBUG,"mod_beep(): forked child process %d.\n",childpid);
			}
}
		





/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_beep_message (int msgtype, char *msg)
{
	char *c1, *c2, *c3=NULL;

	// check to see if this text should be beeped
	if(beepflag) {
		if(cmdlist_search(&beep_on_cmds,msg))
			beep("beep_on");
		if(cmdlist_search(&beep_off_cmds,msg))
			beep("beep_off");
			}

	// other special cases

	// sleep command requires some special processing

	if (msgtype == MSGTYPE_GENERIC) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		if (c1 && !strcasecmp(c1,"sleep")) {
			if (c2[0] == '0')
				beep("beep_off");
			else
				beep("beep_on");
		}
	}

	// handle input messages
	else if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		// process beep commands
		if (c1 && !strcasecmp(c1, "beep")) {
			if (c2 != NULL) {
				if (!strcasecmp(c2,"enable")) {
					log_printf(LOG_NORMAL,"Beeps enabled.\n");
					beepflag = 1;
				} else if (!strcasecmp(c2,"disable")) {
					log_printf(LOG_NORMAL,"Beeps disabled.\n");
					beepflag = 0;
				} else  if (!strcasecmp(c2,"on")) {
					c3=c2+strlen(c2)+1;
					if(c3 && c3[0] != 0) 
						cmdlist_append(&beep_on_cmds,c3);
				} else  if (!strcasecmp(c2,"off")) {
					c3=c2+strlen(c2)+1;
					if(c3 && c3[0] != 0) 
						cmdlist_append(&beep_off_cmds,c3);
				} else log_printf(LOG_NORMAL,"mod_beep(): unknown beep command: beep %s.\n",c2);
			}
		}
	} 
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_beep_init (void)
{
	log_printf(LOG_DEBUG, "mod_beep_init(): initializing\n");

	// register our module
	mod_register(&mod_beep);

	// initialize global variables
	bzero(&beep_on_cmds,sizeof(cmd_list));
	bzero(&beep_off_cmds,sizeof(cmd_list));

	// setup initial beep events

	cmdlist_append(&beep_on_cmds,"shuffle 1");
	cmdlist_append(&beep_off_cmds,"shuffle 0");
	cmdlist_append(&beep_on_cmds,"alarm 1");
	cmdlist_append(&beep_off_cmds,"alarm 0");


	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_beep_deinit (void)
{
	log_printf(LOG_DEBUG, "mod_beep_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
