/*************************************************************************
 * $Id: mod_idle.c,v 1.2 2001/04/18 16:51:48 dpotter Exp $
 *
 * mod_idle.c - a module to monitor periods of idle and notify other
 * 			 	modules.
 *
 * Copyright (C) 2001 by David E. Potter <dp-irmp3@dpotter.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <malloc.h>

#include "tools.h"
#include "config.h"
#include "log.h"
#include "mod.h"
#include "mod_idle.h"


/*************************************************************************
 * GLOBALS AND STATICS
 */

static unsigned int	idle_chunkiness;		// how often to announce idle state
static unsigned long idle_time = 0;			// total amount of time idle


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_idle = {
	mod_idle_deinit,	// deinit
	NULL,				// reload
	&blank_fd,			// watch_fdset
	NULL,				// poll
	NULL,				// update
	mod_idle_message, 	// message
	NULL,				// SIGCHLD handler
};

void mod_idle_message (int msgtype, char *msg)
{
	// it doesn't matter what the message is as long as it's not an IDLE msg.
	if(msgtype != MSGTYPE_IDLE && idle_chunkiness) { 
		idle_time = 0;
		alarm(idle_chunkiness);
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_idle_init (void)
{
	// register our module
	log_printf(LOG_DEBUG, "mod_idle_init(): initializing.\n");
	mod_register(&mod_idle);

	idle_chunkiness = config_getnum("idle_interval",0);
	if (!idle_chunkiness) {
		log_printf(LOG_ERROR, "mod_idle_init(): Cannot find idle_time.\n");
	} else {
		signal(SIGALRM,mod_idle_alarm);
		log_printf(LOG_DEBUG, "mod_idle_init(): idle message granularity is %d second(s).\n",idle_chunkiness);
		alarm(idle_chunkiness);
	}
	return NULL;
}

/*************************************************************************
 * ALARM SIGNAL HANDLER
 */

void mod_idle_alarm(int signo)
{
	log_printf(LOG_NOISYDEBUG, "mod_idle_alarm(): Idle time reached\n");
	idle_time+=idle_chunkiness;
	alarm(idle_chunkiness);
	mod_sendmsgf(MSGTYPE_IDLE,"%d",idle_time);
}

/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_idle_deinit (void)
{
	log_printf(LOG_DEBUG, "mod_idle_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
