/*************************************************************************
 *
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_kbd.c,v $ -- keyboard control
 * $Id: mod_kbd.c,v 1.4 2004/10/10 09:46:37 boucman Exp $
 *
 * Copyright (C) by Michael Behrisch
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <linux/input.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_kbd.h"

/*************************************************************************
 * GLOBALS
 */
char  *mod_kbd_keys[KEY_MAX + 1] = { "Reserved", "Esc", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "Minus", "Equal", "Backspace",
"Tab", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "LeftBrace", "RightBrace", "Enter", "LeftControl", "A", "S", "D", "F", "G",
"H", "J", "K", "L", "Semicolon", "Apostrophe", "Grave", "LeftShift", "BackSlash", "Z", "X", "C", "V", "B", "N", "M", "Comma", "Dot",
"Slash", "RightShift", "KPAsterisk", "LeftAlt", "Space", "CapsLock", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10",
"NumLock", "ScrollLock", "KP7", "KP8", "KP9", "KPMinus", "KP4", "KP5", "KP6", "KPPlus", "KP1", "KP2", "KP3", "KP0", "KPDot", "103rd",
"F13", "102nd", "F11", "F12", "F14", "F15", "F16", "F17", "F18", "F19", "F20", "KPEnter", "RightCtrl", "KPSlash", "SysRq",
"RightAlt", "LineFeed", "Home", "Up", "PageUp", "Left", "Right", "End", "Down", "PageDown", "Insert", "Delete", "Macro", "Mute",
"VolumeDown", "VolumeUp", "Power", "KPEqual", "KPPlusMinus", "Pause", "F21", "F22", "F23", "F24", "KPComma", "LeftMeta", "RightMeta",
"Compose", "Stop", "Again", "Props", "Undo", "Front", "Copy", "Open", "Paste", "Find", "Cut", "Help", "Menu", "Calc", "Setup",
"Sleep", "WakeUp", "File", "SendFile", "DeleteFile", "X-fer", "Prog1", "Prog2", "WWW", "MSDOS", "Coffee", "Direction",
"CycleWindows", "Mail", "Bookmarks", "Computer", "Back", "Forward", "CloseCD", "EjectCD", "EjectCloseCD", "NextSong", "PlayPause",
"PreviousSong", "StopCD", "Record", "Rewind", "Phone", "ISOKey", "Config", "HomePage", "Refresh", "Exit", "Move", "Edit", "ScrollUp",
"ScrollDown", "KPLeftParenthesis", "KPRightParenthesis",
"International1", "International2", "International3", "International4", "International5",
"International6", "International7", "International8", "International9",
"Language1", "Language2", "Language3", "Language4", "Language5", "Language6", "Language7", "Language8", "Language9",
NULL, 
"PlayCD", "PauseCD", "Prog3", "Prog4", "Suspend", "Close",
NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
"Btn0", "Btn1", "Btn2", "Btn3", "Btn4", "Btn5", "Btn6", "Btn7", "Btn8", "Btn9",
NULL, NULL,  NULL, NULL, NULL, NULL,
"LeftBtn", "RightBtn", "MiddleBtn", "SideBtn", "ExtraBtn", "ForwardBtn", "BackBtn",
NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
"Trigger", "ThumbBtn", "ThumbBtn2", "TopBtn", "TopBtn2", "PinkieBtn",
"BaseBtn", "BaseBtn2", "BaseBtn3", "BaseBtn4", "BaseBtn5", "BaseBtn6",
NULL, NULL, NULL, "BtnDead",
"BtnA", "BtnB", "BtnC", "BtnX", "BtnY", "BtnZ", "BtnTL", "BtnTR", "BtnTL2", "BtnTR2", "BtnSelect", "BtnStart", "BtnMode",
"BtnThumbL", "BtnThumbR", NULL,
"ToolPen", "ToolRubber", "ToolBrush", "ToolPencil", "ToolAirbrush", "ToolFinger", "ToolMouse", "ToolLens", NULL, NULL,
"Touch", "Stylus", "Stylus2", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
"Ok", "Select", "Goto", "Clear", "Power2", "Option", "Info", "Time", "Vendor",
"Archive", "Program", "Channel", "Favorites", "EPG", "PVR", "MHP", "Language",
"Title", "Subtitle", "Angle", "Zoom", "Mode", "Keyboard", "Screen", "PC", "TV",
"TV2", "VCR", "VCR2", "Sat", "Sat2", "CD", "Tape", "Radio", "Tuner", "Player", 
"Text", "DVD", "Aux", "MP3", "Audio", "Video", "Directory", "List", "Memo",
"Calendar", "Red", "Green", "Yellow", "Blue", "ChannelUp", "ChannelDown", 
"First", "Last", "AB", "Next", "Restart", "Slow", "Shuffle", "Break",
"Previous", "Digits", "Teen", "Twen" };

int mod_kbd_kbd_fd = -1;
int mod_kbd_mouse_fd = -1;

int mod_kbd_mouse_dx;
int mod_kbd_mouse_dy;
/*************************************************************************
 * MODULE INFO
 */
mod_t mod_kbd = {
    "mod_kbd",
    NULL,             // our deinit function
    mod_kbd_reload,   // called when got SIGHUP
    &mod_kbd_fdset,   // we're watching /dev/input/event...
    mod_kbd_poll,     // and process the sent keys
    NULL,             // we don't need to be called periodically
    mod_kbd_message,  // our message handler
    NULL,             // SIGCHLD handler
    mod_kbd_init,
    NULL,             // avoid warning
};

/*************************************************************************
 * MODULE INIT FUNCTION
 */
char* mod_kbd_init (void)
{
    FD_ZERO(&mod_kbd_fdset);
    log_printf(LOG_DEBUG, "mod_kbd_init(): initializing\n");
    if (!strcasecmp(config_getstr("kbd_enabled", "no"), "yes")) return mod_kbd_activate();
    return NULL;
}

/*************************************************************************
 * MODULE (DE-)ACTIVATION
 */
char* mod_kbd_activate(void)
{
    mod_kbd.poll = mod_kbd_poll;
    char* device = config_getstr("kbd_device", "/dev/input/event0");
    mod_kbd_kbd_fd = open(device, O_RDONLY);
    if (mod_kbd_kbd_fd < 0) return "Cannot open kbd_device";
    FD_SET(mod_kbd_kbd_fd, &mod_kbd_fdset);
    device = config_getstr("mouse_device", "/dev/input/event1");
    mod_kbd_mouse_fd = open(device, O_RDONLY);
    if (mod_kbd_mouse_fd < 0) return "Cannot open mouse_device";
    FD_SET(mod_kbd_mouse_fd, &mod_kbd_fdset);
    log_printf(LOG_NORMAL,"mod_kbd(): mouse activated.\n");
    return NULL;
}

void mod_kbd_deactivate(void)
{
    FD_ZERO(&mod_kbd_fdset);
    mod_kbd.poll = NULL;
    if (mod_kbd_kbd_fd >= 0) {
	    close(mod_kbd_kbd_fd);
	    mod_kbd_kbd_fd = -1;
    }
    if (mod_kbd_mouse_fd >= 0) {
	    close(mod_kbd_mouse_fd);
	    mod_kbd_mouse_fd = -1;
    }
    log_printf(LOG_NORMAL,"mod_kbd(): Keyboard deactivated.\n");
}

void mod_kbd_mouse_move (int* relpos, int sens, char* msg) {
    if (msg == NULL || !strcasecmp(msg,"")) return;
    while (sens * (*relpos) > sens * sens) {
        mod_sendmsg(MSGTYPE_INPUT, msg);
        *relpos -= sens;
    }
}
/*************************************************************************
 * POLL INPUT DATA
 */
void mod_kbd_poll (int fd)
{
    unsigned int i;
    char buf[32];
    char *s;
    struct input_event ev[64];
    int rd = read(fd, ev, sizeof(struct input_event) * 64);
    int sensitivity = config_getnum("mouse_sensitivity", 1);
    
    if (rd < (int) sizeof(struct input_event)) {
        log_printf(LOG_ERROR,"mod_kbd_poll(): error reading");
        return;
    }
    for (i = 0; i < rd / sizeof(struct input_event); i++) {
        if (ev[i].type == EV_KEY && ev[i].value == 1) { // key pressed event
            //log_printf(LOG_NOISYDEBUG, "mod_kbd_poll(): got input: %d (%s)\n", ev[i].code, mod_kbd_keys[ev[i].code]);
            snprintf(buf, sizeof(buf)-1, "kbd_key_%s", mod_kbd_keys[ev[i].code]);
            s = config_getstr(buf, NULL);
            if (s && strcasecmp(s,"")) mod_sendmsg(MSGTYPE_INPUT, s);
        } else if (ev[i].type == EV_REL) { // move event
            if (ev[i].code == REL_X) {
                //log_printf(LOG_NOISYDEBUG, "mod_kbd_poll(): got horizontal move: %d \n", ev[i].value);
                mod_kbd_mouse_dx += ev[i].value;
                mod_kbd_mouse_move(&mod_kbd_mouse_dx, -sensitivity, config_getstr("mouse_move_left", NULL));
                mod_kbd_mouse_move(&mod_kbd_mouse_dx, sensitivity, config_getstr("mouse_move_right", NULL));
            } else if (ev[i].code == REL_Y) {
                //log_printf(LOG_NOISYDEBUG, "mod_kbd_poll(): got vertical move: %d \n", ev[i].value);
                mod_kbd_mouse_dy += ev[i].value;
                mod_kbd_mouse_move(&mod_kbd_mouse_dy, -sensitivity, config_getstr("mouse_move_up", NULL));
                mod_kbd_mouse_move(&mod_kbd_mouse_dy, sensitivity, config_getstr("mouse_move_down", NULL));
            }
        }
    }
}

/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_kbd_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
    char *c1,*c2;

    // handle input messages
    if (msgtype == MSGTYPE_INPUT) {
        c1 =  strtok(msg, " \t");
        if(!c1) return;
        c2 = strtok(NULL, "");
        if(!c2) return;
        // process alarm commands
        if (!strcasecmp(c1, "kbd")) {
            if (!strcasecmp(c2,"1") || !strcasecmp(c2,"enable")) {
                mod_kbd_activate();
            } else if (!strcasecmp(c2,"0") || !strcasecmp(c2,"disable")) {
                mod_kbd_deactivate();
            } else log_printf(LOG_NORMAL,"mod_kbd(): unknown kbd command: %s %s.\n",c1,c2);
        }
    } 
}

/*************************************************************************
 * MODULE RELOAD FUNCTION
 */
char *mod_kbd_reload(void)
{
    mod_kbd_deactivate();
    if (!strcasecmp(config_getstr("kbd_enabled", "no"), "yes")) return mod_kbd_activate();
    return NULL;
}

/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_kbd_deinit (void)
{
    mod_kbd_deactivate();
    log_printf(LOG_DEBUG, "mod_kbd_deinit(): deinitialized\n");
}

/*************************************************************************
 * EOF
 */
