/*
 gui-dcc.c : User interface for DCC

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#define DCC_GUI(a) ((GUI_DCC_REC *) ((a)->gui_data))

typedef struct
{
    DCC_REC *dcc;
    gboolean closed;

    GtkWidget *window;
    GtkWidget *progbar;
    GtkWidget *infotext;
}
GUI_DCC_REC;

/* signal: cancel button clicked */
static gint sig_cancel_dcc(GUI_DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, 0);

    if (!dcc->closed) dcc_destroy(dcc->dcc);
    gtk_widget_destroy(dcc->window);
    g_free(dcc);

    return 0;
}

/* signal: dcc window closed / close button clicked */
static gint sig_close_dcc(GUI_DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, 0);

    if (!dcc->closed) dcc->dcc->gui_data = NULL;
    gtk_widget_destroy(dcc->window);
    g_free(dcc);

    return 0;
}

/* Create DCC transfer window */
void gui_dcc_dialog(DCC_REC *dcc)
{
    GUI_DCC_REC *rec;
    GtkWidget *label;
    GString *str;

    g_return_if_fail(dcc != NULL);

    if ((dcc->type != DCC_TYPE_SEND && dcc->type != DCC_TYPE_GET) || dcc->gui_data != NULL)
        return;

    rec = g_new0(GUI_DCC_REC, 1);
    dcc->gui_data = rec;
    rec->dcc = dcc;
    rec->closed = FALSE;

    str = g_string_new(NULL);
    g_string_sprintf(str, "DCC %s %s",
                     dcc->type == DCC_TYPE_SEND ? "SEND" : "GET", dcc->nick);

    rec->window = gnome_dialog_new(str->str, GNOME_STOCK_BUTTON_CANCEL,
                                   GNOME_STOCK_BUTTON_CLOSE, NULL);

    gtk_signal_connect_object(GTK_OBJECT(rec->window), "delete_event",
                              GTK_SIGNAL_FUNC(sig_close_dcc), (GtkObject *) rec);
    gnome_dialog_button_connect_object(GNOME_DIALOG(rec->window), 0,
                                       GTK_SIGNAL_FUNC(sig_cancel_dcc), (GtkObject *) rec);
    gnome_dialog_button_connect_object(GNOME_DIALOG(rec->window), 1,
                                       GTK_SIGNAL_FUNC(sig_close_dcc), (GtkObject *) rec);

    /* destination nick */
    g_string_sprintf(str, "%s -> %s", dcc->mynick, dcc->nick);
    if (dcc->server != NULL && dcc->server->ircnet != NULL)
        g_string_sprintfa(str, " (%s)", dcc->server->ircnet);

    label = gtk_label_new(str->str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(rec->window)->vbox), label, FALSE, FALSE, 0);

    /* file name */
    label = gtk_label_new(dcc->arg);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(rec->window)->vbox), label, FALSE, FALSE, 0);

    /* progress bar */
    rec->progbar = gtk_progress_bar_new();
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(rec->window)->vbox), rec->progbar, FALSE, FALSE, 10);

    /* statistics */
    rec->infotext = gtk_label_new("(waiting for connection)");
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(rec->window)->vbox), rec->infotext, FALSE, FALSE, 0);

    gtk_widget_show_all(rec->window);

    g_string_free(str, TRUE);
}

/* Close DCC transfer window */
static gboolean gui_dcc_closed(DCC_REC *dcc)
{
    GString *str;
    glong secs;

    g_return_val_if_fail(dcc != NULL, FALSE);
    if ((dcc->type != DCC_TYPE_SEND && dcc->type != DCC_TYPE_GET) || dcc->gui_data == NULL)
        return TRUE;

    str = g_string_new(NULL);

    secs = time(NULL)-dcc->starttime;
    if (dcc->transfd >= dcc->size)
    {
        gtk_progress_bar_update(GTK_PROGRESS_BAR(DCC_GUI(dcc)->progbar), 1);
        g_string_assign(str, _("transfer complete - all"));
    }
    else
    {
	gtk_progress_bar_update(GTK_PROGRESS_BAR(DCC_GUI(dcc)->progbar), (gfloat) dcc->transfd/dcc->size);
	g_string_assign(str, _("transfer aborted -"));
    }

    if (secs <= 0) secs = 1;

    g_string_sprintfa(str, _(" %lu bytes %s - avg %0.2fkB/sec"), dcc->transfd,
                      dcc->type == DCC_TYPE_SEND ? _("sent") : _("received"),
                      (gdouble) (dcc->transfd-dcc->skipped)/secs/1024);
    gtk_label_set(GTK_LABEL(DCC_GUI(dcc)->infotext), str->str);
    g_string_free(str, TRUE);

    DCC_GUI(dcc)->closed = TRUE;
    return TRUE;
}

/* Update DCC transfer window */
static gboolean gui_dcc_update(DCC_REC *dcc)
{
    static gint lastsecs = 0;
    glong secs;
    gchar *str;

    g_return_val_if_fail(dcc != NULL, FALSE);

    if (dcc->gui_data == NULL)
        return TRUE; /* not in a dialog */

    secs = time(NULL)-dcc->starttime;
    if (secs == lastsecs) return TRUE;

    lastsecs = secs;
    if (secs <= 0) secs = 1;

    str = g_strdup_printf(_("%lu of %lu bytes %s - avg %0.2fkB/sec"), dcc->transfd, dcc->size,
                          dcc->type == DCC_TYPE_SEND ? _("sent") : _("received"),
                          (gdouble) (dcc->transfd-dcc->skipped)/secs/1024);
    gtk_label_set(GTK_LABEL(DCC_GUI(dcc)->infotext), str);

    if (dcc->size != 0)
        gtk_progress_bar_update(GTK_PROGRESS_BAR(DCC_GUI(dcc)->progbar), (gfloat) dcc->transfd/dcc->size);

    g_free(str);
    return TRUE;
}

/* gui_dcc_chat() callback: OK button pressed */
static void gui_dcc_chat_open(GtkWidget *widget)
{
    SERVER_REC *server;
    GtkWidget *entry;
    gchar *str, *nick;

    g_return_if_fail(widget != NULL);

    server = gtk_object_get_data(GTK_OBJECT(widget), "data");
    entry = gtk_object_get_data(GTK_OBJECT(widget), "entry");
    g_return_if_fail(entry != NULL);

    nick = gtk_entry_get_text(GTK_ENTRY(entry));
    if (*nick != '\0')
    {
        str = g_strdup_printf("CHAT %s", nick);
        signal_emit("command dcc", 3, str, server, cur_channel);
        g_free(str);
    }

    gtk_widget_destroy(widget);
}

void gui_dcc_chat(SERVER_REC *server)
{
    g_return_if_fail(server != NULL);

    gui_entry_dialog(_("Enter nick to chat with: "), NULL, "DccChatNick", GTK_SIGNAL_FUNC(gui_dcc_chat_open), server);
}

/* ui_dcc_send() callback: OK button pressed */
static void gui_dcc_send_ok(GtkWidget *dialog)
{
    GtkWidget *nickentry, *fileentry;
    gchar *nick, *file;

    g_return_if_fail(dialog != NULL);

    nickentry = gtk_object_get_data(GTK_OBJECT(dialog), "nickentry");
    fileentry = gtk_object_get_data(GTK_OBJECT(dialog), "fileentry");

    nick = gtk_entry_get_text(GTK_ENTRY(nickentry));
    file = gnome_file_entry_get_full_path(GNOME_FILE_ENTRY(fileentry), TRUE);
    if (file == NULL)
    {
        /* file not found */
        gui_dialog(DIALOG_ERROR, _("Couldn't find file:\n%s"),
                   gnome_file_entry_get_full_path(GNOME_FILE_ENTRY(fileentry), FALSE));
    }

    if (file != NULL && *file != '\0' && *nick != '\0')
    {
        SERVER_REC *server;
        gchar *str;

        str = g_strdup_printf("SEND %s %s", nick, file);
        server = gtk_object_get_data(GTK_OBJECT(dialog), "server");
        signal_emit("command dcc", 3, str, server, cur_channel);
        g_free(str);
    }

    gtk_widget_destroy(dialog);
}

void gui_dcc_send(SERVER_REC *server, gchar *nick, gchar *fname)
{
    GtkWidget *dialog, *hbox, *label, *nickentry, *fileentry;
    gchar *path;

    g_return_if_fail(server != NULL);

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, gui_dcc_send_ok, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    nickentry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Nick: "), nick, FALSE);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    label = gtk_label_new(_("File to send: "));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    fileentry = gnome_file_entry_new("DCC SEND", _("Select file to send"));
    gtk_box_pack_start(GTK_BOX(hbox), fileentry, TRUE, TRUE, 0);

    if (fname != NULL)
        gtk_entry_set_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(fileentry))), fname);

    path = convert_home(setup_get_str("dcc_upload_path"));
    gnome_file_entry_set_default_path(GNOME_FILE_ENTRY(fileentry), path);
    g_free(path);

    gtk_object_set_data(GTK_OBJECT(dialog), "server", server);
    gtk_object_set_data(GTK_OBJECT(dialog), "nickentry", nickentry);
    gtk_object_set_data(GTK_OBJECT(dialog), "fileentry", fileentry);

    gtk_widget_show_all(dialog);
}

/* gui_dcc_request() signal: dcc accepted */
static void sig_accept_dcc(GtkWidget *dialog)
{
    DCC_REC *dcc;
    gchar *str;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "dcc");
    gtk_widget_destroy(dialog);

    str = dcc->type == DCC_TYPE_CHAT ?
        g_strdup_printf("CHAT %s", dcc->nick) :
        g_strdup_printf("GET %s %s", dcc->nick, dcc->arg);

    signal_emit("command dcc", 2, str, dcc->server);
    g_free(str);
}

/* gui_dcc_request() signal: dcc rejected */
static void sig_reject_dcc(GtkWidget *dialog)
{
    DCC_REC *dcc;
    gchar *str;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "dcc");
    gtk_widget_destroy(dialog);

    str = dcc->type == DCC_TYPE_CHAT ?
        g_strdup_printf("CLOSE CHAT %s", dcc->nick) :
        g_strdup_printf("CLOSE GET %s %s", dcc->nick, dcc->arg);

    signal_emit("command dcc", 2, str, dcc->server);
    g_free(str);
}

/* gui_dcc_request() signal: dcc get accepted with overwrite */
static void sig_get_overwrite(GtkWidget *dialog)
{
    DCC_REC *dcc;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "dcc");
    dcc->get_type = DCC_GET_OVERWRITE;
    sig_accept_dcc(dialog);
}

static void sig_get_rename_ok(GtkWidget *dialog)
{
    DCC_REC *dcc;
    gchar *str, *fname;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "data");
    fname = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog), "entry")));

    if (*fname != '\0')
    {
        g_free(dcc->arg);
        dcc->arg = g_strdup(fname);
        dcc->get_type = DCC_GET_RENAME;

        str = g_strdup_printf("GET %s %s", dcc->nick, dcc->arg);
        signal_emit("command dcc", 2, str, dcc->server);
        g_free(str);
    }
    gtk_widget_destroy(dialog);
}

static void sig_get_rename_reject(GtkWidget *dialog)
{
    DCC_REC *dcc;
    gchar *str;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "data");
    if (dcc->get_type != DCC_GET_RENAME)
    {
        str = g_strdup_printf("CLOSE GET %s %s", dcc->nick, dcc->arg);
        signal_emit("command dcc", 2, str, dcc->server);
        g_free(str);
    }
}

/* gui_dcc_request() signal: dcc get accepted with rename */
static void sig_get_rename(GtkWidget *dialog)
{
    DCC_REC *dcc;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "dcc");
    gtk_widget_destroy(dialog);

    dialog = gui_entry_dialog(_("Rename file to"), dcc->arg, NULL, GTK_SIGNAL_FUNC(sig_get_rename_ok), dcc);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_get_rename_reject), GTK_OBJECT(dialog));
    gui_widget_depends_data(dialog, "dcc destroyed", dcc);
}

/* gui_dcc_request() signal: dcc get accepted with resume */
static void sig_get_resume(GtkWidget *dialog)
{
    DCC_REC *dcc;
    gchar *str;

    g_return_if_fail(dialog != NULL);

    dcc = gtk_object_get_data(GTK_OBJECT(dialog), "dcc");
    gtk_widget_destroy(dialog);

    str = g_strdup_printf("RESUME %s %s", dcc->nick, dcc->arg);
    signal_emit("command dcc", 2, str, dcc->server);
    g_free(str);
}

static gboolean gui_dcc_request(DCC_REC *dcc)
{
    GtkWidget *dialog, *label;
    GString *str;

    g_return_val_if_fail(dcc != NULL, FALSE);

    dialog = NULL;
    if (dcc->type == DCC_TYPE_GET)
    {
        /* Check if file exists and ask overwrite/rename(/resume) */
        struct stat statbuf;
        gchar *downpath, *ptr;

        ptr = strrchr(dcc->arg, '/');
        if (ptr == NULL) ptr = dcc->arg; else ptr++;

        downpath = convert_home(setup_get_str("dcc_download_path"));
        ptr = g_strdup_printf("%s/%s", downpath, ptr);
        g_free(downpath);

        if (stat(ptr, &statbuf) == 0)
        {
            if (statbuf.st_size < dcc->size)
            {
                dialog = gnome_dialog_new(PACKAGE, _("Overwrite"), _("Rename"), _("Resume"), GNOME_STOCK_BUTTON_CANCEL, NULL);
                gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(sig_get_resume), GTK_OBJECT(dialog));
                gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 3, GTK_SIGNAL_FUNC(sig_reject_dcc), GTK_OBJECT(dialog));
            }
            else
            {
                dialog = gnome_dialog_new(PACKAGE, _("Overwrite"), _("Rename"), GNOME_STOCK_BUTTON_CANCEL, NULL);
                gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(sig_reject_dcc), GTK_OBJECT(dialog));
            }
            gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_get_overwrite), GTK_OBJECT(dialog));
            gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(sig_get_rename), GTK_OBJECT(dialog));
	}
	g_free(ptr);
    }

    if (dialog == NULL)
    {
        dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
        gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_accept_dcc), GTK_OBJECT(dialog));
        gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(sig_reject_dcc), GTK_OBJECT(dialog));
    }

    gui_widget_depends_data(dialog, "dcc destroyed", dcc);

    gtk_signal_connect_object(GTK_OBJECT(dialog), "delete_event",
                              GTK_SIGNAL_FUNC(sig_reject_dcc), GTK_OBJECT(dialog));

    gtk_object_set_data(GTK_OBJECT(dialog), "dcc", dcc);

    str = g_string_new(NULL);
    g_string_sprintf(str, _("Accept DCC %s request from %s (%s)?"),
                     dcc->type == DCC_TYPE_CHAT ? "CHAT" : "SEND", dcc->nick, dcc->addrstr);
    label = gtk_label_new(str->str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, TRUE, TRUE, 5);

    if (dcc->type == DCC_TYPE_GET)
    {
        g_string_sprintf(str, _("File name: %s\nFile size: %ld\n"), dcc->arg, dcc->size);
        label = gtk_label_new(str->str);
        gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, TRUE, TRUE, 5);

    }

    g_string_free(str, TRUE);

    gtk_widget_show_all(dialog);
    return TRUE;
}

static gboolean sig_dcc_connected(DCC_REC *dcc)
{
    g_return_val_if_fail(dcc != NULL, FALSE);

    if (setup_get_bool("toggle_dcc_autodisplay_dialog") && dcc->type != DCC_TYPE_CHAT)
	gui_dcc_dialog(dcc);

    return TRUE;
}

/* command: DCC SEND */
static gboolean cmd_dcc_send(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *target, *fname;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &fname);
    if (*target != '\0' && *fname == '\0')
    {
	/* nick specified but not file name, open the dialog */
	gui_dcc_send(server, target, NULL);
	g_free(params);
	return FALSE;
    }
    g_free(params);
    return TRUE;
}

void gui_dcc_init(void)
{
    signal_add("dcc closed", (SIGNAL_FUNC) gui_dcc_closed);
    signal_add("dcc transfer update", (SIGNAL_FUNC) gui_dcc_update);
    signal_add("dcc request", (SIGNAL_FUNC) gui_dcc_request);
    signal_add("dcc connected", (SIGNAL_FUNC) sig_dcc_connected);
    command_bind("dcc send", NULL, (SIGNAL_FUNC) cmd_dcc_send);
}

void gui_dcc_deinit(void)
{
    signal_remove("dcc closed", (SIGNAL_FUNC) gui_dcc_closed);
    signal_remove("dcc transfer update", (SIGNAL_FUNC) gui_dcc_update);
    signal_remove("dcc request", (SIGNAL_FUNC) gui_dcc_request);
    signal_remove("dcc connected", (SIGNAL_FUNC) sig_dcc_connected);
    command_unbind("dcc send", (SIGNAL_FUNC) cmd_dcc_send);
}
