/*
 setup-bg.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"
#include "setup-themes.h"

static GtkWidget *setup_dialog;

static void save_changes(void)
{
    GtkEntry *fontentry, *bgentry;
    gchar *str;

    fontentry = gtk_object_get_data(GTK_OBJECT(setup_dialog), "fontentry");
    bgentry = gtk_object_get_data(GTK_OBJECT(setup_dialog), "bgentry");

    str = gtk_entry_get_text(fontentry);
    if (temp_current_theme->font != NULL)
	g_free(temp_current_theme->font);
    temp_current_theme->font = *str == '\0' ? NULL : g_strdup(str);

    str = gtk_entry_get_text(bgentry);
    if (temp_current_theme->bg_pixmap != NULL)
	g_free(temp_current_theme->bg_pixmap);
    temp_current_theme->bg_pixmap = *str == '\0' ? NULL : g_strdup(str);
}

static gboolean sig_setup_apply(GtkWidget *dialog)
{
    save_changes();
    return TRUE;
}

static gboolean sig_setup_apply_again(GtkWidget *dialog)
{
    proplist_t tprop, fname;
    GList *tmp;

    g_return_val_if_fail(dialog != NULL, FALSE);

    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
        THEME_REC *rec = tmp->data;

	/* try to open theme.. */
	tprop = PLGetProplistWithPath(rec->path);
	if (tprop == NULL)
	{
	    /* not found, create new theme file */
	    tprop = PLMakeDictionaryFromEntries(NULL, NULL);
	    fname = PLMakeString(rec->path);
	    tprop = PLSetFilename(tprop, fname);
	    PLRelease(fname);
	}

	config_set_bool(tprop, "bg_scrollable", (rec->flags & THEME_FLAG_BG_SCROLLABLE));
	config_set_bool(tprop, "bg_scaled", (rec->flags & THEME_FLAG_BG_SCALED));
	config_set_bool(tprop, "bg_shaded", (rec->flags & THEME_FLAG_BG_SHADED));
	if (!readonly) PLSave(tprop, TRUE);
	PLRelease(tprop);
    }

    return TRUE;
}

static gboolean sig_setup_destroyed(void)
{
    setup_dialog = NULL;
    return TRUE;
}

static void sig_fonsel_ok(GtkFontSelectionDialog *fontsel)
{
    GtkEntry *entry;

    g_return_if_fail(fontsel != NULL);

    if (current_theme->font != NULL) g_free(current_theme->font);
    current_theme->font = g_strdup(gtk_font_selection_dialog_get_font_name(fontsel));

    entry = gtk_object_get_data(GTK_OBJECT(fontsel), "entry");
    gtk_widget_destroy(GTK_WIDGET(fontsel));

    gtk_entry_set_text(GTK_ENTRY(entry), current_theme->font);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_fontsel_destroyed(GtkWidget *widget, GtkWidget **dialog)
{
    *dialog = NULL;
}

static void sig_fontsel_clicked(GtkWidget *button, GtkEntry *entry)
{
    static GtkWidget *fontsel = NULL;

    g_return_if_fail(button != NULL);

    if (fontsel != NULL)
    {
	gdk_window_raise(fontsel->window);
	return;
    }

    fontsel = gtk_font_selection_dialog_new(_("Select new default font"));
    gui_widget_depends(setup_dialog, fontsel);
    gtk_object_set_data(GTK_OBJECT(fontsel), "entry", entry);

    if (current_theme->font != NULL)
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fontsel), current_theme->font);

    gtk_signal_connect(GTK_OBJECT(fontsel), "destroy",
		       GTK_SIGNAL_FUNC(sig_fontsel_destroyed), &fontsel);
    gtk_signal_connect_object(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG (fontsel)->ok_button), "clicked",
                              GTK_SIGNAL_FUNC(sig_fonsel_ok), GTK_OBJECT(fontsel));
    gtk_signal_connect_object(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG (fontsel)->cancel_button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fontsel));

    gtk_widget_show(fontsel);
}

static void sig_change_page(GtkNotebook *notebook, GtkNotebookPage *page, gint pagenum)
{
    if (setup_dialog != NULL && pagenum == 1)
	save_changes();
}

static void sig_scrollable_clicked(GtkToggleButton *button)
{
    if (gtk_toggle_button_get_active(button))
	temp_current_theme->flags |= THEME_FLAG_BG_SCROLLABLE;
    else
	temp_current_theme->flags &= ~THEME_FLAG_BG_SCROLLABLE;
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_scaled_clicked(GtkToggleButton *button)
{
    if (gtk_toggle_button_get_active(button))
	temp_current_theme->flags |= THEME_FLAG_BG_SCALED;
    else
	temp_current_theme->flags &= ~THEME_FLAG_BG_SCALED;
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_shaded_clicked(GtkToggleButton *button)
{
    if (gtk_toggle_button_get_active(button))
	temp_current_theme->flags |= THEME_FLAG_BG_SHADED;
    else
	temp_current_theme->flags &= ~THEME_FLAG_BG_SHADED;
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static gboolean setup_theme_redraw(THEME_REC *theme, gpointer init)
{
    GtkEntry *fontentry, *bgentry;
    GtkToggleButton *button;

    if (theme == temp_current_theme && init == NULL)
	return TRUE;

    fontentry = gtk_object_get_data(GTK_OBJECT(setup_dialog), "fontentry");
    bgentry = gtk_object_get_data(GTK_OBJECT(setup_dialog), "bgentry");

    gtk_entry_set_text(fontentry, theme->font == NULL ? "" : theme->font);
    gtk_entry_set_text(bgentry, theme->bg_pixmap == NULL ? "" : theme->bg_pixmap);

    /* scrollable button */
    button = gtk_object_get_data(GTK_OBJECT(setup_dialog), "bg_scrollable");
    gtk_toggle_button_set_active(button, (theme->flags & THEME_FLAG_BG_SCROLLABLE));
    if (init)
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_scrollable_clicked), NULL);

    /* scaled */
    button = gtk_object_get_data(GTK_OBJECT(setup_dialog), "bg_scaled");
    gtk_toggle_button_set_active(button, (theme->flags & THEME_FLAG_BG_SCALED));
    if (init)
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_scaled_clicked), NULL);

    /* shaded */
    button = gtk_object_get_data(GTK_OBJECT(setup_dialog), "bg_shaded");
    gtk_toggle_button_set_active(button, (theme->flags & THEME_FLAG_BG_SHADED));
    if (init)
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_shaded_clicked), NULL);

    if (init)
    {
	gtk_signal_connect_object(GTK_OBJECT(fontentry), "changed",
				  GTK_SIGNAL_FUNC(gnome_property_box_changed), GTK_OBJECT(setup_dialog));
	gtk_signal_connect_object(GTK_OBJECT(bgentry), "changed",
				  GTK_SIGNAL_FUNC(gnome_property_box_changed), GTK_OBJECT(setup_dialog));
    }

    return TRUE;
}

static gboolean setup_bg(GtkWidget *dialog, GtkWidget *notebook)
{
    GtkWidget *mainwidget, *hbox, *sep;
    GtkWidget *label, *entry, *button, *checkbox;

    setup_dialog = dialog;

    mainwidget = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(mainwidget), 10);

    /* background */
    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(mainwidget), hbox, FALSE, FALSE, 0);

    label = gtk_label_new(_("Background pixmap"));
    gtk_misc_set_alignment(GTK_MISC(label), 0, .5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gnome_file_entry_new("BACKGROUND", _("Background pixmap"));
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_object_set_data(GTK_OBJECT(dialog), "bgentry", gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry)));

    checkbox = gtk_check_button_new_with_label(_("Scrollable pixmap"));
    gtk_object_set_data(GTK_OBJECT(dialog), "bg_scrollable", checkbox);
    gtk_box_pack_start(GTK_BOX(mainwidget), checkbox, FALSE, FALSE, 0);

    checkbox = gtk_check_button_new_with_label(_("Scaled pixmap"));
    gtk_object_set_data(GTK_OBJECT(dialog), "bg_scaled", checkbox);
    gtk_box_pack_start(GTK_BOX(mainwidget), checkbox, FALSE, FALSE, 0);

    checkbox = gtk_check_button_new_with_label(_("Shade pixmap"));
    gtk_object_set_data(GTK_OBJECT(dialog), "bg_shaded", checkbox);
    gtk_box_pack_start(GTK_BOX(mainwidget), checkbox, FALSE, FALSE, 0);

    /* font */
    sep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwidget), sep, FALSE, FALSE, 5);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(mainwidget), hbox, FALSE, FALSE, 0);

    label = gtk_label_new(_("Font"));
    gtk_misc_set_alignment(GTK_MISC(label), 0, .5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gtk_entry_new();
    gtk_object_set_data(GTK_OBJECT(dialog), "fontentry", entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    button = gtk_button_new_with_label(_("Change..."));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_fontsel_clicked), entry);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);


    gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		       GTK_SIGNAL_FUNC(sig_change_page), NULL);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
			     mainwidget, gtk_label_new(_("Background & Fonts")));
    return TRUE;
}

void init_bg(void)
{
    signal_add("setup themes apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_add("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply_again);
    signal_add("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup themes show", (SIGNAL_FUNC) setup_bg);
    signal_add("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}

void deinit_bg(void)
{
    signal_remove("setup themes apply", (SIGNAL_FUNC) sig_setup_apply);
    signal_remove("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply_again);
    signal_remove("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup themes show", (SIGNAL_FUNC) setup_bg);
    signal_remove("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}
