/*
 setup-colors.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../ui-common/txt.h"

#include "setup-int.h"
#include "setup-themes.h"

static gchar *color_names[COLORS] =
{
    N_("Black"),
    N_("Blue"),
    N_("Green"),
    N_("Cyan"),
    N_("Red"),
    N_("Magenta"),
    N_("Brown"),
    N_("Grey"),
    N_("Dark Grey"),
    N_("Light Blue"),
    N_("Light Green"),
    N_("Light Cyan"),
    N_("Light Red"),
    N_("Light Magenta"),
    N_("Yellow"),
    N_("White"),
    N_("Orange"),

    N_("Background color"),
    N_("New data in channel"),
    N_("New msg in channel"),
    N_("New msg for you in channel"),
    N_("Default nick color in nicklist"),
    N_("Gone nick in nicklist"),
    N_("Nicklist background"),
};

/* default colors */
static GdkColor default_colors[COLORS] =
{
    { 0, 0, 0, 0 }, /* black */
    { 0, 0, 0, 0xcccc }, /* blue */
    { 0, 0, 0xcccc, 0 }, /* green */
    { 0, 0, 0xbbbb, 0xbbbb }, /* cyan */
    { 0, 0xcccc, 0, 0 }, /* red */
    { 0, 0xbbbb, 0, 0xbbbb }, /* magenta */
    { 0, 0xbbbb, 0xbbbb, 0 }, /* brown */
    { 0, 0xaaaa, 0xaaaa, 0xaaaa }, /* grey */
    { 0, 0x7777, 0x7777, 0x7777 }, /* dark grey */
    { 0, 0, 0, 0xffff }, /* light blue */
    { 0, 0, 0xffff, 0 }, /* light green */
    { 0, 0, 0xffff, 0xffff }, /* light cyan */
    { 0, 0xffff, 0, 0 }, /* light red */
    { 0, 0xffff, 0, 0xffff }, /* light magenta */
    { 0, 0xffff, 0xffff, 0 }, /* yellow */
    { 0, 0xffff, 0xffff, 0xffff }, /* white */
    { 0, 0xffff, 0xaaaa, 0 }, /* orange */

    { 0, 0, 0, 0 },
    { 0, 0xbfff, 0, 0 },
    { 0, 0xffff, 0, 0 },
    { 0, 0xffff, 0, 0xffff },
    { 0, 0, 0, 0 },
    { 0, 0x7fff, 0x7fff, 0x7fff },
    { 0, 0xffff, 0xffff, 0xffff },
};

static GtkWidget *setup_dialog;
static GtkWidget *mainwidget;

static void draw_color(GtkCList *clist, gint color, gboolean remove, GUI_THEME_REC *theme)
{
    GtkStyle *style;
    gchar *cells[5];
    gint row;

    cells[0] = g_strdup(dgettext(PACKAGE, color_names[color]));
    cells[1] = g_strdup_printf("%04X", theme->colors[color].red);
    cells[2] = g_strdup_printf("%04X", theme->colors[color].green);
    cells[3] = g_strdup_printf("%04X", theme->colors[color].blue);
    cells[4] = _("sample");

    row = color >= BASIC_COLORS ? color-BASIC_COLORS : color+EXTRA_COLORS;
    if (remove)
    {
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, row);
    }
    row = gtk_clist_insert(clist, row, cells);
    if (remove) gtk_clist_thaw(clist);

    g_free(cells[0]); g_free(cells[1]);
    g_free(cells[2]); g_free(cells[3]);

    gtk_clist_set_row_data(clist, row, GINT_TO_POINTER(color));

    /* set "sample" text style */
    style = gtk_style_copy(GTK_WIDGET(clist)->style);
    gtk_style_ref(style);
    style->fg[0] = theme->colors[color];
    if (color < BASIC_COLORS) style->base[0] = theme->colors[COLOR_BACKGROUND];
    gtk_clist_set_cell_style(clist, row, 4, style);
    gtk_style_unref(style);
}

static void set_color(GtkWidget *button, gint color, THEME_REC *theme)
{
    GtkStyle *style;

    g_return_if_fail(button != NULL);

    style = gtk_style_copy(button->style);
    gtk_style_ref(style);
    style->bg[0] = THEME_GUI(theme)->colors[color];
    gtk_widget_set_style(button, style);
    gtk_style_unref(style);

    gtk_object_set_data(GTK_OBJECT(button), "color", GINT_TO_POINTER(color));
}

static void sig_color_selected(GtkWidget *button, GtkWidget *widget)
{
    GtkCList *clist;
    GdkColor color;
    gdouble colors[3];
    gint colornum;

    g_return_if_fail(button != NULL);
    g_return_if_fail(widget != NULL);

    colornum = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), "color"));
    clist = gtk_object_get_data(GTK_OBJECT(setup_dialog), "colorclist");

    gtk_color_selection_get_color(GTK_COLOR_SELECTION(gtk_object_get_data(GTK_OBJECT(button), "colorsel")), colors);
    color.red = (guint16) (colors[0]*65535.0);
    color.green = (guint16) (colors[1]*65535.0);
    color.blue = (guint16) (colors[2]*65535.0);

    gdk_color_alloc(gdk_window_get_colormap(widget->window), &color);
    THEME_GUI(temp_current_theme)->colors[colornum] = color;

    set_color(widget, colornum, temp_current_theme);
    if (clist != NULL) draw_color(clist, colornum, TRUE, THEME_GUI(temp_current_theme));
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));

    gtk_widget_destroy(gtk_object_get_data(GTK_OBJECT(button), "colorseldlg"));
}

static gint sig_colorsel_click(GtkWidget *widget, GdkEventButton *event, GtkCList *clist)
{
    GtkWidget *colorseldlg, *colorsel, *button;
    gdouble colors[3];
    gint color;

    g_return_val_if_fail(widget != NULL, 0);

    color = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    colors[0] = THEME_GUI(temp_current_theme)->colors[color].red/65535.0;
    colors[1] = THEME_GUI(temp_current_theme)->colors[color].green/65535.0;
    colors[2] = THEME_GUI(temp_current_theme)->colors[color].blue/65535.0;

    colorseldlg = gtk_dialog_new();
    gui_widget_depends(setup_dialog, colorseldlg);
    gtk_signal_connect(GTK_OBJECT(colorseldlg), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    colorsel = gtk_color_selection_new();
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(colorseldlg)->vbox), colorsel, TRUE, TRUE, 0);

    button = gtk_button_new_with_label(_("Ok"));
    gtk_object_set_data(GTK_OBJECT(button), "colorsel", colorsel);
    gtk_object_set_data(GTK_OBJECT(button), "colorseldlg", colorseldlg);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(colorseldlg)->action_area), button, TRUE, TRUE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_color_selected), widget);

    button = gtk_button_new_with_label (_("Cancel"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(colorseldlg)->action_area), button, TRUE, TRUE, 10);
    gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(colorseldlg));

    gtk_color_selection_set_color(GTK_COLOR_SELECTION(colorsel), colors);

    gtk_widget_show_all(colorseldlg);
    return 1;
}

static gboolean sig_setup_apply_again(GtkWidget *dialog)
{
    proplist_t fname, tprop, colprop;
    GdkColor *def;
    GString *str;
    GList *tmp;
    gint num;

    g_return_val_if_fail(dialog != NULL, FALSE);

    str = g_string_new(NULL);
    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
	THEME_REC *rec = tmp->data;

	/* try to open theme.. */
	tprop = PLGetProplistWithPath(rec->path);
	if (tprop == NULL)
	{
	    /* not found, create new theme file */
	    tprop = PLMakeDictionaryFromEntries(NULL, NULL);
	    fname = PLMakeString(rec->path);
	    tprop = PLSetFilename(tprop, fname);
	    PLRelease(fname);
	}

	config_set_int(tprop, "default_color", rec->default_color);
	if (rec->font != NULL)
	    config_set_str(tprop, "font", rec->font);
	else
	    config_clean_key(tprop, "font");
	if (rec->bg_pixmap != NULL)
	    config_set_str(tprop, "bg_pixmap", rec->bg_pixmap);
	else
	    config_clean_key(tprop, "bg_pixmap");

	/* write color list */
	tprop = config_make_dict(tprop, "colors");
	colprop = config_get_prop(tprop, "colors");
	for (num = 0; num < COLORS; num++)
	{
	    def = &default_colors[num];

	    if (THEME_GUI(rec)->colors[num].red == def->red &&
		THEME_GUI(rec)->colors[num].green == def->green &&
		THEME_GUI(rec)->colors[num].blue == def->blue)
	    {
		/* same as default, don't bother saving */
		continue;
	    }

	    /* red */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "red_%d", num);
	    else
		g_string_sprintf(str, "extra_red_%d", num-BASIC_COLORS);
	    config_set_int(colprop, str->str, THEME_GUI(rec)->colors[num].red);

	    /* green */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "green_%d", num);
	    else
		g_string_sprintf(str, "extra_green_%d", num-BASIC_COLORS);
	    config_set_int(colprop, str->str, THEME_GUI(rec)->colors[num].green);

	    /* blue */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "blue_%d", num);
	    else
		g_string_sprintf(str, "extra_blue_%d", num-BASIC_COLORS);
	    config_set_int(colprop, str->str, THEME_GUI(rec)->colors[num].blue);
	}
	if (!readonly) PLSave(tprop, TRUE);
	PLRelease(tprop);
    }
    g_string_free(str, TRUE);

    return TRUE;
}

static gboolean sig_setup_destroyed(void)
{
    setup_dialog = NULL;
    return TRUE;
}

/* allocate colors */
static void init_setup_colors(GtkWidget *window)
{
    gint n;

    g_return_if_fail(window != NULL);

    for (n = 0; n < sizeof(THEME_GUI(temp_current_theme)->colors)/sizeof(THEME_GUI(temp_current_theme)->colors[0]); n++)
    {
        THEME_GUI(temp_current_theme)->colors[n].pixel =
            (gulong)((THEME_GUI(temp_current_theme)->colors[n].red & 0xff00)*256 +
                     (THEME_GUI(temp_current_theme)->colors[n].green & 0xff00) +
                     (THEME_GUI(temp_current_theme)->colors[n].blue & 0xff00)/256);
        gdk_color_alloc(gtk_widget_get_colormap(window), &THEME_GUI(temp_current_theme)->colors[n]);
    }
}

static void sig_select_row(GtkCList *clist, gint row, gint column, GdkEvent *event, GtkWidget *eventbox)
{
    gint color;

    g_return_if_fail(clist != NULL);

    color = GPOINTER_TO_INT(gtk_clist_get_row_data(clist, row));
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(color));
    set_color(eventbox, color, temp_current_theme);
}

static void sig_defcolor_changed(GtkSpinButton *spin)
{
    temp_current_theme->default_color =
	gtk_spin_button_get_value_as_int(spin);
}

static gboolean setup_colors_show(GtkWidget *dialog, GtkWidget *notebook)
{
    GtkWidget *table, *hbox, *frame, *eventbox, *scrollbox;
    GtkWidget *clist, *separator, *spin, *label;
    gchar *titles[5], *titles_buggy[5];
    GtkAdjustment *adj;

    g_return_val_if_fail(dialog != NULL, FALSE);

    titles[0] = _("Color"); titles[1] = _("Red"); titles[2] = _("Green");
    titles[3] = _("Blue"); titles[4] = _("Sample text");

    titles_buggy[0] = _("Color                                       ");
    titles_buggy[1] = _("Red  "); titles_buggy[2] = _("Green"); titles_buggy[3] = _("Blue "); titles_buggy[4] = _("Sample text");

    setup_dialog = dialog;

    init_setup_colors(dialog);

    mainwidget = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(mainwidget), 10);

    /* color selection */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(mainwidget), scrollbox, TRUE, TRUE, 5);

    clist = gtk_clist_new_with_titles(5, setup_get_bool("toggle_buggy_gtkthemes") ? titles_buggy : titles);
    gtk_object_set_data(GTK_OBJECT(dialog), "colorclist", clist);
    gtk_widget_set_usize(clist, 400, 100);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_clist_column_titles_passive(GTK_CLIST(clist));
    if (!setup_get_bool("toggle_buggy_gtkthemes"))
    {
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    }
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);

    /* color selector */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(mainwidget), hbox, FALSE, FALSE, 5);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);

    eventbox = gtk_event_box_new();
    gtk_object_set_data(GTK_OBJECT(dialog), "eventbox", eventbox);
    gtk_widget_set_usize(eventbox, 30, 30);
    gtk_container_add(GTK_CONTAINER(frame), eventbox);

    label = gtk_label_new(_("Select the color you want to change from upper list\nand change it from button in left"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 5);

    gtk_signal_connect(GTK_OBJECT(eventbox), "button_press_event",
                       GTK_SIGNAL_FUNC(sig_colorsel_click), clist);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), eventbox);

    /* default color */
    separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwidget), separator, FALSE, FALSE, 5);

    table = gtk_table_new(2, 2, FALSE);
    gtk_box_pack_start(GTK_BOX(mainwidget), table, FALSE, FALSE, 0);

    label = gtk_label_new(_("Default color number"));
    gtk_misc_set_alignment(GTK_MISC(label), 0, .5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, 0, 5, 0);

    hbox = gtk_hbox_new(FALSE, 7);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 0, 1);

    adj = (GtkAdjustment *) gtk_adjustment_new(7, -1, 255, 1, 5, 0);
    spin = gtk_spin_button_new(adj, 0, 0);
    gtk_object_set_data(GTK_OBJECT(setup_dialog), "defaultcolor_spin", spin);
    gtk_signal_connect(GTK_OBJECT(spin), "changed",
                       GTK_SIGNAL_FUNC(sig_defcolor_changed), NULL);
    gtk_widget_set_usize(spin, 50, -1);
    gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 0);


    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
			     mainwidget, gtk_label_new(_("Colors")));
    return TRUE;
}

static gboolean setup_theme_redraw(THEME_REC *theme, gpointer init)
{
    GtkSpinButton *spin;
    GtkCList *clist;
    gint n;

    if (theme == temp_current_theme && init == NULL)
	return TRUE;

    clist = gtk_object_get_data(GTK_OBJECT(setup_dialog), "colorclist");
    spin = gtk_object_get_data(GTK_OBJECT(setup_dialog), "defaultcolor_spin");
    gtk_spin_button_set_value(spin, theme->default_color);

    if (init)
    {
	gtk_signal_connect_object(GTK_OBJECT(spin), "changed",
				  GTK_SIGNAL_FUNC(gnome_property_box_changed), GTK_OBJECT(setup_dialog));
    }

    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);
    n = BASIC_COLORS; /* start from special colors */
    do
    {
	draw_color(clist, n, FALSE, THEME_GUI(theme));
	if (++n == COLORS) n = 0;
    }
    while (n != BASIC_COLORS);
    gtk_clist_thaw(clist);
    gtk_clist_select_row(GTK_CLIST(clist), 0, 0);
    return TRUE;
}

void init_colors(void)
{
    proplist_t tprop, colprop;
    GUI_THEME_REC *gui;
    GString *str;
    GList *tmp;
    gint num;
    GdkColor colors[COLORS];

    setup_dialog = NULL;

    str = g_string_new(NULL);

    memset(colors, 0, sizeof(colors));
    memcpy(colors, default_colors, sizeof(default_colors));

    /* Read [colors] from themes */
    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
	THEME_REC *rec = tmp->data;

	gui = g_new0(GUI_THEME_REC, 1);
	rec->gui_data = gui;

	if (rec->path == NULL)
	{
	    memcpy(gui->colors, colors, sizeof(colors));
	    continue;
	}

	tprop = PLGetProplistWithPath(rec->path);
	colprop = config_get_prop(tprop, "colors");
	for (num = 0; num < COLORS; num++)
	{
	    /* red */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "red_%d", num);
	    else
		g_string_sprintf(str, "extra_red_%d", num-BASIC_COLORS);
	    gui->colors[num].red = config_get_int(colprop, str->str, colors[num].red);

	    /* green */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "green_%d", num);
	    else
		g_string_sprintf(str, "extra_green_%d", num-BASIC_COLORS);
	    gui->colors[num].green = config_get_int(colprop, str->str, colors[num].green);

	    /* blue */
	    if (num < BASIC_COLORS)
		g_string_sprintf(str, "blue_%d", num);
	    else
		g_string_sprintf(str, "extra_blue_%d", num-BASIC_COLORS);
	    gui->colors[num].blue = config_get_int(colprop, str->str, colors[num].blue);
	}
    }

    g_string_free(str, TRUE);

    signal_add("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply_again);
    signal_add("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup themes show", (SIGNAL_FUNC) setup_colors_show);
    signal_add("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}

void deinit_colors(void)
{
    signal_remove("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply_again);
    signal_remove("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup themes show", (SIGNAL_FUNC) setup_colors_show);
    signal_remove("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}
