/*
 gui-readline.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint readtag, sigint_count = 0;

static void window_prev_page(void)
{
    gui_window_scroll(CHANNEL_PARENT(cur_channel), -(last_text_line-first_text_line)/2);
}

static void window_next_page(void)
{
    gui_window_scroll(CHANNEL_PARENT(cur_channel), (last_text_line-first_text_line)/2);
}

void handle_key(gint key)
{
    gchar *text, *str, c;

    /* Quit if we get 5 CTRL-C's in a row. */
    if (key != 3)
	sigint_count = 0;
    else if (++sigint_count >= 5)
	kill(getpid(), SIGTERM);

    switch (key)
    {
	case 27:
	    c = getch();
	    if (c == toupper(c) && c != tolower(c))
		str = g_strdup_printf("ALT-SHIFT-%c", c);
	    else
		str = g_strdup_printf("ALT-%c", toupper(c));
            key_pressed(str, NULL);
	    g_free(str);
            break;

	case KEY_HOME:
            /* home */
            gui_entry_set_pos(0);
            gui_entry_move_pos(0);
            break;
        case KEY_END:
            /* end */
            gui_entry_set_pos(strlen(gui_entry_get_text()));
            gui_entry_move_pos(0);
            break;
        case KEY_PPAGE:
            /* page up */
            window_prev_page();
            break;
        case KEY_NPAGE:
            /* page down */
            window_next_page();
            break;

        case KEY_UP:
            /* up */
            text = ui_history_prev(CHANNEL_PARENT(cur_channel), gui_entry_get_text());
            gui_entry_set_text(text);
            break;
        case KEY_DOWN:
            /* down */
            text = ui_history_next(CHANNEL_PARENT(cur_channel), gui_entry_get_text());
            gui_entry_set_text(text);
            break;
        case KEY_RIGHT:
            /* right */
            gui_entry_move_pos(1);
            break;
        case KEY_LEFT:
            /* left */
            gui_entry_move_pos(-1);
            break;

	case 21:
            /* Ctrl-U, clear line */
            gui_entry_set_text("");
	    break;

	case 9:
	    key_pressed("Tab", NULL);
	    break;

	case 8:
	case 127:
        case KEY_BACKSPACE:
            gui_entry_erase(1);
            break;

	case 0:
	    /* Ctrl-space - ignore */
	    break;
        case 1:
            /* C-A, home */
            gui_entry_set_pos(0);
            gui_entry_move_pos(0);
            break;
        case 5:
            /* C-E, end */
            gui_entry_set_pos(strlen(gui_entry_get_text()));
            gui_entry_move_pos(0);
            break;

        case '\n':
	case 13:
	    key_pressed("Return", NULL);

	    str = gui_entry_get_text();
	    if (*str == '\0') break;

	    ui_translate_output(str);
	    signal_emit("send command", 3, str, cur_channel->server, cur_channel);

            ui_history_add(CHANNEL_PARENT(cur_channel), gui_entry_get_text(), FALSE);
            gui_entry_set_text("");
            ui_history_clear_pos(CHANNEL_PARENT(cur_channel));
            break;

	default:
	    if (key > 0 && key < 32)
	    {
		str = g_strdup_printf("CTRL-%c", key == 31 ? '-' : key+'A'-1);
		key_pressed(str, NULL);
		g_free(str);
		break;
	    }

	    if (key < 256)
	    {
		gchar str[2];

		str[0] = toupper(key); str[1] = '\0';
		key_pressed(str, NULL);
		gui_entry_insert_char((gchar) key);
	    }
            break;
    }
}

void readline(void)
{
    gint key;

    for (;;)
    {
        key = getch();
        if (key == ERR) break;

        handle_key(key);
    }
}

static gboolean sig_prev_page(void)
{
    window_prev_page();
    return TRUE;
}

static gboolean sig_next_page(void)
{
    window_next_page();
    return TRUE;
}

static gboolean sig_change_window(gchar *data)
{
    signal_emit("command window goto", 3, data, cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_completion(void)
{
    gchar *line;
    gint pos;

    pos = gui_entry_get_pos();

    line = ui_completion_line(cur_channel, gui_entry_get_text(), &pos);
    if (line != NULL)
    {
	gui_entry_set_text(line);
	gui_entry_set_pos(pos);
	g_free(line);
    }

    return TRUE;
}

static gboolean sig_replace(void)
{
    gchar *line;
    gint pos;

    pos = gui_entry_get_pos();

    line = ui_auto_completion(gui_entry_get_text(), &pos);
    if (line != NULL)
    {
	gui_entry_set_text(line);
	gui_entry_set_pos(pos);
	g_free(line);
    }

    return TRUE;
}

static gboolean sig_prev_window(void)
{
    signal_emit("command window prev", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_next_window(void)
{
    signal_emit("command window next", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_window_goto_active(void)
{
    signal_emit("command window goto", 3, "active", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_prev_channel(void)
{
    signal_emit("command channel prev", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_next_channel(void)
{
    signal_emit("command channel next", 3, "", cur_channel->server, cur_channel);
    return TRUE;
}

static gboolean sig_addchar(gchar *data)
{
    gui_entry_insert_char(*data);
    return TRUE;
}

void gui_readline_init(void)
{
    static gchar changekeys[] = "1234567890QWERTYUIO";
    gchar *key, *data;
    gint n;

    readtag = gui_input_add(0, GUI_INPUT_READ, (GUIInputFunction) readline, NULL);

    key_bind("completion", NULL, "Nick completion", "Tab", (SIGNAL_FUNC) sig_completion);
    key_bind("check replaces", NULL, "Check word replaces", " ", (SIGNAL_FUNC) sig_replace);
    key_bind("check replaces", NULL, NULL, "Return", (SIGNAL_FUNC) sig_replace);
    key_bind("window prev", NULL, "Previous window", "CTRL-P", (SIGNAL_FUNC) sig_prev_window);
    key_bind("window next", NULL, "Next window", "CTRL-N", (SIGNAL_FUNC) sig_next_window);
    key_bind("window active", NULL, "Go to next window with the highest activity", "ALT-A", (SIGNAL_FUNC) sig_window_goto_active);
    key_bind("channel next", NULL, "Next channel", "CTRL-X", (SIGNAL_FUNC) sig_next_channel);
    key_bind("channel prev", NULL, "Next channel", NULL, (SIGNAL_FUNC) sig_prev_channel);

    key_bind("redraw", NULL, "Redraw window", "CTRL-L", (SIGNAL_FUNC) irssi_redraw);
    key_bind("prev page", NULL, "Previous page", "ALT-P", (SIGNAL_FUNC) sig_prev_page);
    key_bind("next page", NULL, "Next page", "ALT-N", (SIGNAL_FUNC) sig_next_page);

    key_bind("special char", "\x02", "Insert special character", "CTRL-B", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x1f", NULL, "CTRL--", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x03", NULL, "CTRL-C", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x16", NULL, "CTRL-V", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x07", NULL, "CTRL-G", (SIGNAL_FUNC) sig_addchar);
    key_bind("special char", "\x0f", NULL, "CTRL-O", (SIGNAL_FUNC) sig_addchar);

    for (n = 0; changekeys[n] != '\0'; n++)
    {
        key = g_strdup_printf("ALT-%c", changekeys[n]);
        data = g_strdup_printf("%d", n+1);
	key_bind("change window", data, "Change window", key, (SIGNAL_FUNC) sig_change_window);
	g_free(data); g_free(key);
    }
}

void gui_readline_deinit(void)
{
    key_unbind("completion", (SIGNAL_FUNC) sig_completion);
    key_unbind("check replaces", (SIGNAL_FUNC) sig_replace);
    key_unbind("window prev", (SIGNAL_FUNC) sig_prev_window);
    key_unbind("window next", (SIGNAL_FUNC) sig_next_window);
    key_unbind("window active", (SIGNAL_FUNC) sig_window_goto_active);
    key_unbind("channel next", (SIGNAL_FUNC) sig_next_channel);
    key_unbind("channel prev", (SIGNAL_FUNC) sig_prev_channel);

    key_unbind("redraw", (SIGNAL_FUNC) irssi_redraw);
    key_unbind("prev page", (SIGNAL_FUNC) sig_prev_page);
    key_unbind("next page", (SIGNAL_FUNC) sig_next_page);

    key_unbind("special char", (SIGNAL_FUNC) sig_addchar);
    key_unbind("change window", (SIGNAL_FUNC) sig_change_window);
    gui_input_remove(readtag);
}
