/*
 channels-query.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*

 How the thing works:

 - After channel is joined and NAMES list is got, send "channel query" signal
 - "channel query" : add channel to server->quries lists

loop:
 - Wait for NAMES list from all channels before doing anything else..
 - After got the last NAMES list, start sending the queries ..
 - find the query to send, check where server->queries list isn't NULL
   (mode, who, banlist, ban exceptions, invite list)
 - if not found anything -> all channels are synced
 - send "command #chan1,#chan2,#chan3,.." command to server
 - wait for reply from server, then check if it was last query to be sent to
   server. If it was, send "channel sync" signal
 - check if the reply was for last channel. If so, goto loop
*/

#include "irssi.h"

static GList *queries;

static gboolean sig_connected(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    server->queries = g_hash_table_new((GHashFunc) g_str_hash, (GCompareFunc) g_str_equal);
    for (tmp = queries; tmp != NULL; tmp = tmp->next)
	g_hash_table_insert(server->queries, tmp->data, NULL);
    return TRUE;
}

static void sig_query_destroy_hash(gpointer key, GList *list)
{
    g_list_free(list);
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    g_hash_table_foreach(server->queries, (GHFunc) sig_query_destroy_hash, NULL);
    g_hash_table_destroy(server->queries);
    if (server->last_query_chan != NULL) g_free(server->last_query_chan);
    return TRUE;
}

/* Add channel to query list */
static void channel_query_add(CHANNEL_REC *channel, gchar *key)
{
    GList *list;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(key != NULL);

    list = g_hash_table_lookup(channel->server->queries, key);
    list = g_list_append(list, channel);

    /* Need to change the hash table value */
    g_hash_table_remove(channel->server->queries, key);
    g_hash_table_insert(channel->server->queries, key, list);
}

/* Remove channel from query list */
static void channel_query_remove(CHANNEL_REC *channel, gchar *key)
{
    GList *list;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(key != NULL);

    list = g_hash_table_lookup(channel->server->queries, key);
    if (g_list_find(list, channel) == NULL)
	return;

    list = g_list_remove(list, channel);

    /* Need to change the hash table value */
    g_hash_table_remove(channel->server->queries, key);
    g_hash_table_insert(channel->server->queries, key, list);
}

static gboolean sig_channel_destroyed(CHANNEL_REC *channel)
{
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    if (channel->server != NULL && channel->type == CHANNEL_TYPE_CHANNEL)
    {
	/* remove channel from query lists */
	for (tmp = queries; tmp != NULL; tmp = tmp->next)
	    channel_query_remove(channel, tmp->data);
    }
    return TRUE;
}

static void channels_query_check(SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    GList *tmp, *chans, *qchans;
    GString *str;
    gchar *key, *cmd;
    gboolean onlyone;

    g_return_if_fail(server != NULL);

    if (server->last_query_chan != NULL)
    {
	g_free(server->last_query_chan);
	server->last_query_chan = NULL;
    }

    /* check if we've received /NAMES list from all joined channels */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (rec->server == server && !rec->names_got && rec->type == CHANNEL_TYPE_CHANNEL)
	{
	    /* nope - wait a bit more.. */
	    return;
	}
    }

    /* find what to ask */
    key = NULL; chans = NULL;
    for (tmp = queries; tmp != NULL; tmp = tmp->next)
    {
	chans = g_hash_table_lookup(server->queries, tmp->data);
	if (chans != NULL)
	{
	    key = tmp->data;
	    break;
	}
    }

    if (chans == NULL)
    {
	/* no queries.. */
	return;
    }

    str = g_string_new(NULL);
    if ((server->no_multi_mode && strstr(key, "mode") != NULL) ||
	(server->no_multi_who && strcmp(key, "who") == 0))
    {
        onlyone = TRUE;
        chanrec = chans->data;
        qchans = g_list_append(NULL, chanrec);

        g_string_assign(str, chanrec->name);
    }
    else
    {
        /* Send the request of all channels. */
        for (tmp = chans; tmp != NULL; tmp = tmp->next)
        {
            chanrec = tmp->data;

            if (tmp->next == NULL)
                g_string_append(str, chanrec->name);
            else
                g_string_sprintfa(str, "%s,", chanrec->name);
        }
        onlyone = FALSE;
        qchans = chans;
    }

    if (strcmp(key, "mode") == 0)
    {
	gchar *elist;

	cmd = g_strdup_printf("MODE %s", str->str);
	for (tmp = qchans; tmp != NULL; tmp = tmp->next)
	{
	    chanrec = tmp->data;

	    elist = g_strdup_printf("%s %s", chanrec->name, str->str);
	    server_redirect_event(server, elist, 3,
				  "event 403", "chanquery mode abort", 1,
				  "event 442", "chanquery mode abort", 1, /* "you're not on that channel" */
				  "event 324", "chanquery mode", 1, NULL);
	    g_free(elist);
	}
    }
    else if (strcmp(key, "who") == 0)
    {
	gchar *elist;

	cmd = g_strdup_printf("WHO %s", str->str);

	for (tmp = qchans; tmp != NULL; tmp = tmp->next)
	{
	    chanrec = tmp->data;

	    elist = g_strdup_printf("%s %s", chanrec->name, str->str);
	    server_redirect_event(server, elist, 2,
                                  "event 401", "chanquery who abort", 1,
				  "event 315", "chanquery who end", 1,
				  "event 352", "silent event who", 1, NULL);
	    g_free(elist);
	}
    }
    else if (strcmp(key, "bmode") == 0)
    {
	cmd = g_strdup_printf("MODE %s b", str->str);
	for (tmp = qchans; tmp != NULL; tmp = tmp->next)
	{
	    chanrec = tmp->data;

	    server_redirect_event(server, chanrec->name, 2,
				  "event 403", "chanquery mode abort", 1,
				  "event 368", "chanquery ban end", 1,
				  "event 367", "chanquery ban", 1, NULL);
	}
    }
    else if (strcmp(key, "emode") == 0)
    {
	cmd = g_strdup_printf("MODE %s e", str->str);
	for (tmp = qchans; tmp != NULL; tmp = tmp->next)
	{
	    chanrec = tmp->data;

	    server_redirect_event(server, chanrec->name, 4,
				  "event 403", "chanquery mode abort", 1,
				  "event 472", "chanquery emode unknown", -1,
				  "event 482", "chanquery emode unknown chanop", 1,
				  "event 349", "chanquery eban end", 1,
				  "event 348", "chanquery eban", 1, NULL);
	}
    }
    else if (strcmp(key, "imode") == 0)
    {
	cmd = g_strdup_printf("MODE %s I", str->str);
	for (tmp = qchans; tmp != NULL; tmp = tmp->next)
	{
	    chanrec = tmp->data;

	    server_redirect_event(server, chanrec->name, 4,
				  "event 403", "chanquery mode abort", 1,
				  "event 472", "chanquery emode unknown", -1,
				  "event 482", "chanquery emode unknown chanop", 1,
				  "event 347", "chanquery ilist end", 1,
				  "event 346", "chanquery ilist", 1, NULL);
	}
    }
    else
    {
	g_warning("channels_query_check() : shouldn't happen!! key = '%s'", key);
	g_string_free(str, TRUE);
	return;
    }

    /* chans = new channel list */
    if (!onlyone)
    {
        /* all channels queried, set to NULL */
        chans = NULL;
    }
    else
    {
        /* remove the first channel from list */
        chans = g_list_remove(chans, chans->data);
    }
    g_hash_table_remove(server->queries, key);
    g_hash_table_insert(server->queries, key, chans);

    /* Get the channel of last query */
    chanrec = g_list_last(qchans)->data;
    g_list_free(qchans);

    server->last_query_chan = g_strdup(chanrec->name);

    /* send the command */
    irc_send_cmd(server, cmd);
    g_string_free(str, TRUE);
    g_free(cmd);
}

static gboolean sig_channel_query(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    /* Add channel to query lists */
    if (!channel->no_modes)
	channel_query_add(channel, "mode");
    channel_query_add(channel, "who");
    if (!channel->no_modes)
    {
	channel_query_add(channel, "bmode");
	if (!channel->server->emode_not_known)
	{
	    channel_query_add(channel, "emode");
	    channel_query_add(channel, "imode");
	}
    }

    if (channel->server->last_query_chan == NULL)
	channels_query_check(channel->server);
    return TRUE;
}

static void channel_checksync(CHANNEL_REC *channel)
{
    GList *tmp;

    g_return_if_fail(channel != NULL);

    /* if there's no more queries in queries in buffer, send the sync signal */
    if (channel->synced)
	return; /* already synced */

    for (tmp = queries; tmp != NULL; tmp = tmp->next)
    {
	if (g_list_find(g_hash_table_lookup(channel->server->queries, tmp->data), channel) != NULL)
	    return;
    }

    channel->synced = TRUE;
    signal_emit("channel sync", 1, channel);
}

static void channel_got_query(SERVER_REC *server, CHANNEL_REC *chanrec, gchar *channel)
{
    g_return_if_fail(server != NULL);
    g_return_if_fail(channel != NULL);
    g_return_if_fail(server->last_query_chan != NULL);

    /* check if we need to get another query.. */
    if (g_strcasecmp(server->last_query_chan, channel) == 0)
	channels_query_check(server);

    /* check if channel is synced */
    if (chanrec != NULL)
	channel_checksync(chanrec);
}

static gboolean event_channel_mode(gchar *data, SERVER_REC *server, gchar *nick)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *mode;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3 | PARAM_FLAG_GETREST, NULL, &channel, &mode);
    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
	modes_parse_channel(chanrec, nick, mode);
	if (!chanrec->mode_key && chanrec->key != NULL)
	{
	    /* join was used with key but there's no key set in channel modes.. */
	    g_free(chanrec->key);
	    chanrec->key = NULL;
	}
    }
    channel_got_query(server, chanrec, channel);

    g_free(params);
    return TRUE;
}

static void multi_query_remove(SERVER_REC *server, gchar *event, gchar *data)
{
    GList *queue;

    while ((queue = server_redirect_getqueue(server, event, data)) != NULL)
	server_redirect_remove_next(server, event, queue);
}

static gboolean event_end_of_who(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;
    GList *chans, *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);

    if (strchr(channel, ',') != NULL)
    {
	/* instead of multiple End of WHO replies we get only this one... */
	server->one_endofwho = TRUE;
	multi_query_remove(server, "event 315", data);

	chans = str2list(channel, ',');
	for (tmp = chans; tmp != NULL; tmp = tmp->next)
	{
	    /* Send the wholist event */
	    chanrec = channel_find(server, tmp->data);
	    if (chanrec == NULL)
		continue;

	    /* check that the WHO actually did return something
	       (that it understood #chan1,#chan2,..) */
	    if (tmp == chans)
	    {
		NICK_REC *nick;

		/* check if our own nick has host .. */
		nick = nicklist_find(chanrec, server->nick);
		if (nick->host == NULL)
		{
		    /* nope, send the WHO commands again separately. */
		    server->no_multi_who = TRUE;
		}
	    }

	    if (server->no_multi_who)
	    {
		channel_query_add(chanrec, "who");
		continue;
	    }

	    chanrec->wholist = TRUE;
	    signal_emit("channel wholist", 1, chanrec);

	    /* check if we need can send another query */
	    channel_got_query(server, chanrec, tmp->data);
	}

	if (server->no_multi_who)
	    channels_query_check(server);

	if (chans != NULL)
	{
	    g_free(chans->data);
	    g_list_free(chans);
	}
    }
    else
    {
	chanrec = channel_find(server, channel);
	if (chanrec != NULL)
	{
	    /* End of /WHO query */
	    chanrec->wholist = TRUE;
	    signal_emit("channel wholist", 1, chanrec);
	}
	channel_got_query(server, chanrec, channel);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_banlist(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *ban, *setby, *tims;
    glong secs, tim;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 5, NULL, &channel, &ban, &setby, &tims);
    chanrec = channel_find(server, channel);

    if (sscanf(tims, "%ld", &tim) != 1) tim = (glong) time(NULL);
    secs = (glong) time(NULL)-tim;

    if (chanrec != NULL)
        ban_add(chanrec, ban, setby, (time_t) tim);

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_banlist(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    chanrec = channel_find(server, channel);

    /* End of banlist query */
    channel_got_query(server, chanrec, channel);

    g_free(params);
    return TRUE;
}

static gboolean event_ebanlist(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *ban, *setby, *tims;
    glong secs, tim;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 5, NULL, &channel, &ban, &setby, &tims);
    chanrec = channel_find(server, channel);

    if (sscanf(tims, "%ld", &tim) != 1) tim = (glong) time(NULL);
    secs = (glong) time(NULL)-tim;

    if (chanrec != NULL)
        ban_exception_add(chanrec, ban, setby, (time_t) tim);

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_ebanlist(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    chanrec = channel_find(server, channel);

    /* End of exception banlist query */
    channel_got_query(server, chanrec, channel);

    g_free(params);
    return TRUE;
}

static gboolean event_invite_list(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel, *invite;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, NULL, &channel, &invite);
    chanrec = channel_find(server, channel);

    if (chanrec != NULL)
        invitelist_add(chanrec, invite);

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_invitelist(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    chanrec = channel_find(server, channel);

    /* End of invite list query */
    channel_got_query(server, chanrec, channel);

    g_free(params);
    return TRUE;
}

static void remove_echanmode(SERVER_REC *server)
{
    GList *tmp, *chans;

    server->emode_not_known = TRUE;

    /* Remove e and I mode queries from server's query list */
    chans = g_hash_table_lookup(server->queries, "emode");
    g_list_free(chans);
    g_hash_table_remove(server->queries, "emode");
    g_hash_table_insert(server->queries, "emode", NULL);

    chans = g_hash_table_lookup(server->queries, "imode");
    g_list_free(chans);
    g_hash_table_remove(server->queries, "imode");
    g_hash_table_insert(server->queries, "imode", NULL);

    /* Check for channel syncs */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *chanrec = tmp->data;

	if (chanrec->server != server || chanrec->synced || chanrec->type != CHANNEL_TYPE_CHANNEL)
	    continue;

	/* check if channel is synced.. */
	channel_checksync(chanrec);
    }

    /* go to next query */
    channels_query_check(server);
}

static gboolean event_unknown_mode(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    gchar *params, *mode;

    g_return_val_if_fail(data != NULL, FALSE);
    params = event_get_params(data, 2, NULL, &mode);

    if (*mode == 'e' || *mode == 'I')
    {
	/* Didn't understand channel mode e/I */
	remove_echanmode(server);
    }
    else
    {
	/* Something else, send the normal signal */
	signal_emit("event 472", 4, data, server, nick, address);
    }
    g_free(params);
    return TRUE;
}

static gboolean event_not_chanop(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);
    params = event_get_params(data, 2, NULL, &channel);

    chanrec = channel_find(server, channel);
    g_free(params);

    if (chanrec != NULL)
    {
	/* Didn't understand channel mode e/I */
	remove_echanmode(server);
    }

    return TRUE;
}

static gboolean event_no_such_channel(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    params = event_get_params(data, 2, NULL, &channel);

    chanrec = channel_find(server, channel);
    g_free(params);

    if (chanrec != NULL)
    {
	/* channel not found - probably created a new channel
	   and left it immediately. Remove channel from queries */
	for (tmp = queries; tmp != NULL; tmp = tmp->next)
	    channel_query_remove(chanrec, tmp->data);

    }

    if (strchr(channel, ',') != NULL)
    {
        /* server doesn't understand MODE #chan1,#chan2,...  */
	multi_query_remove(server, "event 324", data);

        for (tmp = channels; tmp != NULL; tmp = tmp->next)
        {
            chanrec = tmp->data;

            if (chanrec->server == server && chanrec->type == CHANNEL_TYPE_CHANNEL)
                channel_query_add(chanrec, "mode");
        }

        server->no_multi_mode = TRUE;
        channels_query_check(server);
        return TRUE;
    }

    channel_got_query(server, chanrec, channel);
    return TRUE;
}

static gboolean event_who_no_such_channel(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    params = event_get_params(data, 2, NULL, &channel);

    chanrec = channel_find(server, channel);
    g_free(params);

    if (chanrec != NULL)
    {
	/* channel not found - probably created a new channel
	   and left it immediately. Remove channel from queries */
	for (tmp = queries; tmp != NULL; tmp = tmp->next)
	    channel_query_remove(chanrec, tmp->data);

    }

    if (strchr(channel, ',') != NULL)
    {
        /* server doesn't understand WHO #chan1,#chan2,...  */
	multi_query_remove(server, "event 315", data);

        for (tmp = channels; tmp != NULL; tmp = tmp->next)
        {
            chanrec = tmp->data;

            if (chanrec->server == server && chanrec->type == CHANNEL_TYPE_CHANNEL)
                channel_query_add(chanrec, "who");
        }

        server->no_multi_who = TRUE;
        channels_query_check(server);
        return TRUE;
    }

    channel_got_query(server, chanrec, channel);
    return TRUE;
}

void channels_query_init(void)
{
    queries = g_list_append(NULL, "mode");
    queries = g_list_append(queries, "who");
    queries = g_list_append(queries, "bmode");
    queries = g_list_append(queries, "emode");
    queries = g_list_append(queries, "imode");

    signal_add("server connected", (SIGNAL_FUNC) sig_connected);
    signal_add("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    signal_add("channel query", (SIGNAL_FUNC) sig_channel_query);
    signal_add("channel destroyed", (SIGNAL_FUNC) sig_channel_destroyed);

    signal_add("chanquery mode", (SIGNAL_FUNC) event_channel_mode);
    signal_add("chanquery who end", (SIGNAL_FUNC) event_end_of_who);

    signal_add("chanquery eban", (SIGNAL_FUNC) event_ebanlist);
    signal_add("chanquery eban end", (SIGNAL_FUNC) event_end_of_ebanlist);
    signal_add("chanquery ban", (SIGNAL_FUNC) event_banlist);
    signal_add("chanquery ban end", (SIGNAL_FUNC) event_end_of_banlist);
    signal_add("chanquery ilist", (SIGNAL_FUNC) event_invite_list);
    signal_add("chanquery ilist end", (SIGNAL_FUNC) event_end_of_invitelist);
    signal_add("chanquery emode unknown", (SIGNAL_FUNC) event_unknown_mode);
    signal_add("chanquery emode unknown chanop", (SIGNAL_FUNC) event_not_chanop);
    signal_add("chanquery mode abort", (SIGNAL_FUNC) event_no_such_channel);
    signal_add("chanquery who abort", (SIGNAL_FUNC) event_who_no_such_channel);
}

void channels_query_deinit(void)
{
    g_list_free(queries);
    signal_remove("server connected", (SIGNAL_FUNC) sig_connected);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    signal_remove("channel query", (SIGNAL_FUNC) sig_channel_query);
    signal_remove("channel destroyed", (SIGNAL_FUNC) sig_channel_destroyed);

    signal_remove("chanquery mode", (SIGNAL_FUNC) event_channel_mode);
    signal_remove("chanquery who end", (SIGNAL_FUNC) event_end_of_who);

    signal_remove("chanquery eban", (SIGNAL_FUNC) event_ebanlist);
    signal_remove("chanquery eban end", (SIGNAL_FUNC) event_end_of_ebanlist);
    signal_remove("chanquery ban", (SIGNAL_FUNC) event_banlist);
    signal_remove("chanquery ban end", (SIGNAL_FUNC) event_end_of_banlist);
    signal_remove("chanquery ilist", (SIGNAL_FUNC) event_invite_list);
    signal_remove("chanquery ilist end", (SIGNAL_FUNC) event_end_of_invitelist);
    signal_remove("chanquery emode unknown", (SIGNAL_FUNC) event_unknown_mode);
    signal_remove("chanquery emode unknown chanop", (SIGNAL_FUNC) event_not_chanop);
    signal_remove("chanquery mode abort", (SIGNAL_FUNC) event_no_such_channel);
    signal_remove("chanquery who abort", (SIGNAL_FUNC) event_who_no_such_channel);
}
