/*
 network-nonblock.c : Nonblocking net_connect()

    Copyright (C) 1998-1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../common.h"

#include "network.h"
#include "network-nonblock.h"

static gint childcheck_tag;

typedef struct
{
    NET_CALLBACK func;
    gpointer data;

    gint pipes[2];
    gint port;
    IPADDR *my_ip;
    gint tag;
}
SIMPLE_THREAD_REC;

/* nonblocking gethostbyname(), ip (gulong) + error (gint, -1 = error) is
   written to pipe when found PID of the resolver child is returned */
gint net_gethostname_nonblock(gchar *addr, gint pipe)
{
    gchar *errorstr;
    IPADDR ip;
    gint error, len;
    gint pid;

    g_return_val_if_fail(addr != NULL, FALSE);

    pid = fork();
    if (pid > 0)
    {
        /* parent */
        return pid;
    }

    if (pid != 0)
    {
        /* failed! */
        g_warning("net_connect_thread(): fork() failed! Using blocking resolving");
    }

    /* child */
    error = net_gethostname(addr, &ip);
    write(pipe, &ip, sizeof(ip));
    write(pipe, &error, sizeof(error));
    if (error == -1)
    {
	/* write the error message */
	errorstr = net_gethosterror();
	len = strlen(errorstr)+1;
	write(pipe, &len, sizeof(len));
	write(pipe, errorstr, len);
    }
    if (pid == 0) _exit(99);

    return -1;
}

/* Kill the resolver child */
void net_disconnect_nonblock(gulong pid)
{
    if (pid > 0)
        kill(pid, SIGKILL);
}

static void simple_init(SIMPLE_THREAD_REC *rec, gint handle)
{
    g_return_if_fail(rec != NULL);

    gui_input_remove(rec->tag);

    if (net_geterror(handle) != 0)
    {
        /* failed */
	rec->func(-1, rec->data);
	close(handle);
        g_free(rec);
        return;
    }

    rec->func(handle, rec->data);
    if (rec->my_ip != NULL) g_free(rec->my_ip);
    g_free(rec);
}

static void simple_readpipe(SIMPLE_THREAD_REC *rec, gint pipe)
{
    IPADDR ip;
    gint error;
    gint handle;

    g_return_if_fail(rec != NULL);

    gui_input_remove(rec->tag);

    read(pipe, &ip, sizeof(ip));
    read(pipe, &error, sizeof(error));
    close(rec->pipes[0]);
    close(rec->pipes[1]);

    handle = error == -1 ? -1 : net_connect_ip(&ip, rec->port, rec->my_ip);
    if (handle == -1)
    {
        /* failed */
	rec->func(-1, rec->data);
        if (rec->my_ip != NULL) g_free(rec->my_ip);
	g_free(rec);
        return;
    }

    rec->tag = gui_input_add(handle, GUI_INPUT_WRITE/* | GUI_INPUT_EXCEPTION*/,
                             (GUIInputFunction) simple_init, rec);
}

/* Connect to server, call func when finished */
gboolean net_connect_nonblock(gchar *server, gint port, IPADDR *my_ip, NET_CALLBACK func, gpointer data)
{
    SIMPLE_THREAD_REC *rec;
    gint fd[2];

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(func != NULL, FALSE);

    if (pipe(fd) != 0)
    {
        g_warning("net_connect_thread_simple(): pipe() failed.");
        return FALSE;
    }

    net_gethostname_nonblock(server, fd[1]);

    rec = g_new0(SIMPLE_THREAD_REC, 1);
    rec->port = port;
    if (my_ip != NULL)
    {
	rec->my_ip = g_malloc(sizeof(IPADDR));
	memcpy(rec->my_ip, my_ip, sizeof(IPADDR));
    }
    rec->func = func;
    rec->data = data;
    rec->pipes[0] = fd[0];
    rec->pipes[1] = fd[1];
    rec->tag = gui_input_add(fd[0], GUI_INPUT_READ, (GUIInputFunction) simple_readpipe, rec);
    return TRUE;
}

static gint child_check(void)
{
    gint status;

    while (waitpid(-1, &status, WNOHANG) > 0) ;
    return 1;
}

void network_nonblock_init(void)
{
    childcheck_tag = gui_timeout_add(1000, (GUITimeoutFunction) child_check, NULL);
}

void network_nonblock_deinit(void)
{
    gui_timeout_remove(childcheck_tag);
}
