/****************************************************************************
 *                           DFIOFilter.h
 *
 * Author: Matthew Ballance
 * Desc:   Base class for the DFIO Filter class. Each DFIO Filter will
 *         provide an implementation of this class
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_DFIO_FILTER_INFO_H
#define INCLUDED_DFIO_FILTER_INFO_H
#include <stdio.h>
#include "ivi_String.h"
#include <string.h>
#include <tcl.h>
#include "FileAccObj.h"

class DFIO;
class DFIOFilter {
    public:
        DFIOFilter(
                Tcl_Interp    *interp,
                const char    *inst_name,
                const char    *name, 
                const char    *desc, 
                const char    *suffixes);

        virtual DFIOFilter *newDFIOFilter(void)   { return 0;  }

        /************************************************************
         * isValidFile()
         *
         * Checks to see if the file pointed to by 'fp' is a valid 
         * file of the type handled by this filter
         *
         * Returns:
         * - 1 if the file is of the type handled by this filter...
         * - 0 if the file cannot be handled...
         ************************************************************/
        virtual Int32       isValidFile(FileAccObj  &file) { return 0; }

        /******************************************************************
         * Export()
         *
         * Parameters:
         * - dfio
         *   Handle to the DFIO that contains the data to export
         *
         * - file
         *   Handle to the file that data should be exported to.
         *
         * Returns:
         * - -1
         *   Error occurred while processing file. 
         *
         * - 0
         *   File saved without error...
         ******************************************************************/
        virtual Int32 Export(DFIO **dfio, FileAccObj &file) { return -1; }

        /******************************************************************
         * Import()
         *
         * Parameters:
         * - dfio
         *   Handle to the DFIO to initialize and use
         *
         * - file
         *   Handle to the file that contains the data to import
         *
         * Returns:
         * - -1
         *   Error occurred while processing file. 
         *
         * - 0
         *   File loaded without error...
         ******************************************************************/
        virtual Int32 Import(DFIO **dfio, FileAccObj &file) { return -1; }

        /******************************************************************
         * InstCmd()
         ******************************************************************/
        virtual Int32 InstCmd(Uint32 argc, Char **argv);

        const char *get_filterName() { return filterName.value(); }
        const char *get_filterDesc() { return filterDesc.value(); }
        const char *get_instName()   { return instName.value();   }
        int doesImport() {return (filterCapability&FiltCapabilty_Import)?1:0;}
        int doesExport() {return (filterCapability&FiltCapabilty_Export)?1:0;}

        enum {
            FiltCapabilty_Import   = 1,
            FiltCapabilty_Export   = 2
        };

    protected:
        Tcl_Interp  *interp;
        Uint32       filterCapability;

    public:

    private:
        String       instName;
        String       filterName;
        Tcl_Obj     *filtSuffixes;
        String       filterDesc;
};

#endif /* INCLUDED_DFIO_FILTER_INFO_H */
