/*****************************************************************************
 *                              DesignDB.h
 *
 * Author: Matthew Ballance
 * Desc:   Data structures for browsing a design or waveform-database
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_DESIGN_DB_H
#define INCLUDED_DESIGN_DB_H

#include <tcl.h>
#include <tk.h>
#include "tclOldConfig.h"
#include "types.h"
#include "tree_node.h"
#include "vpi_user.h"
#include "IviSim.h"
#include "vector.h"
#include "DFIO.h"

class DesignDB {

public:
    DesignDB(Tcl_Interp *interp, int argc, char **argv);
    void scanDFIO();
    void scanTrace(DFIOTrace *trace);

    /**** ****/
    virtual void scanDesign();

    void dumpDesign(void);
    void Init(Tcl_Interp *interp);
    int      makeTclHandle(char *desPath, TreeNode *stPt, char *tclHandle);
    Tcl_Obj *tclSigNameList(char *modPath, TreeNode *stPt, char *resource);
    int getSignals(TreeNode *parent, Vector<TreeNode> *sigs, 
                   Char **patterns, Uint32 numPatterns);

    int DesignDB::tclCmdGlob(
        int         argc,
        char      **argv
        );

    /*********************************************************
     * globElems
     *
     * Parameters:
     * - Parent
     *   Specifies an alternate root-node to use. This can
     *   be used to allow scope-relative searching
     *   Parent will be used if:
     *   - If the path is not '/'-rooted
     *   - Parent is non-null
     * - PathSpec
     *   A '/'-delimited path specifying the path to a 
     *   resource.
     * - elemType
     *   Specifies the type of element of interest. May be:
     *   - @modules@
     *   - @signals@
     *   - @functions@
     *   - @tasks@
     * - flags
     *   Allows user-specified search options such as -r
     *********************************************************/
    Vector<TreeNode> *DesignDB::globElems(
        TreeNode     *Parent, 
        const char   *PathSpec,
        const char   *elemType,
        Uint32        flags);
    
    Char             sepChar;

    virtual int InstCmd(int argc, char **argv);


    IviSim *get_sim();
    DFIO   *get_dfio();

protected:
    void populateTree(TreeNode *tree);
    int Configure(Uint32 argc, Char **argc, Uint32 flags);


    TreeNode *scanModule(TreeNode *parent, vpiHandle modHndl);
    TreeNode *scanSignal(TreeNode *parent, vpiHandle modHndl, Uint32 type);

    ClientData          LoadDesignCbId;
    TreeNode           *designTree;
    Tcl_Interp         *interp;
    Vector<TreeNode>   *matchList;
    Vector<TreeNode>   *nextMatchList;
    Vector<TreeNode>   *tmpGlobVect;

    String              instName;
    String              tmpString;

    Char               *simName;
    IviSim             *sim;

    Char               *dfioName;
    DFIO               *dfio;

    enum {
        Cfg_Sim,
        Cfg_Dfio,
        Cfg_NumCfg
    };

    /*************************************************************************
     * getConfigSpec()
     *************************************************************************/
    Tk_ConfigSpec *getConfigSpec(void);
                
    /*************************************************************************
     * optionSpecified()
     *************************************************************************/
    Uint32 optionSpecified(Uint32 idx) {
        return (getConfigSpec()[idx].specFlags & TK_CONFIG_OPTION_SPECIFIED);
    }

};

extern "C" int DesignDB_Init(Tcl_Interp *interp);

#endif /*INCLUDED_DESIGN_DB_H*/
