/****************************************************************************
 *                                IviSim.h
 *
 * Author: Matthew Ballance
 * Desc:   Description of the simulator-interface object used by IVI
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_IVI_SIM_H
#define INCLUDED_IVI_SIM_H
#include "vpi_user.h"
#include "types.h"
#include <stdarg.h>
#include <tcl.h>
#include <tk.h>
#include "ivi_String.h"

/********************************************************************
 * mcd_cb_func()
 * type for a message-cb function...
 ********************************************************************/
typedef void (*mcd_cb_func)(voidPtr userData, Char *msg);

class DesignDB;
class IviSim {
    public:
        /*****************************************************************
         * IviSim()
         *
         * Parameters:
         * - logFileName
         *   Name of the logfile to log simulation activity to. If 
         *   NULL or zero-length, no logfile will be created...
         *****************************************************************/
        IviSim(Tcl_Interp *interp, int argc, char **argv);

        /*****************************************************************
         * ~IviSim()
         *****************************************************************/
         virtual ~IviSim(void);

        /************************************************************
         * InstCmd()
         ************************************************************/
         virtual int InstCmd(int argc, char **argv);

        /************************************************************
         * RegisterMsgCB()
         *
         * Registers a message-callback function on a specified set 
         * of mcd channels. The given function will be called 
         * whenever data is sent to the specified channel.
         *
         * Parameters:
         * - mcd
         *
         * - cbFunc
         *
         * - userData
         *
         * - clear_others
         *   If this flag is set, any existing files or functions to 
         *   which the output is directed will be closed or removed.
         ************************************************************/
        virtual int RegisterMsgCB(
                Uint32                   mcd,
                mcd_cb_func              cbFunc,
                void                    *userData,
                Uint32                   clear_others);

        /************************************************************
         * Run()
         *
         * Run the simulation ahead for some number of time units.
         ************************************************************/
        virtual int Run(Uint32 max_time);

        /************************************************************
         * ErrorStr()
         *
         * Returns the most recent error, or NULL if there are no
         * errors.
         ************************************************************/
        virtual char *ErrorStr();

        /************************************************************
         * LoadDesign()
         *
         * Loads a design into this instance of the VVPSim.
         ************************************************************/
        virtual int LoadDesign(char *design_path);

        /************************************************************
         * AppendModulePath()
         *
         * Appends an path to the module-path
         ************************************************************/
        virtual int AppendModulePath(char *path);

        /************************************************************
         * LoadVPIModule()
         *
         * Loads a VPI module into this instance of VVPSim
         ************************************************************/
        virtual int LoadVPIModule(char *vpi_module);

        /************************************************************
         * GetStdoutChannel()
         *
         * Returns the name of the stdout TCL channel.
         ************************************************************/
        virtual const char *GetStdoutChannel();

        /************************************************************
         * Close()
         ************************************************************/
        virtual int Close();

        /************************************************************
         * Delete()
         ************************************************************/
        void Delete();

        /************************************************************
         * setSimResolution()
         ************************************************************/
        virtual void setSimResolution(Int32 resolution);

        /************************************************************
         * getSimResolution()
         */
        /** Returns the internal simulation resolution. This will
         *  be a negative integer corresponding to 
         *  fs, ps, ns, us, ms, or s. 
         ************************************************************/
        virtual Int32 getSimResolution();

        /************************************************************
         * setResMultiplier()
         */
        /** Sets the resolution multiplier. The multiplier will be
         *  1, 10, or 100 and corresponds to the minimum timestep of
         *  the simulator
         ************************************************************/
        virtual void setResMultiplier(Int32 mult);

        /************************************************************
         * getResMultiplier()
         */
        /** Returns the resolution multiplier. This will be 
         *  1, 10, or 100 and corresponds to the minimum timestep of
         *  the simulator
         ************************************************************/
        virtual Int32 getResMultiplier();

        /*****************************************************************
         * vpi_register_systf()
         *****************************************************************/
        virtual void vpi_register_systf(const struct t_vpi_systf_data *ss);

        /*****************************************************************
         * vpi_printf()
         *****************************************************************/
        virtual void vpi_printf(const char*fmt, ...);

        /*****************************************************************
         * vpi_mcd_close()
         *****************************************************************/
        virtual Uint32 vpi_mcd_close(unsigned int mcd);

        /*****************************************************************
         * vpi_mcd_name()
         *****************************************************************/
        virtual Char *vpi_mcd_name(unsigned int mcd);

        /*****************************************************************
         * vpi_mcd_open()
         *****************************************************************/
        virtual Uint32 vpi_mcd_open(char *name);

        /*****************************************************************
         * vpi_mcd_open_x()
         *****************************************************************/
        virtual Uint32 vpi_mcd_open_x(char *name, char *mode);

        /*****************************************************************
         * vpi_mcd_printf()
         *****************************************************************/
        virtual Int32 vpi_mcd_printf(Uint32 mcd, const Char *fmt, ...);

        /*****************************************************************
         * vpi_mcd_vprintf(
         *****************************************************************/
        virtual Int32 vpi_mcd_vprintf(Uint32 mcd, const Char *fmt, va_list ap);

        /*****************************************************************
         * vpi_mcd_fputc()
         *****************************************************************/
        virtual Int32 vpi_mcd_fputc(Uint32 mcd, UChar x);

        /*****************************************************************
         * vpi_mcd_fgetc()
         *****************************************************************/
        virtual Int32 vpi_mcd_fgetc(Uint32 mcd);

        /*****************************************************************
         * vpi_register_cb()
         *****************************************************************/
        virtual vpiHandle vpi_register_cb(p_cb_data data);

        /*****************************************************************
         * vpi_remove_cb
         *****************************************************************/
        virtual Int32 vpi_remove_cb(vpiHandle ref);

        /*****************************************************************
         * vpi_sim_control()
         *****************************************************************/
        virtual void vpi_sim_control(Int32 operation, ...);

        /*****************************************************************
         * vpi_handle()
         *****************************************************************/
        virtual vpiHandle vpi_handle(Int32 type, vpiHandle ref);

        /*****************************************************************
         * vpi_iterate()
         *****************************************************************/
        virtual vpiHandle vpi_iterate(Int32 type, vpiHandle ref);

        /*****************************************************************
         * vpi_scan()
         *****************************************************************/
        virtual vpiHandle vpi_scan(vpiHandle iter);

        /*****************************************************************
         * vpi_handle_by_index()
         *****************************************************************/
        virtual vpiHandle vpi_handle_by_index(vpiHandle ref, Int32 index);

        /*****************************************************************
         * vpi_get_time()
         *****************************************************************/
        virtual void vpi_get_time(vpiHandle obj, s_vpi_time *t);

        /*****************************************************************
         * vpi_get()
         *****************************************************************/
        virtual Int32 vpi_get(Int32 property, vpiHandle ref);

        /*****************************************************************
         * vpi_get_str()
         *****************************************************************/
        virtual Char *vpi_get_str(Int32 property, vpiHandle ref);

        /*****************************************************************
         * vpi_get_value()
         *****************************************************************/
        virtual void vpi_get_value(vpiHandle expr, p_vpi_value value);

        /*****************************************************************
         * vpi_put_value()
         *****************************************************************/
        virtual vpiHandle vpi_put_value(vpiHandle obj, p_vpi_value value,
				     p_vpi_time when, Int32 flags);

        /*****************************************************************
         * vpi_free_object()
         *****************************************************************/
        virtual Int32 vpi_free_object(vpiHandle ref);

        /*****************************************************************
         * vpi_get_vlog_info()
         *****************************************************************/
        virtual Int32 vpi_get_vlog_info(p_vpi_vlog_info vlog_info_p);

        static char *GetVpiObjString(Uint32 objCode);

        inline const char *getInstName() { return instName.value(); }
        inline DesignDB *get_ddb() { return ddb; }
        Uint32         ok;
        Uint32         finished;

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Tcl_Interp    *interp;
        Tcl_Command    d_command;
        String         instName;

        Char          *ddbName;
        DesignDB      *ddb;

        /** Specifies the simulation resolution. This will be
         *  -15, -12, -9, -6, -3, or 0 (fs,ps,ns,us,ms,s). The 
         *  minimum simulator time-step is simResolution*resMultiplier
         */
        Int32          d_simResolution;

        /** Specifies the size of the minimum timestep in terms of
         *  simulation resolution (d_simResolution). Will be 
         *  1, 10, or 100.
         */
        Int32          d_resMultiplier;

        int Configure(int argc, char **argv, int flags);

        int Inspect(int argc, char **argv);

        /************************************************************
         * Delete()
         ************************************************************/
        static void Delete(char *userData);

        /************************************************************
         * FormatTimeCmd()
         ************************************************************/
        void FormatTimeCmd();

        /************************************************************
         *
         ************************************************************/
        enum {
            OPT_DDB,
            OPT_NumOpts
        };

        /************************************************************
         * getConfigSpec()
         ************************************************************/
        Tk_ConfigSpec *getConfigSpec(void);
                
        /************************************************************
         * optionSpecified()
         ************************************************************/
        Uint32 optionSpecified(Uint32 idx) {
            return (getConfigSpec()[idx].specFlags&TK_CONFIG_OPTION_SPECIFIED);
        }
};


#endif /* INCLUDED_IVI_SIM_H */
