
/*
 * The JTS Topology Suite is a collection of Java classes that
 * implement the fundamental operations required to validate a given
 * geo-spatial data set to a known topological specification.
 *
 * Copyright (C) 2001 Vivid Solutions
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, contact:
 *
 *     Vivid Solutions
 *     Suite #1A
 *     2328 Government Street
 *     Victoria BC  V8T 5G5
 *     Canada
 *
 *     (250)385-6040
 *     www.vividsolutions.com
 */
package com.vividsolutions.jts.noding.snapround;

import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.noding.*;


/**
 * @version 1.6
 */
public class SegmentSnapper {

  private static double TOLERANCE = 0.5;
  /**
   * @return true if the point p is within the snap tolerance of the line p0-p1
   */
  public static boolean isWithinTolerance(Coordinate p, Coordinate p0, Coordinate p1)
  {
    double minx = p.x - TOLERANCE;
    double maxx = p.x + TOLERANCE;
    double miny = p.y - TOLERANCE;
    double maxy = p.y + TOLERANCE;
    double segMinx = Math.min(p0.x, p1.x);
    double segMaxx = Math.max(p0.x, p1.x);
    double segMiny = Math.min(p0.y, p1.y);
    double segMaxy = Math.max(p0.y, p1.y);
    if ( maxx < segMinx
      || minx > segMaxx
      || maxy < segMiny
      || miny > segMaxy) return false;

    double dx = p1.x - p0.x;
    double dy = p1.y - p0.y;

    double px = p.x - p0.x;
    double py = p.y - p0.y;

    double dely = px * dy / dx - py;
    double delx = py * dx / dy - px;
    double discy = px * dy - py * dx;

    if (Math.abs(discy) < Math.abs(0.5 * dx) ) return true;
    double discx = py * dx - px * dy;
    if (Math.abs(discx) < Math.abs(0.5 * dy) ) return true;
/*
    double dely = px * dy / dx - py;
    if (dely > 0.5 || dely < -0.5) return false;

    double delx = py * dx / dy - px;
    if (delx > 0.5 || delx < -0.5) return false;
*/
    return false;
  }

  public SegmentSnapper() {
  }

  /**
   * Adds a new node (equal to the snap pt) to the segment
   * if the snapPt is
   * within tolerance of the segment
   *
   * @param snapPt
   * @param segStr
   * @param segIndex
   * @return <code>true</code> if a node was added
   */
  public boolean addSnappedNode(
      Coordinate snapPt,
      SegmentString segStr,  int segIndex
      )
  {
    Coordinate p0 = segStr.getCoordinate(segIndex);
    Coordinate p1 = segStr.getCoordinate(segIndex + 1);

    // no need to snap if the snapPt equals an endpoint of the segment
    if (snapPt.equals(p0)) return false;
    if (snapPt.equals(p1)) return false;

    if (isWithinTolerance(snapPt, p0, p1)) {
      segStr.addIntersection(snapPt, segIndex);
      return true;
    }
    return false;
  }
}
