#ifndef K3DUI_DAG_CONTROL_H
#define K3DUI_DAG_CONTROL_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <k3dsdk/uuid.h>
#include <k3dsdk/vectors.h>
#include <vector>

// Forward declarations
class sdpGtkWidget;

namespace k3d
{

// Forward declarations
class idocument;
class iproperty;
class iunknown;

namespace dag_control
{

/// Encapsulates an "edge" (a directed connection between two objects) to be visualized by the control
class edge
{
public:
	iunknown* from;
	iunknown* to;
};

/// Encapsulates a "node" (zero-to-many document objects) to be visualized by the control
class node
{
public:
	std::string label;
	
	typedef std::vector<iunknown*> objects_t;
	objects_t objects;
};

/// Defines a collection of nodes and dependencies
class graph
{
public:
	~graph();
	
	typedef std::vector<node*> nodes_t;
	nodes_t nodes;
	
	typedef std::vector<edge*> edges_t;
	edges_t edges;
};

/// Abstract interface for objects that populate graphs - these control what the user ends-up seeing
class filter_policy
{
public:
	virtual void populate_graph(graph& Graph) = 0;
};

/// Concrete implementation of filter_policy that does nothing
class null_filter_policy :
	public filter_policy
{
public:
	void populate_graph(graph& Graph);
};

/// Concrete implementation of filter_policy that displays every object in the document
class all_objects_filter_policy :
	public filter_policy
{
public:
	all_objects_filter_policy(idocument& Document);

	void populate_graph(graph& Graph);

private:
	idocument& m_document;
};

/// Concrete implementation of filter_policy that displays every object that matches given class ID
class class_id_filter_policy :
	public filter_policy
{
public:
	class_id_filter_policy(idocument& Document, const uuid& ClassID);
	
	void populate_graph(graph& Graph);

private:
	idocument& m_document;
	const uuid m_class_id;
};

/// Abstract interface for objects that adjust the layout of a graph
class layout_policy
{
public:
	virtual void update_layout(graph& Graph) = 0;
};

/// Concrete implementation of layout_policy that does nothing
class null_layout_policy :
	public layout_policy
{
public:
	void update_layout(graph& Graph);
};

/// Concrete implementation of layout_policy that sorts nodes alphabetically by label
class sort_by_label_layout_policy :
	public layout_policy
{
public:
	void update_layout(graph& Graph);
};

/// Concrete implementation of layout_policy that sorts nodes based on the type of objects they front for
class sort_by_type_layout_policy :
	public layout_policy
{
public:
	void update_layout(graph& Graph);
};

/// Abstract interface for objects that draw the graph
class drawing_policy
{
public:
	virtual void draw_graph(const graph& Graph) = 0;
};

/// Concrete implementation of drawing_policy that does nothing
class null_drawing_policy :
	public drawing_policy
{
public:
	void draw_graph(const graph& Graph);
};

class control
{
public:
	control(idocument& Document, iunknown* ParentCommandNode, const std::string& CommandNodeName);
	~control();

	/// Sets a new policy for filtering the contents of the control - note: the control assumes responsibility for the lifetime of the policy object
	void set_filter_policy(filter_policy* const Policy);
	/// Sets a new policy for automatic layout of the contents of the control - note: the control assumes responsibility for the lifetime of the policy object
	void set_layout_policy(layout_policy* const Policy);
	/// Sets a new policy for drawing the contents of the control - note: the control assumes responsibility for the lifetime of the policy object
	void set_drawing_policy(drawing_policy* const Policy);

	sdpGtkWidget root_widget();

private:
	class implementation;
	implementation* const m_implementation;
};

} // namespace dag_control

} // namespace k3d

#endif // !K3DUI_DAG_CONTROL_H

