// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Filter object that takes a mesh as input and returns the Catmull-Clark subdivision of this mesh.
		\author Bart Janssens <bart.janssens@polytechnic.be>
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include "subdiv_algorithms.h"

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// catmull_clark_subdivider_implementation

class catmull_clark_subdivider_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	catmull_clark_subdivider_implementation(k3d::idocument& Document) :
		base(Document),
		m_levels(k3d::init_name("levels") + k3d::init_description("Levels [integer]") + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_document(Document) + k3d::init_value(1) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_levels));
		register_property(m_levels);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &catmull_clark_subdivider_implementation::on_reset_geometry));
		m_levels.changed_signal().connect(SigC::slot(*this, &catmull_clark_subdivider_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &catmull_clark_subdivider_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		const bool ignore_selection = !k3d::contains_selection(*input);

		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		subdiv::catmull_clark(m_levels.property_value(), *input, *mesh, ignore_selection);
		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<catmull_clark_subdivider_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xdc73f6e3, 0x744b4b51, 0xac535307, 0xc82d261c),
				"CatmullClark",
				"Generates a Catmull-Clark subdivision mesh",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_levels;
};

/////////////////////////////////////////////////////////////////////////////
// catmull_clark_subdivision_factory

k3d::iplugin_factory& catmull_clark_subdivision_factory()
{
	return catmull_clark_subdivider_implementation::get_factory();
}

} // namespace libk3dmesh


