// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/geometry.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_circle_implementation

class nurbs_circle_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_circle_implementation(k3d::idocument& Document) :
		base(Document),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_thetamax(k3d::init_name("thetamax") + k3d::init_description("End angle [number]") + k3d::init_value(k3d::radians(360.0)) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle))),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Segments [integer]") + k3d::init_value(4) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_radius));
		enable_serialization(k3d::persistence::proxy(m_thetamax));
		enable_serialization(k3d::persistence::proxy(m_u_segments));

		register_property(m_radius);
		register_property(m_thetamax);
		register_property(m_u_segments);

		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_circle_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &nurbs_circle_implementation::on_reset_geometry));
		m_thetamax.changed_signal().connect(SigC::slot(*this, &nurbs_circle_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &nurbs_circle_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_circle_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double radius = m_radius.property_value();
		const double thetamax = m_thetamax.property_value();
		const unsigned long u_segments = m_u_segments.property_value();

		k3d::mesh* const mesh = new k3d::mesh();
		k3d::nucurve_group* const nucurve_group = new k3d::nucurve_group();
		k3d::nucurve* const nucurve = new k3d::nucurve();
		mesh->nucurve_groups.push_back(nucurve_group);
		nucurve_group->curves.push_back(nucurve);
		nucurve_group->material = m_material.interface();
		nucurve->order = 3;

		std::vector<double> weights;
		std::vector<k3d::vector3> control_points;
		k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), 0, thetamax, u_segments, nucurve->knots, weights, control_points);
		return_val_if_fail(weights.size() == control_points.size(), 0);

		for(unsigned long i = 0; i != control_points.size(); ++i)
			{
				mesh->points.push_back(new k3d::point(radius * control_points[i]));
				nucurve->control_points.push_back(k3d::nucurve::control_point(mesh->points.back(), weights[i]));
			}

		assert_warning(is_valid(*nucurve));
		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_circle_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
		k3d::uuid(0x82c3a0ef, 0x4e074f5c, 0xb3926e7f, 0xb725b571),
			"NurbsCircle",
			"Generates a NURBS circle",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_thetamax;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_circle_factory

k3d::iplugin_factory& nurbs_circle_factory()
{
	return nurbs_circle_implementation::get_factory();
}

} // namespace libk3dnurbs


