#include <string>
#include <vector>
#include <iostream>
using namespace std;

#include "category.h"
#include "exprnode.h"
#include "expression.h"
#include "utility.h"

#define TRUE 1
#define FALSE 0

//  expr := expr ( "|" | "&" | "^" ) clause | clause
//  clause := clause ( "=" | "<" | ">" | "!=" | "<=" | ">=" ) term | term
//  term := ["!"] "(" expr ")" | ["!"] identifier

ExprNode *createExprTree( string expr )
{
    // parse expression
    vector<string> tokenv;
    if( !exprToTokens( expr, tokenv ) )
    {
        cout << "*** malformed expression: " << expr << endl;
        return 0;
    }
//    for( int n = 0; n < tokenv.size(); n+=2 )
//        cout << tokenv[n] << "\t" << tokenv[n+1] << endl;

    ExprNode *node = new ExprNode;
    node->leftChild = 0;
    node->rightChild = 0;

    int tokennr = tokenv.size()-2;
    if( !buildExpression( tokenv, node, tokennr ) ||
        ( tokennr != -2 ) )
    {
        cout << "*** malformed expression: " << expr << endl;
        delete node;
        return 0;
    }

//    printNodeRec( node, 0 );
//    cout << "done." << endl;

    return node;
}

int exprTrueForCategory( ExprNode *node, Category *cat )
{
    string result = evaluateExprRec( cat, node );
    if( ( result == ",TRUE" ) || ( result == ",FALSE" ) )
        return( result == ",TRUE" );
    return cat->unaryProperty( result );
}

string evaluateExprRec( Category *cat, ExprNode *node )
{
    if( node->sort == "ID/NR" )
        return node->value;

    if( node->value == "!" )
    {
        string result = evaluateExprRec( cat, node->leftChild );
        if( result == ",TRUE" )
            return ",FALSE";
        if( result == ",FALSE" )
            return ",TRUE";
        if( cat->unaryProperty( result ) )
            return ",FALSE";
        return ",TRUE";
    }

    string left = evaluateExprRec( cat, node->leftChild ),
           right = evaluateExprRec( cat, node->rightChild );

    if( ( node->value == "=" ) ||
        ( node->value == ">" ) ||
        ( node->value == "<" ) ||
        ( node->value == "!=" ) ||
        ( node->value == ">=" ) ||
        ( node->value == "<=" ) )
        return cat->hasPropertyWithValue( right, node->value, left ) ?
            ",TRUE" : ",FALSE";

    if( ( left != ",TRUE" ) && ( left != ",FALSE" ) )
        left = cat->unaryProperty( left ) ? ",TRUE" : ",FALSE";
    if( ( right != ",TRUE" ) && ( right != ",FALSE" ) )
        right = cat->unaryProperty( right ) ? ",TRUE" : ",FALSE";

    int lefttrue = left == ",TRUE",
        righttrue = right == ",TRUE",
        result;

    if( node->value == "|" )
        result = lefttrue || righttrue;
    else if( node->value == "&" )
        result = lefttrue && righttrue;
    else if( node->value == "^" )
        result = lefttrue ^ righttrue;

    return( result ? ",TRUE" : ",FALSE" );
}

void printNodeRec( ExprNode *node, int depth )
{
    for( int n = 0; n < depth; n++ )
        cout << " ";
    cout << node->sort << " " << node->value << endl;

    depth++;
    if( node->rightChild != 0 )
        printNodeRec( node->rightChild, depth );
    if( node->leftChild != 0 )
        printNodeRec( node->leftChild, depth );
}

int buildExpression( vector<string> &tokenv, ExprNode *node, int &tokennr )
{
    if( tokennr < 0 )
        return FALSE;
    int savetokennr = tokennr;

    node->leftChild = new ExprNode;
    node->leftChild->leftChild = 0;
    node->leftChild->rightChild = 0;
    node->rightChild = new ExprNode;
    node->rightChild->leftChild = 0;
    node->rightChild->rightChild = 0;

    // expr := expr |& clause
    if( buildClause( tokenv, node->leftChild, tokennr ) &&
        ( tokennr >= 2 ) &&
        ( tokenv[tokennr] == "OP" ) &&
        ( ( tokenv[tokennr+1] == "|" ) ||
          ( tokenv[tokennr+1] == "&" ) ||
          ( tokenv[tokennr+1] == "^" ) ) )
    {
        string operation = tokenv[tokennr+1];
        tokennr -= 2;

        if( buildExpression( tokenv, node->rightChild, tokennr ) )
        {
            //cout << "expr := expr |& clause" << endl;

            node->sort = "OP";
            node->value = operation;
            return TRUE;
        }
    }
    tokennr = savetokennr;

    delete node->leftChild;
    delete node->rightChild;
    node->leftChild = 0;
    node->rightChild = 0;

    // expr := clause
    if( buildClause( tokenv, node, tokennr ) )
        //cout << "expr := clause" << endl;
        return TRUE;

    tokennr = savetokennr;
    return FALSE;
}

int buildClause( vector<string> &tokenv, ExprNode *node, int &tokennr )
{
    if( tokennr < 0 )
        return FALSE;
    int savetokennr = tokennr;

    node->leftChild = new ExprNode;
    node->leftChild->leftChild = 0;
    node->leftChild->rightChild = 0;
    node->rightChild = new ExprNode;
    node->rightChild->leftChild = 0;
    node->rightChild->rightChild = 0;

    // clause := clause = term
    if( buildTerm( tokenv, node->leftChild, tokennr ) &&
        ( tokennr >= 2 ) &&
        ( tokenv[tokennr] == "OP" ) &&
        ( ( tokenv[tokennr+1] == "=" ) ||
          ( tokenv[tokennr+1] == "!=" ) ||
          ( tokenv[tokennr+1] == "<" ) ||
          ( tokenv[tokennr+1] == ">" ) ||
          ( tokenv[tokennr+1] == "<=" ) ||
          ( tokenv[tokennr+1] == ">=" ) ) )
    {
        string op = tokenv[tokennr+1];

        tokennr -= 2;
        if( buildClause( tokenv, node->rightChild, tokennr ) )
        {
            //cout << "clause := clause = term" << endl;

            node->sort = "OP";
            node->value = op;
            return TRUE;
        }
    }
    tokennr = savetokennr;

    delete node->leftChild;
    delete node->rightChild;
    node->leftChild = 0;
    node->rightChild = 0;

    // clause := term
    if( buildTerm( tokenv, node, tokennr ) )
        return TRUE;

    tokennr = savetokennr;
    return FALSE;
}

int buildTerm( vector<string> &tokenv, ExprNode *node, int &tokennr )
{
    if( tokennr < 0 )
        return FALSE;
    int savetokennr = tokennr;

    // term := ["!"] "(" expr ")"
    if( ( tokennr >= 4 ) &&
        ( tokenv[tokennr] == ")" ) )
    {
        tokennr -= 2;
        if( buildExpression( tokenv, node, tokennr ) &&
            ( tokennr >= 0 ) &&
            ( tokenv[tokennr] == "(" ) )
        {
            tokennr -= 2;
            //cout << "term := ( expr )" << endl;

            while( ( tokennr >= 0 ) &&
                   ( tokenv[tokennr] == "OP" ) &&
                   ( tokenv[tokennr+1] == "!" ) )
            {
                tokennr -= 2;
                ExprNode *newnode = new ExprNode;

                newnode->sort = node->sort;
                newnode->value = node->value;
                newnode->leftChild = node->leftChild;
                newnode->rightChild = node->rightChild;

                node->sort = "OP";
                node->value = "!";
                node->leftChild = newnode;
                node->rightChild = 0;
            }
            return TRUE;
        }
    }
    tokennr = savetokennr;

    // term := identifier
    if( tokenv[tokennr] == "ID/NR" )
    {
        //cout << "term := identifier " << tokenv[tokennr+1] << endl;

        node->sort = "ID/NR";
        node->value = tokenv[tokennr+1];

        tokennr -= 2;

        while( ( tokennr >= 0 ) &&
               ( tokenv[tokennr] == "OP" ) &&
               ( tokenv[tokennr+1] == "!" ) )
        {
            tokennr -= 2;
            ExprNode *newnode = new ExprNode;

            newnode->sort = node->sort;
            newnode->value = node->value;
            newnode->leftChild = node->leftChild;
            newnode->rightChild = node->rightChild;

            node->sort = "OP";
            node->value = "!";
            node->leftChild = newnode;
            node->rightChild = 0;
        }

        return TRUE;
    }

    tokennr = savetokennr;
    return FALSE;
}

int exprToTokens( string expr, vector<string> &tokenv )
{
    int k = -1;
    string s;

    while( ( k+1 < expr.length() ) &&
           ( ( k = expr.find_first_not_of( ' ', k+1 ) ) != -1 ) )
    {
        if( ( k+1 < expr.length() ) &&
            ( expr[k] == '>' ) &&
            ( expr[k+1] == '=' ) )
        {
            k++;
            tokenv.push_back( "OP" );
            tokenv.push_back( ">=" );
            continue;
        }

        if( ( k+1 < expr.length() ) &&
            ( expr[k] == '<' ) &&
            ( expr[k+1] == '=' ) )
        {
            k++;
            tokenv.push_back( "OP" );
            tokenv.push_back( "<=" );
            continue;
        }

        if( ( k+1 < expr.length() ) &&
            ( expr[k] == '&' ) &&
            ( expr[k+1] == '&' ) )
        {
            k++;
            tokenv.push_back( "OP" );
            tokenv.push_back( "&" );
            continue;
        }

        if( ( k+1 < expr.length() ) &&
            ( expr[k] == '|' ) &&
            ( expr[k+1] == '|' ) )
        {
            k++;
            tokenv.push_back( "OP" );
            tokenv.push_back( "|" );
            continue;
        }

        if( expr[k] == '!' )
        {
            if( ( k+1 < expr.length() ) &&
                ( expr[k+1] == '=' ) )
            {
                k++;
                tokenv.push_back( "OP" );
                tokenv.push_back( "!=" );
            }
            else
            {
                tokenv.push_back( "OP" );
                tokenv.push_back( "!" );
            }
            continue;
        }

        if( ( expr[k] == '(' ) ||
            ( expr[k] == ')' ) )
        {
            s = " ";
            s[0] = expr[k];
            tokenv.push_back( s );
            tokenv.push_back( "" );
            continue;
        }

        if( ( ( expr[k] >= 'A' ) && ( expr[k] <= 'Z' ) ) ||
            ( ( expr[k] >= 'a' ) && ( expr[k] <= 'z' ) ) ||
            ( ( expr[k] >= '0' ) && ( expr[k] <= '9' ) ) )
        {
            s = "";
            while( ( k < expr.length() ) &&
                   ( ( ( expr[k] >= 'a' ) && ( expr[k] <= 'z' ) ) ||
                     ( ( expr[k] >= 'A' ) && ( expr[k] <= 'Z' ) ) ||
                     ( ( expr[k] >= '0' ) && ( expr[k] <= '9' ) ) ||
                     ( expr[k] == ' ' ) ) )
                s += expr[k++];

            tokenv.push_back( "ID/NR" );
            tokenv.push_back( trim( s ) );
            k--;
            continue;
        }

        if( ( expr[k] == '=' ) ||
            ( expr[k] == '<' ) ||
            ( expr[k] == '>' ) ||
            ( expr[k] == '|' ) ||
            ( expr[k] == '&' ) ||
            ( expr[k] == '^' ) )
        {
            tokenv.push_back( "OP" );
            s = " ";
            s[0] = expr[k];
            tokenv.push_back( s );
            continue;
        }

        return FALSE;
    }

    return TRUE;
}



