/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */

#include <kdebug.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <kurl.h>
#include <kio/netaccess.h>
#include <qpopupmenu.h>

#include "catalogsettings.h"
#include "editcmd.h"
#include "tagextractor.h"
#include "kbabelview.h"
#include "kbabeldictbox.h"
#include "mymultilineedit.h"
#include "roughtransdlg.h"

#include "resources.h"

QList<ModuleInfo> KBabelView::dictionaries()
{
   QList<ModuleInfo> list = dictBox->moduleInfos();

   return list;
}

void KBabelView::configureDictionary(const QString id)
{
   dictBox->configure(id);
}

void KBabelView::editDictionary(const QString id)
{
   dictBox->edit(id);
}


void KBabelView::aboutDictionary(const QString id)
{
   dictBox->aboutModule(id);
}

void KBabelView::informDictionary()
{
    dictBox->setTextChanged(_catalog->msgid(_currentIndex),
            _catalog->msgstr(_currentIndex),_catalog->comment(_currentIndex));
}

void KBabelView::setNewLanguage()
{
    IdentitySettings s = _catalog->identitySettings();

    dictBox->setLanguage(s.languageCode, s.languageName);
}


void KBabelView::wheelEvent(QWheelEvent *e)
{
    if( (e->state() & ControlButton) && (e->state() & AltButton))
    {
        if(e->delta() > 0)
        {
            gotoPrevFuzzyOrUntrans();
        }
        else
        {
            gotoNextFuzzyOrUntrans();
        }
    }
    else if(e->state() & ControlButton)
    {
        if(e->delta() > 0)
        {
            gotoPrevFuzzy();
        }
        else
        {
            gotoNextFuzzy();
        }
    }
    else if(e->state() & AltButton)
    {
        if(e->delta() > 0)
        {
            gotoPrevUntranslated();
        }
        else
        {
            gotoNextUntranslated();
        }
    }
    else
    {
        if(e->delta() > 0)
        {
            gotoPrev();
        }
        else
        {
            gotoNext();
        }
    }
   
    e->accept();
}



void KBabelView::roughTranslation()
{
    RoughTransDlg *dlg = new RoughTransDlg(dictBox, _catalog, this
            , "roughtransDlg");

    dlg->exec();

    delete dlg;
}


void KBabelView::updateTags()
{
    bool hadTags = _tags.count() > 0;

    _tags = _catalog->tagList(_currentIndex);

    if(_tagsMenu)
    {
        _tagsMenu->clear();
     
        QStringList tList;
        QStringList::Iterator it;
        int counter=0;
        for(it=_tags.begin(); it!=_tags.end(); ++it)
        {
            if(!tList.contains(*it))
            {
                _tagsMenu->insertItem(*it,counter);
                tList.append(*it);
            }
            counter++;
        }
    }

    bool haveTags = (_tags.count() > 0);
    
    if(isReadOnly())
        haveTags=false;

    if(haveTags != hadTags)
    {
        emit signalNextTagAvailable(haveTags);
        emit signalTagsAvailable(haveTags);
    }
}

void KBabelView::setTagsMenu(QPopupMenu *menu)
{
    _tagsMenu=menu;
    
    connect(_tagsMenu,SIGNAL(activated(int)),this,SLOT(insertTag(int)));
}


void KBabelView::insertTag(int n)
{
    EditCommand* tmp = new BeginCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(_currentIndex);
    _catalog->applyEditCommand(tmp,this);


    int line,col;
    msgstrEdit->getCursorPosition(&line,&col);
	int offset = msgstrEdit->pos2Offset(line,col);
    InsTextCmd* insCmd = new InsTextCmd(offset,_tagsMenu->text(n));
    insCmd->setPart(EditCommand::Msgstr);
    insCmd->setIndex(_currentIndex);
   
    msgstrEdit->processCommand(insCmd,false);
    forwardMsgstrEditCmd(insCmd);

    tmp = new EndCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(_currentIndex);
    _catalog->applyEditCommand(tmp,this);
}

void KBabelView::insertNextTag()
{
    int line,col;
    msgstrEdit->getCursorPosition(&line,&col);
	int offset = msgstrEdit->pos2Offset(line,col);

    QString s = _catalog->msgstr(_currentIndex).left(offset);
    
    TagExtractor te(s);
    uint num=te.countTags();
    if(num >= _tags.count())
    {
    	KNotifyClient::beep();
        return;
    }
     
    QString t=_tags[num];

    EditCommand* tmp = new BeginCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(_currentIndex);
    _catalog->applyEditCommand(tmp,this);

    InsTextCmd* insCmd = new InsTextCmd(offset,t);
    insCmd->setPart(EditCommand::Msgstr);
    insCmd->setIndex(_currentIndex);
   
    msgstrEdit->processCommand(insCmd,false);
    forwardMsgstrEditCmd(insCmd);
   
    tmp = new EndCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(_currentIndex);
    _catalog->applyEditCommand(tmp,this);
}

void KBabelView::showTagsMenu()
{
    if(_tagsMenu && _tags.count() > 0)
    {
        int y=msgstrEdit->height()/2;
        int x=msgstrEdit->width()/2;
        _tagsMenu->exec(msgstrEdit->mapToGlobal( QPoint(x,y) ) );

        return;
    }
}


void KBabelView::diff()
{
    diffInternal(false);
}

void KBabelView::diffShowOrig()
{
    msgidLabel->setText(_catalog->msgid(_currentIndex));
}

void KBabelView::toggleAutoDiff(bool on)
{
    if(on != _diffEnabled)
    {
        _diffEnabled = on;

        if(on)
        {
            diff();
        }
        else
        {
            diffShowOrig();
        }
    }
}

void KBabelView::autoDiff()
{
    diffInternal(true);
}

void KBabelView::diffInternal(bool autoDf)
{    
    if(_diffing || _loadingDiffFile)
    {
        return;
    }

    _diffing = true;
    uint diffIndex = _currentIndex;
    
    QString diffString;
    
    Catalog::DiffResult r = _catalog->diff(_currentIndex, &diffString);

    if(r == Catalog::DiffNeedList)
    {
        if(_settings.useDBForDiff)
        { 
            _loadingDiffFile=true;
            bool wasEnabled=_diffEnabled;
            _diffEnabled=false;        

            QValueList<DiffEntry> diffList;
            QString error;
            QString package = _catalog->packageName()+".po";
            kdDebug(KBABEL) << "getting list for " << package << endl;    
            
            if(dictBox->messagesForPackage(package,diffList,error))
            {
                kdDebug(KBABEL) << "got " << diffList.count() 
                    << " messages" << endl;
                _catalog->setDiffList(diffList);
            }
            else
            {
                KMessageBox::sorry(this
                    ,i18n("An error occured while trying to get list\n"
                        "of messages for this file from the database:\n"
                        "%1").arg(error));
                
                _diffing=false;
                _diffEnabled=false;
                _loadingDiffFile=false;
                emit signalDiffEnabled(false);

                return;
            }
            
            _diffEnabled=wasEnabled;
            _loadingDiffFile=false;
        }
        else
        {
            _diffing=false;
            if(!openDiffFile(true))
            {
                _diffEnabled=false;
                 emit signalDiffEnabled(false);
            
                _diffing=false;
                return;
            }
        
            _diffing = true;
        }
        
        diffIndex = _currentIndex;        
        r = _catalog->diff(_currentIndex, &diffString);
    }

    // if the index changed in the meanwhile
    while(diffIndex != _currentIndex)
    {
        diffIndex=_currentIndex;
        r = _catalog->diff(_currentIndex,&diffString);
    }

    if(r == Catalog::DiffOk)
    {
        msgidLabel->setText(diffString);

        if(diffString == _catalog->msgid(_currentIndex) )
        {
            emit signalChangeStatusbar(i18n("No difference found"));
        }
        else
        {
            emit signalChangeStatusbar(i18n("Difference found"));
        }
    }
    else
    {
        if(!autoDf)
        {
            KMessageBox::information(this
                ,i18n("No corresponding message found"));
        }
        else
        {
            emit signalChangeStatusbar(
                    i18n("No corresponding message found"));
        }
    }

    _diffing = false;
}

bool KBabelView::openDiffFile()
{
    return openDiffFile(false);
}

bool KBabelView::openDiffFile(bool autoDiff)
{
    if(_diffing || _loadingDiffFile)
        return false;
    
    KURL url;

    if( autoDiff && !_settings.diffBaseDir.isEmpty() )
    {
        KURL fileURL = _catalog->currentURL();
        KURL poBaseURL(_catManSettings.poBaseDir);
        QString poBase = poBaseURL.path();
        int len = poBase.length();
        if(fileURL.path().left(len) == poBase)
        {
           QString fileRelPath = fileURL.path().mid(len);
           if(fileRelPath[0] == '/')
               fileRelPath=fileRelPath.mid(1);

           if(_settings.diffBaseDir.right(1) != "/")
               _settings.diffBaseDir += '/';

           QString diffFilePath = _settings.diffBaseDir + fileRelPath;

           
           KURL diffFileURL(diffFilePath);

           if(diffFileURL.isValid() && KIO::NetAccess::exists(diffFileURL))
           {
               url = diffFileURL;

               kdDebug() << "using file " << diffFileURL.prettyURL() 
                         << " as diff file" << endl;
           }
        }
    }
        
 
    if(url.isEmpty())
    {
        url = KFileDialog::getOpenURL(_settings.diffBaseDir, "*.po *.pot"
               ,this, i18n("Select file to diff with"));
    }

    if(url.isEmpty())
        return false;
 
    _loadingDiffFile=true;
    bool wasEnabled=_diffEnabled;
    _diffEnabled=false;
        

    Catalog cat;
    bool dummy;
    
    connect(&cat,SIGNAL(signalProgress(int)),this,SIGNAL(signalProgress(int)));
    emit signalResetProgressBar(i18n("loading file for diff"),100);
    
    Catalog::IOStatus stat = cat.openURL(url,dummy);
    
    emit signalClearProgressBar();

    
    if(stat != Catalog::OK && stat != Catalog::RECOVERED_PARSE_ERROR)
    {
        switch(stat)
        {
            case Catalog::PARSE_ERROR:
            {
                KMessageBox::sorry(this
                    ,i18n("Error while trying to read file:\n %1\n"
                   "Maybe it's not a valid PO-file.").arg(url.prettyURL()));
                break;
            }
            case Catalog::NO_PERMISSIONS:
            {
                KMessageBox::sorry(this,i18n(
                "You don't have permissions to read file:\n %1")
                        .arg(url.prettyURL()));
                break;
            }
            case Catalog::NO_FILE:
            {
                 KMessageBox::sorry(this,i18n(
                 "You have not specified a valid file:\n %1")
                         .arg(url.prettyURL()));
                break;
            }
            default:
            {
                 KMessageBox::sorry(this,i18n(
                  "Error while trying to open file:\n %1")
                         .arg(url.prettyURL()));
                 break;
            }

        }
        
        _diffEnabled=wasEnabled;
        _loadingDiffFile=false;

        return false; 
    }

    _catalog->setDiffList( cat.asDiffList() );
    
    _diffEnabled=wasEnabled;
    _loadingDiffFile=false;

    return true;
}

