/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2004 by Stanislav Visnovsky
                            <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**************************************************************************** */

#include "kbproject.h"
#include "kbprojectmanager.h"

#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kspell.h>

namespace KBabel
{

Project::Project( const QString& file ) :
    QObject ()
    , _filename (file)
    , _name (i18n("unnamed"))
{
    _config = new KConfig (_filename);
    kdDebug () << "Opened project " << _filename << endl;
    
    // read the project name
    _config->setGroup( "Project" );
    _name = _config->readEntry( "Name", "" );
}

Project::~Project ()
{
    // store the project name
    _config->setGroup( "Project" );
    _config->writeEntry( "Name", _name );
    _config->sync();

    // unregister in project manager
    kdDebug () << "Freeing project " << _filename << endl;
    ProjectManager::remove (this);
    delete _config;
}
	    
KConfig* Project::config ()
{
    return _config;
}

IdentitySettings Project::identitySettings ()
{    
    Defaults::Identity defaultIdentity;
    IdentitySettings settings;

    {
	_config->setGroup("Header");
    
	settings.authorName=_config->readEntry("Author-Name"
            ,defaultIdentity.authorName());
	settings.authorLocalizedName=_config->readEntry("Local-Author-Name"
            ,defaultIdentity.authorName());
	settings.authorEmail=_config->readEntry("Author-Email"
            ,defaultIdentity.authorEmail());
	settings.languageName=_config->readEntry("Language"
            ,defaultIdentity.languageName());
	settings.languageCode=_config->readEntry("LanguageCode"
            ,defaultIdentity.languageCode());
	settings.mailingList=_config->readEntry("Mailinglist"
            ,defaultIdentity.mailingList());
	settings.timeZone=_config->readEntry("Timezone"
    	    ,defaultIdentity.timezone());
    }

    {    
	KConfigGroupSaver (_config, "Misc");
	settings.numberOfPluralForms=_config->readNumEntry("PluralForms"
            , defaultIdentity.numberOfPluralForms);
	    
	if( settings.numberOfPluralForms < -1 )
	{
	    kdWarning() << "Invalid number of plural forms, ignoring: " << settings.numberOfPluralForms << endl;
	    settings.numberOfPluralForms = 2;
	}
	settings.checkPluralArgument=_config->readBoolEntry("CheckPluralArgument",true);
	settings.gnuPluralFormHeader=_config->readEntry("PluralFormsHeader"
           , defaultIdentity.gnuPluralFormHeader);
    }
    return settings;
}

SaveSettings Project::saveSettings ()
{
    SaveSettings settings;
    
    KConfigGroupSaver groupSaver(_config,"Header");

    settings.autoUpdate=_config->readBoolEntry("AutoUpdate"
	    ,Defaults::Save::autoUpdate);
    settings.updateLastTranslator=_config->readBoolEntry("Update-Last-Translator"
            ,Defaults::Save::updateLastTranslator);
    settings.updateRevisionDate=_config->readBoolEntry("Update-Revision-Date"
            ,Defaults::Save::updateRevisionDate);
    settings.updateLanguageTeam=_config->readBoolEntry("Update-Language-Team"
            ,Defaults::Save::updateLanguageTeam);
    settings.updateCharset=_config->readBoolEntry("Update-Charset"
            ,Defaults::Save::updateCharset);
    settings.updateEncoding=_config->readBoolEntry("Update-Encoding"
            ,Defaults::Save::updateEncoding);
    settings.encoding=(FileEncoding)(_config->readNumEntry("Encoding"
            ,(int)Defaults::Save::encoding));
    settings.useOldEncoding=_config->readBoolEntry("UseOldEncoding"
            ,Defaults::Save::useOldEncoding);

    settings.updateProject=_config->readBoolEntry("Update-Project"
            ,Defaults::Save::updateProject);
    settings.projectString=_config->readEntry("ProjectString"
            ,Defaults::Save::projectString());

    settings.autoSyntaxCheck = _config->readBoolEntry("AutoSyntaxCheck"
            ,Defaults::Save::autoSyntaxCheck);
    settings.saveObsolete = _config->readBoolEntry("SaveObsolete"
            ,Defaults::Save::saveObsolete);
    settings.customDateFormat = _config->readEntry("CustomDateFormat"
	    ,Defaults::Save::customDateFormat());
    settings.dateFormat = (Qt::DateFormat)( _config->readNumEntry("DateFormat"
	    ,(int)Defaults::Save::dateFormat) );
    settings.updateDescription = _config->readBoolEntry("UpdateDescription"
	    ,Defaults::Save::updateDescription);
    settings.descriptionString = _config->readEntry("DescriptionString" 
	    ,Defaults::Save::descriptionString());
    settings.updateTranslatorCopyright = _config->readBoolEntry("UpdateTranslatorCopyright"
            ,Defaults::Save::updateTranslatorCopyright);
    settings.FSFCopyright=(CopyrightUpdate)(_config->readNumEntry("FSFCopyright"
            ,(int)Defaults::Save::FSFCopyright));

    settings.autoSaveDelay=_config->readNumEntry("AutoSaveDelay"
	    , Defaults::Save::autoSaveDelay);
    
    return settings;
}


MiscSettings Project::miscSettings ()
{
    KConfigGroupSaver saver(_config,"Misc");
    
    MiscSettings settings;

    Defaults::Misc defaultMisc;
    QString temp=_config->readEntry("AccelMarker"
                                   ,defaultMisc.accelMarker());
    if(temp.length() > 0)
    {
	settings.accelMarker=temp[0];
    }
    
    temp = _config->readEntry("ContextInfo"
	    ,defaultMisc.contextInfo().pattern());

    settings.contextInfo.setPattern(temp);

    temp = _config->readEntry("SingularPlural"
	    ,defaultMisc.singularPlural().pattern());
    settings.singularPlural.setPattern(temp);

    settings.useBzip = _config->readBoolEntry("BZipCompression", defaultMisc.useBzip);
    settings.compressSingleFile = _config->readBoolEntry("CompressSingleFile", defaultMisc.compressSingleFile);
    
    return settings;
}

SpellcheckSettings Project::spellcheckSettings ()
{
    KConfigGroupSaver gs(_config,"Spellcheck");
    
    SpellcheckSettings settings;
    
    KSpellConfig spellConf(0,"spellconfig");

    settings.noRootAffix=_config->readBoolEntry("NoRootAffix"
	    ,spellConf.noRootAffix());
    settings.runTogether=_config->readBoolEntry("RunTogether"
	    ,spellConf.runTogether());
    settings.spellEncoding=_config->readNumEntry("SpellEncoding"
            ,spellConf.encoding());
    settings.spellClient=_config->readNumEntry("SpellClient"
            ,spellConf.client());
    settings.spellDict=_config->readEntry("SpellDictionary"
            ,spellConf.dictionary());
    settings.rememberIgnored=_config->readBoolEntry("RememberIgnored"
            ,KBabel::Defaults::Spellcheck::rememberIgnored);
    settings.ignoreURL=_config->readPathEntry("IgnoreURL"
            ,KBabel::Defaults::Spellcheck::ignoreURL());
    settings.onFlySpellcheck=_config->readBoolEntry("OnFlySpellCheck"
            ,KBabel::Defaults::Spellcheck::onFlySpellcheck);

    settings.valid=true;
    
    return settings;
}

SourceContextSettings Project::sourceContextSettings ()
{
    SourceContextSettings settings;
    
    KConfigGroupSaver saver(_config,"SourceContext");

    settings.codeRoot=_config->readPathEntry("CodeRoot", KBabel::Defaults::SourceContextSettings::codeRoot());
    if( _config->hasKey("Paths") )
        settings.sourcePaths=_config->readPathListEntry("Paths");
    else
        settings.sourcePaths=KBabel::Defaults::SourceContextSettings::sourcePaths();
	
    return settings;
}

CatManSettings Project::catManSettings ()
{
    CatManSettings _settings;
    
    KConfigGroupSaver cs(_config,"CatalogManager");

    _settings.poBaseDir=_config->readPathEntry("PoBaseDir"
               ,KBabel::Defaults::CatalogManager::poBaseDir());
    _settings.potBaseDir=_config->readPathEntry("PotBaseDir"
               ,KBabel::Defaults::CatalogManager::potBaseDir());
    _settings.openWindow=_config->readBoolEntry("OpenWindow"
               ,KBabel::Defaults::CatalogManager::openWindow);

    _settings.killCmdOnExit=_config->readBoolEntry("KillCmdOnExit"
               ,KBabel::Defaults::CatalogManager::killCmdOnExit);
    _settings.indexWords=_config->readBoolEntry("IndexWords"
               ,KBabel::Defaults::CatalogManager::indexWords);


    if(_config->hasKey("DirCommands"))
    {
          _settings.dirCommands = _config->readListEntry("DirCommands");
          _settings.dirCommandNames =
              _config->readListEntry("DirCommandNames");
    }
    else
    {
          _settings.dirCommands = KBabel::Defaults::CatalogManager::dirCommands();
          _settings.dirCommandNames =
              KBabel::Defaults::CatalogManager::dirCommandNames();
    }
    if(_config->hasKey("FileCommands"))
    {
          _settings.fileCommands = _config->readListEntry("FileCommands");
          _settings.fileCommandNames =
              _config->readListEntry("FileCommandNames");
    }
    else
    {
          _settings.fileCommands=KBabel::Defaults::CatalogManager::fileCommands();
          _settings.fileCommandNames =
              KBabel::Defaults::CatalogManager::fileCommandNames();
    }

    _settings.ignoreURL=_config->readPathEntry("ValidationIgnoreURL"
        , KBabel::Defaults::CatalogManager::ignoreURL());

    _settings.flagColumn=_config->readBoolEntry("ShowFlagColumn"
        , KBabel::Defaults::CatalogManager::flagColumn);
    _settings.fuzzyColumn=_config->readBoolEntry("ShowFuzzyColumn"
        , KBabel::Defaults::CatalogManager::fuzzyColumn);
    _settings.untranslatedColumn=_config->readBoolEntry("ShowUntranslatedColumn"
        , KBabel::Defaults::CatalogManager::untranslatedColumn);
    _settings.totalColumn=_config->readBoolEntry("ShoTotalColumn"
        , KBabel::Defaults::CatalogManager::totalColumn);
    _settings.cvsColumn=_config->readBoolEntry("ShowCVSColumn"
        , KBabel::Defaults::CatalogManager::cvsColumn);
    _settings.revisionColumn=_config->readBoolEntry("ShowRevisionColumn"
        , KBabel::Defaults::CatalogManager::revisionColumn);
    _settings.translatorColumn=_config->readBoolEntry("ShowTranslatorColumn"
        , KBabel::Defaults::CatalogManager::translatorColumn);

    return _settings;
}
	    
void Project::setSettings (IdentitySettings settings)
{    
    {
	KConfigGroupSaver groupSaver(_config,"Header");
    
	_config->writeEntry("Author-Name", settings.authorName);
	_config->writeEntry("Local-Author-Name", settings.authorLocalizedName);
	_config->writeEntry("Author-Email", settings.authorEmail);
	_config->writeEntry("Language", settings.languageName);
	_config->writeEntry("LanguageCode", settings.languageCode);
	_config->writeEntry("Mailinglist", settings.mailingList);
	_config->writeEntry("Timezone", settings.timeZone);
    }

    {
	KConfigGroupSaver (_config, "Misc");

	_config->writeEntry("PluralForms", settings.numberOfPluralForms);
	_config->writeEntry("CheckPluralArgument", settings.checkPluralArgument);
	_config->writeEntry("PluralFormsHeader", settings.gnuPluralFormHeader);
    }
	
    _config->sync();
    
    emit signalIdentitySettingsChanged();
    emit signalSettingsChanged();
}

void Project::setSettings (SaveSettings settings)
{
    KConfigGroupSaver groupSaver(_config,"Header");

    _config->writeEntry("AutoUpdate", settings.autoUpdate);
    _config->writeEntry("Update-Last-Translator", settings.updateLastTranslator);
    _config->writeEntry("Update-Revision-Date", settings.updateRevisionDate);
    _config->writeEntry("Update-Language-Team", settings.updateLanguageTeam);
    _config->writeEntry("Update-Charset", settings.updateCharset);
    _config->writeEntry("Update-Encoding", settings.updateEncoding);
    _config->writeEntry("Encoding", settings.encoding);
    _config->writeEntry("UseOldEncoding", settings.useOldEncoding);

    _config->writeEntry("Update-Project", settings.updateProject);
    _config->writeEntry("ProjectString", settings.projectString);

    _config->writeEntry("AutoSyntaxCheck", settings.autoSyntaxCheck);
    _config->writeEntry("SaveObsolete", settings.saveObsolete);
    _config->writeEntry("CustomDateFormat", settings.customDateFormat);
    _config->writeEntry("DateFormat", settings.dateFormat);
    _config->writeEntry("UpdateDescription", settings.updateDescription);
    _config->writeEntry("DescriptionString" , settings.descriptionString);
    _config->writeEntry("UpdateTranslatorCopyright", settings.updateTranslatorCopyright);
    _config->writeEntry("FSFCopyright", settings.FSFCopyright);

    _config->writeEntry("AutoSaveDelay", settings.autoSaveDelay);

    _config->sync();    
    
    emit signalSaveSettingsChanged();
    emit signalSettingsChanged();
}


void Project::setSettings (MiscSettings settings)
{
    KConfigGroupSaver saver(_config,"Misc");
    
    _config->writeEntry("AccelMarker", QString(settings.accelMarker));
    _config->writeEntry("ContextInfo", settings.contextInfo.pattern());
    _config->writeEntry("SingularPlural", settings.singularPlural.pattern());
    _config->writeEntry("BZipCompression", settings.useBzip );
    _config->writeEntry("CompressSingleFile", settings.compressSingleFile );
    
    _config->sync();
    
    emit signalMiscSettingsChanged();
    emit signalSettingsChanged();
}

void Project::setSettings (SpellcheckSettings settings)
{
    KConfigGroupSaver gs(_config,"Spellcheck");
    
    _config->writeEntry("NoRootAffix", settings.noRootAffix);
    _config->writeEntry("RunTogether", settings.runTogether);
    _config->writeEntry("SpellEncoding", settings.spellEncoding);
    _config->writeEntry("SpellClient", settings.spellClient);
    _config->writeEntry("SpellDictionary", settings.spellDict);
    _config->writeEntry("RememberIgnored", settings.rememberIgnored);
    _config->writeEntry("IgnoreURL", settings.ignoreURL);
    _config->writeEntry("OnFlySpellCheck", settings.onFlySpellcheck);
    
    _config->sync();
    
    emit signalSpellcheckSettingsChanged();
    emit signalSettingsChanged();
}

void Project::setSettings (SourceContextSettings _settings)
{
    KConfigGroupSaver saver(_config,"SourceContext");

    _config->writePathEntry("CodeRoot", _settings.codeRoot);
    _config->writePathEntry("Paths", _settings.sourcePaths);

    _config->sync();
    
    emit signalSourceContextSettingsChanged();
    emit signalSettingsChanged();
}

void Project::setSettings (CatManSettings _settings)
{
    KConfigGroupSaver cs(_config,"CatalogManager");

    _config->writePathEntry("PoBaseDir", _settings.poBaseDir);
    _config->writePathEntry("PotBaseDir", _settings.potBaseDir);
    _config->writeEntry("OpenWindow", _settings.openWindow);

    _config->writeEntry("KillCmdOnExit", _settings.killCmdOnExit);
    _config->writeEntry("IndexWords", _settings.indexWords);

    _config->writeEntry("DirCommands", _settings.dirCommands );
    _config->writeEntry("DirCommandNames", _settings.dirCommandNames );

    _config->writeEntry("FileCommands", _settings.fileCommands );
    _config->writeEntry("FileCommandNames", _settings.fileCommandNames );

    _config->writePathEntry("ValidationIgnoreURL", _settings.ignoreURL);

    _config->writeEntry("ShowFlagColumn", _settings.flagColumn);
    _config->writeEntry("ShowFuzzyColumn", _settings.fuzzyColumn);
    _config->writeEntry("ShowUntranslatedColumn", _settings.untranslatedColumn);
    _config->writeEntry("ShoTotalColumn", _settings.totalColumn);
    _config->writeEntry("ShowCVSColumn", _settings.cvsColumn);
    _config->writeEntry("ShowRevisionColumn", _settings.revisionColumn);
    _config->writeEntry("ShowTranslatorColumn", _settings.translatorColumn);
	
    _config->sync();
    
    emit signalCatManSettingsChanged();
    emit signalSettingsChanged();
}
    
}

#include "kbproject.moc"
