/**************************************************************************
 *                                                                        *
 *  KDoomsDay - Copyright (C) 2004-2005 Helen Faulkner                    *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   * 
 *                                                                        *
 *                                                                        *              
 * Parts of this code are based on the KDE clock applet, which has the    *
 * following license:                                                     *
 *                                                                        *
 *                                                                        *    
 *  Copyright (c) 1996-2002 the kicker authors. See file KICKER-AUTHORS.  *
 *                                                                        *    
 *  Permission is hereby granted, free of charge, to any person obtaining *
 * a copy of this software and associated documentation files (the        *
 * "Software"), to deal in the Software without restriction, including    *
 * without limitation the rights to use, copy, modify, merge, publish,    * 
 * distribute, sublicense, and/or sell copies of the Software, and to     *
 * permit persons to whom the Software is furnished to do so, subject to  *
 * the following conditions:                                              *
 *                                                                        *
 * The above copyright notice and this permission notice shall be         *
 * included in all copies or substantial portions of the Software.        *
 *                                                                        * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,        *
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF     *
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. * 
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR      *
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,  *
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR  *
 * OTHER DEALINGS IN THE SOFTWARE.                                        *
 *                                                                        * 
 **************************************************************************/


#include <qtooltip.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kmessagebox.h>
#include <kglobalsettings.h>
#include <kdatetimewidget.h>
#include <kfontrequester.h>
#include <kcolorcombo.h>
#include <kaboutdata.h>
#include <kaboutapplication.h>

#include <qtimer.h>
#include <qlineedit.h>
#include <qbuttongroup.h>

#include "kdoomsday.h"


kdoomsday::kdoomsday(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name)
{
    setBackgroundOrigin( QWidget::AncestorOrigin ); 
    QToolTip::setGloballyEnabled ( true );  
    
    config = new KConfig(configFile);
    counter = new CounterWidget(this);
    
    if (config->readBoolEntry("New_Widget", true)) {
        config->writeEntry("New_Widget", false);
        config->sync();
        preferences();        
    }      
    
    counter->show();   
    
    // this needed to force an immediate background update  
    QTimer::singleShot(1, this, SLOT(slotUpdate()) );   
    emit(updateLayout());            

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), SLOT(slotUpdate()));
    timer->start(500); 

}


kdoomsday::~kdoomsday()
{
}


void kdoomsday::about()
{
    KAboutData about("KDoomsDay", "Countdown Applet", "0.2",
                     "KDoomsDay", 
                     KAboutData::License_GPL_V2, "(C) 2004-2005 Helen Faulkner",
                     "A Countdown Applet for kicker");

    about. addAuthor("Helen Faulkner", "Developer", "projects@thousand-ships.com");
    about. addCredit("Ben Burton", "for helping with stuff in general!", "bab@debian.org");

    KAboutApplication a(&about, this);
    a.exec();
}


void kdoomsday::preferences()
{
    if (counter->mySettings->exec()==QDialog::Accepted) {
        counter->writeConfig(); 
        counter->loadSettings();
        // this to force an immediate background update
        QTimer::singleShot(1, this, SLOT(slotUpdate()) );    
        
        counter->updateCounter();
        counter->updateToolTip();
         
        emit(updateLayout()); 
    } 
}



// essentially from kicker clockapplet
int kdoomsday::widthForHeight(int height) const
{
    bool applyGeometry = (position() == pBottom) || (position() == pTop);

    int width = counter->preferedWidthForHeight(height);
    
    if (applyGeometry) {
        counter->setFixedSize(width, height);
        counter->setMinimumSize(width,height);
    }
    return width;
}

// essentially from kicker clockapplet
int kdoomsday::heightForWidth(int width) const
{
    bool applyGeometry = (position() == pLeft) || (position() == pRight);
    int height = counter->preferedHeightForWidth(width);

    if (applyGeometry) {
        counter->setFixedSize(width, height);
    }

    return height;
}



void kdoomsday::slotUpdate()
{
  counter->updateCounter();
  adjustSize();
}

//************************************************************


CounterWidget::CounterWidget(kdoomsday *appletIn, const char *name)
    : QLabel(appletIn, name)
{

    // default values for settings
    defaultDateTime = QDateTime( QDate( 2005, 12, 31 ), QTime( 12, 0, 0 ) );
    defaultFont = KGlobalSettings::generalFont();
    defaultFontColor = QColor(0, 0, 0);   
    defaultEventName = QString("My Event");
    defaultUnit = 1;
    
    applet = appletIn;

    setBackgroundOrigin( QWidget::AncestorOrigin );       
    
    mySettings = new Settings(this , "Settings", true, 0); 
    loadSettings();
    updateCounter();
}


int CounterWidget::preferedWidthForHeight(int height) const
{
    QString maxLengthTime = QString::number(timeLeft)+ "-";
    return fontMetrics().width( maxLengthTime );
}


int CounterWidget::preferedHeightForWidth(int width) const
{
    return fontMetrics().lineSpacing();
}


void CounterWidget::updateCounter()
{      

    bool wasNegative;
    
    wasNegative = false;

    if (timeLeft < 0) wasNegative = true;
    
    // calculate amount of time left
    switch (applet->config->readNumEntry("Count_unit", 1 )) {

        case 0:     // weeks
            timeLeft = (QDateTime::currentDateTime()).daysTo(applet->config->readDateTimeEntry("Date_time", &defaultDateTime));
            timeLeft = floor (timeLeft /7);
            break;  
            
        case 1:     // days
            timeLeft = (QDateTime::currentDateTime()).daysTo(applet->config->readDateTimeEntry("Date_time", &defaultDateTime));
            break;  
            
        case 2:     // hours
            timeLeft = (QDateTime::currentDateTime()).secsTo(applet->config->readDateTimeEntry("Date_time", &defaultDateTime));    
            timeLeft = floor (timeLeft /3600.0);
            break;          
        
        case 3:     // minutes
            timeLeft = (QDateTime::currentDateTime()).secsTo(applet->config->readDateTimeEntry("Date_time", &defaultDateTime));
            timeLeft = floor(timeLeft /60.0);
            break;
    
        default:    // same as days
            timeLeft = (QDateTime::currentDateTime()).daysTo(applet->config->readDateTimeEntry("Date_time", &defaultDateTime));
            break;
            
    }
    
    // set text of counter
    setText(QString::number(timeLeft));
    adjustSize();
    applet->adjustSize(); 
    
    // set background to be like applet background
    const QPixmap* brush = applet->paletteBackgroundPixmap();
    
    if (brush)
    {
        setPaletteBackgroundPixmap(*brush);
    }   
    else {
        setPaletteBackgroundPixmap(QPixmap());
    }         
    
    if (wasNegative && (timeLeft > 0 )) {
        updateToolTip();
    }
    
}

void CounterWidget::loadSettings()
{

    readConfig();

    // set general things
    setFrameStyle(Panel | (true ? Sunken : 0));
    setAlignment(AlignVCenter | AlignHCenter | SingleLine);

    setFont(applet->config->readFontEntry("Font", &defaultFont));

    setPaletteForegroundColor ( applet->config->readColorEntry("Font_colour",  &defaultFontColor) );
    

    
    switch (applet->config->readNumEntry("Count_unit", defaultUnit ) ) {

        case 0:         // weeks
                        unitString = "Weeks ";
                        break;  
            
        case 1:         // days
                        unitString = "Days ";
                        break;  
            
        case 2:         // hours
                        unitString = "Hours ";
                        break;          
        
        case 3:         // minutes
                        unitString = "Minutes ";
                        break;
    
        default:        // same as days
                        unitString = "Days ";
                        break;   
    }   
    
    
    updateToolTip();
    
    adjustSize();
    applet->adjustSize();
    update();
}


void CounterWidget::updateToolTip(){

    QString prefixString;
    
    
    if (timeLeft < 0) {
        prefixString = unitString + "since ";
    } 
    else {
        prefixString = unitString + "until ";
    }   

    // set the new tooltip
    QToolTip::remove(this);     
    
    QToolTip::add( this, prefixString + applet->config->readEntry("Event_name", defaultEventName));

}


void CounterWidget::readConfig(){
 
    // read in settings
    mySettings->EventName->setText(applet->config->readEntry("Event_name", defaultEventName) );
    mySettings->Count_Style->setProperty( "selectedId", applet->config->readNumEntry("Count_unit", defaultUnit ) );
    mySettings->dateTimeWidget->setDateTime(applet->config->readDateTimeEntry("Date_time", &defaultDateTime) );
    mySettings->fontRequester->setFont(applet->config->readFontEntry("Font", &defaultFont) );
    mySettings->fontColour->setColor(applet->config->readColorEntry("Font_colour",  &defaultFontColor) );
}

    
void CounterWidget::writeConfig(){

    // write settings to config file
    applet->config->writeEntry("Event_name", mySettings->EventName->text() );
    applet->config->writeEntry("Count_unit", mySettings->Count_Style->selectedId() );
    applet->config->writeEntry("Date_time", mySettings->dateTimeWidget->dateTime() );
    applet->config->writeEntry("Font", mySettings->fontRequester->font() );
    applet->config->writeEntry("Font_colour", mySettings->fontColour->color() );
    
    applet->config->sync();

}


//************************************************************



extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile)
    {
    
        KGlobal::locale()->insertCatalogue("KDoomsDay");
        return new kdoomsday(configFile, KPanelApplet::Normal,
                             KPanelApplet::About | KPanelApplet::Preferences,
                             parent, "KDoomsDay");
    }
}

#include "kdoomsday.moc"
