/*
* Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "preferences.h"
#include "kino_common.h"

// #include <string.h>
#include <libgnome/gnome-config.h>

#include <string>
using std::string;
using std::make_pair;
#include <sstream>
#include <iostream>

/** The global prefs object.
    
    It is accessible to other classes via the getInstance method.
*/

Preferences *Preferences::instance = NULL;

/** Singleton getInstance method. Standard pattern followed, non threadsafe
	but not required (at the moment) since it gets called before any threading
	is activated. Keep an eye on it though.
 
	\returns the single instance of Preferences.
*/

Preferences &Preferences::getInstance()
{
	if ( instance == NULL )
		instance = new Preferences;
	return *instance;
}

/** The constructor loads the user preferences.
 
    Since this uses gnome-config, preferences are typically
    loaded from ~/.gnome/kino.
*/
Preferences::Preferences()
{
	gchar key[ 256 ];
	gchar *tempString;

	gnome_config_push_prefix( "/kino/general/" );

	tempString = gnome_config_get_string( "file=capture" );
	if ( tempString )
	{
		strncpy( file, tempString, 511 );
		g_free( tempString );
	}
	defaultNormalisation = gnome_config_get_int( "defaultNormalisation=0" );
	defaultAudio = gnome_config_get_int( "defaultAudio=0" );
	defaultAspect = gnome_config_get_int( "defaultAspect=0" );
	g_snprintf( key, 255, "fileFormat=%d", RAW_FORMAT );
	fileFormat = gnome_config_get_int( key );
	autoSplit = gnome_config_get_bool( "autoSplit=true" );
	timeStamp = gnome_config_get_bool( "timeStamp=false" );
	frames = gnome_config_get_int( "frames=7000" );
	every = gnome_config_get_int( "every=1" );
	interface = gnome_config_get_int( "interface=0" );
	channel = gnome_config_get_int( "channel=63" );
	phyID = -1;
	tempString = gnome_config_get_string( "avcGUID=" );
	if ( tempString )
	{
		strncpy( avcGUID, tempString, 64 );
		g_free( tempString );
	}
	g_snprintf( key, 255, "displayMode=%d", DISPLAY_XV );
	displayMode = gnome_config_get_int( key );
	displayQuality = gnome_config_get_float( "displayQuality=2" );
	displayFixed = gnome_config_get_bool( "displayFixed=false" );
	enableAudio = gnome_config_get_bool( "enableAudio=true" );
	cip_n = gnome_config_get_int( "cip_increment=0" );
	cip_d = gnome_config_get_int( "cip_threshold=0" );
	syt_offset = gnome_config_get_int( "syt_offset=0" );
	preview_capture = gnome_config_get_bool( "preview_capture=false" );
	dropFrame = gnome_config_get_bool( "dropFrame=false" );
	tempString = gnome_config_get_string( "audioDevice=/dev/dsp" );
	if ( tempString )
	{
		strncpy( audioDevice, tempString, 511 );
		g_free( tempString );
	}

	/// Jogshuttle stuff
	enableJogShuttle = gnome_config_get_bool( "enableJogShuttle=false" );
	_isGrabbing = false;
	/* Fill array of actions */
	/* Element 0 is empty */
	int count = 0; /** Ease moving code around */
	/* "<none>" should always go first */
	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "<none>" ),
	                       "",
	                       "" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++,
	                       _( "Toggle play" ),
	                       _( "Toggle between play and pause" ),
	                       " " ) );

	/* --- NAVIGATION --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Forward one frame" ),
	                       _( "Move one frame forward" ),
	                       "l" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Back one frame" ),
	                       _( "Move one frame backward" ),
	                       "h" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Forward one second" ),
	                       _( "Move one second forward" ),
	                       "w" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Forward 30 seconds" ),
	                       _( "Move 30 seconds forward" ),
	                       "30w" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Back one second" ),
	                       _( "Move one second backward" ),
	                       "b" ) );


	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Back 30 seconds" ),
	                       _( "Move 30 seconds backward" ),
	                       "30b" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Start of scene" ),
	                       _( "Move to start of current scene" ),
	                       "0" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "End of scene" ),
	                       _( "Move to end of current scene" ),
	                       "$" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Previous scene" ),
	                       _( "Move to start of previous scene" ),
	                       "k" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Next scene" ),
	                       _( "Move to start of next scene" ),
	                       "j" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "First frame" ),
	                       _( "Move to beginning of movie" ),
	                       "gg" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Last frame" ),
	                       _( "Move to end of movie" ),
	                       "G" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Forward 5 scenes" ),
	                       _( "Move forward 5 scenes" ),
	                       "5$" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Back 5 scene" ),
	                       _( "Move backward 5 scenes" ),
	                       "5^" ) );


	/* --- CUT --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut current frame" ),
	                       _( "Cut current frame" ),
	                       "x" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut a second" ),
	                       _( "Cut a second" ),
	                       "dw" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut current scene" ),
	                       _( "Cut current scene" ),
	                       "dd" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut to end of current scene" ),
	                       _( "Cut to end of current scene" ),
	                       "d$" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut to end of movie" ),
	                       _( "Cut to end of movie" ),
	                       "dG" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut from start of current scene" ),
	                       _( "Cut from start of current scene" ),
	                       "d0" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Cut from start of movie" ),
	                       _( "Cut from start of movie" ),
	                       "dgg" ) );

	/* --- COPY --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Copy current scene" ),
	                       _( "Copy current scene" ),
	                       "yy" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Copy to end of scene" ),
	                       _( "Copy to end of scene" ),
	                       "y$" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Copy from start of scene" ),
	                       _( "Cut from start of scene" ),
	                       "y0" ) );

	/* --- PASTE --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Paste after current frame" ),
	                       _( "Paste after current frame" ),
	                       "p" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Paste before current frame" ),
	                       _( "Paste before current frame" ),
	                       "P" ) );

	/* --- MODE SWITCHING --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to Edit" ),
	                       _( "Switch to the edit interface" ),
	                       "F2" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to Capture" ),
	                       _( "Switch to the capture interface" ),
	                       "A" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to Timeline" ),
	                       _( "Switch to the capture interface" ),
	                       "v" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to Export" ),
	                       _( "Switch to the capture interface" ),
	                       ":W" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to FX" ),
	                       _( "Switch to the capture interface" ),
	                       "C" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Switch to Trim" ),
	                       _( "Switch to trim mode" ),
	                       "t" ) );
	/* --- GENERAL --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Repeat last command" ),
	                       _( "Repeat last command" ),
	                       "." ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Split scene" ),
	                       _( "Split scene before current frame" ),
	                       "Ctrl+J" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Join scenes" ),
	                       _( "Join this scene with the following" ),
	                       "J" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Undo" ),
	                       _( "Undo last command" ),
	                       "u" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Redo" ),
	                       _( "Redo last command" ),
	                       "Ctrl+R" ) );
						   
	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Capture/Overwrite/Export/Render" ),
	                       _( "Start Capture, Export, or Render or Apply Overwrite" ),
	                       "Enter" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Stop" ),
	                       _( "Stop transport, capture, export, or render" ),
	                       "Esc" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Publish SMIL" ),
	                       _( "Publish SMIL" ),
	                       "Ctrl+W" ) );

	/* --- TRIM --- */

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Set In" ),
	                       _( "Set the In Point to current position" ),
	                       "i" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Set Out" ),
	                       _( "Set the Out Point to current position" ),
	                       "o" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Insert Before" ),
	                       _( "Insert Before" ),
	                       ":r" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Insert After" ),
	                       _( "Insert After" ),
	                       ":a" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Toggle Insert/Overwrite" ),
	                       _( "Toggle Insert/Overwrite" ),
	                       "Ins" ) );
	                       
	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Toggle Looping" ),
	                       _( "Toggle Looping" ),
	                       "\\" ) );

	_JogShuttleActions.push_back(
	    JogShuttleActions( count++, _( "Toggle Link" ),
	                       _( "Toggle Link" ),
	                       "=" ) );

	// TODO: Add more actions?


	/* Actually get the preferences */
	count = gnome_config_get_int( "jogShuttleNumMappings=0" );
	if ( 0 != count )
	{
		for ( int i = 0; i < count ; i++ )
		{
			char key[ 512 ];
			sprintf( key, "jogShuttleMappingFirstButton%i=0", i );
			unsigned int first_code
			= gnome_config_get_int( key );
			sprintf( key, "jogShuttleMappingSecondButton%i=0", i );
			unsigned int second_code
			= gnome_config_get_int( key );
			sprintf( key, "jogShuttleMappingAction%i=", i );
			// g_print ( " Key = %s\n", key );
			tempString =
			    gnome_config_get_string( key );

			// TODO: Free
			// g_print( "tmpstring Read action %s\n", tempString );
			string action = string( tempString );
			// g_print( "Read action %s\n", action.c_str() );
			/* Insert */
			_JogShuttleMappings[ make_pair ( first_code, second_code ) ]
			= JogShuttleMapping( action );
		}
	}
	else
	{
		// Set up defaults manuall, if 0 count
		// These are my own default mappings
		_JogShuttleMappings[ make_pair ( 0x100, 0x0 ) ]
		= JogShuttleMapping( "i" );
		_JogShuttleMappings[ make_pair ( 0x101, 0x0 ) ]
		= JogShuttleMapping( "o" );
		_JogShuttleMappings[ make_pair ( 0x102, 0x0 ) ]
		= JogShuttleMapping( "h" );
		_JogShuttleMappings[ make_pair ( 0x103, 0x0 ) ]
		= JogShuttleMapping( "l" );
		
		_JogShuttleMappings[ make_pair ( 0x113, 0x0 ) ]
		= JogShuttleMapping( "b" );
		_JogShuttleMappings[ make_pair ( 0x111, 0x0 ) ]
		= JogShuttleMapping( "w" );
		_JogShuttleMappings[ make_pair ( 0x110, 0x0 ) ]
		= JogShuttleMapping( " " );
		_JogShuttleMappings[ make_pair ( 0x112, 0x0 ) ]
		= JogShuttleMapping( "k" );
		_JogShuttleMappings[ make_pair ( 0x114, 0x0 ) ]
		= JogShuttleMapping( "j" );
		_JogShuttleMappings[ make_pair ( 0x115, 0x0 ) ]
		= JogShuttleMapping( "gg" );
		_JogShuttleMappings[ make_pair ( 0x116, 0x0 ) ]
		= JogShuttleMapping( "G" );
		
		_JogShuttleMappings[ make_pair ( 0x10, 0x0 ) ]
		= JogShuttleMapping( " " );
	}




	enableV4L = gnome_config_get_bool( "enableV4L=false" );
	disableKeyRepeat = gnome_config_get_bool( "disableKeyRepeat=false" );
	audioRendering = gnome_config_get_int( "audioRendering=0" );
	previewRendering = gnome_config_get_int( "previewRendering=1" );
	maxUndos = gnome_config_get_int( "maxUndos=50" );
	//tempString = gnome_config_get_string("video1394Device=/dev/video1394");
	//if (tempString) {
	//    strncpy( video1394Device, tempString, 511);
	//    g_free(tempString);
	//}
	dvCaptureBuffers = gnome_config_get_int( "dvCaptureBuffers=50" );
	dvExportBuffers = gnome_config_get_int( "dvExportBuffers=10" );
	dvDecoderClampLuma = gnome_config_get_bool( "dvDecoderClampLuma=false" );
	dvDecoderClampChroma = gnome_config_get_bool( "dvDecoderClampChroma=false" );
	maxFileSize = gnome_config_get_int( "maxFileSize=2000" );
	audioScrub = gnome_config_get_bool( "audioScrub=false" );
	tempString = gnome_config_get_string( "v4lVideoDevice=/dev/video0" );
	if ( tempString )
	{
		strncpy( v4lVideoDevice, tempString, 511 );
		g_free( tempString );
	}
	tempString = gnome_config_get_string( "v4lAudioDevice=/dev/dsp" );
	if ( tempString )
	{
		strncpy( v4lAudioDevice, tempString, 511 );
		g_free( tempString );
	}
	tempString = gnome_config_get_string( "v4lInput=PAL" );
	if ( tempString )
	{
		strncpy( v4lInput, tempString, 31 );
		g_free( tempString );
	}
	tempString = gnome_config_get_string( "v4lAudio=32000" );
	if ( tempString )
	{
		strncpy( v4lAudio, tempString, 31 );
		g_free( tempString );
	}
	isOpenDML = gnome_config_get_bool( "isOpenDML=false" );
	tempString = gnome_config_get_string( "defaultDirectory=~/" );
	if ( tempString )
	{
		strncpy( defaultDirectory, tempString, 511 );
		g_free( tempString );
	}
	displayExtract = gnome_config_get_int( "displayExtract=0" );
	relativeSave = gnome_config_get_bool( "relativeSave=false" );

	dvCaptureDriver = gnome_config_get_int( "dvCaptureDriver=0" );
	tempString = gnome_config_get_string( "dvCaptureDevice=/dev/raw1394" );
	if ( tempString )
	{
		strncpy( dvCaptureDevice, tempString, 511 );
		g_free( tempString );
	}
	dv1394Preview = gnome_config_get_bool( "dv1394Preview=false" );
	tempString = gnome_config_get_string( "dvExportDevice=/dev/ieee1394/dv/host0/PAL/out" );
	if ( tempString )
	{
		strncpy( dvExportDevice, tempString, 511 );
		g_free( tempString );
	}
	avcPollIntervalMs = gnome_config_get_int( "avcPollIntervalMs=200" );
	dvExportPrerollSec = gnome_config_get_int( "dvExportPrerollSec=4" );
	dvTwoPassEncoder = gnome_config_get_bool( "dvTwoPassEncoder=false" );
	windowWidth = gnome_config_get_int( "windowWidth=-1" );
	windowHeight = gnome_config_get_int( "windowHeight=-1" );
	storyboardPosition = gnome_config_get_int( "storyboardPosition=-1" );
	previewSize = gnome_config_get_float( "previewSize=0.0" );
	timeFormat = gnome_config_get_int( "timeFormat=1" );
	
	tempString = gnome_config_get_string( "metaNames=*title,author,copyright,id,abstract" );
	if ( tempString )
	{
		strncpy( metaNames, tempString, 511 );
		g_free( tempString );
	}
	
	vector< string > metaNamesList;
	vector< string >::iterator metaNamesIter;

	// build a metaValues key for each metaName
	// metaValues is a range of values to pick from
	// metaValue * = open-ended, values outside range permitted, can be used alongside values
	string_utils::split( metaNames, ",", metaNamesList );
	for ( metaNamesIter = metaNamesList.begin(); metaNamesIter != metaNamesList.end(); ++metaNamesIter )
	{
		std::ostringstream str;
		std::string metaname = *metaNamesIter;

		if ( metaname.at( 0 ) == '*' )
			metaname.erase( 0, 1 );

		str << "metaValues_" << metaname << "=*";
		tempString = gnome_config_get_string( str.str().c_str() );
		if ( tempString )
		{
			vector< string > metaValuesSimpleList;
			vector< string >::iterator iter;
			vector< pair< string, string > > metaValuesList;

			string_utils::split( tempString, ",", metaValuesSimpleList );
			for ( iter = metaValuesSimpleList.begin(); iter != metaValuesSimpleList.end(); ++iter )
			{
				vector< string > labelValueVector;
				pair< string, string > labelValuePair;
				
				string_utils::split( *iter, "=", labelValueVector );
				if ( labelValueVector.size() > 1 )
					labelValuePair = make_pair( labelValueVector[0], labelValueVector[1] );
				else
					labelValuePair = make_pair( *iter, *iter );
				metaValuesList.push_back( labelValuePair );
			}
			metaValues.insert( make_pair( metaname, metaValuesList ) );
			g_free( tempString );
		}
	}

	enablePublish = gnome_config_get_bool( "enablePublish=false" );
	expandStoryboard = gnome_config_get_bool( "expandStoryboard=false" );
	tempString = gnome_config_get_string( "newProjectURI=" );
	if ( tempString )
	{
		strncpy( newProjectURI, tempString, 511 );
		g_free( tempString );
	}
	tempString = gnome_config_get_string( "newProjectXPath=" );
	if ( tempString )
	{
		strncpy( newProjectXPath, tempString, 511 );
		g_free( tempString );
	}

	trimModeInsert = gnome_config_get_bool( "trimModeInsert=false" );
	
	gnome_config_pop_prefix();
}

/** Save the user preferences.
*/

/// Save the user preferences.
void
Preferences::Save()
{
	gnome_config_push_prefix( "/kino/general/" );

	gnome_config_set_int( "defaultNormalisation", defaultNormalisation );
	gnome_config_set_int( "defaultAudio", defaultAudio );
	gnome_config_set_int( "defaultAspect", defaultAspect );
	gnome_config_set_string( "file", file );
	gnome_config_set_int( "fileFormat", fileFormat );
	gnome_config_set_bool( "autoSplit", autoSplit );
	gnome_config_set_bool( "timeStamp", timeStamp );
	gnome_config_set_int( "frames", frames );
	gnome_config_set_int( "every", every );
	gnome_config_set_int( "interface", interface );
	gnome_config_set_int( "channel", channel );
	gnome_config_set_string( "avcGUID", avcGUID );
	gnome_config_set_int( "displayMode", displayMode );
	gnome_config_set_float( "displayQuality", displayQuality );
	gnome_config_set_bool( "displayFixed", displayFixed );
	gnome_config_set_bool( "enableAudio", enableAudio );
	gnome_config_set_int( "cip_increment", cip_n );
	gnome_config_set_int( "cip_threshold", cip_d );
	gnome_config_set_int( "syt_offset", syt_offset );
	gnome_config_set_bool( "preview_capture", preview_capture );
	gnome_config_set_bool( "dropFrame", dropFrame );
	gnome_config_set_string( "audioDevice", audioDevice );

	/* JogShuttle Stuff */
	gnome_config_set_bool( "enableJogShuttle", enableJogShuttle );
	gnome_config_set_int( "jogShuttleNumMappings", _JogShuttleMappings.size() );
	int count = 0;
	map<pair<unsigned short, unsigned short>, JogShuttleMapping>::iterator i;
	pair<unsigned short, unsigned short> map_key;
	for ( i = _JogShuttleMappings.begin();
	        i != _JogShuttleMappings.end() ; i++, count++ )
	{
		map_key = ( *i ).first;
		char key[ 512 ];
		sprintf( key, "jogShuttleMappingFirstButton%i", count );
		gnome_config_set_int( key, map_key.first );
		sprintf( key, "jogShuttleMappingSecondButton%i", count );
		gnome_config_set_int( key, map_key.second );
		sprintf( key, "jogShuttleMappingAction%i", count );
		gnome_config_set_string( key, ( *i ).second._action.c_str() );
	}

	gnome_config_set_bool( "enableV4L", enableV4L );
	gnome_config_set_bool( "disableKeyRepeat", disableKeyRepeat );
	gnome_config_set_int( "audioRendering", audioRendering );
	gnome_config_set_int( "previewRendering", previewRendering );
	gnome_config_set_int( "maxUndos", maxUndos );
	gnome_config_set_string( "video1394Device", video1394Device );
	gnome_config_set_int( "dvCaptureBuffers", dvCaptureBuffers );
	gnome_config_set_int( "dvExportBuffers", dvExportBuffers );
	gnome_config_set_bool( "dvDecoderClampLuma", dvDecoderClampLuma );
	gnome_config_set_bool( "dvDecoderClampChroma", dvDecoderClampChroma );
	gnome_config_set_int( "maxFileSize", maxFileSize );
	gnome_config_set_bool( "audioScrub", audioScrub );
	gnome_config_set_string( "v4lVideoDevice", v4lVideoDevice );
	gnome_config_set_string( "v4lAudioDevice", v4lAudioDevice );
	gnome_config_set_string( "v4lInput", v4lInput );
	gnome_config_set_string( "v4lAudio", v4lAudio );
	gnome_config_set_bool( "isOpenDML", isOpenDML );
	gnome_config_set_string( "defaultDirectory", defaultDirectory );
	gnome_config_set_int( "displayExtract", displayExtract );
	gnome_config_set_bool( "relativeSave", relativeSave );

	gnome_config_set_int( "dvCaptureDriver", dvCaptureDriver );
	gnome_config_set_string( "dvCaptureDevice", dvCaptureDevice );
	gnome_config_set_bool( "dv1394Preview", dv1394Preview );
	gnome_config_set_string( "dvExportDevice", dvExportDevice );
	gnome_config_set_int( "avcPollIntervalMs", avcPollIntervalMs );
	gnome_config_set_int( "dvExportPrerollSec", dvExportPrerollSec );
	gnome_config_set_bool( "dvTwoPassEncoder", dvTwoPassEncoder );
	gnome_config_set_int( "windowWidth", windowWidth );
	gnome_config_set_int( "windowHeight", windowHeight );
	gnome_config_set_int( "storyboardPosition", storyboardPosition );
	gnome_config_set_float( "previewSize", previewSize );
	gnome_config_set_int( "timeFormat", timeFormat );
	gnome_config_set_string( "metaNames", metaNames );
	gnome_config_set_bool( "enablePublish", enablePublish );
	gnome_config_set_bool( "expandStoryboard", expandStoryboard );
	gnome_config_set_string( "newProjectURI", newProjectURI );
	gnome_config_set_string( "newProjectXPath", newProjectXPath );
	gnome_config_set_bool( "trimModeInsert", trimModeInsert );

	gnome_config_pop_prefix();

	gnome_config_sync();
}

/// Provide access to the audioDevice propery from C modules.
gchar *Preferences_getAudioDevice( void )
{
	return Preferences::getInstance().audioDevice;
}
