#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <time.h>

#include "kppreferencesdialog.h"
#include "kpcalendarview.h"
#include "kpstatusbar.h"
#include "kpguiutils.h"

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kpworkout.h"
#include "../kpplugin.h"
#include "../kputil.h"

static void     kp_preferences_dialog_class_init  (KPPreferencesDialogClass *klass);
static void     kp_preferences_dialog_init        (KPPreferencesDialog *dialog);
static void     kp_preferences_dialog_finalize    (GObject *object);
static void     kp_preferences_dialog_response_cb (GtkDialog *gtkdialog,
                                                   int response_id,
                                                   KPPreferencesDialog *dialog);
static void     detail_list_row_selected          (GtkTreeSelection *selection,
                                                   gpointer data);
static void     on_detail_add_button_clicked      (GtkButton *button,
                                                   gpointer data);
static void     on_detail_remove_button_clicked   (GtkButton *button,
                                                   gpointer data);
static void     set_detail_tree_view              (KPPreferencesDialog *dialog);
static gboolean set_detail_list_row               (GtkTreeModel *model,
                                                   GtkTreePath *path,
                                                   GtkTreeIter *iter,
                                                   gpointer data);
static void     sport_list_row_selected           (GtkTreeSelection *selection,
                                                   gpointer data);
static void     on_sport_add_button_clicked       (GtkButton *button,
                                                   gpointer data);
static void     on_sport_remove_button_clicked    (GtkButton *button,
                                                   gpointer data);
static void     set_sport_tree_view               (KPPreferencesDialog *dialog);
static gboolean set_sport_list_row                (GtkTreeModel *model,
                                                   GtkTreePath *path,
                                                   GtkTreeIter *iter,
                                                   gpointer data);
static void     set_plugin_tree_view              (KPPreferencesDialog *dialog);
static void     plugin_load_state_toggled         (GtkCellRendererToggle *renderer,
                                                   gchar *path,
                                                   KPPreferencesDialog *dialog);
static void     populate_tree_with_settings_list  (GtkTreeView *treeview,
                                                   const gchar *setting);
static void     init_values                       (KPPreferencesDialog *dialog);
static gboolean set_values                        (KPPreferencesDialog *dialog);
static void     on_statusbar_show_sth_button_clicked
                                                  (GtkCheckButton *button,
                                                   KPPreferencesDialog *dialog);
static void     on_browse_button_clicked          (GtkButton *button,
                                                   KPPreferencesDialog *dialog);

static GuiModuleSignalsData signals_data[] = {
{"detail_add_button",   "clicked",CB(on_detail_add_button_clicked),   NULL},
{"detail_remove_button","clicked",CB(on_detail_remove_button_clicked),NULL},
{"sport_add_button",    "clicked",CB(on_sport_add_button_clicked),    NULL},
{"sport_remove_button", "clicked",CB(on_sport_remove_button_clicked), NULL},
{"html_dir_browse",     "clicked",CB(on_browse_button_clicked),       NULL},  
{"default_dir_browse",  "clicked",CB(on_browse_button_clicked),       NULL},
{ NULL,                  NULL,    NULL,                               NULL},
};


typedef struct KPPreferencesDialogPrivateData_
{
  KPTrainingLog *log;
  KPCalendarView *cv;
  KPWorkout *wo;

  GtkWidget *detail_add_button;
  GtkWidget *detail_remove_button;
  GtkWidget *detail_treeview;
  GtkWidget *detail_entry;
  
  GtkWidget *sport_add_button;
  GtkWidget *sport_remove_button;
  GtkWidget *sport_treeview;
  GtkWidget *sport_entry;

  GtkWidget *plugin_treeview;
  
  GtkWidget *sb[KP_STATUSBAR_F_N];
  
  GtkWidget *window_geometry_button;
  GtkWidget *default_dir_entry;
  GtkWidget *html_dir_entry;
  GtkWidget *exit_button;
} KPPreferencesDialogPrivateData;

#define KP_PREFERENCES_DIALOG_PRIVATE_DATA(widget) (((KPPreferencesDialogPrivateData*) \
        (KP_PREFERENCES_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_preferences_dialog_get_type (void)
{
  static GType kp_preferences_dialog_type = 0;

  if (kp_preferences_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPPreferencesDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_preferences_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPPreferencesDialog),
      0,
      (GInstanceInitFunc) kp_preferences_dialog_init,
      NULL,
    };

    kp_preferences_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                         "KPPreferencesDialog",
                                                         &our_info, 0);
  }

  return kp_preferences_dialog_type;
}

static void
kp_preferences_dialog_class_init (KPPreferencesDialogClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_preferences_dialog_finalize;
}

static void
kp_preferences_dialog_init (KPPreferencesDialog * dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  KPStatusbarFieldData *field;
  GtkWidget *notebook;
  GladeXML *xml;
  guint i;

  xml = kp_gui_load ("preferences", "preferences");
  
  dialog->private_data = g_new0 (KPPreferencesDialogPrivateData, 1);
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  g_signal_connect (G_OBJECT (dialog),
                   "response",
                    G_CALLBACK (kp_preferences_dialog_response_cb), dialog);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 7);
  
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog),
                     _("Kipin\303\244 Preferences"));

  p_data->exit_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                               GTK_STOCK_CLOSE,
                                               GTK_RESPONSE_CLOSE);

  notebook = KP_W (xml, "preferences");
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), notebook);
  gtk_container_set_border_width (GTK_CONTAINER (notebook), 5);

  p_data->wo = kp_workout_new ();
  
  /* Put widgets to private data */
  p_data->window_geometry_button = KP_W (xml, "window_geometry_button");
  p_data->detail_remove_button = KP_W (xml, "detail_remove_button");  
  p_data->detail_add_button = KP_W (xml, "detail_add_button");  
  p_data->detail_treeview = KP_W (xml, "detail_treeview");
  p_data->detail_entry = KP_W (xml, "detail_entry");

  p_data->sport_remove_button = KP_W (xml, "sport_remove_button");  
  p_data->sport_add_button = KP_W (xml, "sport_add_button");  
  p_data->sport_treeview = KP_W (xml, "sport_treeview");
  p_data->sport_entry = KP_W (xml, "sport_entry");

  p_data->plugin_treeview = KP_W (xml, "plugin_treeview");

  p_data->default_dir_entry = KP_W (xml, "default_dir_entry");
  p_data->html_dir_entry = KP_W (xml, "html_dir_entry");

  init_values (dialog);
 
  for (i=0; i < KP_STATUSBAR_F_N; i++) {
  
    field = kp_statusbar_get_field (i);
       
    p_data->sb[i] = KP_W (xml, field->name);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p_data->sb[i]),
                                  kp_settings_get_bool (field->setting));
    
    g_object_set_data (G_OBJECT (p_data->sb[i]), "setting_name",
                       field->setting);
    
    g_signal_connect (G_OBJECT (p_data->sb[i]), "clicked",
                      G_CALLBACK (on_statusbar_show_sth_button_clicked),
                      dialog);
    
    g_free (field->name);
    g_free (field);
  }
  
  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  g_object_unref (G_OBJECT (xml));
}

static void
kp_preferences_dialog_finalize (GObject * object)
{
  KPPreferencesDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_PREFERENCES_DIALOG (object));

  dialog = KP_PREFERENCES_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
kp_preferences_dialog_new (void)
{
  KPPreferencesDialog *dialog;

  dialog = g_object_new (KP_TYPE_PREFERENCES_DIALOG, NULL);

  set_detail_tree_view (dialog);
  set_sport_tree_view (dialog);
  set_plugin_tree_view (dialog);

  return GTK_WIDGET (dialog);
}

static void
kp_preferences_dialog_response_cb (GtkDialog * gtk_dialog, int response_id,
  KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  if (response_id == GTK_RESPONSE_CLOSE) 
    if (set_values (dialog)) {
      kp_settings_save ();
      g_print("Settings saved.\n");
    }
}

static void
init_values (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  gchar *default_dir;
  gchar *html_op_dir;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  html_op_dir = kp_settings_get_str ("html_output_dir");
  default_dir = kp_settings_get_str ("default_dir");

  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (p_data->window_geometry_button),
      kp_settings_get_bool ("save_window_geometry"));
  
  if (default_dir) {
    gtk_entry_set_text (GTK_ENTRY (p_data->default_dir_entry), default_dir);
    g_free (default_dir);
  }
  if (html_op_dir) {
    gtk_entry_set_text (GTK_ENTRY (p_data->html_dir_entry), html_op_dir);
    g_free (html_op_dir);
  }
}

static gboolean
set_values (KPPreferencesDialog *dialog)
{
  GtkTreeModel *model;
  KPPreferencesDialogPrivateData *p_data;
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
 
  kp_settings_set_str ("default_dir",
                       gtk_entry_get_text (GTK_ENTRY (p_data->default_dir_entry)),
                       TRUE);
  kp_settings_set_str ("html_output_dir",
                       gtk_entry_get_text (GTK_ENTRY (p_data->html_dir_entry)),
                       TRUE);

  kp_settings_set_bool ("save_window_geometry",
                        gtk_toggle_button_get_active (
                        GTK_TOGGLE_BUTTON (p_data->window_geometry_button)));
  
  /* Set sport list items */
  kp_settings_remove_setting ("preset_sport");
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->sport_treeview));
  gtk_tree_model_foreach (model, set_sport_list_row, dialog);

  /* Set detail list items */
  kp_settings_remove_setting ("preset_detail");
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->detail_treeview));
  gtk_tree_model_foreach (model, set_detail_list_row, dialog);

  return TRUE;
}

static void
on_statusbar_show_sth_button_clicked (GtkCheckButton *button,
                                      KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  gboolean state;
  gchar *setting;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
 
  setting = g_object_get_data (G_OBJECT (button), "setting_name");
  g_return_if_fail (setting != NULL);
  
  state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

  kp_settings_set_bool (setting, state);
}


static void
on_browse_button_clicked (GtkButton *button, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  const gchar *wname;
  gchar *dirname;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  
  dirname = kp_gui_get_dir (GTK_WINDOW (dialog));
  wname = gtk_widget_get_name (GTK_WIDGET (button));
  
  if (strcmp (wname, "html_dir_browse") == 0)
    gtk_entry_set_text (GTK_ENTRY (p_data->html_dir_entry), dirname);
  else if (strcmp (wname, "default_dir_browse") == 0)
    gtk_entry_set_text (GTK_ENTRY (p_data->default_dir_entry), dirname);
}

static gboolean
set_sport_list_row (GtkTreeModel * model,
                    GtkTreePath * path,
                    GtkTreeIter * iter,
                    gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  const gchar *val;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  g_return_val_if_fail (p_data != NULL, TRUE); 

  gtk_tree_model_get (model, iter, 0, &val, -1);
  kp_settings_set_str ("preset_sport", val, FALSE);

  return FALSE;
}

/*
 * notebook's note 'DETAILS'
 * 
 */
static void
set_detail_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->detail_treeview);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Detail"), renderer,
                                                     "text", 0, NULL);
  gtk_tree_view_append_column (treeview, column1);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);

  populate_tree_with_settings_list (treeview, "preset_detail");
}

static void
populate_tree_with_settings_list (GtkTreeView *treeview, const gchar *setting)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GSList *list;

  list = kp_settings_get_list (setting);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  while (list) {
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, list->data, -1);
    list = list->next;
  }
}

static void
detail_list_row_selected (GtkTreeSelection * selection,
                          gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *detail;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  gtk_widget_set_sensitive (p_data->detail_remove_button, TRUE);

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 0, &detail, -1);

    gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), detail);
    g_free (detail);
  }
}


static void
on_detail_remove_button_clicked (GtkButton * button,
                                 gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->detail_treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), "");
}

static void
on_detail_add_button_clicked (GtkButton * button,
                              gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *entry;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  entry = gtk_entry_get_text (GTK_ENTRY (p_data->detail_entry));
  if (strlen (entry) == 0)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->detail_treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, entry, -1);
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), "");
}

static gboolean
set_detail_list_row (GtkTreeModel * model,
                     GtkTreePath * path,
                     GtkTreeIter * iter,
                     gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  const gchar *val;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  g_return_val_if_fail (p_data != NULL, TRUE); 

  gtk_tree_model_get (model, iter, 0, &val, -1);
  kp_settings_set_str ("preset_detail", val, FALSE);
  g_print("Setting string (really): %s=%s\n", "preset_detail", val);

  return FALSE;
}

/*
 * notebook's note 'SPORTS'
 * 
 */
static void
set_sport_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->sport_treeview);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Sport"), renderer,
                                                     "text", 0, NULL);
  gtk_tree_view_append_column (treeview, column1);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (sport_list_row_selected), dialog);
  
  populate_tree_with_settings_list (treeview, "preset_sport");
}

static void
sport_list_row_selected (GtkTreeSelection * selection,
                         gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *sport;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  gtk_widget_set_sensitive (p_data->sport_remove_button, TRUE);

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 0, &sport, -1);

    gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), sport);
    g_free (sport);
  }
}


static void
on_sport_remove_button_clicked (GtkButton * button,
                                gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->sport_treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), "");
}

static void
on_sport_add_button_clicked (GtkButton * button,
                              gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *entry;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  entry = gtk_entry_get_text (GTK_ENTRY (p_data->sport_entry));
  if (strlen (entry) == 0)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->sport_treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, entry, -1);
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), "");
}

static void set_plugin_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *col1;
  GtkTreeViewColumn *col2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;
  GtkTreeIter iter;
  GList *list;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->plugin_treeview);
  
  store = gtk_list_store_new (2, G_TYPE_BOOLEAN, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_toggle_new ();
  
  g_signal_connect (G_OBJECT (renderer), "toggled",
                    G_CALLBACK (plugin_load_state_toggled), dialog);
  
  col1 = gtk_tree_view_column_new_with_attributes (_("Loaded"), renderer,
                                                     "active", 0, NULL);
  renderer = gtk_cell_renderer_text_new ();

  col2 = gtk_tree_view_column_new_with_attributes (_("Plugin"), renderer,
                                                     "text", 1, NULL);

  gtk_tree_view_column_set_clickable (col1, TRUE);
  
  gtk_tree_view_append_column (treeview, col1);
  gtk_tree_view_append_column (treeview, col2);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  /*g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (sport_list_row_selected), dialog);*/
  
  list = kp_plugin_get_plugins ();
  
  while (list) {
    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter, 0, KP_PLUGIN (list->data)->loaded, -1);
    gtk_list_store_set (store, &iter, 1, KP_PLUGIN (list->data)->info->name, -1);
    list = list->next;
  }
}

static void plugin_load_state_toggled (GtkCellRendererToggle *renderer,
                                       gchar *path,
                                       KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *p;
  gboolean ind;
  gchar *module;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  p = gtk_tree_path_new_from_string (path);

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->plugin_treeview));
 
  gtk_tree_model_get_iter (model, &iter, p);
  gtk_tree_model_get (model, &iter, 0, &ind, 1, &module, -1);

  ind ^= 1;

  if (ind) {
    kp_plugin_load_by_name (module);
    kp_settings_add_to_list ("load_plugin", module);
  } else {
    kp_plugin_unload_by_name (module);
    kp_settings_remove_from_list ("load_plugin", module);
  }

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, ind, -1);
  gtk_tree_path_free (p);
}

