#include <config.h>

#include <jni.h>
#include <assert.h>
#include <string.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#endif

#include "newobject.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "java_lang_Class.h"
#include "Class.h"
#include "Class_Reflection.h"
#include "VMClassLoader.h"
#include "../java.lang.reflect/Method.h"
extern tOBREF OOMExceptionObject;
extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class


/* Takes an existing exception and turns it into an InvocationTargetException that wraps the original
 *
 *
 *
 *
 */

tOBREF CLASS_CreateInvocationTargetException(JNIEnv* env)
{
  /* We have to construct this puppy carefully, it doesn't have the normal String constructor*/
  jclass exClass;
  jmethodID consMethodID;
  jclass origException;
  
  origException = (*env)->ExceptionOccurred(env);
  (*env)->ExceptionClear(env);
  
  exClass = (*env)->FindClass(env, "java/lang/reflect/InvocationTargetException");
  if(exClass == NULL) { (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InternalError")); return NULL; }
  consMethodID = (*env)->GetMethodID(env, exClass, "<init>", "(Ljava/lang/Throwable;Ljava/lang/String;)V");
  if(consMethodID == NULL) { (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InternalError")); return NULL; }
  (*env)->Throw(env, (*env)->NewObject(env, exClass, consMethodID, origException, INTERP_NewStringFromAsciz(env, "An exception was thrown while calling java.lang.reflect.Method.invoke")));
  return NULL;	
}

/* Makes a class object for a Class that doesn't have one yet
 *
 *
 *
 */

tOBREF CLASS_MakeClassObject(JNIEnv* env, tClassLoaderTuple* tuple)
{
  tOBREF pstClassObject;
  assert(pstClassType);
  if(tuple->classObject == NULL)
    {
      pstClassObject = INTERP_NewObject(env, pstClassType);
      // CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstClassObject);
      // do we need this -- yes I think so, a GC could occur before tihs is returned

      /* set _classStruct field in new class object */
      CLASS_SetClassStruct(pstClassObject, tuple);
      assert( CLASS_GetClassStruct(env, pstClassObject));
      /* set _classLoader field in new class object to NULL */
      CLASS_SetClassLoader(pstClassObject, tuple->classLoader);
      /* set class struct to point to class object */
      tuple->classObject = pstClassObject;

    }
 else
     {
	 assert( CLASS_GetClassStruct(env, tuple->classObject) == tuple);
     }
  return tuple->classObject;
}


/* Checks that the array of objects matches the types required by the method 
 *
 *
 *
 */

int CLASS_CheckNumReflectionMethodArguments(JNIEnv* env, tMethod* method, jobjectArray args)
{
  int expectedArguments = CLASSFILE_CalcNumArguments( method->uidSignature);
  if(expectedArguments != 0)
    {
      if(args == NULL)
	{
	  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Class[] passed to invoke/newInstance is NULL"));
	  return -1;
	}
	assert( ADEREF(args)->i32Number >= 0);
	/* Check the number of arguments is correct */
	if((*env)->GetArrayLength(env, args) != expectedArguments)
	  {
	    char* errMessage = (char*) sys_malloc(1024);
	    if(errMessage == NULL) { (*env)->Throw(env, OOMExceptionObject); return -1; }
	    sprintf(errMessage,"%i arguments passed to invoke, expecting %i", (int) (*env)->GetArrayLength(env, args), expectedArguments);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", errMessage));
	    sys_free(errMessage);
	    return -1;
	  }
      }
  else
    {
      if(args != NULL)
       {
	 //we must check they didn't give us arguments by mistake
	 if( (*env)->GetArrayLength(env, args) != 0)
	{
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "array of arguments passed to invoke had more than 0 elements (we were expecting 0)"));
	    return -1;
	}
       }
    }
 return 0;
}

int CLASS_CreateReflectedMethodArguments(JNIEnv* env, tMethod* method, jobjectArray args, jvalue** ret_array )
{
   jvalue* jvalue_array;
   int expectedArguments = CLASSFILE_CalcNumArguments( method->uidSignature);
    int i = 0;

    //We have to convert the array of object into an array of jvalues
   if(expectedArguments != 0)
     {
       jvalue_array = sys_malloc((*env)->GetArrayLength(env, args) * sizeof(jvalue));
       if(jvalue_array == NULL)
	 {
	   (*env)->Throw(env, OOMExceptionObject);
	   return -1;
	 }
       for(i = 0; i < expectedArguments;i++) //we must unwrap primitive arguments
	 {
	   tOBREF an_arg = (*env)->GetObjectArrayElement(env, args, i);
	   jclass targetClass = NULL;
	   jclass suppliedClass = NULL;

	   targetClass = CLASS_GetClassFromSig(env, method->uidSignature, method->pstClass, i);
	   if(Java_java_lang_Class_isPrimitive(env, targetClass) == JNI_TRUE)
	     {
		targetClass = CLASS_GetWrapperClass(env, targetClass);
		if(an_arg == NULL)
		{
		    sys_free(jvalue_array);
		    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) was NULL"));
		    return -1;
		}
		suppliedClass = CLASS_GetClass(env, an_arg);
		if(suppliedClass != targetClass)
		{
		   if( suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_VOID))
		    {
		    sys_free(jvalue_array);
		    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Void passed as an argument to invoke"));
		    return -1;
		    }
		   if(Java_java_lang_Class_isPrimitiveWrapper(env, suppliedClass) == JNI_FALSE)
		    {
		    sys_free(jvalue_array);
		    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Object passed as an argument to invoke is not of primitive type!"));
		    return -1;
		    }
		   if(CLASS_CheckWrapperWidening(env, targetClass, suppliedClass) == JNI_FALSE)
		    {
		    sys_free(jvalue_array);
		    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Object passed as an argument to invoke is not of the expected type"));
		    return -1;
		    }
		}
		    if(CLASS_UnwrapObject(env, an_arg, jvalue_array + i) != 0) 
			{
			sys_free(jvalue_array);
			(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Failed to unwrap a primitive object"));
			return -1;
		 	}
		if(suppliedClass != targetClass)
		{
		//now we must do the widening of the result
		    if(CLASS_WidenResult(env, targetClass, suppliedClass, jvalue_array + i) != 0) 
			{
			sys_free(jvalue_array);
			(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Failed to widen a primitive object during invoke (even though the widening check succeeded)"));
			return -1;
		 	}
		}

	     }
	  else
	     {
	       if(an_arg != NULL)
		{
		
		if( Java_java_lang_Class_isInstance(env, targetClass, an_arg) == JNI_FALSE)
		{
	        char* errMessage;
		tClassLoaderTuple* pstDesiredClass;
		tClassLoaderTuple* pstSuppliedClass;

	       sys_free(jvalue_array);
	       pstDesiredClass = CLASS_GetClassStruct(env, targetClass );
	       pstSuppliedClass = CLASS_GetClassStruct(env, CLASS_GetClass(env, an_arg) );

		errMessage = sys_malloc( 1024 + strlen(pstDesiredClass->uidName) + strlen(pstSuppliedClass->uidName));
		if(errMessage == NULL) { (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InternalError")); return -1; }

		sprintf(errMessage, "Object passed to invoke as an argument was type %s, expected type %s\n", pstSuppliedClass->uidName, pstDesiredClass->uidName);
		(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", errMessage));
		sys_free(errMessage);
		return -1;
		 }
		}
		jvalue_array[i].l = an_arg;
	     }
	 }
     }
    else
      jvalue_array = NULL;

//success
  *ret_array = jvalue_array;
  return 0;
}

int CLASS_CompareClassArrays(JNIEnv* env, jobjectArray a, jobjectArray b)
{
  int lengtha = (*env)->GetArrayLength(env, a);
  if(lengtha == (*env)->GetArrayLength(env, b))
    {
      int i;
      for(i = 0; i < lengtha;i++)
	{
	  if( (*env)->GetObjectArrayElement(env, a, i) != (*env)->GetObjectArrayElement(env, b, i))
	    {
	      return 2;
	    }
	}
      return 0; //match
    }
  else
    return 1;
}

int CLASS_getDeclaringClass(JNIEnv* env,tClassLoaderTuple* startingClass, tMethod* method, jclass* origClazz, int* slotIndex)
{
	tClassLoaderTuple* searchClass = startingClass;
	while(searchClass != NULL)
	{
	 int i = 0;
	 for(i = 0; i < searchClass->pstClass->u16MethodsCount;i++)
	  {
	   if(searchClass->pstClass->pstMethods + i == method)
	    {
		*slotIndex = i;
		*origClazz = CLASS_MakeClassObject(env, searchClass);
		return 0;
	    }
	  }
	 searchClass = searchClass->pstClass->pstSuperClass;
	}
	*slotIndex = -1;
	*origClazz = NULL;
	return -1;
}


/* Takes an array of Class objects (a tArray* of their jclass objects) 
 
   And returns the signature 
 
 */ 
 
static char sigbuffer[4096]; 
 
int ConvertClassToSig(JNIEnv* env, tClassLoaderTuple* clazz, int primitive) 
{ 
    if(primitive) 
	{ 
		    if(uidcmp(clazz->uidName, UID_GetUid("int")) == 0) 
			strcat(sigbuffer, "I"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("byte")) == 0) 
			strcat(sigbuffer, "B"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("short")) == 0) 
			strcat(sigbuffer, "S"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("char")) == 0) 
			strcat(sigbuffer, "C"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("float")) == 0) 
			strcat(sigbuffer, "F"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("double")) == 0) 
			strcat(sigbuffer, "D"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("boolean")) == 0) 
			strcat(sigbuffer, "Z"); 
		    else if(uidcmp(clazz->uidName, UID_GetUid("long")) == 0) 
			strcat(sigbuffer, "J"); 
	} 
    else 
	{ 
	    //		    fprintf(stderr, "class is not primitive %s\n", clazz->uidName); 
		 if(clazz->uidName[0] == '[') 
		     { 
			 int looper = 0; 
			 int isClass = 0; 
			 while( looper < strlen(clazz->uidName)) 
			     { 
				 if( clazz->uidName[looper] != '[') 
				     break; 
				 else 
				     strcat(sigbuffer, "["); 
				 looper++; 
			     } 
 
			 if( clazz->uidName[looper] == 'L') 
			     { 
				 isClass = 1; 
				// strcat(sigbuffer, "L"); 
			     } 
			 while(  looper < strlen(clazz->uidName)) 
			     { 
				 char buf[2]; 
				 buf[0] = clazz->uidName[looper]; 
			         buf[1] = 0; 
				 strcat(sigbuffer, buf); 
				 looper++; 
			     } 
			 //			 if(isClass) 
			 //			     strcat(sigbuffer, ";"); 
		     } 
		 else 
		     { 
			 //It's an object 
			 strcat(sigbuffer, "L"); 
			 strcat(sigbuffer, clazz->uidName); 
			 strcat(sigbuffer, ";"); 
		     } 
 
 
	} 
 
    return 0; 
} 
 


/* Takes an array Class[] and returns a java-style signature string */ 
 
Uid CLASS_GetSigFromClassArray(JNIEnv* env, tArray* args) 
{ 
    int numargs = args->i32Number; 
    int i; 
     
    //    fprintf(stderr, "Got an array with %i els, type %s\n", (int) args->i32Number, args->pstType->uidName); 
    strcpy(sigbuffer, "("); 
    for(i = 0; i < numargs;i++) 
	{ 
	    jclass clazzObj = ((jclass*) args->pvElements)[i]; 
	    jclass clazzClazz = (*env)->GetObjectClass(env, clazzObj); 
	    jfieldID fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
	    tClassLoaderTuple* classPtr; 
	    assert(fieldID); 
     
	    classPtr = (tClassLoaderTuple*) (*env)->GetIntField(env, clazzObj, fieldID); 
 
	    if( Java_java_lang_Class_isPrimitive(env, clazzObj)) 
		{ 
		    ConvertClassToSig(env, classPtr, 1); 
		} 
	    else 
		{ 
		    ConvertClassToSig(env, classPtr, 0); 
		} 
	} 
    strcat(sigbuffer, ")"); 
    //Now the return type 
 
    return UID_GetUid(sigbuffer); 
} 


/* Since we're using these primitive class objects so often, I think we should create a method to access them directly out of their, with an enumerated type as index */

/* Checks whether this is a wrapper class for a primitive value (like java/lang/Integer java/lang/Byte etc)
 *
 * Note this returns true for java/lang/Void too
 *
 */
jboolean Java_java_lang_Class_isPrimitiveWrapper( JNIEnv* env,  jobject clazz) { 

  if( (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_CHAR)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BOOLEAN)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_VOID)))
    {
      return JNI_TRUE;
    }
  return JNI_FALSE;
}

/* Checks whether the supplied wrapper type can be widened to the target type. We assume the types aren't the same 

   eg java/lang/Short can be widened to java/lang/Integer java/lang/Long
*/

jboolean CLASS_CheckWrapperWidening(JNIEnv* env, jclass targetClass, jclass suppliedClass)
{
  if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))
     {
       return JNI_FALSE; //no widening
     }
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE))
     {
       return JNI_FALSE; //no widening
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
     {
       //int can be widened to long 
       if((targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)))
	  return JNI_TRUE;
       return JNI_FALSE;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT))
     {
       //short can be widened to int and long
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))
	  return JNI_TRUE;
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
	  return JNI_TRUE;
       return JNI_FALSE;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE))
     {
       //byte can be widened to short, int and long
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))
	  return JNI_TRUE;
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
	  return JNI_TRUE;
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT))
	  return JNI_TRUE;
       return JNI_FALSE;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_CHAR))
     {
       return JNI_FALSE; //no widening
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BOOLEAN))
     {
       return JNI_FALSE; //no widening
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT))
     {
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE))
	  return JNI_TRUE;
       return JNI_FALSE; 
     } 
  else {
    assert(1 == 2);
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Unknown Wrapper class in CLASS_CheckWrapperWidening"));
      return JNI_FALSE;
       }
}

jclass CLASS_GetWrapperClass(JNIEnv* env, jclass primClass)
{
  tClassLoaderTuple* pstType = CLASS_GetClassStruct(env, primClass);

  if(strcmp(pstType->uidName, "int") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Integer");
    }
  else if(strcmp(pstType->uidName, "byte") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Byte");
    }
  else if(strcmp(pstType->uidName, "char") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Character");
    }
  else if(strcmp(pstType->uidName, "short") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Short");
    }
  else if(strcmp(pstType->uidName, "long") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Long");
    }
  else if(strcmp(pstType->uidName, "double") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Double");
    }
  else if(strcmp(pstType->uidName, "float") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Float");
    }
  else if(strcmp(pstType->uidName, "boolean") == 0)
    {
      return (*env)->FindClass(env, "java/lang/Boolean");
    }
  else
    {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Couldn't get wrapper class for primitive class in CLASS_GetWrapperClass"));
      return NULL;
    }
}

int CLASS_UnwrapObject(JNIEnv* env, jobject object, jvalue* retval)
{
  tClassLoaderTuple* pstType = ODEREF(object)->pstType;
  jclass clazz = CLASS_GetClass(env, object);
  jfieldID fieldID;

  fieldID = (*env)->GetFieldID(env, clazz, "value", NULL);
  if(fieldID == NULL)
    {
      retval->j = 0;
      return -1;
    }
  if(strcmp(pstType->uidName, "java/lang/Integer") == 0)
    {
      retval->i = (*env)->GetIntField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Byte") == 0)
    {
      retval->b = (*env)->GetByteField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Short") == 0)
    {
      retval->s = (*env)->GetShortField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Character") == 0)
    {
      retval->c = (*env)->GetCharField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Long") == 0)
    {
      retval->j = (*env)->GetLongField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Double") == 0)
    {
      retval->d = (*env)->GetDoubleField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Float") == 0)
    {
      retval->f = (*env)->GetFloatField(env, object, fieldID);
      return 0;
    }
  else if(strcmp(pstType->uidName, "java/lang/Boolean") == 0)
    {
      retval->z = (*env)->GetBooleanField(env, object, fieldID);
      return 0;
    }
  else
    {
      return -1;
    }
}

/* The target class is something like java/lang/Integer and the suppliedClass is something like java/lang/Short
 *
 * The check to see if widening is possible must already have been performed
 *
 */

int CLASS_WidenResult(JNIEnv* env, jclass targetClass, jclass suppliedClass, jvalue* retval)
{
  if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
     {
       //int can be widened to long 
       if((targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)))
	 {
	   retval->j = retval->i;
	   return 0;
	 }
       return -1;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT))
     {
       //short can be widened to int and long
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))
	 {
	   retval->j = retval->s;
	   return 0;
	 }
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
	 {
	   retval->i = retval->s;
	   return 0;
	 }
       return -1;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE))
     {
       //byte can be widened to short, int and long
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))
	 {
	   retval->j = retval->b;
	   return 0;
	 }
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT))
	 {
	   retval->i = retval->b;
	   return 0;
	 }
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT))
	 {
	   retval->s = retval->b;
	   return 0;
	 }
       return -1;
     } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT))
     {
       if(targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE))
	 {
	   retval->d = retval->f;
	   return 0;
	 }
       return -1;
     } 
  else {
    assert(1 == 2);
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Unknown widening operation for Wrapper class in CLASS_WidenResult"));
      return -1;
       }
}

jclass CLASS_GetClassFromSig(JNIEnv* env, char* pszSignature, tClassLoaderTuple* loadingClass, int index) 
{ 
 
  int pos,len; 
  int itemcount = 0; 
  uint16 jint_count=0; 
  byte *Bytes; 
  int looking_at_return_type = 0; //Just so we can examine the last item 
 
  #ifdef DEBUG 
  assert(pszSignature); 
  #endif 
 
  #ifdef CLASSFILE_TRACE 
  eprintf("Parsing: %s\n", pszSignature); 
  #endif 
 
  len = strlen(pszSignature); 
  Bytes = pszSignature; 
  for (pos = 0; pos < len; pos++) 
  { 
    if(itemcount == index) 
      { 
	//This is the item we want 
	    switch(Bytes[pos]) 
	      { 
	      case '(': 
		{ 
		  break; 
		} 
	      case ')': 
		{ 
		  looking_at_return_type = 1; 
		  break; 
		} 
	      case 'V':
	      case 'B': 		
	      case 'C': 
	      case 'F': 
	      case 'S': 
	      case 'Z': 
	      case 'I': 
	      case 'D': 
	      case 'J': 
		return CLASS_GetPrimitiveClass(env, Bytes[pos]);
		break;
	      case 'L': 
		{ 
		  char* orig = sys_malloc(4096); 
		  char* start = Bytes + pos; 
		  jclass retval; 
		  int origpos = pos; 
		  
		  assert(orig != NULL); 
		  assert(len < 4096); 
		  
		  for ( ; Bytes[pos] != ';' && pos < len; pos++); 

		  assert(((pos - origpos) - 1) > 0);
		  strncpy(orig, start + 1,(pos - origpos) -1); 
		  orig[ pos - origpos - 1] = 0; 
		  
		  retval = Java_java_lang_Class_forNameHelperFromName(env, orig);
		  //fprintf(stderr, "DOING GetClassFromSig %s\n", orig); 
		  //  temp_string = INTERP_NewStringFromAsciz(env, orig);
		  //retval = Java_java_lang_Class_forName(env, NULL , temp_string);
		// sys_free(orig); 		  
		  return retval; 
		  break; 
		} 
	      case '[': 
		{ 
		  int start = pos; 
		  tClassLoaderTuple* result;

		  char buf[2048]; 
		  while ((Bytes[pos] == '[') && (pos < len))  
		    { 
		      pos++; 
		    } 
		  
		  if (Bytes[pos] == 'L')	/* object signature */ 
		    { 
		      while (Bytes[pos] != ';') 
			{ 
			  pos++; 
			} 
		    } 
		  //	else 
		    pos++; 
		  
		  memset(buf,0, 2048); 
		  strncpy(buf, pszSignature + start, pos - start); 
	
		  result = CLASSFILE_FindOrLoad(env, buf, loadingClass);
		  if(result)
		    {
		      return CLASS_MakeClassObject(env, result);
		    }
		  else
		    return NULL;
		  break; 
		} 
	      default: 
		{ 
		  //        #ifdef DEBUG 
			       assert(2 == 7); 
		  eprintf("Error: Parse_Argument_Number: illegal signature %s, index %i, itemcount %i\n", Bytes + pos, index, itemcount); 
		  exit(1); 
		  //        #endif 
			       } 
	      } //end switch 
			
			} 
    else 
      { 
	//Just count 
	    switch(Bytes[pos]) 
	      { 
	      case '(': 
		{ 
		  break; 
		} 
	      case ')': 
		{ 
		  looking_at_return_type = 1; 
		  //        goto LabelLoopEnd;  /* force a jump out of the loop */ 
		} 
	      case 'B': 
	      case 'C': 
	      case 'F': 
	      case 'S': 
	      case 'Z': 
	      case 'I': 
		{ 
		  itemcount++; 
		  jint_count++; 
		  break; 
		} 
	      case 'D': 
	      case 'J': 
		{ 
		  itemcount++; 
		  jint_count += 2; 
		  break; 
		} 
	      case 'L': 
		{ 
		  itemcount++; 
		  jint_count++; 
		  for ( ; Bytes[pos] != ';' && pos < len; pos++); 
		  break; 
		} 
	      case '[': 
		{ 
		  itemcount++; 
		  jint_count++; 
		  while ((Bytes[pos] == '[' || (Bytes[pos] >= '0' && Bytes[pos] <= '9')) && pos < len) /* multiple arrays with sizes*/ 
		    { 
		      pos++; 
		    } 
		  if (Bytes[pos] == 'L')	/* object signature */ 
		    { 
		      while (Bytes[pos] != ';') 
			{ 
			  pos++; 
			} 
		    } 
		  break; 
		} 
	      default: 
		{ 
		  //        #ifdef DEBUG 
			       assert(2 == 7); 
		  eprintf("Error: Parse_Argument_Number: illegal signature index %i, pos %i, itemcount %i, char is %c\n", index, pos, itemcount, Bytes[pos]); 
		  exit(1); 
		  //        #endif 
			       } 
	      } //end second switch 
			       } //end else 
				 
				 
				 
#ifdef CLASSFILE_TRACE 
				 eprintf("int32 argument len: %d\n",jint_count); 
#endif 
 
 
  } 
  
  return NULL; 
} 

jclass CLASS_GetPrimitiveClass(JNIEnv* env, char type)
{
  return java_lang_VMClassLoader_getPrimitiveClass(env, NULL, type);
  /*  switch(type)
    {
	      case 'V': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env,"void");
		}
	      case 'B': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "byte");
		}
	      case 'C': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "char");
		}
	      case 'F': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "float");
		}
	      case 'S': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "short");
		}
	      case 'Z': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "boolean");
		}
	      case 'I': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "int");
		}
	      case 'D': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "double");
		}
	      case 'J': 
		{
		  return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, "long");
		}
    default:
      {
	eprintf("Invalid type \"%c\" encountered in CLASS_GetPrimitiveClass\n", type);
	return NULL;
      }
      }*/
}

jobjectArray CLASS_GetClassArrayFromSig(JNIEnv* env, char* sig, tClassLoaderTuple* loadingClass)
{
  int count;
  jobjectArray parameters;
  int j = 0;

  count = CLASSFILE_CalcNumArguments( sig );

  parameters = (*env)->NewObjectArray(env, count, (*env)->FindClass(env, "java/lang/Class"), NULL);

  for(j = 0; j < (count);j++)
    {
      jclass type;
      type = CLASS_GetClassFromSig(env, sig, loadingClass, j);

      if(type == NULL)
	{
	  (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/NoClassDefFoundError"), sig);
	  return NULL;
	}
      
      (*env)->SetObjectArrayElement(env, parameters, j, type);
    }
   return parameters;
}

//	public native Field getField(String name) throws NoSuchFieldException, SecurityException; 
 
jobject Java_java_lang_Class_getField( JNIEnv* env,  jclass clazz, jstring name)  
{ 
    int isStatic = 0; 
    jfieldID fieldID; 
    jobject retField; 
 
    jfieldID decClassField; 
    jfieldID nameField; 
    jfieldID slotField; 
 
    jclass fieldClazz; 
 
    char* fieldName;
    tClassLoaderTuple* pstClass;

    /* The Field.java class has the following members: 
 
	private Class declaringClass; 
	private String name; 
	private int slot; 
     */ 
 
    fieldName = INTERP_AscizFromString(env, name);
//    eprintf( "Class.getField: Looking for field %s\n", fieldName);
 
    fieldID = (*env)->GetFieldID(env, clazz, fieldName, NULL); 
 
    pstClass = CLASS_GetClassStruct(env, clazz);

    if(fieldID == NULL) 
	{ 
	    fieldID = (*env)->GetStaticFieldID(env, clazz, fieldName, NULL); 
	    isStatic = 1; 
	    if(fieldID == NULL) 
		{ 
//		    eprintf("Didn't find static or instance field %s, looking in %s \n", INTERP_AscizFromString(env, name), pstClass->uidName); 
		    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchFieldException")); 
		    return NULL; 
		} 
 
	} 

    sys_free(fieldName);
 
    //    printf("got field %x\n", fieldID); 
     
    retField = INTERP_NewObjectFromName(env, "java/lang/reflect/Field"); 
    (*env)->NewGlobalRef( env, retField );
    fieldClazz = CLASS_GetClass(env, retField); 
 
   nameField = (*env)->GetFieldID(env, fieldClazz, "name", NULL); 
     
//    eprintf("here in getField\n"); 
 
    (*env)->SetObjectField(env, retField, nameField, name); 
 
 
   decClassField = (*env)->GetFieldID(env, fieldClazz, "declaringClass", NULL); 
 
    (*env)->SetObjectField(env, retField, decClassField, clazz); 
 
   slotField = (*env)->GetFieldID(env, fieldClazz, "slot", NULL); 
 
    (*env)->SetIntField(env, retField, slotField, -1); 
 
    return retField; 
    //    return NULL; 
} 
 
 
jobject Java_java_lang_Class_getDeclaredField( JNIEnv* env,  jclass clazz, jstring name) 
{ 
 
    return Java_java_lang_Class_getField(env, clazz, name); 
	#ifndef USEOSKIT 
    assert(1 == 2); 
#endif 
    return NULL; 
} 
 
 
 
void constructFieldObject( JNIEnv* env, tOBREF fieldObject, jclass declaringClazz, char* name, int slot, int isStatic) 
{ 
    jclass fieldClazz; 
    jmethodID consID; 
    fieldClazz = (*env)->GetObjectClass(env, fieldObject); 
    assert(fieldClazz); 
     
    consID = (*env)->GetMethodID(env, fieldClazz, "<init>", "(Ljava/lang/Class;Ljava/lang/String;I)V"); 
    assert(consID); 
 
    (*env)->CallNonvirtualVoidMethod(env, fieldObject, fieldClazz, consID, declaringClazz, INTERP_NewStringFromAsciz(env, name), slot); 
 
} 
 
    /** 
     * Get all public fields from this class. 
     * 
     * Do we have to look at each superclass ??? XXX 
     *
     * @return all public fields in this class. 
     * @exception SecurityException if you do not have access to public 
     *            members of this class. 
     */ 
 
jobjectArray Java_java_lang_Class_getFields( JNIEnv* env,  jclass clazz)  
{ 
    jfieldID fieldID; 
    tClassLoaderTuple* classPtr; 
    jclass clazzClazz, fieldClazz; 
    int numInst = 0, numStatic = 0, i = 0, numPublicInst = 0, numPublicStatic = 0; 
    int j = 0; 
    int startStats; 
    jarray retArray; 
 
    //We need the clazz clazz!!! 
    clazzClazz = (*env)->GetObjectClass(env, clazz); 
 
    assert(clazzClazz); 

    fieldClazz = (*env)->FindClass(env, "java/lang/reflect/Field");

    assert(fieldClazz);
 
    fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
 
    assert(fieldID); 
    classPtr = (tClassLoaderTuple*) (*env)->GetIntField(env, clazz, fieldID); 
    assert(classPtr); 
 
    numInst = classPtr->pstClass->u16InstCount; 
    numStatic = classPtr->pstClass->u16StatCount; 
 
    //We must first see which ones are public! 
    for(i = 0; i < numInst;i++) 
	    if( classPtr->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		    numPublicInst++; 
 
    for(i = 0; i < numStatic;i++) 
	    if( classPtr->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		    numPublicStatic++; 
 
    retArray = (*env)->NewObjectArray(env,  numPublicInst + numPublicStatic, fieldClazz, NULL);
    assert(retArray); 
    (*env)->NewGlobalRef(env, (jobject) retArray);

    startStats = numPublicInst; 
    i = 0; 
    j = 0; 
    while(numPublicInst > 0) 
	{ 
	    assert(i < classPtr->pstClass->u16InstCount); 
	    if( classPtr->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		{ 
		    tOBREF fieldObject = INTERP_NewObjectFromName(env, "java/lang/reflect/Field"); 
//		    eprintf("%i field %s\n", i, (classPtr->pstClass->pstInstOffsets[i]).uidFieldName); 
		    constructFieldObject( env, fieldObject, clazz, (classPtr->pstClass->pstInstOffsets[i]).uidFieldName, i , 0 /*not static */); 
		    ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
		    numPublicInst--; 
		} 
	    i++; 
	} 
 
    i = 0; 
    while(numPublicStatic > 0) 
	{ 
	    assert(i < classPtr->pstClass->u16StatCount); 
	    if( classPtr->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		{ 
		    tOBREF fieldObject = INTERP_NewObjectFromName(env, "java/lang/reflect/Field"); 
//		    eprintf( "%i field %s\n", i, (classPtr->pstClass->pstStatOffsets[i]).uidFieldName); 
		    constructFieldObject( env, fieldObject, clazz, (classPtr->pstClass->pstStatOffsets[i]).uidFieldName, i, 1 /*static */ ); 
		    ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
		    numPublicStatic--; 
		} 
	    i++; 
	} 
 
    return retArray; 
}
 
jobjectArray Java_java_lang_Class_getDeclaredFields( JNIEnv* env,  jclass clazz)  
{ 
  //just duplicating the code from getFields
    jfieldID fieldID; 
    tClassLoaderTuple* classPtr; 
    jclass clazzClazz, fieldClazz; 
    int numInst = 0, numStatic = 0, i = 0, numPublicInst = 0, numPublicStatic = 0; 
    int j = 0; 
    int startStats; 
    tARREF retArray; 
 
    //We need the clazz clazz!!! 
    clazzClazz = (*env)->GetObjectClass(env, clazz); 
 
    assert(clazzClazz); 

    fieldClazz = (*env)->FindClass(env, "java/lang/reflect/Field");

    assert(fieldClazz);
 
    fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
 
    assert(fieldID); 
    classPtr = (tClassLoaderTuple*) (*env)->GetIntField(env, clazz, fieldID); 
    assert(classPtr); 
 
    numInst = classPtr->pstClass->u16InstCount; 
    numStatic = classPtr->pstClass->u16StatCount; 
 
    //We must first see which ones are public! 
    for(i = 0; i < numInst;i++) 
	    if( classPtr->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		    numPublicInst++; 
 
    for(i = 0; i < numStatic;i++) 
	    if( classPtr->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		    numPublicStatic++; 
 
    retArray = (*env)->NewObjectArray(env,  numPublicInst + numPublicStatic, fieldClazz, NULL);
    assert(retArray); 

    startStats = numPublicInst; 
    i = 0; 
    j = 0; 
    while(numPublicInst > 0) 
	{ 
	    assert(i < classPtr->pstClass->u16InstCount); 
	    if( classPtr->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		{ 
		    tOBREF fieldObject = INTERP_NewObjectFromName(env, "java/lang/reflect/Field"); 
//		    eprintf("%i field %s\n", i, (classPtr->pstClass->pstInstOffsets[i]).uidFieldName); 
		    constructFieldObject( env, fieldObject, clazz, (classPtr->pstClass->pstInstOffsets[i]).uidFieldName, i , 0 /*not static */); 
		    ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
		    numPublicInst--; 
		} 
	    i++; 
	} 
 
    i = 0; 
    while(numPublicStatic > 0) 
	{ 
	    assert(i < classPtr->pstClass->u16StatCount); 
	    if( classPtr->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) 
		{ 
		    tOBREF fieldObject = INTERP_NewObjectFromName(env, "java/lang/reflect/Field"); 
//		    eprintf("%i field %s\n", i, (classPtr->pstClass->pstStatOffsets[i]).uidFieldName); 
		    constructFieldObject( env, fieldObject, clazz, (classPtr->pstClass->pstStatOffsets[i]).uidFieldName, i, 1 /*static */ ); 
		    ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
		    numPublicStatic--; 
		} 
	    i++; 
	} 
 
    return retArray; 
} 
 
 
/* Helper methods for forName
 */

jclass Java_java_lang_Class_forNameHelperFromStruct(JNIEnv* env, tClassLoaderTuple* pstOurClassType)
{
  tOBREF   pstClassObject; 
  
  if(pstOurClassType == NULL) 
    { 
      (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassNotFoundException")); 
      return NULL; 
    } 
  if(pstOurClassType->classObject)                  
    pstClassObject= pstOurClassType->classObject; 
  else 
    { 
      pstClassObject = INTERP_NewObjectFromName(env, "java/lang/Class"); 
      if(pstClassObject == NULL)
	{
	  (*env)->Throw(env, OOMExceptionObject);
	  return NULL; 
	}
      CLASS_SetClassStruct(pstClassObject, pstOurClassType); 
      pstOurClassType->classObject = pstClassObject;
    } 
  return pstClassObject; 
}

#include "loading_errors.h"

jclass Java_java_lang_Class_forNameHelperFromName(JNIEnv* env, char* pszClassName)
{
  tClassLoaderTuple* pstOurClassType;
  jclass ret;
  uint32 error;
  tJNIData* pstJNIData;
  tClassLoaderTuple* loadingClass = NULL;

  /*  pstJNIData = JNI_getJNIData(sys_current_thread());
  if(pstJNIData != NULL)
      {
	  if(pstJNIData->pstTopFrame != NULL)
	      loadingClass = pstJNIData->pstTopFrame->pstCurrClass;
      }
  */
  pstOurClassType = CLASSFILE_FindOrLoadWithError(env, pszClassName, loadingClass, &error); //XXX tuple
  
  if(pstOurClassType != NULL)
    {
      ret = Java_java_lang_Class_forNameHelperFromStruct(env, pstOurClassType);
      sys_free(pszClassName);
      return ret;
    }
  else 
    { 
      //eprintf("INTERP failed to load class %s\n", pszClassName); 
      (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
      sys_free(pszClassName);
      return NULL; 
    } 
}


 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public static native Class forName(String); 
 * 
 * Returns the Class object associated with the class with the given 
 * string name. 
 *
 * When a primitive type is given as "I" or "Z" or "int" or "void", we look for a user-defined class of that name. (Use the .TYPE field to get these classes, eg java.lang.Boolean.TYPE)
 *
 * If the string starts with a "[" it may contain a type of the form Lsometype;
 * If it doesn't start with a "[", anything of that form is rejected.
 * 
 */ 
 
jclass Java_java_lang_Class_forName 
( 
  JNIEnv* env, 
  jclass clazz, 
  jobject stringObj 
) 
{ 
  char*    pszClassName; 
  tJNIData* pstJNIData;

  //NB don't use the class var here, it may be NULL 
 
  if(stringObj == NULL) 
      { 
	  (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassNotFoundException")); 
	  return NULL; 
      } 

  pstJNIData = JNI_getJNIData(sys_current_thread());
 
#ifndef USEOSKIT 
  assert(stringObj != NULL); 
#endif 
  pszClassName =   INTERP_AscizFromString(env, stringObj); 

  if(strcmp(pszClassName, "") == 0)
  {
    (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
    sys_free(pszClassName);
    return NULL;
  }

  //check if it's an array they want
  if(pszClassName[0] == '[')
    {
      int typeIndex = 0;
      while( typeIndex < (strlen(pszClassName) - 1) && pszClassName[typeIndex] == '[')
	{
	  typeIndex++;
	}
      if((typeIndex - 1) == (strlen(pszClassName) - 1))
	{
	  (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	  sys_free(pszClassName);
	  return NULL;
	}
      else if(pszClassName[typeIndex] == 'L')
	{
	  if(strchr(pszClassName, ';') == NULL)
	    {
	      (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	      sys_free(pszClassName);
	      return NULL;
	    }
	  return Java_java_lang_Class_forNameHelperFromName(env, pszClassName);
	}
      else {
	tClassLoaderTuple* pstOurClassType;
	jclass ret;
	if(strchr(pszClassName, ';') != NULL)
	    {
	      (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	      sys_free(pszClassName);
	      return NULL;
	    }
	//just check that the next character is that of a primitive array
	if(!( (pszClassName[typeIndex] == 'V') ||
	      (pszClassName[typeIndex] == 'B') ||
	      (pszClassName[typeIndex] == 'C') ||
	      (pszClassName[typeIndex] == 'F') ||
	      (pszClassName[typeIndex] == 'S') ||
	      (pszClassName[typeIndex] == 'Z') ||
	      (pszClassName[typeIndex] == 'I') ||
	      (pszClassName[typeIndex] == 'D') ||
	      (pszClassName[typeIndex] == 'J')))
	  {
	      (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	      sys_free(pszClassName);
	      return NULL;
	  }
	 //a primitive array
	 pstOurClassType = CLASSFILE_FindOrLoad(env, pszClassName, pstJNIData->pstTopFrame->pstCurrClass);
	 ret = Java_java_lang_Class_forNameHelperFromStruct(env, pstOurClassType);
	 sys_free(pszClassName);
	 return ret;
      }
    }
  else //not an array
    {
      if(strchr(pszClassName, ';') != NULL)
	{
	  (*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	  sys_free(pszClassName);
	  return NULL;
	}
      return Java_java_lang_Class_forNameHelperFromName(env, pszClassName);
    }
} 

jobjectArray Java_java_lang_Class_getDeclaredConstructors( JNIEnv* env,  jclass clazz)
{  
  return Java_java_lang_Class_getConstructors_Helper(env, clazz, JNI_TRUE); 
}

/* Gets an array of Ljava/lang/reflect/Constructor which represent the (public) constructors for the clazz Class object */ 

jobjectArray Java_java_lang_Class_getConstructors( JNIEnv* env,  jclass clazz) 
{
  return Java_java_lang_Class_getConstructors_Helper(env, clazz, JNI_FALSE); 
}
 

/* private Class clazz;
   private int slot;
   private Class[] parameterTypes;
   private Class[] exceptionTypes;
*/

/* Helper method for the get*Constructors methods
 *
 * if declared is true, then we return private and protected constructors too
 */

jobjectArray Java_java_lang_Class_getConstructors_Helper( JNIEnv* env,  jclass clazz, int declared) { 
 jclass classClazz = CLASS_GetClass(env, clazz);

 tClassLoaderTuple* targetClass = CLASS_GetClassStruct(env, clazz);
 jclass consClazz = (*env)->FindClass(env, "java/lang/reflect/Constructor");
 jobjectArray retval;
 tMethod* method;

 int i = 0;
 int j = 0;
 int numConstructors = 0;

 /* first count them */
 for(i = 0; i < targetClass->pstClass->u16MethodsCount;i++)
   {
     method = targetClass->pstClass->pstMethods + i;
     if(strcmp(method->uidName, "<init>") == 0) 
       { 
	 if( (declared == JNI_TRUE) || ((declared == JNI_FALSE) && (method->u16AccessFlags & ACC_PUBLIC)))
	   {     
	     numConstructors++;
	   }
       }
   }

 /* create our return array */
 retval = (*env)->NewObjectArray(env, numConstructors, consClazz, NULL);
 if(retval == NULL) { (*env)->Throw(env, OOMExceptionObject); return NULL; }

 //now make objects for each one
 for(i = 0; i < targetClass->pstClass->u16MethodsCount;i++)
   {
     int k = 0;
     jobjectArray exceptionTypeArray;

     method = targetClass->pstClass->pstMethods + i;
     if(strcmp(method->uidName, "<init>") == 0) 
       { 
	 if( (declared == JNI_TRUE) || ((declared == JNI_FALSE) && (method->u16AccessFlags & ACC_PUBLIC)))
	   { 
	     jfieldID clazzField;
	     jfieldID slotField;
	     jfieldID parametersField;
	     jfieldID exceptionsField;

	     jobject consObject;

	     clazzField = (*env)->GetFieldID(env, consClazz, "clazz", "Ljava/lang/Class;");
	     slotField = (*env)->GetFieldID(env, consClazz, "slot", "I");
	     parametersField = (*env)->GetFieldID(env, consClazz, "parameterTypes", "[Ljava/lang/Class;");
	     exceptionsField = (*env)->GetFieldID(env, consClazz, "exceptionTypes", "[Ljava/lang/Class;");

	     if((clazzField == NULL) || (slotField == NULL) || (parametersField == NULL) || (exceptionsField == NULL))
	       {
		 (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Couldn't get field for java.lang.reflect.Constructor"));
		 return NULL;
	       }


	     consObject = INTERP_NewObject(env, CLASS_GetClassStruct(env, consClazz));

	     if(consObject == NULL) { (*env)->Throw(env, OOMExceptionObject); return NULL; }

	     (*env)->SetObjectField(env, consObject, clazzField, clazz);
	     (*env)->SetIntField(env, consObject, slotField, i);
	     (*env)->SetObjectField(env, consObject, parametersField, (jobject) CLASS_GetClassArrayFromSig(env, method->uidSignature, method->pstClass));

	     if(method->bHaveResolvedExceptions != 1)
	       {
		 if(CLASSFILE_ResolveMethodExceptions(env, method) != 0)
		   {
		     return NULL; //exception has been thrown
		   }
	       }
	     assert(method->bHaveResolvedExceptions == 1);

	     exceptionTypeArray = (*env)->NewObjectArray(env, method->u16NumExceptions, classClazz, NULL);
	     for(k = 0; k < method->u16NumExceptions;k++)
	       {
		 (*env)->SetObjectArrayElement(env, exceptionTypeArray, k, CLASS_MakeClassObject(env, method->pstExceptions[k]));
	       }
	     (*env)->SetObjectField(env, consObject, exceptionsField, (jobject) exceptionTypeArray);
	     (*env)->SetObjectArrayElement(env, retval, j++, consObject);
	   }
       }
   }
 return retval;
}

/* Returns a single constructor object */

jobject Java_java_lang_Class_getConstructor_Helper( JNIEnv* env, jclass clazz, jobjectArray args, int declared) { 
 jclass classClazz = CLASS_GetClass(env, clazz);

 tClassLoaderTuple* targetClass = CLASS_GetClassStruct(env, clazz);
 jclass consClazz = (*env)->FindClass(env, "java/lang/reflect/Constructor");
 tMethod* method;
 int i = 0;

 jobject consObject = INTERP_NewObject(env, CLASS_GetClassStruct(env, consClazz));
 if(consObject == NULL) { (*env)->Throw(env, OOMExceptionObject); return NULL; }


 for(i = 0; i < targetClass->pstClass->u16MethodsCount;i++)
   {
     int k = 0;
     jobjectArray exceptionTypeArray;

     method = targetClass->pstClass->pstMethods + i;
     if(strcmp(method->uidName, "<init>") == 0) 
       { 
	 if( (declared == JNI_TRUE) || ((declared == JNI_FALSE) && (method->u16AccessFlags & ACC_PUBLIC)))
	   { 
	     jfieldID clazzField;
	     jfieldID slotField;
	     jfieldID parametersField;
	     jfieldID exceptionsField;

	     /* check whether the args match */
	     if( CLASS_CompareClassArrays(env, args, CLASS_GetClassArrayFromSig(env, method->uidSignature, method->pstClass)) == 0)
	       {
	       
	     clazzField = (*env)->GetFieldID(env, consClazz, "clazz", "Ljava/lang/Class;");
	     slotField = (*env)->GetFieldID(env, consClazz, "slot", "I");
	     parametersField = (*env)->GetFieldID(env, consClazz, "parameterTypes", "[Ljava/lang/Class;");
	     exceptionsField = (*env)->GetFieldID(env, consClazz, "exceptionTypes", "[Ljava/lang/Class;");

	     if((clazzField == NULL) || (slotField == NULL) || (parametersField == NULL) || (exceptionsField == NULL))
	       {
		 (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Couldn't get field for java.lang.reflect.Constructor"));
		 return NULL;
	       }


	     if(consObject == NULL) { (*env)->Throw(env, OOMExceptionObject); return NULL; }

	     (*env)->SetObjectField(env, consObject, clazzField, clazz);
	     (*env)->SetIntField(env, consObject, slotField, i);
	     (*env)->SetObjectField(env, consObject, parametersField, (jobject) CLASS_GetClassArrayFromSig(env, method->uidSignature,method->pstClass));

	     if(method->bHaveResolvedExceptions == 0)
	       {
		 CLASSFILE_ResolveMethodExceptions(env, method);
	       }
	     assert(method->bHaveResolvedExceptions == 1);

	     exceptionTypeArray = (*env)->NewObjectArray(env, method->u16NumExceptions, classClazz, NULL);
	     for(k = 0; k < method->u16NumExceptions;k++)
	       {
		 (*env)->SetObjectArrayElement(env, exceptionTypeArray, k, CLASS_MakeClassObject(env, method->pstExceptions[k]));
	       }
	     (*env)->SetObjectField(env, consObject, exceptionsField, (jobject) exceptionTypeArray);
	     return consObject;
	       }
	   }
       }
   }

 (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NoSuchMethodException", "Constructor not found"));
 return NULL;
}

jobject Java_java_lang_Class_getConstructor( JNIEnv* env,  jclass clazz, jobjectArray args) { 
  return Java_java_lang_Class_getConstructor_Helper(env, clazz, args, JNI_FALSE);
}

jobject Java_java_lang_Class_getDeclaredConstructor( JNIEnv* env,  jclass clazz, jobjectArray args) { 
  return Java_java_lang_Class_getConstructor_Helper(env, clazz, args, JNI_TRUE);
}


 
/* Gets a public method. This means we should look in the VT.

   It can also be a static method, do we have to check superclasses? 

   The slot index must be relative to the declaringClass
 */
 
jobject Java_java_lang_Class_getMethod( JNIEnv* env,  jclass clazzObj, jstring name, jobjectArray argsObj)  
{ 
    tArray* args; 
    jfieldID classStruct_fieldID; 
    tClassLoaderTuple* classPtr; 
    jclass clazzClazz; 
    int i; 
    jobject retmethod = INTERP_NewObjectFromName(env, "java/lang/reflect/Method"); 
    int iFound = 0; 
    char* uidName ; 
    char* nameTemp;
    tMethod* method;
    char* testsig;

    //Quickly check if any of the parameter args is null 
    if(argsObj == NULL) 
	{ 
	    (*env)->ThrowNew(env, (*env)->FindClass(env,"java/lang/NoSuchMethodException"), "null argument passed for parameter types"); 
	    return NULL; 
	} 
 
    args = ADEREF(argsObj); 
 
    for(i = 0; i < args->i32Number;i++) 
	{ 
	    if( ((tOBREF*) args->pvElements)[i] == NULL) 
		{ 
		    (*env)->ThrowNew(env, (*env)->FindClass(env,"java/lang/NoSuchMethodException"), "null argument passed in the parameter type array"); 
		    return NULL; 
		} 
	} 
 
    //We need the clazz clazz!!! 
    clazzClazz = (*env)->GetObjectClass(env, clazzObj); 
 
    assert(clazzClazz); 
 
    classStruct_fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
 
    assert(classStruct_fieldID); 
    classPtr = (tClassLoaderTuple*) (*env)->GetIntField(env, clazzObj, classStruct_fieldID); 
    assert(classPtr); 
    //Scan through the methods and look for the Name 

    nameTemp = INTERP_AscizFromString(env, name);
    uidName = UID_GetUid(nameTemp);
    sys_free(nameTemp);
//new version start
    testsig = CLASS_GetSigFromClassArray(env, args); 
    method = CLASSFILE_FindMethodInVT(env, classPtr, uidName, testsig );
    
    if(method == NULL)
      {
	tClassLoaderTuple* searchClass = classPtr;
	iFound = 0;
	//look for static method, we search the superclasses too
	while(searchClass != NULL)
	{
	for(i = 0; i < searchClass->pstClass->u16MethodsCount;i++)
	{
	 if(searchClass->pstClass->pstMethods[i].u16AccessFlags & ACC_STATIC)
	  {
	  if( strncmp(testsig, searchClass->pstClass->pstMethods[i].uidSignature, strlen(testsig)) == 0)
	    {	
	       	jclass methodClazz = CLASS_GetClass(env, retmethod);  	//Construct a new Method Object and fill in vars 
		jfieldID fieldID = (*env)->GetFieldID(env, methodClazz, "declaringClass","Ljava/lang/Class;"); 
		jfieldID slot_fieldID = (*env)->GetFieldID(env, methodClazz, "slot","I"); 
		jfieldID isstatic_fieldID = (*env)->GetFieldID(env, methodClazz, "isstatic","I"); 
		jfieldID name_fieldID = (*env)->GetFieldID(env, methodClazz, "name","Ljava/lang/String;"); 
		int slotIndex = i;

		iFound = 1;
	
		assert(slotIndex >= 0);
		assert(slotIndex < searchClass->pstClass->u16MethodsCount);
		(*env)->SetObjectField(env, retmethod, fieldID, CLASS_MakeClassObject(env, searchClass)); 
		(*env)->SetIntField(env, retmethod, slot_fieldID, slotIndex);
		(*env)->SetIntField(env, retmethod, isstatic_fieldID, 1);
	        (*env)->SetIntField(env, retmethod, name_fieldID, (int) INTERP_NewStringFromAsciz( env, uidName)); 
		return retmethod;
	    }

	  }
	}
	 searchClass = searchClass->pstClass->pstSuperClass;
	}
      }
    else
      {
  	jclass methodClazz = CLASS_GetClass(env, retmethod);  	//Construct a new Method Object and fill in vars 
	jfieldID fieldID = (*env)->GetFieldID(env, methodClazz, "declaringClass","Ljava/lang/Class;"); 
	jfieldID slot_fieldID = (*env)->GetFieldID(env, methodClazz, "slot","I"); 
	jfieldID isstatic_fieldID = (*env)->GetFieldID(env, methodClazz, "isstatic","I"); 
	jfieldID name_fieldID = (*env)->GetFieldID(env, methodClazz, "name","Ljava/lang/String;"); 
	int slotIndex = -1;
	jclass origClazz;
	iFound = 1;
	
	assert(CLASS_getDeclaringClass(env, classPtr, method, &origClazz, &slotIndex) == 0);
	assert(slotIndex >= 0);
	(*env)->SetObjectField(env, retmethod, fieldID, origClazz); 
	(*env)->SetIntField(env, retmethod, slot_fieldID, slotIndex);
	(*env)->SetIntField(env, retmethod, isstatic_fieldID, 0);
        (*env)->SetIntField(env, retmethod, name_fieldID, (int) INTERP_NewStringFromAsciz( env, uidName)); 
	return retmethod;
      }

    if(iFound) 
	return retmethod; 
//    eprintf("METHOD NOT FOUND %s.%s\n", classPtr->uidName, uidName); 
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodException")); 
    return NULL; 
} 
 
jobject Java_java_lang_Class_getDeclaredMethod( JNIEnv* env,  jclass clazzObj, jstring name, jobjectArray argsObj)  
{ 
    jfieldID classStruct_fieldID; 
    tClassLoaderTuple* classPtr; 
    jclass clazzClazz; 
    tArray* args;
    jobject retmethod;
    int i = 0;
    int iFound = 0;
    Uid uidName = NULL;
    char* pszName = NULL;
    Uid testsig = NULL;

    //Quickly check if any of the parameter args is null 
    if(argsObj == NULL) 
	{ 
	    (*env)->ThrowNew(env, (*env)->FindClass(env,"java/lang/NoSuchMethodException"), "null argument passed for parameter types"); 
	    return NULL; 
	} 
 
    args = ADEREF(argsObj); 
 
    for(i = 0; i < args->i32Number;i++) 
	{ 
	    if( ((tOBREF*) args->pvElements)[i] == NULL) 
		{ 
		    (*env)->ThrowNew(env, (*env)->FindClass(env,"java/lang/NoSuchMethodException"), "null argument passed in the parameter type array"); 
		    return NULL; 
		} 
	} 

    //We need the clazz clazz!!! 
    clazzClazz = (*env)->GetObjectClass(env, clazzObj); 
 
    assert(clazzClazz); 
 
    classStruct_fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
 
    assert(classStruct_fieldID); 
    classPtr = (tClassLoaderTuple*) (*env)->GetIntField(env, clazzObj, classStruct_fieldID); 
    assert(classPtr); 

    pszName = INTERP_AscizFromString(env, name);
    uidName = UID_GetUid(pszName);

    assert(uidName);

    testsig = CLASS_GetSigFromClassArray(env, args); 

    for(i = 0; i < classPtr->pstClass->u16MethodsCount;i++) 
      { 
	tMethod* method = classPtr->pstClass->pstMethods + i; 
	if(uidcmp(method->uidName, uidName) == 0) 
	  { 
	    if(strncmp(method->uidSignature, testsig, strlen(testsig)) == 0) 
	      { 
		jfieldID fieldID;
		jfieldID slot_fieldID;
		jfieldID isstatic_fieldID;
		jfieldID name_fieldID;
		//Construct a new Method Object and fill in vars 
		    
		jclass methodClazz = (*env)->FindClass(env, "java/lang/reflect/Method");
		retmethod = (*env)->AllocObject(env, methodClazz);
		
		
		fieldID = (*env)->GetFieldID(env, methodClazz, "declaringClass","Ljava/lang/Class;"); 
		slot_fieldID = (*env)->GetFieldID(env, methodClazz, "slot","I"); 
		isstatic_fieldID = (*env)->GetFieldID(env, methodClazz, "isstatic","I"); 
		name_fieldID = (*env)->GetFieldID(env, methodClazz, "name","Ljava/lang/String;"); 

		iFound  = 1; 
		
		//We must find the correct slot in the virtual table
	       if(method->u16AccessFlags & ACC_STATIC)
		 {
		   (*env)->SetIntField(env, retmethod, isstatic_fieldID, 1); 
		   (*env)->SetIntField(env, retmethod, slot_fieldID, i); 
		   (*env)->SetObjectField(env, retmethod, fieldID, clazzObj); 
		 }
	       else
		 {
		   jclass origClazz;
		   int slotIndex = -1;
		   assert(CLASS_getDeclaringClass(env, classPtr, method, &origClazz, &slotIndex) == 0);
		   assert(slotIndex >= 0);
		   (*env)->SetIntField(env, retmethod, isstatic_fieldID, 0); 
		   (*env)->SetIntField(env, retmethod, slot_fieldID, slotIndex); 
		   (*env)->SetObjectField(env, retmethod, fieldID, origClazz); 
		 }	     

		(*env)->SetObjectField(env, retmethod, name_fieldID, INTERP_NewStringFromAsciz(env,  method->uidName)); 
		return retmethod;
	      }
	  }
      } //end for
	
     //no such method found 
     (*env)->ThrowNew(env, (*env)->FindClass(env,"java/lang/NoSuchMethodException"), "no method found for this name and argument set"); 
     return NULL;
} 

	/* Get all public methods from this class. 
	 * 
	 * That means: 
	 * 
	 * Don't get constructors 
	 * Don't get non-public methods 
	 * Get methods from all superclasses (!!) 
	 * 
	 */ 

jobjectArray Java_java_lang_Class_getMethods( JNIEnv* env,  jclass clazz)  
{ 
  return Java_java_lang_Class_getMethods_Helper(env, clazz, 0); //public methods
}

jobjectArray Java_java_lang_Class_getDeclaredMethods( JNIEnv* env,  jclass clazz)  
{
 return Java_java_lang_Class_getMethods_Helper(env, clazz, 1); //declared methods
}
/* Helper method for getting methods */ 

jobjectArray Java_java_lang_Class_getMethods_Helper( JNIEnv* env,  jclass clazz, jint declared)  
{ 
    tClassLoaderTuple* theClass, *saveClass; 
    tMethod* method; 
    jobjectArray retval; 
    jclass methodClazz;

    int methodCounter = 0; 
    int numFound = 0; 
    int j = 0; 
 
    jclass clazzClazz ; 
    jfieldID fieldID; 
 
    clazzClazz = (*env)->GetObjectClass(env, clazz); 
    fieldID = (*env)->GetFieldID(env, clazzClazz, "_classStruct", "I"); 
     
    assert(fieldID); 

    methodClazz = (*env)->FindClass(env, "java/lang/reflect/Method");
    assert(methodClazz);

    saveClass = theClass = (tClassLoaderTuple*) (*env)->GetIntField(env, clazz, fieldID); 
 
    assert(theClass); 
 
    while(theClass != NULL) 
      { 
	for(methodCounter = 0; methodCounter < theClass->pstClass->u16MethodsCount;methodCounter++) 
	  { 
	    method = (tMethod*) (&theClass->pstClass->pstMethods[methodCounter]); 
	    if( (strcmp(method->uidName, "<init>") == 0) || (strcmp(method->uidName, "<clinit>") == 0) ) 
	      { 
		
	      } 
	    else 
	      { 
		if(declared)
		  {
		  if(method->u16AccessFlags & ACC_PUBLIC) 
		    {
		      if(method->pstClass == saveClass)
			numFound++; 
		    }
		  }
		else
		  numFound++; 
	      } 
	  } 
	theClass = theClass->pstClass->pstSuperClass; 
      } 
 
    theClass = saveClass;  
    
    retval = (*env)->NewObjectArray(env, numFound, methodClazz, NULL);
    
    while(numFound > 0) 
      { 
	while(theClass != NULL) 
	  { 
	    
	    for(methodCounter = 0; methodCounter < theClass->pstClass->u16MethodsCount;methodCounter++) 
	      { 
		  method = (tMethod*) (&theClass->pstClass->pstMethods[methodCounter]); 
		  if((strcmp(method->uidName, "<init>") == 0) || (strcmp(method->uidName, "<clinit>") == 0) ) 
		    { 
		      
		    } 
		  else  
		    { 
		      int useMethod = 0;
		      if(declared)
			{
			  if( method->u16AccessFlags & ACC_PUBLIC) 
			    {
			      if(method->pstClass == saveClass)
				useMethod = 1;
			    }
			}
		      else
			{
			useMethod = 1;
			}

		      if(useMethod)
			{ 
 
			  jfieldID fieldID; 
			  
 
			  jobject o = INTERP_NewObjectFromName(env, "java/lang/reflect/Method"); 
			  jclass methodClazz = (*env)->GetObjectClass(env, o); 
			  
			  numFound --; 
			  //	   fprintf(stderr, "Adding method %s at %i\n", method->uidName, j); 
			  
			  fieldID = (*env)->GetFieldID(env, methodClazz, "declaringClass", "Ljava/lang/Class;"); 
			  assert(fieldID); 
			  (*env)->SetObjectField(env, o, fieldID, CLASS_MakeClassObject(env, theClass));
			  
			  //Set the name 
			  fieldID = (*env)->GetFieldID(env, methodClazz, "name", "Ljava/lang/String;"); 
			  assert(fieldID); 
 
			  (*env)->SetObjectField(env, o, fieldID, INTERP_NewStringFromAsciz(env, method->uidName)); 
			  //Set the slot 
			  fieldID = (*env)->GetFieldID(env, methodClazz, "slot", "I"); 
			  assert(fieldID); 
 
			  (*env)->SetIntField(env, o, fieldID, methodCounter); //(int32) method); 
		      
		          (*env)->SetObjectArrayElement(env, retval, j, o);
			  j++; 

			  //we call this to make sure the exception types can be loaded
			  if(java_lang_reflect_Method_getExceptionTypes(env, o) == (jobjectArray) NULL)
			    {
			      //an exception has occurred
			      return NULL;
			    }
		    }
		    } //end else (init / /clinit)
	      }
	    theClass = theClass->pstClass->pstSuperClass; 
	    
	  }
      } 
    
    assert(retval != NULL); 
   return retval;    
}
