/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "jni.h"
#include "assert.h"
#include "newobject.h"
#include "interp_methods.h"
#include <string.h>

#include "lib/indigenous/java.lang/java_lang_Class.h"  
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "lib/indigenous/java.lang/VMClassLoader.h" 
#include "garbage.h"

#define VOIDSIG (void (*) (JNIEnv*, ...))
#define BYTESIG (jbyte (*) (JNIEnv*, ...))
#define SHORTSIG (jshort (*) (JNIEnv*, ...))
#define INTSIG (jint (*) (JNIEnv*, ...))
#define LONGSIG (jlong (*) (JNIEnv*, ...))
#define DOUBLESIG (jdouble (*) (JNIEnv*, ...))

//methods stolen for gcj for creating the method names

// Add a character to the buffer, encoding properly.
static void
add_char (char *buf, jchar c, int *here)
{
  int i ;

  if (c == '_')
    {
      buf[(*here)++] = '_';
      buf[(*here)++] = '1';
    }
  else if (c == ';')
    {
      buf[(*here)++] = '_';
      buf[(*here)++] = '2';
    }
  else if (c == '[')
    {
      buf[(*here)++] = '_';
      buf[(*here)++] = '3';
    }

  // Also check for `.' here because we might be passed an internal
  // qualified class name like `foo.bar'.
  else if (c == '/' || c == '.')
    buf[(*here)++] = '_';
  else if ((c >= '0' && c <= '9')
           || (c >= 'a' && c <= 'z')
           || (c >= 'A' && c <= 'Z'))
    buf[(*here)++] = (char) c;
  else
    {
      // "Unicode" character.
      buf[(*here)++] = '_';
      buf[(*here)++] = '0';
      for (i = 0; i < 4; ++i)
        {
          int val = c & 0x0f;
          buf[(*here) + 3 - i] = (val > 10) ? ('a' + val - 10) : ('0' + val);
          c >>= 4;
        }
      *here += 4;
    }
}



// Compute a mangled name for a native function.  This computes the
// long name, and also returns an index which indicates where a NUL
// can be placed to create the short name.  This function assumes that
// the buffer is large enough for its results.
static void
mangled_name (char* buf, char* chars, char* func_name, char* signature, long* long_start)
     //jclass klass, _Jv_Utf8Const *func_name,
     //              _Jv_Utf8Const *signature, char *buf, int *long_start)
{
  int here = 5;
  int i = 0;

  const unsigned char *fn = (const unsigned char *) func_name;
  const unsigned char *limit = fn + strlen(func_name); 
  const unsigned char *sig = (const unsigned char *) signature;

  jint len = strlen(chars);

  strcpy (buf, "Java_");
  for (i = 0; i < len; ++i)
    add_char (buf, chars[i], &here);

  // Don't use add_char because we need a literal `_'.
  buf[here++] = '_';


  for (i = 0; ; ++i)
    {
      int ch = fn[i]; //UTF8_GET (fn, limit);
      if (ch <= 0)
        break;
      add_char (buf, ch, &here);
    }

  // This is where the long signature begins.
  *long_start = here;

  if( (buf[here - 1]  == '(') && (buf[here] == ')'))
    {
      buf[here + 1] = '\0';
      return;
    }

  buf[here++] = '_';
  buf[here++] = '_';

  

  limit = sig + strlen(signature);
  assert(sig[0] == '(');
  ++sig;
  while (1)
    {
      int ch = sig[0];
      if (ch == ')' || ch <= 0)
        break;
      add_char (buf, ch, &here);
      sig++;
    }

  buf[here] = '\0';
}



extern int INTERP_FirstThreadRunning;
extern tJNIData* tempJNIData; //We keep this around to have JNI data while the first thread hasn't started yet
/*
 * @doc FUNC
 * @func
 * This function is called when a native method needs to be invoked. The
 * tMethod pointer is passed to it, along with the current stack frame. A
 * search is done to find the method is the class's native methods. When it
 * has been found it has to be cast to the correct type (return type and
 * number of parameters). The parameters that are passed to it are taken from
 * the operand stack (accessed via the current stack frame). The method is
 * invoked and its return value pushed back onto the operand stack. If an
 * exception is thrown, the function returns -1 and the interpreter catches
 * the exception.
 *
 */

//#ifdef DEBUG_TRACE 
void StackInspect(tStackFrame*, int32*);
//#endif

extern tClassLoaderTuple* pstObjectType; // this can be globally used to get the type for object
extern tAllocatorHeap* system_heap;

int JNI_CallNativeMethod
(
  tStackFrame* pstFrame,  /* @parm Current Java stack frame */
  tMethod* pstMethod      /* @parm Method to invoke */
)
{
  void* pFunction = NULL;
  void (*pVoidFunc) (JNIEnv*, ...);
  jbyte (*pByteFunc) (JNIEnv*, ...);
  jshort (*pShortFunc) (JNIEnv*, ...);
  jint (*pIntFunc) (JNIEnv*, ...);
  jlong (*pLongFunc) (JNIEnv*, ...);
  jdouble (*pDoubleFunc) (JNIEnv*, ...);
  int i, j, k;
  int iFound = 0;
  int jniiter = 0;
  char* pszClassName = pstMethod->pstClass->uidName;
  jvalue jvRetVal;
  tJNIData* pstJNIData = JNI_getJNIData( sys_current_thread());
  int32* ot;
  char* pszRetChar;
  tClassLoaderTuple* theClass;
  JNIEnv* env;

  env = &pstJNIData->functions;

  assert(pstJNIData);
 
  /* record stack frame */
  if (pstJNIData)
  {
    pstJNIData->pstTopFrame = pstFrame;
  }
  else
    {
	eprintf( "No pstJNIData in call to %s.%s\n", pstMethod->pstClass->uidName, pstMethod->uidName); 
    }
  
  theClass = pstMethod->pstClass;

  if(pszClassName == NULL)
    {
        (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodException"));
	return -1;
    }

  if (pszClassName[0] == '[')
  {
      theClass = pstObjectType;
      pszClassName = "java/lang/Object";
  }

  //moved preload to back
  /* first find the method... use UID comparisons */
  
  
  //      fprintf(stderr, "{-------------- looking for method %s (%i methods)\n",  pstMethod->uidName, theClass->u16NumNatives);
  for (i = 0; i < theClass->pstClass->u16NumNatives; i++)
  {
    //assert( strcmp( theClass->uidName, 
    if (theClass->pstClass->pstNativeMethods[i].fnPtr)
    {
//      eprintf( "%i Comping %s and %s\n", i, pstMethod->pstClass->pstNativeMethods[i].name , pstMethod->uidName);
      if (theClass->pstClass->pstNativeMethods[i].name == pstMethod->uidName)
      {
        if (theClass->pstClass->pstNativeMethods[i].signature == pstMethod->uidSignature)
        {
          iFound = 1;
          pFunction = theClass->pstClass->pstNativeMethods[i].fnPtr;
          break;
        }
      }
    }
  }

  //      fprintf(stderr, "--------------} found method %s: %i\n",  pstMethod->uidName, iFound);
  if (!iFound)
  {
    



    #ifdef DLOPEN
    {
      int loop = 0;
      int jFound = 0;
      long offset = 0;
      char symbolName[256];
      char* iter;

      memset(symbolName, 0, 256);

      //      eprintf( "Loading native method dynamically");

      mangled_name(symbolName, pstMethod->pstClass->uidName, pstMethod->uidName, pstMethod->uidSignature, &offset);
      /* my old code
      strcpy(symbolName, "Java_");
      iter = strtok(pstMethod->pstClass->uidName, "/");
	  while(iter != NULL)
	    {
	      strcat(symbolName, iter);
	      iter = strtok(NULL, "/");
	      //      if(iter)
 	      strcat(symbolName, "_");

	    }
	  strcat(symbolName, pstMethod->uidName);
      */
      //	  eprintf("Looking up method %s\n", symbolName);

      for(loop = 0; loop < JNI_NumberLoadedLibraries;loop++)
	{
	  void* result = dlsym ( JNI_DynLoadedLibraries[loop], symbolName);
	  if(result != NULL)
	      {
		//	  eprintf( "Got symbol! \n");
		  jFound = 1;
		  pFunction = result;
		  break;
	      }
	}
      
      if(jFound)
	{

	}
      else
	{
	  //try the short name
	  symbolName[offset] = 0;
	  for(loop = 0; loop < JNI_NumberLoadedLibraries;loop++)
	    {
	      void* result = dlsym ( JNI_DynLoadedLibraries[loop], symbolName);
	      if(result != NULL)
		{
		  //		  eprintf( "Got symbol! \n");
		  jFound = 1;
		  pFunction = result;
		  break;
		}
	    } 

	  if(!jFound)
	    {
	      eprintf( "Error: unknown native method called: %s.%s%s %s\n", pszClassName, pstMethod->uidName, pstMethod->uidSignature, pstMethod->pstClass->uidName);
	      eprintf( "There were %i methods for this class\n", pstMethod->pstClass->pstClass->u16NumNatives);
	      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/LinkageError", "Could not link to native method"));
	      return -1;
	    }
	}
    }

    #else
        (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodException"));
	return -1;
    #endif


  }

  #ifdef LOG_NATIVE_CALLS
  //log this method call
  {
    FILE* logfile = fopen("./log/native_call.log.txt", "a");
    assert(logfile);
    fprintf(logfile, "%s.%s\n", pstMethod->pstClass->uidName, pstMethod->uidName);
    fclose(logfile);
  }
  #endif

  /* reset optop to just before params */
  pstFrame->pi32OpTop -= pstMethod->u16ArgSize;
  ot = pstFrame->pi32OpTop;

  if (pstJNIData)
  {
    /* allocate memory for local references */
    pstJNIData->u16NumLocalRefs = pstMethod->u16ArgSize + 1;
    pstJNIData->ppstLocalRefs = (tOBREF*) sys_malloc(sizeof(tOBREF) * (pstMethod->u16ArgSize + 1));
    assert(pstJNIData->ppstLocalRefs);
    pstJNIData->u16NumLocalRefsUsed = 1;
    memset(pstJNIData->ppstLocalRefs, 0, sizeof(tOBREF) * (pstMethod->u16ArgSize + 1));

    /* put object and array references into the local reference thingy */
    if (pstMethod->u16AccessFlags & ACC_STATIC)
    {
	//Make sure we have a reference to the pstClassObject
	if(pstMethod->pstClass->classObject == NULL)
	    {
	      if((strcmp(pstMethod->pstClass->uidName, "java/lang/Class") == 0) && (pstMethod->pstClass->classLoader == NULL))
		{
		  //don't make a class object
		}
		else
		{
		CLASS_MakeClassObject(env, pstMethod->pstClass);
		assert(pstMethod->pstClass->classObject != NULL);
		}
	      //FindClass(env, pstMethod->pstClass->uidName);

	    }
       if(pstJNIData->ppstLocalRefs)
	   {
	       pstJNIData->ppstLocalRefs[0] = pstMethod->pstClass->classObject;
	   }
       else
	 {
	  eprintf( "No local refs in native call?\n");
	  assert( 3 == 2);
	 }
    }
    else
    {
      /* put the "this" pointer into the local reference table */
      pstJNIData->ppstLocalRefs[0] = (tOBREF) ot[1];
    }

#ifndef TEASEME
   assert(GARBAGE_InHeap(env, pstJNIData->ppstLocalRefs[0]) || (pstJNIData->ppstLocalRefs[0] == NULL));
#endif
   if(pstMethod->u16AccessFlags & ACC_STATIC)
	j = 1; 
    else
	j = 2;

     k = 1;

    for (i = 1; pstMethod->uidSignature[i] != ')'; i++)
    {
    assert(pstJNIData->u16NumLocalRefsUsed <= pstJNIData->u16NumLocalRefs);      
      switch (pstMethod->uidSignature[i])
      {
      case 'J':
      case 'D':
	  j++;
	  break;
        case '[':
        {
	    //	    fprintf(stderr, "j is now %i (in [)\n", j);
	    //	    fprintf(stderr, "Copying local ref [ from %i to %i, value %x\n", j, k, (int) ot[j]);
	    if(strcmp(pstMethod->uidName, "initFromTwosCompByteArray") == 0)
	      {
		eprintf("k is %i\n", k);
	      }
          pstJNIData->ppstLocalRefs[k++] = (tOBREF) ot[j];
         if( pstJNIData->ppstLocalRefs[k - 1])
		assert(GARBAGE_InHeap(env, pstJNIData->ppstLocalRefs[k - 1]));
	  pstJNIData->u16NumLocalRefsUsed = k;
          /* skip all the ['s */
          while (pstMethod->uidSignature[i] == '[')
          {
            i++;
          }
          /* skip the type - either object or a single letter */
          if (pstMethod->uidSignature[i] == 'L')
          {
            while (pstMethod->uidSignature[i] != ';')
            {
              i++;
            }
          }
          else
          {
//            i++;
//i will be incremented by the loop
          }
          break;
        }
        case 'L':
        {
	    //	    fprintf(stderr, "Copying local ref from %i to %i, value %x\n", j, k, (int) ot[j]);
	    pstJNIData->ppstLocalRefs[k++] = (tOBREF) ot[j];
	    if((tOBREF) ot[j])
	      if(GARBAGE_InHeap(env, pstJNIData->ppstLocalRefs[k - 1]) != 1)
		{
		  eprintf("Garbage InHeap() failed for %p\n", pstJNIData->ppstLocalRefs[k - 1]);
		}
	    pstJNIData->u16NumLocalRefsUsed = k;
	    while (pstMethod->uidSignature[i] != ';')
	      {
		i++;
	      }
          break;
        }
      default:
	break;
      } //end switch

      j++;
      //      eprintf("j is now %i\n", j);
    }
    pstJNIData->u16NumLocalRefsUsed = k;
    assert(pstJNIData->u16NumLocalRefsUsed <= pstJNIData->u16NumLocalRefs);
//    fprintf(stderr, "Setting num local refs used to %i, avail %i\n", pstJNIData->u16NumLocalRefsUsed, pstJNIData->u16NumLocalRefs);

#ifndef TEASEME
  for(jniiter = 0; jniiter < pstJNIData->u16NumLocalRefsUsed;jniiter++)
      if(pstJNIData->ppstLocalRefs[jniiter] != NULL)
	  {
	    if(GARBAGE_InSpecifiedHeap(system_heap, pstJNIData->ppstLocalRefs[jniiter]) == 0)
	      assert( GARBAGE_InHeap(env, pstJNIData->ppstLocalRefs[jniiter]));
	  }
#endif
  }




  /* find return type char */
  pszRetChar = strchr(pstMethod->uidSignature, ')');
  pszRetChar += 1;

  
  switch (pstMethod->u16ArgSize)
  {
    case 0:
    {
      switch (*pszRetChar)
      {
        case 'V':
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject);
	      }
	  else
	      (*pVoidFunc) (env);
          break;
        }
        case 'Z':
        case 'B':
        {
          pByteFunc = BYTESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject);
	      }
	  else
	      jvRetVal.b = (*pByteFunc) (env);
          break;
        }
        case 'C':
        case 'S':
        {
          pShortFunc = SHORTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.s = (*pShortFunc) ( env, pstMethod->pstClass->classObject );
	      }
	  else
	    jvRetVal.s = (*pShortFunc) ( env );
          break;
        }
        case 'I':
        case 'F':
        case 'L':
        case '[':
        {
          pIntFunc =  INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) ( env, pstMethod->pstClass->classObject);
	      }
	  else 
	      {
	          jvRetVal.i = (*pIntFunc) (  env );
	      }
          break;
        }
        case 'J':
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.j = (*pLongFunc) ( env , pstJNIData,pstMethod->pstClass->classObject);

	      }
	  else
          jvRetVal.j = (*pLongFunc) (  env);
          break;
        }
        case 'D':
        {
          pDoubleFunc = DOUBLESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  assert(1 == 2);
	      }
          jvRetVal.d = (*pDoubleFunc) ( env );
          break;
        }
      }
      break;
    }
    case 1:
    {
      switch (*pszRetChar)
      {
        case 'V':
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		(*pVoidFunc) (  env, pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
	      (*pVoidFunc) (  env, ot[1]);
          break;
        }
        case 'Z':
        case 'B':
        {
          pByteFunc = BYTESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.b = (*pByteFunc) ( env , pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
          jvRetVal.b = (*pByteFunc) ( env , ot[1]);
          break;
        }
        case 'C':
        case 'S':
        {
          pShortFunc = SHORTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.s = (*pShortFunc) (  env, pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
	    jvRetVal.s = (*pShortFunc) (  env, ot[1]);
          break;
        }
        case 'I':
        case 'F':
        case 'L':
        case '[':
        {
          pIntFunc =  INTSIG pFunction;

	  //Some experimentation here
	  
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) ( env, pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
	        jvRetVal.i = (*pIntFunc) (  env , ot[1]);

          break;
        }
        case 'J':
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.j = (*pLongFunc) (  env , pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
	    jvRetVal.j = (*pLongFunc) (  env , ot[1]);
          break;
        }
        case 'D':
        {
          pDoubleFunc = DOUBLESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, ot[1]);
	      }
	  else
	      jvRetVal.d = (*pDoubleFunc) (env, ot[1]);
          break;
        }
      }
      break;
    }
    case 2:
    {
      switch (*pszRetChar)
      {
        case 'V':
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
                (*pVoidFunc) (env, ot[1], ot[2]);

          break;
        }
        case 'Z':
        case 'B':
        {
          pByteFunc = BYTESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
	    jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2]);
          break;
        }
        case 'C':
        case 'S':
        {
          pShortFunc = SHORTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
	    jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2]);
          break;
        }
        case 'I':
        case 'F':
        case 'L':
        case '[':
        {
          pIntFunc =  INTSIG pFunction;

	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
	        jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2]);


          break;
        }
        case 'J':
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
	      jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2]);
          break;
        }
        case 'D':
        {
          pDoubleFunc = DOUBLESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	      }
	  else
	      jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2]);

          break;
        }
      }
      break;
    }
    case 3:
    {
      switch (*pszRetChar)
      {
        case 'V':
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3]);
	      }
	  else
	      (*pVoidFunc) (env, ot[1], ot[2], ot[3]);
          break;
        }
        case 'Z':
        case 'B':
        {
          pByteFunc = BYTESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3]);
	      }
	  else
	    jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2], ot[3]);
          break;
        }
        case 'C':
        case 'S':
        {
          pShortFunc = SHORTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3]);

	      }
	  else
	    {
	      jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2], ot[3]);
	    }
          break;
        }
        case 'I':
        case 'F':
        case 'L':
        case '[':
        {
          pIntFunc = INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.i = (*pIntFunc) (env,pstMethod->pstClass->classObject, ot[1], ot[2], ot[3]);
	      }
	  else
	    {
	      jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3]);
	    }
          break;
        }
        case 'J':
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  assert(1 == 2);
	      }
	  else 
	    jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3]);
          break;
        }
        case 'D':
        {
          pDoubleFunc = DOUBLESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3]);
	      }
	  else
	    jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2], ot[3]);
          break;
        }
      }
      break;
    }
    case 4:
    {
      switch (*pszRetChar)
      {
        case 'V':
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
	      }
	  else
	      (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4]);
          break;
        }
        case 'Z':
        case 'B':
        {
          pByteFunc = BYTESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
	      }
	  else
	      jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2], ot[3], ot[4]);


          break;
        }
        case 'C':
        case 'S':
        {
          pShortFunc = SHORTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
	      }
	  else 
	      jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2], ot[3], ot[4]);
          break;
        }
        case 'I':
        case 'F':
        case 'L':
        case '[':
        {
          pIntFunc =  INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
	      }
	  else
	                jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4]);
          break;
        }
        case 'J':
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
	      }
	  else
	      jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4]);


          break;
        }
        case 'D':
        {
          pDoubleFunc = DOUBLESIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4]);
		  //  assert(1 == 2);
	      }
	  else
	      jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2], ot[3], ot[4]);
          break;
        }
      }
      break;
    }
    case 5:
    {

      switch (pstMethod->u16RetSize)
      {
        case 0:
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5]);
	      }
	  else
	      (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
          
          break;
        }
        case 1:
        {
          pIntFunc = INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject,ot[1], ot[2], ot[3], ot[4], ot[5]);
	      }
	  else
	      jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
          break;
        }
        case 2:
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5]);
	      }
	  else
	      jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
          break;
        }
      }
      break;
    }
    case 6:
    {

      switch (pstMethod->u16RetSize)
      {
        case 0:
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	      }
	  else
                (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);

          break;
        }
        case 1:
        {
          pIntFunc = INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	      }
	  else
	      jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
          break;
        }
        case 2:
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	      }
	  else
	      jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
          break;
        }
      }
      break;
    }
    case 7:
    {

      switch (pstMethod->u16RetSize)
      {
        case 0:
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		(*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
	      }
	  else
          (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
          break;
        }
        case 1:
        {
          pIntFunc = INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
	      }
	  else
	      jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
          break;
        }
        case 2:
        {
          pLongFunc = LONGSIG pFunction;

	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	      {
		  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
	      }
	  else
	      jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
          break;
        }
      }
      break;
    }
    default:
    {
      switch (pstMethod->u16RetSize)
      {
        case 0:
        {
          pVoidFunc = VOIDSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	    (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
	  else
	    (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
          break;
        }
        case 1:
        {
          pIntFunc = INTSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	    jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
	  else
	    jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
          break;
        }
        case 2:
        {
          pLongFunc = LONGSIG pFunction;
	  if(pstMethod->u16AccessFlags & ACC_STATIC)
	    jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
	  else
	    jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
          break;
        }
      }
    }
  }

  switch (*pszRetChar)
  {
    case 'V':
    {
      break;
    }
    case 'Z':
    case 'B':
    {
      pstFrame->pi32OpTop[1] = jvRetVal.b;
      pstFrame->pi32OpTop += 1;
      break;
    }
    case 'C':
    case 'S':
    {
      pstFrame->pi32OpTop[1] = jvRetVal.s;
      pstFrame->pi32OpTop += 1;
      break;
    }
    case 'I':
    case 'F':
    {
      pstFrame->pi32OpTop[1] = jvRetVal.i;
      pstFrame->pi32OpTop += 1;
      break;
    }
    case 'L':
    case '[':
    {
      #ifdef PERSIST
      #ifdef DEBUG
#ifdef EVICTIONS
      fprintf(stderr, "Retval.i is %i, meth %s\n", jvRetVal.i, pstMethod->uidName);
      assert(ISPID((PID) jvRetVal.i) == 0);
#endif
      #endif
      #endif
      pstFrame->pi32OpTop[1] = jvRetVal.i;
      pstFrame->pi32OpTop += 1;
      break;
    }
    case 'J':
    {
      pstFrame->pi32OpTop[1] = jvRetVal.j&0xFFFFFFFF; //hi
      pstFrame->pi32OpTop[2] = jvRetVal.j>>32; //lo
      pstFrame->pi32OpTop += 2;
      break;
    }
    case 'D':
    {
      tDConv dconv;
      dconv.d = jvRetVal.d;
      pstFrame->pi32OpTop[1] = dconv.i.hi;
      pstFrame->pi32OpTop[2] = dconv.i.lo;
      pstFrame->pi32OpTop += 2;
      break;
    }
  }

  if (pstJNIData)
  {
    /* get rid of local references */
    sys_free(pstJNIData->ppstLocalRefs);
    pstJNIData->ppstLocalRefs = NULL;
    pstJNIData->u16NumLocalRefs = 0;
    pstJNIData->u16NumLocalRefsUsed = 0;
  }

  /* check if we need to throw an exception */
  if (pstJNIData && pstJNIData->pstException)
  {
      #ifdef DEBUG_EXCEPTIONS
       printf("JNI_CallNativeMethod is returning -1 with pstJNIData %x and exception %x(%s)\n",pstJNIData, pstJNIData->pstException, DEREF(pstJNIData->pstException)->pstType->uidName);
      #endif
    pstFrame->pi32OpTop[1] = (int32) pstJNIData->pstException;
    pstFrame->pi32OpTop += 1;
    pstJNIData->pstException = NULL;
    return -1;
  }

  return 0;
}

