#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>

#include "knap_player.h"
#include <klocale.h>
#include <qslider.h>
#include <qbuttongroup.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>

#include <qdir.h>
#include <qfile.h>

#include <kconfig.h>
#include <kstddirs.h>  
#include <kmessagebox.h>
#include <dispatcher.h>  

#include "prevbtn.xpm"
#include "playbtn.xpm"
#include "pausebtn.xpm"
#include "stopbtn.xpm"
#include "nextbtn.xpm"

#include "support_funcs.h"

using namespace Arts;  

Arts::Dispatcher dispatcher; 

   KNAP_Player::KNAP_Player(QWidget *parent,const char *name)
: QFrame(parent,name)
{
   setFrameStyle( QFrame::Panel | QFrame::Sunken ); 
//   setSizePolicy(QSizePolicy(QSizePolicy::Maximum,QSizePolicy::Maximum));
   QGridLayout *grid = new QGridLayout(this,4,2,4,4);
   grid->setColStretch(0,0);
   grid->setColStretch(1,0);

   /* Widget layout
    *
    * |----------------------------|-|
    * |  songname                  | | 
    * |                            | |
    * -----------------------------| | <-- volume ctrl
    * |  and info                  |_|
    * |                            |X|
    * |                            |-|
    * |----------------------------| |
    * |  |X| songpos               | |    
    * |----------------------------| |
    * | [  ] [  ] [  ] [  ] [  ]   | | <-- song play/stop etc
    * |------------------------------|
    */

   // songname
   songname = new QLabel("",this,"songname");
   grid->addWidget(songname,0,0);  
   songname->setSizePolicy(QSizePolicy(QSizePolicy::Maximum,QSizePolicy::Maximum));

   // songinfo
   songinfo = new QLabel("info",this,"songname");
   grid->addWidget(songinfo,1,0);  
   grid->setRowStretch(1,2);
   songinfo->setSizePolicy(QSizePolicy(QSizePolicy::Maximum,QSizePolicy::Maximum));

   // volume
   volume = new QSlider(Qt::Vertical,this,"volume");
   grid->addMultiCellWidget(volume,0,3,1,1);  
   volume->setRange(0,100);
   //connect(volume,SIGNAL( sliderMoved( int ) ), this, SLOT( sliderVol( int ) ));
   connect(volume,SIGNAL( valueChanged( int ) ), this, SLOT( sliderVol( int ) ));

   // not using ticks - makes the slider too big
   //  volume->setTickmarks(QSlider::Right);

   // songpos 
   songpos = new QSlider(Qt::Horizontal,this,"songpos");
   grid->addWidget(songpos,2,0);  
   songpos->setRange(0,100);
   songpos->setTracking(false);
   connect(songpos,SIGNAL( sliderReleased() ), this, SLOT( calcSongPos() ));
   connect(songpos,SIGNAL( sliderPressed() ), this, SLOT( songPosDown() ));
   showpos = true;

   // the controls
   QButtonGroup * control = new QButtonGroup( this, "control" );
   control->setTitle( "" /*i18n( ""  )*/ );
   control->setFrameShape( QButtonGroup::NoFrame );
   //	    control->setExclusive( TRUE );
   control->setColumnLayout(0, Qt::Vertical );
   control->layout()->setSpacing( 0 );
   control->layout()->setMargin( 0 );                                                 

   QHBoxLayout *controlLayout = new QHBoxLayout( control->layout() );
   controlLayout->setAlignment( Qt::AlignTop );
   controlLayout->setSpacing( 0 );
   controlLayout->setMargin( 0 );                                                     

   grid->addWidget(control,3,0);  

   prevbtn = new QPushButton(control,"prev");
   prevbtn->setPixmap(QPixmap((const char **)prevbtn_xpm));
   controlLayout->addWidget(prevbtn);
   connect(prevbtn,SIGNAL( clicked() ),this, SLOT( getprev() ));

   playbtn = new QPushButton(control,"play");
   playbtn->setPixmap(QPixmap((const char **)playbtn_xpm));
   controlLayout->addWidget(playbtn);
   connect(playbtn,SIGNAL( clicked() ),this, SLOT( play() ));

   pausebtn = new QPushButton(control,"pause");
   pausebtn->setPixmap(QPixmap((const char **)pausebtn_xpm));
   controlLayout->addWidget(pausebtn);
   connect(pausebtn,SIGNAL( clicked() ),this, SLOT( pause() ));

   stopbtn = new QPushButton(control,"stop");
   stopbtn->setPixmap(QPixmap((const char **)stopbtn_xpm));
   controlLayout->addWidget(stopbtn);
   connect(stopbtn,SIGNAL( clicked() ),this, SLOT( stop() ));

   nextbtn = new QPushButton(control,"next");
   nextbtn->setPixmap(QPixmap((const char **)nextbtn_xpm));
   controlLayout->addWidget(nextbtn);
   connect(nextbtn,SIGNAL( clicked() ),this, SLOT( getnext() ));

   // initialize aRts references
   _playObject = Arts::PlayObject::null();
   _playObjectFactory = Arts::PlayObjectFactory::null();
   _volumeControl = Arts::StereoVolumeControl::null();
   _soundServer = Arts::SimpleSoundServer::null();
   _lastPolledState = Arts::posIdle;                                                      

   // start update timer
   _artsPollTimer.start( 500 );
   connect( &_artsPollTimer, SIGNAL(timeout()), this, SLOT(pollArts()));
             
   //              _statusPollTimer.start( 500 );
   //                  connect( &_statusPollTimer, SIGNAL(timeout()), this,
   //                  SLOT(updateStatusDisplay())); 

   setVolume(100); 
}
/*
 * well its plain to see - initArts is i think almost a direct plagiarism of the noatun
 * code of the kdemultimedia player
 * as from kde2.0
 */
bool KNAP_Player::initArts()
{
   if ( _soundServer.isNull() || _soundServer.error()  ) {
      _volumeControl = Arts::StereoVolumeControl::null();
      _soundServer = Reference("global:Arts_SimpleSoundServer");
      _playObjectFactory = Arts::Reference("global:Arts_PlayObjectFactory");

      if( _soundServer.isNull() || _soundServer.error() )
      {
	 // aRts seems not to be running, let's try to run it

	 // First, let's read the configuration as in kcmarts
	 KConfig *config = new KConfig("kcmartsrc");
	 QCString cmdline;

	 config->setGroup("Arts");

	 bool rt = config->readBoolEntry("StartRealTime",false);
	 bool x11Comm = config->readBoolEntry("X11GlobalComm",false);

	 /* put the value of x11Comm into .mcoprc */
	 KConfig *X11CommConfig = new KConfig(QDir::homeDirPath()+"/.mcoprc");

	 if(x11Comm)
	    X11CommConfig->writeEntry("GlobalComm","Arts::X11GlobalComm");
	 else
	    X11CommConfig->writeEntry("GlobalComm","Arts::TmpGlobalComm");

	 X11CommConfig->sync();
	 delete X11CommConfig;

	 cmdline = QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("kdeinit_wrapper")));
	 cmdline += " ";

	 if (rt)
	    cmdline += QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("artswrapper")));
	 else
	    cmdline += QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("artsd")));

	 cmdline += " ";
	 cmdline += config->readEntry("Arguments", "-F 5 -S 8192").utf8();

	 int status=system(cmdline);

	 if ( status!=-1 && WIFEXITED(status) )
	 {
	    // We could have a race-condition here. The correct way to do it is to
	    // make artsd fork-and-exit after starting to listen to connections
	    // (and running artsd directly instead of using kdeinit),
	    // but this is better than nothing.
	    int time = 0;
	    do
	    {
	       sleep(1);
	       _soundServer = Reference("global:Arts_SimpleSoundServer");
	       _playObjectFactory = Arts::Reference("global:Arts_PlayObjectFactory");
	    } while(++time < 5 && (_soundServer.isNull() || _playObjectFactory.isNull()));
	 }

	 if( _playObjectFactory.isNull() )
	 {
	    KMessageBox::error( 0, i18n("Connection to the soundserver failed - make sure that artsd is really running.") );
	    return false;
	 }
      }

      if ( !_soundServer.isNull() ) {
	 // create a new stereo volume control object on the server
	 _volumeControl = DynamicCast(_soundServer.createObject("Arts::StereoVolumeControl"));
	 _volumeControl.start();
	 _volumeEffectID = _soundServer.outstack().insertBottom(_volumeControl, "Volume Control");
	 _volumeControl.scaleFactor( _volume/100.0 );
      }
   }

   _playObject = Arts::PlayObject::null();

   return true;
}




KNAP_Player::~KNAP_Player()
{
   // remove effect from effect chain
   if ( !_soundServer.isNull() && !_soundServer.error() &&
	 !_volumeControl.isNull() && !_volumeControl.error() )
   {
      _soundServer.outstack().remove( _volumeEffectID );
   }

   // destroy aRts objects
   _playObject =
      Arts::PlayObject::null();
   _playObjectFactory =
      Arts::PlayObjectFactory::null();
   _volumeControl =
      Arts::StereoVolumeControl::null();
   _soundServer =
      Arts::SimpleSoundServer::null();                                        
}
//-- the slots --------------------
void KNAP_Player::play()
{
   //  emit playSong();
   //  play("/home/jade/mp3/Rick Price - Heaven Knows.mp3");
   if( !_playObject.isNull() ) {
//      if ( _playObject.state()==Arts::posPlaying ) stop();
//	 pause();
//      else
	 _playObject.play();
   }
   else 
   {
      emit playSong();
      // TODO remove this !!!!!!
      //play("/home/jade/mp3/Rick Price - Heaven Knows.mp3");
   }
}
void KNAP_Player::play(const char * file)
{
   //if(file.isNull()) file=currfile;
   if(file ==0) return;

   bool i = initArts();
   if(!i) 
   {
      stop();
      cerr <<"***************************** could not initialise arts\n";
      return;
   }
   else
      cerr<<"***************************** initialized arts\n";

   if ( !_playObjectFactory.isNull() ) {
      _playObject = _playObjectFactory.createPlayObject( file );
      if ( !_playObject.isNull() ) {
	 _playObject.play();
	 songname->setText(getFileName(file,true));
	 return; // _playObject.state()==Arts::posPlaying;
      } 

   }
   songname->setText("");
   return;   
}
void KNAP_Player::stop()
{
   if( !_playObject.isNull() ) {
      Arts::poTime l_t_zero( 0, 0, 0, "samples" );
      _playObject.seek( l_t_zero );
      _playObject.halt();
   }

   _playObject = Arts::PlayObject::null();
   _lastPolledState = Arts::posIdle;                                                      
   songname->setText("");
}
void KNAP_Player::pause()
{
   if( !_playObject.isNull() ) 
   {
      if ( _lastPolledState==Arts::posPaused )
	 _playObject.play();
      else
	 _playObject.pause();
   } 
}

void KNAP_Player::getnext()
{
  emit nextSong();
}

void KNAP_Player::getprev()
{
  emit prevSong();
}

void KNAP_Player::setVolume(int vol)
{
   if(vol<0) vol = 0;
   if(vol>100) vol = 100; 
   _volume = vol;

   if ( !_volumeControl.isNull() )
   {
      _volumeControl.scaleFactor(vol/100.0);
   }  
   volume->setValue(100 -_volume);
   cerr << "set vol " << 100 - _volume << endl;
}
/**
 * slot to catch vol slider changes 
 * we then reverse this number to get the actual volume
 * so that the volume control works top to bottom
 */
void KNAP_Player::sliderVol(int vol)
{
   setVolume(100-vol);
}

void KNAP_Player::calcSongPos()
{
   int i = songpos->value();
   seek(i);
   showpos = true;
}
void KNAP_Player::songPosDown()
{
   showpos = false;
}
void KNAP_Player::seek( int sec_per )
{
   if(sec_per<0) sec_per=0;
   if(sec_per>100) sec_per=100;

   if ( !_playObject.isNull() ) 
   {

      long all = _playObject.overallTime().seconds;

      sec_per = (all * sec_per )/100;
      Arts::poTime t;
      t.seconds = sec_per;
      t.ms = 0;
      t.custom = sec_per;
      t.customUnit = "seconds";
      _playObject.seek( t );
   }
} 


void KNAP_Player::pollArts()
{
   if ( !_playObject.isNull() ) {
      // check play object state and update buttons
      int oldPolledState = _lastPolledState;
      _lastPolledState = _playObject.state();
      switch ( _lastPolledState ) {
	 case Arts::posIdle:
	    songinfo->setText( "Idle" );

	    if ( oldPolledState==Arts::posPlaying ) {
	       //if ( _mediaManager->active() )
	       emit nextSong();
	    }
	    else stop(); // ??????? ******************************************
	    break;

	 case Arts::posPaused:
	    songinfo->setText( "Paused" );
	    break;

	 case Arts::posPlaying:
	    songinfo->setText( "Playing" );
	    break;
      }
   } 
   else 
   {
      if ( _lastPolledState!=Arts::posIdle ) stop();

      songinfo->setText("Stopped");
   }

   // now the position that we are at
   long sec = 0;
   long all = 0;
   int percent = 0;
   if ( !_playObject.isNull() ) {
      int newPolledState = _playObject.state();
      if ( newPolledState==Arts::posPaused || newPolledState==Arts::posPlaying )
      {
	 sec = _playObject.currentTime().seconds;
	 all = _playObject.overallTime().seconds;
	 percent = 100*sec;
	 if ( all ) percent/=all; else percent=0;

	 //if(songpos->tracking()==false) 
	 if(showpos) songpos->setValue(percent);
      }
   }
   else
      songpos->setValue(0);
}

QSize KNAP_Player::sizeHint()
{
  return QFrame::sizeHint(); //QSize(200,100); 
}
