/***************************************************************************
 *   Copyright (C) 2005-2006 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdatetime.h>

#include <kglobal.h>
#include <klistview.h>
#include <klocale.h>
#include <kmdcodec.h>

#include "kfileitemext.h"

#include "klistviewitemdups.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KListViewItemDups::KListViewItemDups( KListView *parent, KFileItemExt *_item,
		const QString &parent_url1_str, const QString &parent_url2_str )
	: QListViewItem( (QListView*)parent ), m_filter( NONE )
{
	item = _item;

	m_parent_url1 = KURL::fromPathOrURL( parent_url1_str );
	m_parent_url2 = KURL::fromPathOrURL( parent_url2_str );

	m_color = QColor( 0, 0, 0 );
	m_my_blue = QColor( 15, 25, 200 );

	QDateTime dt;
	dt.setTime_t( item->time( KIO::UDS_MODIFICATION_TIME ) );
	QString date = KGlobal::locale()->formatDateTime( dt );

	// Fill the item with data
	KURL tmp_url = item->url(); // This is necessary to display a pretty url on non-local file systems.
	tmp_url.setPath( "" );
	setText( 0, tmp_url.pathOrURL() + item->url().upURL().path( -1 ) );
	setText( 1, KIO::decodeFileName( item->url().fileName() ) );
	setText( 2, /*KGlobal::locale()->formatNumber( item->size(), 0 )*/KIO::convertSize( item->size() ) );
	setText( 3, date );
	setText( 4, item->permissionsString() );
	setText( 5, item->user() );
	setText( 6, item->group() );

	// put the icon into the leftmost column
	setPixmap( 0, item->pixmap( 16 ) );

	updateColor();
}

KListViewItemDups::~KListViewItemDups()
{
}

QString KListViewItemDups::key( int column, bool ) const
{
	switch ( column )
	{
		case 2:  // Returns size in bytes. Used for sorting
			return QString().sprintf( "%010d", (uint)item->size() );

		case 3:  // Returns time in secs from 1/1/1970. Used for sorting
			return QString().sprintf( "%010ld", item->time( KIO::UDS_MODIFICATION_TIME ) );

		default:
			break;
	}
	return text( column );
}

void KListViewItemDups::paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int alignment )
{
	QColorGroup tmpcolorgroup( cg );

	tmpcolorgroup.setColor( QColorGroup::Text, m_color );
	tmpcolorgroup.setColor( QColorGroup::HighlightedText, QColor( 255-m_color.red(), 255-m_color.green(), 255-m_color.blue() ) ); // complementary color
	QListViewItem::paintCell( p, tmpcolorgroup, column, width, alignment );
}

bool KListViewItemDups::isDuplicatesPair()
{
	KFileItemExt *tmp_item = item->duplicates_size;
	int count = 1;
	bool hasother = false;
	while ( tmp_item ) // check whether this duplicate list contains more than two files or duplicates within one folder.
	{
		if ( count > 2 ) break; // If there are more than two files, we surely have a duplicate that is no "brother" file.

		if ( compareKURLs( item->url(), tmp_item->url(), item->dir ) )
		{
			hasother = true;
		}

		count++;
		tmp_item = tmp_item->duplicates_size;
	}

	return ( count > 2 || count < 2 || !hasother ) ? false : true;
}

void KListViewItemDups::updateColor()
{
	m_color = isDuplicatesPair() ? QColor( 0, 0, 0 ) : m_my_blue;
}

bool KListViewItemDups::compareKURLs( const KURL &url1, const KURL &url2, int parent_dir_of_first )
{
	KURL tmp_url;

	if ( parent_dir_of_first == 0 )
	{
		if ( KURL::relativePath( m_parent_url1.path( -1 ), url1.upURL().path( -1 ) ) ==
				KURL::relativePath( m_parent_url2.path( -1 ), url2.upURL().path( -1 ) ) )
		{
			return true;
		}
	}
	else
	{
		if ( KURL::relativePath( m_parent_url2.path( -1 ), url1.upURL().path( -1 ) ) ==
				KURL::relativePath( m_parent_url1.path( -1 ), url2.upURL().path( -1 ) ) )
		{
			return true;
		}
	}

	return false;
}

void KListViewItemDups::addFilter( FILTER filter )
{
	m_filter |= filter;
}

void KListViewItemDups::removeFilter( FILTER filter )
{
	if ( m_filter & filter ) m_filter ^= filter;
}

void KListViewItemDups::setVisibility()
{
	setVisible( ( m_filter != NONE ) ? false : true );
}
