
#ifndef VERSION_H
#define VERSION_H

#include <kore/kore.h>

namespace kore
{

/**
 * class kore::Version - common version information (major/minor/revision/string)
 * helper class
 */
class KORE_API Version
{
public:
    /**
     * Create a Version instance: <major>.<minor>.<revision>
     * @param major - Major version number (default = 0)
     * @param minor - Minor version number (default = 0)
     * @param revision - Revision number (default = 0)
     * @param version - Version string (usually "<major>.<minor>.<revision>")
     */
    Version(const int major ,const int minor = 0,const int revision = 0,const char* version = 0);
//    /**
//     * Copy constructor
//     * @param other - version to be copied
//     */
//    Version(Version& other);
    virtual ~Version();
    /**
     * Get the major version number.
     * @return - the major version number
     */
    virtual const int major() const;
    /**
     * Get the minor version number.
     * @return - the minor version number
     */
    virtual const int minor() const;
    /**
     * Get the revision number.
     * @return - the revision number
     */
    virtual const int revision() const;
    /**
     * Get the version string (usually "<major>.<minor>.<revision>").
     * @return - the version number string
     */
    virtual const char* string() const;
    /**
     * Get the version string (usually "<major>.<minor>.<revision>").
     * @return - the version number string
     */
    virtual operator const char*() const;
    /**
     * Compares to version to see if they match
     * Two versions match if they have the same major, minor and revision
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the two versions match (i.e. are the same), false otherwise
     */
    virtual const bool operator ==(const Version& other) const;
    /**
     * Compares to version to see if they don't match
     * v1 != v2 if and only if !(v1 == v2)
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the two versions match (i.e. are the same), false otherwise
     */
    virtual const bool operator !=(const Version& other) const;
    /**
     * Checks of the current version is older (smaller) than the specified one
     * The comparison is performed on major, minor and revision, in this order
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the corrent version is older (smaller), false otherwise
     */
    virtual const bool operator <(const Version& other) const;
    /**
     * The combined result of == and <
     * v1 <= v2 if and only if (v1 == v2) || (v1 < v2)
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the corrent version is older (smaller), false otherwise
     */
    virtual const bool operator <=(const Version& other) const;
    /**
     * Checks of the current version is newer (bigger) than the specified one
     * The comparison is performed on major, minor and revision, in this order
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the corrent version is newer (bigger), false otherwise
     */
    virtual const bool operator >(const Version& other) const;
    /**
     * The combined result of == and >
     * v1 >= v2 if and only if (v1 == v2) || (v1 > v2)
     * Note: the version string is ignored when performing version comparison
     * @param other - the other version to compare against
     * @return true - if the corrent version is older (smaller), false otherwise
     */
    virtual const bool operator >=(const Version& other) const;
    /**
     * Checks if the current version is binary compatible with the specified one
     * @param other - the other version to compare against
     * @return true - if the two versions are binary compatible, false otherwise
     */
    virtual const bool operator &(const Version& other) const;
    /**
     * Checks if the current version is source compatible with the specified one
     * @param other - the other version to compare against
     * @return true - if the two versions are source compatible, false otherwise
     */
    virtual const bool operator &&(const Version& other) const;

protected:
    /**
     * Default constructor.
     */
    Version();
    /**
     * Set the major version number.
     * @param major - the major version number
     */
    void setMajor(const int major = 0 );
    /**
     * Set the minor version number.
     * @param minor - the minor version number
     */
    void setMinor(const int minor = 0 );
    /**
     * Set the revision number.
     * @param revision - the revision number
     */
    void setRevision(const int revision = 0);
    /**
     * Set the version string.
     * @param version - the version string
     */
    void setString(const char* version = 0 );
    /**
     * Set the version information: major/minor/revision/string
     * @param major - the major version number
     * @param minor - the minor version number
     * @param revision - the revision number
     * @param version - the version string
     */
    void setVersion(const int major = 0,const int minor = 0,const int revision = 0,const char* version = 0);
private:
    // major version number
    int _major;
    // minor version number
    int _minor;
    // revision number
    int _revision;
    // version string
    const char* _version;
};

}

#endif
