;;;; -*- mode: lisp -*-
;;;;
;;;; $Id: hmac.lisp,v 1.2 2004/09/15 09:52:30 sven Exp $
;;;;
;;;; Implementation of RFC 2104 - HMAC: Keyed-Hashing for Message Authentication
;;;;
;;;; Copyright (C) 2004 Sven Van Caekenberghe, Beta Nine BVBA. All Rights Reserved.
;;;;
;;;; You are granted the rights to distribute and use this software
;;;; as governed by the terms of the Lisp Lesser GNU Public License
;;;; (http://opensource.franz.com/preamble.html), also known as the LLGPL.
;;;;

(in-package :kpax)

(export '(hmac 
          make-byte-vector
          string->byte-vector byte-vector->string
          byte-vector->hex-string hex-string->byte-vector))

(defun make-byte-vector (size &optional (initial-element 0))
  "Make a byte vector of size bytes filled with initial-element"
  (make-array size :element-type '(unsigned-byte 8) :initial-element initial-element))

(defun string->byte-vector (string)
  "Convert an 8-bit ASCII string to a byte vector"
  (let ((bytes (make-byte-vector (length string))))
    (loop :for char :across string
          :for i :upfrom 0
          :do (setf (aref bytes i) (mod (char-code char) 256)))
    bytes))

(defun byte-vector->string (bytes)
  "Convert a byte vector to an 8-bit ASCII string"
  (let ((string (make-string (length bytes))))
    (loop :for byte :across bytes
          :for i :upfrom 0
          :do (setf (char string i) (code-char byte)))
    string))

(defparameter +hex-digits+ "0123456789ABCDEF")

(defun byte-vector->hex-string (bytes)
  "Convert a byte vector to a hex string, as in #(1 15 255) becomes '010FFF'"
  (let ((string (make-string (* 2 (length bytes)))))
    (loop :for byte :across bytes
          :for i :upfrom 0
          :do (let ((upper (ash byte -4))
                    (lower (logand byte #xF)))
                (setf (char string (* i 2)) (char +hex-digits+ upper)
                      (char string (1+ (* i 2))) (char +hex-digits+ lower))))
    string))

(defun hex-string->byte-vector (hex-string)
  "Convert a hex string to a byte vector, as '010FFF' becomes #(1 15 255)"
  (assert (evenp (length hex-string)))
  (let ((bytes (make-byte-vector (/ (length hex-string) 2))))
    (loop :for i :upfrom 0 :below (length bytes)
          :do (setf (aref bytes i) (+ (ash (position (char hex-string (* i 2)) +hex-digits+ :test #'char-equal) 4)
                                      (position (char hex-string (1+ (* i 2))) +hex-digits+ :test #'char-equal))))
    bytes))

(defun hash-start-place-holder ()
  (make-byte-vector 16))

(defun hash-update-place-holder (state sequence)
  (etypecase sequence
    (string
     (loop :for i :upfrom 0 :below (length sequence)
           :do (setf (aref state (mod i 16)) (logxor (aref state (mod i 16))
                                                     (mod (char-code (char sequence i)) 256)))))
    ((simple-array (unsigned-byte 8) (*))
     (loop :for i :upfrom 0 :below (length sequence)
           :do (setf (aref state (mod i 16)) (logxor (aref state (mod i 16))
                                                     (aref sequence i)))))))

(defun hash-finish-place-holder (state)
  state)

(defun do-hash (hash-start-function hash-update-function hash-finish-function &rest sequences)
  (let ((state (funcall hash-start-function)))
    (loop :for sequence :in sequences
          :do (funcall hash-update-function state sequence))
    (funcall hash-finish-function state)))

(defun normalize-hmac-key (key hash-block-length hash-start-function hash-update-function hash-finish-function)
  (if (= (length key) hash-block-length)
      (etypecase key
        (string (string->byte-vector key))
        ((simple-array (unsigned-byte 8) (*)) key))
    (let ((padded-bytes (make-byte-vector hash-block-length 0)))
      (if (< (length key) hash-block-length)
          (etypecase key
            (string 
             (loop :for char :across key
                   :for i :upfrom 0
                   :do (setf (aref padded-bytes i) (mod (char-code char) 256))))
            ((simple-array (unsigned-byte 8) (*)) 
             (loop :for byte :across key
                   :for i :upfrom 0
                   :do (setf (aref padded-bytes i) byte))))
        (let ((hash (do-hash hash-start-function hash-update-function hash-finish-function key)))
          (loop :for byte :across hash
                :for i :upfrom 0
                :do (setf (aref padded-bytes i) byte))))
      padded-bytes)))

(defun hmac (data 
             key 
             &key 
             (hash-start-function #'hash-start-place-holder) 
             (hash-update-function #'hash-update-place-holder)
             (hash-finish-function #'hash-finish-place-holder)
             (hash-block-length 64))
  "Compute an RFC 2104 HMAC from the data and key sequences using hash-[start|update|finish]-function"
  (let ((ipad (make-byte-vector hash-block-length #x36))
        (opad (make-byte-vector hash-block-length #x5C))
        (normalized-key (normalize-hmac-key key hash-block-length 
                                            hash-start-function hash-update-function hash-finish-function))
        (result (make-byte-vector hash-block-length))
        tmp)
    ;; result = k XOR ipad
    (loop :for i :upfrom 0 :below hash-block-length
          :do (setf (aref result i) (logxor (aref normalized-key i) (aref ipad i))))
    ;; tmp = H (result, data)
    (setf tmp (do-hash hash-start-function hash-update-function hash-finish-function 
                       result data))
    ;; result = k XOR opad
    (loop :for i :upfrom 0 :below hash-block-length
          :do (setf (aref result i) (logxor (aref normalized-key i) (aref opad i))))
    ;; H (result, tmp)
    (do-hash hash-start-function hash-update-function hash-finish-function
             result tmp)))

;; these are the test vectors from the appendix sample code from RFC 2104 for HMAC-MD5 

#+NIL
(assert (string-equal (byte-vector->hex-string 
                       (hmac "Hi There" 
                             (hex-string->byte-vector "0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b0b")
                             :hash-start-function #'md5:make-md5-state
                             :hash-update-function #'md5:update-md5-state
                             :hash-finish-function #'md5:finalize-md5-state))
                      "9294727a3638bb1c13f48ef8158bfc9d"))

#+NIL
(assert (string-equal (byte-vector->hex-string 
                       (hmac "what do ya want for nothing?"
                             "Jefe"
                             :hash-start-function #'md5:make-md5-state
                             :hash-update-function #'md5:update-md5-state
                             :hash-finish-function #'md5:finalize-md5-state))
                      "750c783e6ab0b503eaa86e310a5db738"))

#+NIL
(assert (string-equal (byte-vector->hex-string 
                       (hmac (make-byte-vector 50 #xDD)
                             (hex-string->byte-vector "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA")
                             :hash-start-function #'md5:make-md5-state
                             :hash-update-function #'md5:update-md5-state
                             :hash-finish-function #'md5:finalize-md5-state))
                      "56be34521d144c88dbb8c733f0e8b3f6"))

;;;; eof