// KreateCD - CD recording software for the K desktop environment
//
// 1999-2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "AudioFileInfo.h"
#include "AudioConverter.h"
#include "AudioConversion.h"
#include "ProgressDialog.h"
#include "CDTrack.h"

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

AudioFileInfo::AudioFileInfo() {
  SampleChannels=1;
  SampleRate=44100;
  SampleEndian=Endian_Big;
  SampleFormat=Samples_PCM;
  SampleSize=Samples_16Bit;
  updateSampleBytes();
  Format[0]=0;
  DataPos=0;
  DataSize=0;
  updateDataSamples();
  resetSelection();
  Balance=0;
  MaxBoost=-1;
  BoostFactor=1;
  Converter=0;
  FileName[0]=0;
  DecodedFileName[0]=0;
  NeedsRipping=false;
}

AudioFileInfo::AudioFileInfo(const AudioFileInfo &object) {
  SampleEndian=object.SampleEndian;
  SampleFormat=object.SampleFormat;
  SampleChannels=object.SampleChannels;
  SampleSize=object.SampleSize;
  SampleRate=object.SampleRate;
  SampleBytes=object.SampleBytes;
  DataPos=object.DataPos;
  DataSize=object.DataSize;
  DataSamples=object.DataSamples;
  strcpy(Format,object.Format);
  strcpy(FileName,object.FileName);
  DecodedFileName[0]=0;
  Converter=object.Converter;
  if (Converter!=0) {
    Converter->refCount++;
  }
  ConverterFunction=object.ConverterFunction;
  SelectedStart=object.SelectedStart;
  SelectedEnd=object.SelectedEnd;
  BoostFactor=object.BoostFactor;
  Balance=object.Balance;
  MaxBoost=object.MaxBoost;
  NeedsRipping=object.NeedsRipping;
}

AudioFileInfo::~AudioFileInfo(void) {
  if (Converter) { 
    Converter->refCount--;
    if (Converter->refCount==0) {
      delete Converter;
    }
  }
  if (DecodedFileName[0]!=0) {
      remove(DecodedFileName);
  }		 
}

void AudioFileInfo::setEndian(Sample_Endian end) {
  if ( (end==Endian_Big) || (end==Endian_Little) ) SampleEndian=end;
}

void AudioFileInfo::setFormat(Sample_Format form) {
  SampleFormat=form;
}

void AudioFileInfo::setNeedsRipping(bool nr) {
  NeedsRipping=nr;
}

void AudioFileInfo::setSize(Sample_Size siz) {
  if ( (siz!=Samples_8Bit) && (siz!=Samples_16Bit) &&
       (siz!=Samples_24Bit) && (siz!=Samples_32Bit) ) return;
  SampleSize=siz;
  updateSampleBytes();
}

void AudioFileInfo::setChannels(int chan) {
  if ( (chan<1) || (chan>16)) return;
  SampleChannels=chan;
  updateSampleBytes();
}

void AudioFileInfo::setRate(int rat) {
  if ( (rat<1) || (rat>1000000) ) return;
  SampleRate=rat;
}

void AudioFileInfo::setDataPos(long pos) {
  if (pos<0) return;
  DataPos=pos;
}

void AudioFileInfo::setDataSize(long siz) {
  if (siz<0) return;
  DataSize=siz;
  updateDataSamples();
  resetSelection();
}

void AudioFileInfo::setDataSamples(long samp) {
  if (samp<0) return;
  DataSamples=samp;
  DataSize=DataSamples*SampleBytes;
  resetSelection();
}

void AudioFileInfo::setStartPos(long int pos) {
  if (pos>(DataSamples-1)) pos=DataSamples-1;
  if (pos<0) pos=0;
  SelectedStart=pos;
  if (SelectedEnd<SelectedStart) SelectedEnd=DataSamples-1;
}

void AudioFileInfo::setEndPos(long int pos) {
  if (pos>(DataSamples-1)) pos=DataSamples-1;
  if (pos<0) pos=0;
  SelectedEnd=pos;
  if (SelectedEnd<SelectedStart) SelectedStart=0;
}

void AudioFileInfo::setBoost(float boo) {
  if ( (boo<0) || (boo>100)) return;
  BoostFactor=boo;
}

void AudioFileInfo::setBalance(float bal) {
  if ( (bal<-1) || (bal>1)) return;
  Balance=bal;
}

void AudioFileInfo::setMaxBoost(float boo) {
  if ( (boo<0) || (boo>100)) return;
  MaxBoost=boo;
}

void AudioFileInfo::setConverter(AudioConverter *conv,int (*function)(int argc,char **argv)) {
  if (Converter!=0) {
    Converter->refCount--;
    if (Converter->refCount==0) {
      delete Converter;
    }
  }
  Converter=conv;
  ConverterFunction=function;
  Converter->refCount++;
}

void AudioFileInfo::setFormatName(const char *fn) {
  Format[15]=0;
  strncpy(Format,fn,15);
}

void AudioFileInfo::setFilename(const char *fn) {
  FileName[1023]=0;
  strncpy(FileName,fn,1023);
}

AudioFileInfo::Sample_Endian AudioFileInfo::endian(void) {
  return(SampleEndian);
}

AudioFileInfo::Sample_Format AudioFileInfo::format(void) {
  return(SampleFormat);
}

AudioFileInfo::Sample_Size AudioFileInfo::size(void) {
  return(SampleSize);
}

int AudioFileInfo::channels(void) {
  return(SampleChannels);
}

int AudioFileInfo::rate(void) {
  return(SampleRate);
}

int AudioFileInfo::sampleBytes(void) {
  return(SampleBytes);
}

long AudioFileInfo::dataPos(void) {
  return(DataPos);
}

long AudioFileInfo::dataSize(void) {
  return(DataSize);
}

long AudioFileInfo::dataSamples(void) {
  return(DataSamples);
}

long int AudioFileInfo::startPos(void) {
  return(SelectedStart);
}

long int AudioFileInfo::endPos(void) {
  return(SelectedEnd);
}

float AudioFileInfo::boost(void) {
  return(BoostFactor);
}

float AudioFileInfo::maxBoost(void) {
  return(MaxBoost);
}

float AudioFileInfo::balance(void) {
  return(Balance);
}

bool AudioFileInfo::needsRipping(void) {
  return(NeedsRipping);
}

const char *AudioFileInfo::formatName(void) {
  return(Format);
}

AudioConverter *AudioFileInfo::converter(void) {
  return(Converter);
}

int(*AudioFileInfo::converterFunction(void))(int argc,char **argv) {
  return(ConverterFunction);
}


void AudioFileInfo::updateSampleBytes(void) {
  SampleBytes=SampleChannels;
  if (SampleSize==Samples_16Bit) SampleBytes*=2;
  if (SampleSize==Samples_24Bit) SampleBytes*=3;
  if (SampleSize==Samples_32Bit) SampleBytes*=4;
  updateDataSamples();
  resetSelection();
}

void AudioFileInfo::updateDataSamples(void) {
  DataSamples=DataSize/SampleBytes;
}

void AudioFileInfo::resetSelection(void) {
  SelectedStart=0;
  SelectedEnd=DataSamples-1;  
}

bool AudioFileInfo::validateFile(void) {
  ProgressDialog *prog;
  AudioConversion *conv;
  if (FileName[0]==0) return(false);
  if (Converter!=0) {
    if (DecodedFileName[0]!=0) return(true);
    if (ConverterFunction!=0) {
      conv=new AudioConversion(this,Converter,ConverterFunction);
    } else {
      conv=new AudioConversion(this,Converter);
    } 
    prog=new ProgressDialog();
    prog->attachProcess(conv);
    CDTrack::createName(DecodedFileName,true,rand());
    if (!conv->convertAudio(DecodedFileName)) {
      delete prog;
      delete conv;
      DecodedFileName[0]=0;
      return(false);   
    }
    if (prog->exec()==0) {
      delete prog;
      delete conv;
      DecodedFileName[0]=0;
      return(false);
    }
    delete conv;
    delete prog;
  }
  return(true);
}

const char *AudioFileInfo::getBurnFile(void) {
  if (FileName[0]==0) return(0);
  if (Converter==0) {
    return(FileName);
  }
  if (DecodedFileName[0]==0) return(0);
  return(DecodedFileName);
}

const char *AudioFileInfo::getSourceName(void) {
  return(FileName);
}

int AudioFileInfo::needImage(void) {
  if (channels()!=2) return(1);
  if (rate()!=44100) return(1);
  if (dataPos()!=0) return(1);
  if (endian()!=AudioFileInfo::Endian_Big) return(1);
  if ( (startPos()!=0) || (endPos()!=(dataSamples()-1))) return(1);
  if ( (boost()!=1) || (balance()!=0) ) return(1);
  if ( dataSamples()%588!=0) return (1);
  return(0);
}

int AudioFileInfo::needFile(void) {
  if (channels()!=2) return(1);
  if (rate()!=44100) return(1);
  if (dataPos()!=0) return(1);
  if (format()!=AudioFileInfo::Samples_PCM) return(1);
  if (size()!=AudioFileInfo::Samples_16Bit) return(1);
  if (endian()!=AudioFileInfo::Endian_Big) return(1);
  if ( (startPos()!=0) || (endPos()!=(dataSamples()-1))) return(1);
  if ( (boost()!=1) || (balance()!=0) ) return(1);
  if ( dataSamples()%588!=0) return (1);
  if ( converter()!=0) return(1);
  return(0);
}

int AudioFileInfo::getAbsoluteDuration(void) {
  long long xdur;
  xdur=((long long) dataSamples())*75;
  return((xdur+rate()-1)/rate());
}

int AudioFileInfo::getSelectedDuration(void) {
  long long xdur;
  xdur=(((long long )endPos())-((long long )startPos())+1)*75;
  return((xdur+rate()-1)/rate());
} 
