// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "IsoFile.h"
#include "IsoFile.moc"

#include <qstring.h>


#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>

ISOFile::ISOFile(ISOFile::ISOType type,QString *name,QString *reference) {
  parentObject=0;
  subObjects=new ISOFileList();
  objectType=type;
  objectName=new QString(name->data());
  if (reference==0) {
    objectReference=0;
  } else {
    objectReference=new QString(reference->data());
  }
}

ISOFile::~ISOFile(void) {
  ISOFile *ifile;

  emit treeChanged();
 
  while ( (ifile=subObjects->first()) !=0) {
    delete ifile;
  }

  if (parentObject!=0) {
    parentObject->subObjects->remove(this);
  }
  delete objectName;
  if (objectReference!=0) delete objectReference;
  delete subObjects;
}

ISOFile::ISOFile(const ISOFile &obj):QObject() {
  ISOFile *subobj,*subobj2;

  objectType=obj.objectType;
  objectName=new QString(*obj.objectName);
  if (obj.objectReference!=0) {
    objectReference=new QString(*obj.objectReference);
  } else {
    objectReference=0;
  }
  parentObject=0;
  subObjects=new ISOFileList();
  subobj=obj.subObjects->first();
  while ( subobj!= 0) {
    subobj2=new ISOFile(*subobj);
    subObjects->append(subobj2);
    subobj2->parentObject=this; 
    subobj=obj.subObjects->next();
  }
}

void ISOFile::addObject(class ISOFile *object) {
  this->subObjects->append(object);
  object->parentObject=this;
  emit treeChanged();
  connect(object,SIGNAL(treeChanged()),this,SLOT(notifyChange()));
}

const class QString *ISOFile::name(void) {
  return (objectName);
}

const class QString *ISOFile::reference(void) {
  return (objectReference);
}

ISOFile::ISOType ISOFile::type(void) {
  return (objectType);
}

ISOFile *ISOFile::getChildren(ISOFile *walker) {
  if (walker==0) {
    return(subObjects->first());
  }
  subObjects->findRef(walker);
  return(subObjects->next());
}


ISOFile *ISOFile::findChildren(const QString *name)
{
  ISOFile *file;
  file=subObjects->first();
  while (file!=0) {
    if (strcmp(name->data(),file->objectName->data())==0) return(file);
    file=subObjects->next();
  }
  return(0);
}

bool ISOFile::createTree(const char *prefix,FILE *outfile,FILE *excludeout,QString dummyDir) {
  ISOFile *file;
  QString filename;
  char *ps1,*ps2;
  if (subObjects->isEmpty()) {
    fprintf(outfile,"%s/=%s\n",prefix,dummyDir.data());
  }
  file=subObjects->first();
  while (file!=0) {
    filename=prefix;
    filename+="/";
    filename+=file->objectName->data();
    switch (file->objectType) {
    case ISO_ISODir:
      if (!file->createTree(filename.data(),outfile,excludeout,dummyDir)) {
	return(false);
      }
      break;
    case ISO_RealFile:
      ps1=makeescstr(filename.data());
      ps2=makeescstr(file->objectReference->data());
      fprintf(outfile,"%s=%s\n",ps1,ps2);
      delete [] ps1;
      delete [] ps2;
      break;

    case ISO_RealDir:
      ps1=makeescstr(filename.data());
      ps2=makeescstr(file->objectReference->data());
      fprintf(outfile,"%s/=%s\n",ps1,ps2);
      delete [] ps1;
      delete [] ps2;
      break;
    }
    file=subObjects->next();
  }
  return(true);
}

ISOFile *ISOFile::getParent(void) {
  return parentObject;
}

char *ISOFile::makeescstr(const char *src) {
  char *dest,*dptr;
  const char *sptr;
  int slen=1;
  sptr=src;
  while (*sptr!=0) {
    slen++;
    if ( (*sptr=='\\') || (*sptr=='=')) {
      slen++;
    }
    ++sptr;
  }
  dest=new char[slen];
  sptr=src;
  dptr=dest;
  while (*sptr!=0) {
    if ( (*sptr=='\\') || (*sptr=='=')) {
      *dptr++='\\';
    }
    *dptr++=*sptr++;
  }
  *dptr=0;
  return(dest);
}

void ISOFile::notifyChange(void) {
  emit treeChanged(); 
}
