// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#define BUFFER_SIZE 256*1024
#define MIN_BUFFER 128*1024

#include "PipeCopy.h"
#include "PipeCopy.moc"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>


#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

void PipeCopy::setupPipe(void) {
  prepareProcess();
  *this<<"pipecopy";
}

bool PipeCopy::copyPipe(const char *source,const char *dest,long int size) {
  char size_str[128];

  sprintf(size_str,"%ld",size);
  *this<<source<<dest<<size_str;
  return(true);
}

void PipeCopy::delayedOpen(void) {
  *this<<"MAGIC_DELAY_OPEN"<<"dummy"<<"dummy";
}

void PipeCopy::startPipe(void) {
  startProcess();
  return;
}

PipeCopy::PipeCopy(void):ProcessInterface(pipecopy) {

}

bool PipeCopy::processStderrLine(char *linebuffer) {
  if (strncmp("#FOUT#",linebuffer,6)==0) {
    emit(firstOutput());
    return(true);
  }
  return(true);
}

static int infile[256];
static int outfile[256];
static long int size[256];
static char *source[256];
static char *dest[256];
static char *buffer;
static int pipes;  

void PipeCopy::cleanup(void) {
  int i;
  free(buffer);
  for (i=0;i<pipes;++i) {
    if (infile[i]!=-1) close(infile[i]);
    if (outfile[i]!=-1) close(outfile[i]);
  }
}

  
bool PipeCopy::openOut(void) {
  int i;
  for (i=0;i<pipes;++i) {

    outfile[i]=open(dest[i],O_WRONLY);
    if (outfile[i]==-1) {
      return(false);
    }  
  } 
  return(true);
}

int PipeCopy::pipecopy(int argc,char **argv) {
  bool lateopen=false;
  long int readby;
  int i,j;
  bool fout=false;

  if ((argc-1)%3!=0) return(20);
  pipes=(argc-1)/3;

  for (i=0,j=0;i<pipes;++i) {
    if (strcmp(argv[1+3*i],"MAGIC_DELAY_OPEN")==0) {
      lateopen=true;
      continue;
    } 
    source[j]=argv[1+3*i];
    dest[j]=argv[2+3*i];
    size[j]=atol(argv[3+3*i]);
    infile[j]=-1;
    outfile[j]=-1;
    ++j;
  }

  pipes=j;

  buffer=(char *) malloc(BUFFER_SIZE);
  if (buffer==0) {
    return(20);
  }

  if ( (!lateopen) && (!openOut()) ) {
      cleanup();
      return(20);
  }  

  for (i=0;i<pipes;++i) {

    long int stored;

    infile[i]=open(source[i],O_RDONLY);
    if (infile[i]==-1) {
        cleanup();
        return(20);
    }

    stored=0;  
    while (size[i]>0) { 
      readby=read(infile[i],buffer+stored,BUFFER_SIZE-stored);

      if ( (readby==0) || (readby==-1) ) break;
      if (readby>size[i]) readby=size[i];
      size[i]-=readby;
      stored+=readby;
      if (stored>MIN_BUFFER) {
        if (!fout) {
          fout=true;
          fprintf(stderr,"#FOUT#\n");
          fflush(stderr);
          if ( (lateopen) && (!openOut()) ) {
            cleanup();
            return(20);
          }
        }
        write(outfile[i],buffer,stored);
        stored=0;
      }
    }
    if (stored!=0) {
      write(outfile[i],buffer,stored);
    }

    memset(buffer,0,BUFFER_SIZE);
    while (size[i]>0) {
      readby=(size[i]>BUFFER_SIZE)?BUFFER_SIZE:size[i];
      write(outfile[i],buffer,readby);
      size[i]-=readby;
    }
    close(infile[i]);
    close(outfile[i]);
    infile[i]=-1;
    outfile[i]=-1;
  }

  cleanup();
  return(0);
}
