// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "TrackWindow.h"
#include "CDTrack.h"
#include "AudioFileInfo.h"
#include "TOCReader.h"
#include "TrackDialog.h"
#include "AudioOptionsDialog.h"
#include "IsoOptionsDialog.h"
#include "IsoWindow.h"
#include "IsoImage.h"
#include "AppChooser.h"

#include <qlabel.h>
#include <qcheckbox.h>
#include <qsize.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qwindowdefs.h>
#include <qstring.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qcombobox.h>

#include <kconfig.h>
#include <kapp.h>
#include <klocale.h>
#include <klineedit.h>
#include <kfiledialog.h>
#include <kurl.h>

#include "TrackWindow.moc"

TrackWindow::TrackWindow( QWidget *parent, const char *name,
			  CDTrack *track,int TLMType):QDialog(parent,name,true) {
  char buffer[32];
  KConfig *config;
  QRect DefaultRect(100,50,500,300);
  class QVBoxLayout *winlayout;
  class QHBoxLayout *mainlayout;
  class QHBoxLayout *buttonlayout;
  class QVBoxLayout *leftlayout; 
  class QVBoxLayout *rightlayout;
  class QGridLayout *leftgridlayout;
  _TLMType=TLMType;
 
  ignsigs=0;
  ThisTrack=track;
  config=kapp->config();
  setCaption(kapp->makeStdCaption(i18n("Track Properties")));
  //  setCaption(i18n("KreateCD - track window"));

  winlayout=new QVBoxLayout(this,10); 
  mainlayout=new QHBoxLayout();
  winlayout->addLayout(mainlayout,1);
  leftlayout=new QVBoxLayout();
  mainlayout->addLayout(leftlayout,1);
  buttonlayout = new QHBoxLayout();
  winlayout->addLayout(buttonlayout, 1);

  leftgridlayout=new QGridLayout(7,2);
  leftlayout->addLayout(leftgridlayout);

  // Build track description widget & label

  DescriptionLabel=new QLabel(i18n("Description"),this,"description label");
  DescriptionLabel->setMinimumSize(DescriptionLabel->sizeHint());
  DescriptionEdit=new KLineEdit(this,"description edit");
  DescriptionEdit->setMaxLength(39);
  DescriptionEdit->setMinimumSize(DescriptionEdit->sizeHint());
  leftgridlayout->addWidget(DescriptionLabel,1,0,1);
  leftgridlayout->addWidget(DescriptionEdit,1,1,1);

  // Build track type widget & label
  LabTrackType=new QLabel(i18n("Track type"),this,"track type label");
  LabTrackType->setMinimumSize(LabTrackType->sizeHint());
  CombTrackType=new QComboBox(this,"track type");

  CDTrack::getTypeString(CDTrack::Track_Audio,buffer);
  CombTrackType->insertItem(buffer,-1);
  CDTrack::getTypeString(CDTrack::Track_AudioPreEmph,buffer);
  CombTrackType->insertItem(buffer,-1);
  if (_TLMType!=TLMAudio)
    {
      CDTrack::getTypeString(CDTrack::Track_DataMode1,buffer);
      CombTrackType->insertItem(buffer,-1);
      CDTrack::getTypeString(CDTrack::Track_DataMode2,buffer);
      CombTrackType->insertItem(buffer,-1);
      CDTrack::getTypeString(CDTrack::Track_XA1,buffer);
      CombTrackType->insertItem(buffer,-1);
      CDTrack::getTypeString(CDTrack::Track_XA2,buffer);
      CombTrackType->insertItem(buffer,-1);
      CDTrack::getTypeString(CDTrack::Track_CDI,buffer);
      CombTrackType->insertItem(buffer,-1);
    }

  CombTrackType->setMinimumSize(CombTrackType->sizeHint());
  leftgridlayout->addWidget(LabTrackType,3,0,1);
  leftgridlayout->addWidget(CombTrackType,3,1,1);

  // Build source type widget & label
  LabSourceType=new QLabel(i18n("Track source"),this,"track source label");
  LabSourceType->setMinimumSize(LabSourceType->sizeHint());
  CombSourceType=new QComboBox(this,"track type");
  CombSourceType->insertItem(i18n("None"),-1);
  CombSourceType->insertItem(i18n("CD track"),-1);
  CombSourceType->insertItem(i18n("File"),-1);
  CombSourceType->insertItem(i18n("Filesystem"),-1);
  CombSourceType->setMinimumSize(CombSourceType->sizeHint());
  leftgridlayout->addWidget(LabSourceType,5,0,1);
  leftgridlayout->addWidget(CombSourceType,5,1,1);

  // Build checkbox wheather to use image file 
  CheckUseImage=new QCheckBox(i18n("Use image file"),this,"use file image");
  CheckUseImage->setFixedSize(CheckUseImage->sizeHint());
  leftlayout->addWidget(CheckUseImage);
  
  // Build button for reselection of source
  ButtonSelectSource=new QPushButton(i18n("Select source"),this,"source select");
  ButtonSelectSource->setMinimumSize(ButtonSelectSource->sizeHint());
  leftlayout->addWidget(ButtonSelectSource);

  // Build options button
  ButtonOptions=new QPushButton(i18n("Options..."),this,"track options");
  ButtonOptions->setMinimumSize(ButtonOptions->sizeHint());
  leftlayout->addWidget(ButtonOptions);

  PropertyFrame=new QGroupBox(i18n("Properties"),this,"props frame");
  PropertyFrame->setAlignment(AlignHCenter);
  rightlayout=new QVBoxLayout(PropertyFrame,30);  

  mainlayout->addWidget(PropertyFrame,2);
  PropertyLabel=new QLabel(PropertyFrame,"props label");
  PropertyFrame->setMinimumSize(80,80);


  rightlayout->addWidget(PropertyLabel);


  connect(CombTrackType,SIGNAL(activated(int)),
	  this,SLOT(changeTrackMode(int)));
  connect(CombSourceType,SIGNAL(activated(int)),
	  this,SLOT(changeSourceType(int)));
  connect(DescriptionEdit,SIGNAL(textChanged(const QString &)),
          this,SLOT(changeDescription(const QString &)));
  connect(CheckUseImage,SIGNAL(toggled(bool)),this,SLOT(changeUseImage(bool)));
  connect(ButtonSelectSource,SIGNAL(clicked()),this,SLOT(changeSource()));
  connect(ButtonOptions,SIGNAL(clicked()),this,SLOT(openOptions()));

  OkButton=new QPushButton(i18n("OK"),this);
  OkButton->setMinimumSize(OkButton->sizeHint());
  CancelButton = new QPushButton(i18n("Cancel"), this);
  CancelButton->setMinimumSize(CancelButton->sizeHint());

  buttonlayout->addWidget(OkButton);
  buttonlayout->addWidget(CancelButton);

  connect(OkButton, SIGNAL(clicked()), this, SLOT(hide()));
  connect(CancelButton, SIGNAL(clicked()), this, SIGNAL(canceled()));

  config->setGroup("Geometry");
  this->setGeometry(config->readRectEntry("TrackWindowGeo",&DefaultRect));
  winlayout->activate();

  updateWindow();
  
}

TrackWindow::~TrackWindow(void) {
  KConfig *config;
  QRect SaveRect(this->pos(),this->size());
  config=kapp->config();
  config->setGroup("Geometry");
  config->writeEntry("TrackWindowGeo",SaveRect);
}

void TrackWindow::closeEvent(QCloseEvent *) {
  emit canceled();
}
  
void TrackWindow::changeTrackMode(int modeindex) {
  if (ignsigs) return;

  switch (modeindex) {
  case 0:
    ThisTrack->setTrackType(CDTrack::Track_Audio);
    break;
  case 1:
    ThisTrack->setTrackType(CDTrack::Track_AudioPreEmph);
    break;
  case 2:
    ThisTrack->setTrackType(CDTrack::Track_DataMode1);
    break;
  case 3:
    ThisTrack->setTrackType(CDTrack::Track_DataMode2);
    break;
  case 4:
    ThisTrack->setTrackType(CDTrack::Track_XA1);
    break;
  case 5:
    ThisTrack->setTrackType(CDTrack::Track_XA2);
    break;
  case 6:
    ThisTrack->setTrackType(CDTrack::Track_CDI);
    break;
  }
  updateWindow();
}

void TrackWindow::changeSource(void) {
  changeSourceType(CombSourceType->currentItem());
}

void TrackWindow::changeSourceType(int sourcetype) {
  KConfig *config;

  config=kapp->config();
  if (ignsigs) return;

  switch (sourcetype) {
  case 0:
    ThisTrack->clearSource();
    break;
  case 1:
    {
      TOCReader *toc;
      int seltrack;
      if ((toc=AppChooser::readTOC())==0) {
	QMessageBox::critical(this,QString::null,i18n("No disc in drive!"));
	break;
      }
      
      {
	TrackDialog *dd;
	dd=new TrackDialog(this,0,toc);
	seltrack=dd->exec();
	delete(dd);
	if (seltrack!=0) {
	  ThisTrack->setTrackType(toc->getTOCType(seltrack));
	  ThisTrack->setSourceTrack(seltrack,toc->getTOCSum(),
				    toc->getTOCDuration(seltrack),
				    toc->getTOCLBA(seltrack));
	}
	delete toc;
      }
      break;
    }
  case 2:
    {
      KURL url;
      QString qs;
      config->setGroup("Path");
      url=KFileDialog::getOpenURL(config->readEntry("PathDatafile",""))
	.path();
      qs=url.path();
      if (qs.isEmpty()) break;
      ThisTrack->setSourceFile(qs.data());
      config->setGroup("Path");
      config->writeEntry("PathDatafile",qs.data());
      break;
    }
  case 3:
    {
      class ISOImage *isoim;
      ISOWindow *ISOTreeWindow;
      isoim=ThisTrack->getISOImage();
      ThisTrack->setSourceFileSystem(isoim);
      if (ThisTrack->getISOImage()==0) break;
      if (isoim==0) {
	connect(ThisTrack->getISOImage(),SIGNAL(imageChanged()),
		this,SLOT(makeImageDirty()));
      }
      ISOTreeWindow=new ISOWindow(0,0,ThisTrack->getISOImage(),ThisTrack);
      connect(ISOTreeWindow,SIGNAL(needUpdate()),
	      this,SLOT(updateWindowContents()));
      connect(ISOTreeWindow,SIGNAL(imageChanged()),
            this,SLOT(makeImageDirty()));
      ISOTreeWindow->exec();
      delete ISOTreeWindow;
      break;
    }
  }
  updateWindow();
}

void TrackWindow::changeUseImage(bool useimage) {
  if (ignsigs) return;
  if (useimage) {
    ThisTrack->enableImage();
  } else {
    ThisTrack->disableImage();
  }
  updateWindow();
}

void TrackWindow::changeDescription(const QString &descr) {
  ThisTrack->setDescription(descr.data());
  updateWindow();
}


void TrackWindow::updateWindowContents(void) {
  emit (updateTrack());
  updateWindow();
}

void TrackWindow::updateWindow(void) {
  char properties[2048];
  char tempstring[512];
  int item=0;

  ignsigs=1;
  switch (ThisTrack->getTrackType()) {
  case CDTrack::Track_Audio:
    item=0;
    break;
  case CDTrack::Track_AudioPreEmph:
    item=1;
    break;
  case CDTrack::Track_DataMode1:
    item=2;
    break;
  case CDTrack::Track_DataMode2:
    item=3;
    break;
  case CDTrack::Track_XA1:
    item=4;
    break;
  case CDTrack::Track_XA2:
    item=5;
    break;
  case CDTrack::Track_CDI:
    item=6;
    break;
  }
  CombTrackType->setCurrentItem(item);
  switch(ThisTrack->getTrackSource()) {
  case CDTrack::Source_None:
    item=0;
    break;
  case CDTrack::Source_CD:
    item=1;
    break;
  case CDTrack::Source_File:
    item=2;
    break;
  case CDTrack::Source_Filesystem: 
    item=3;
    break;
  }
  CombSourceType->setCurrentItem(item);
  ThisTrack->getTrackDescription(tempstring);
  DescriptionEdit->setText(tempstring);
  CheckUseImage->setChecked(ThisTrack->getUseImage());
  ignsigs=0;

  ButtonSelectSource->setEnabled(ThisTrack->getTrackSource()!=
				 CDTrack::Source_None);
  ButtonOptions->setEnabled(
			    (ThisTrack->getTrackSource()!=CDTrack::Source_None) &&
			    (ThisTrack->isAudio()) ||
			    (ThisTrack->getTrackSource()==CDTrack::Source_Filesystem)
			    );


  emit updateTrack();
  
  if (ThisTrack->getTrackSource()==CDTrack::Source_None) {
    PropertyLabel->setText("");
    return;
  }

  if (ThisTrack->isAudio()) {
    AudioFileInfo *afile;
    strcpy(properties,i18n("Audio track\n"));
    afile=ThisTrack->getAudioFile();
    if (afile==0) {
      strcat(properties,i18n("WARNING: unknown format!\n\n"));
    } else {
      enum AudioFileInfo::Sample_Endian endi;
      enum AudioFileInfo::Sample_Size sasi;
      int realsiz;
      char dura[32];

      sprintf(tempstring,i18n("Format: %s\n\n"),afile->formatName());
      strcat(properties,tempstring);
      endi=afile->endian();
      if (endi==AudioFileInfo::Endian_Little) {
        strcat(properties,i18n("Intel byte order\n"));
      } else {
        strcat(properties,i18n("Motorola byte order\n"));
      }
      sasi=afile->size();
      realsiz=8;
      if (sasi==AudioFileInfo::Samples_16Bit) realsiz=16;
      if (sasi==AudioFileInfo::Samples_24Bit) realsiz=24;
      if (sasi==AudioFileInfo::Samples_32Bit) realsiz=32;
      sprintf(tempstring,i18n("%d Channels - %d Bits\n"),
	      afile->channels(),realsiz);
      strcat(properties,tempstring);
      sprintf(tempstring,i18n("Sample rate: %ld\n\n"),afile->rate());
      CDTrack::getDurationHMSB(dura,afile->getAbsoluteDuration(),true);
      strcat(properties,tempstring);
      sprintf(tempstring,i18n("Absolute length: %s\n"),dura);
      strcat(properties,tempstring);
      CDTrack::getDurationHMSB(dura,afile->getSelectedDuration(),true);
      sprintf(tempstring,i18n("Selected length: %s\n\n"),dura);
      strcat(properties,tempstring);
      sprintf(tempstring,i18n("Boost factor   : %f\n\n"),
	      afile->boost());
      strcat(properties,tempstring);
      sprintf(tempstring,i18n("Balance        : %f\n\n"),
	      afile->balance());
      strcat(properties,tempstring);

    }
  } else {
    strcpy(properties,"Data track\n\n");     
  }
  
  PropertyLabel->setText(properties);
}

void TrackWindow::openOptions(void) {
  if (ThisTrack->isAudio()) {
    AudioOptionsDialog *AudioOptWindow;

    if ( (ThisTrack->getAudioFile()!=0) && (ThisTrack->getAudioFile()->needsRipping())) {
      ThisTrack->validateTrack();
      if (ThisTrack->getAudioFile()->needsRipping()) {
        QMessageBox::critical(this,QString::null,
			      i18n("Cannot read source track!"));
        return;
      }
    }

    if ( (ThisTrack->getAudioFile()==0) || (!ThisTrack->getAudioFile()->validateFile()) ) {
      QMessageBox::critical(this,QString::null,
		      i18n("Cannot read source track!"));
      return;
    }

    updateWindow();
    emit(updateTrack());
    AudioOptWindow=new AudioOptionsDialog(0,0,ThisTrack,true);
    connect(AudioOptWindow,SIGNAL(needUpdate()),
	    this,SLOT(updateWindowContents()));
    connect(AudioOptWindow,SIGNAL(needUpdate()),this,SLOT(makeImageDirty()));
    AudioOptWindow->exec();
    delete AudioOptWindow;
  } else {
    ISOOptionsDialog *ISOOptWindow;
    if (ThisTrack->getTrackSource()!=CDTrack::Source_Filesystem) return;
    if (ThisTrack->getISOImage()==0) return;
    ISOOptWindow=new ISOOptionsDialog(0,0,ThisTrack->getISOImage());
    connect(ISOOptWindow,SIGNAL(needUpdate()),this,
	    SLOT(updateWindowContents()));
    ISOOptWindow->exec();
    delete ISOOptWindow;
  }
}

void TrackWindow::makeImageDirty(void) {
  ThisTrack->clearImage();
}

bool TrackWindow::isOpenedTrack(CDTrack *track)
{
  return (ThisTrack==track);
}
