// KreateCD - CD recording software for the K desktop environment
//
// 2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "WaveView.h"
#include "WaveData.h"
#include "WaveView.moc"

#include <qwidget.h>
#include <qpainter.h>
#include <qpoint.h>
#include <qtimer.h>

#include <math.h>
#include <stdio.h>

static double waveview_scale_label[]={15*60,10*60,5*60,2*60,60,30,10,5,2,1,
                                      0.5,0.2,0.1,0.05,0.02,0.01,0.005,
                                      0.002,0.001,0.0005,0.0002,0.0001,0};
static double waveview_scale_main[]={60,60,60,60,60,10,10,1,1,1,
                                     0.1,0.1,0.1,0.01,0.01,0.01,0.001,
                                     0.001,0.001,0.0001,0.0001,0.0001,0};
static double waveview_scale_sub[]={30,30,30,30,30,5,5,0.5,0.5,0.5,
                                    0.05,0.05,0.05,0.005,0.005,0.005,0.0005,
                                    0.0005,0.0005,0.00005,0.00005,0.00005,0};
static int waveview_scale_prec[]={0,0,0,0,0,0,0,0,0,0,1,1,1,2,2,2,3,3,3,
                                  4,4,4,0};

WaveView::WaveView(QWidget *parent,WaveData *wdat):QWidget(parent) {
  int i;

  minimumY=wdat->getMinY();
  minimumX=0;
  maximumY=wdat->getMaxY();
  maximumX=wdat->getMaxX();
  sampleRate=wdat->getSampleRate();
  curves=wdat->getCurves();
  waveData=wdat;
  if (curves>WAVEVIEW_MAXCURVES) {
    curves=WAVEVIEW_MAXCURVES;
  }
  displayMin=minimumX;
  displayMax=maximumX;

  for (i=0;i<WAVEVIEW_MAXCURVES;++i) {
    cachePositive[i]=0;
    cacheNegative[i]=0;
  }
  cachedWidth=0;
  myTimer=new QTimer();
  connect(myTimer,SIGNAL(timeout()),this,SLOT(timerEvent()));

  curveColorsV[0]=QColor(196,0,0);
  curveColorsV[1]=QColor(0,196,0);
  curveColorsV[2]=QColor(0,0,196);
  curveColorsV[3]=QColor(196,196,0);
  curveColorsV[4]=QColor(196,0,196);
  curveColorsV[5]=QColor(0,196,196);
  curveColorsV[6]=QColor(196,196,196);
  curveColorsV[7]=QColor(128,128,128);
  startMarkColorV=QColor(128,255,128);
  endMarkColorV=QColor(255,128,128);
  playingColorV=QColor(128,128,255);
  axisColorV=QColor(255,255,255);
  backColorV=QColor(0,0,0);
  selectColorV=QColor(128,0,0);
  scaleColorV=QColor(196,196,196);

  startMarkSample=-1;
  endMarkSample=-1;
  playMarkSample=-1;
  selectStart=-1;
  selectEnd=-1;

  mouseState=No_Action;
  mouseButton=Button_None;

  autoYZoom=true;
  autoFollow=true;
  showTimescale=true;
  showTimerange=true;
  showVolumerange=true;

  updateGeometry();
}

WaveView::~WaveView(void) {
  flushCache();
}

void WaveView::setStartMark(long sample) {
  QRect updRect;
  int ox,nx;
  if ( (sample<-1) || (sample>maximumX) ) return;  
  if (sample==startMarkSample) return;
  ox=coordX(startMarkSample);
  startMarkSample=sample;
  if (startMarkSample>endMarkSample) {
    setEndMark(startMarkSample);
    emit endChanged(endMarkSample);
  }
  nx=coordX(sample);
  if ( (nx==-1) && (ox==-1) ) return;
  if ( (ox==-1) || (nx==-1) ) {
    if (ox==-1) {
      updRect=QRect(nx,reservedTop,1,realHeight);
    } else {
      updRect=QRect(ox,reservedTop,1,realHeight);
    }
  } else if (ox>=nx) {
      updRect=QRect(nx,reservedTop,ox-nx+1,realHeight);
  } else {
      updRect=QRect(ox,reservedTop,nx-ox+1,realHeight);
  }

  repaint(updRect,false);
}


void WaveView::setEndMark(long sample) {
  QRect updRect;
  int ox,nx;
  if ( (sample<-1) || (sample>maximumX) ) return;  
  if (sample==endMarkSample) return;
  ox=coordX(endMarkSample);
  endMarkSample=sample;
  if (endMarkSample<startMarkSample) {
    setStartMark(endMarkSample);
    emit startChanged(startMarkSample);
  }
  nx=coordX(sample);
  if ( (ox==-1) && (nx==-1) ) return;
  if ( (ox==-1) || (nx==-1) ) {
    if (ox==-1) {
      updRect=QRect(nx,reservedTop,1,realHeight);
    } else {
      updRect=QRect(ox,reservedTop,1,realHeight);
    }
  } else if (ox>=nx) {
      updRect=QRect(nx,reservedTop,ox-nx+1,realHeight);
  } else {
      updRect=QRect(ox,reservedTop,nx-ox+1,realHeight);
  }

  repaint(updRect,false);
}

void WaveView::setPlayMark(long sample) {
  QRect updRect;
  int ox,nx;
  if ( (sample<-1) || (sample>maximumX) ) return;  
  if (sample==playMarkSample) return;
  ox=coordX(playMarkSample);
  playMarkSample=sample;
  if ( (playMarkSample!=-1) && (
       (displayMin>playMarkSample) || (displayMax<playMarkSample)
     ) ) {
    setVisibleRange(playMarkSample,playMarkSample+displayMax-displayMin);
    return;
  }
  nx=coordX(sample);
  if ( (ox==-1) && (nx==-1) ) return;
  if ( (ox==-1) || (nx==-1) ) {
    if (ox==-1) {
      updRect=QRect(nx,reservedTop,1,realHeight);
    } else {
      updRect=QRect(ox,reservedTop,1,realHeight);
    }
  } else if (ox>=nx) {
      updRect=QRect(nx,reservedTop,ox-nx+1,realHeight);
  } else {
      updRect=QRect(ox,reservedTop,nx-ox+1,realHeight);
  }

  repaint(updRect,false);
}

void WaveView::setSelectRegion(long ssample,long esample) {
  QRect updregion;

  if (selectEnd!=-1) {
    int x1;
    int x2;
    x1=coordX(selectStart);
    x2=coordX(selectEnd);
    if ( (x1!=-1) || (x2!=-1) ) {
      if (x1==-1) x1=0;
      if (x2==-1) x2=realWidth-1;
      updregion=QRect(x1,reservedTop,x2-x1+1,realHeight);
    }
  }

  if ( (ssample==-1) || (esample==-1) || (ssample>esample) || 
       (ssample<minimumX) || (esample>maximumX) ) {
    selectStart=-1;
    selectEnd=-1;
  } else {
    int x1;
    int x2;
    selectStart=ssample;
    selectEnd=esample;
    x1=coordX(selectStart);
    x2=coordX(selectEnd);
    if ( (x1!=-1) || (x2!=-1) ) {
      if (x1==-1) x1=0;
      if (x2==-1) x2=realWidth-1;
      updregion|=QRect(x1,0,x2-x1+1,realHeight);
    }
  }
  repaint(updregion,false);
}


void WaveView::setVisibleRange(long ssample,long esample) {
  if ( (ssample>maximumX) || (esample<minimumX) ||
       (ssample>esample) ) return;

  if ( (ssample==displayMin) && (esample==displayMax) ) return;
  if ( (ssample+realWidth>esample) ) {
    ssample=esample-realWidth+1;
    if (ssample<minimumX) ssample=minimumX;
  }
  displayMin=ssample;
  displayMax=esample;
  updateGeometry();
}

void WaveView::zoomIn(int percent) {
  long long xsize,ns,ne;
  xsize=displayMax-displayMin;
  ns=displayMin+((xsize*percent)/100);
  ne=displayMax-((xsize*percent)/100);
  setVisibleRange(ns,ne);
}

void WaveView::zoomOut(int percent) {
  long xsize,ns,ne;
  xsize=displayMax-displayMin;
  ns=displayMin-((xsize*percent)/100);
  ne=displayMax+((xsize*percent)/100);
  if (ns<minimumX) ns=minimumX;
  if (ne>maximumX) ne=maximumX;
  setVisibleRange(ns,ne);
}

void WaveView::setAutoZoom(bool azoom) {
  autoYZoom=azoom;
  update();
}

void WaveView::setFollowPlay(bool fplay) {
  autoFollow=fplay;
  update();
}

void WaveView::setTimeScale(bool tscale) {
  showTimescale=tscale;
  update();
}

void WaveView::setTimeRange(bool trange) {
  showTimerange=trange;
  update();
}

void WaveView::setVolumeRange(bool vrange) {
  showVolumerange=vrange;
  update();
}

void WaveView::setZoom(double zoom) {
  if (zoom<=0) return;
  if (zoom>100) return;
  zoomY=zoom;
  update();
}

void WaveView::setStartMarkColor(QColor &col) {
  startMarkColorV=col;
  update();
}

void WaveView::setEndMarkColor(QColor &col) {
  endMarkColorV=col;
  update();
}

void WaveView::setPlayingColor(QColor &col) {
  playingColorV=col;
  update();
}

void WaveView::setAxisColor(QColor &col) {
  axisColorV=col;
  update();
}

void WaveView::setBackColor(QColor &col) {
  backColorV=col;
  update();
}

void WaveView::setSelectColor(QColor &col) {
  selectColorV=col;
  update();
}

void WaveView::setScaleColor(QColor &col) {
  scaleColorV=col;
  update();
}

void WaveView::setFirstChannelColor(QColor &col) {
  curveColorsV[0]=col;
  update();
}

void WaveView::setSecondChannelColor(QColor &col) {
  curveColorsV[1]=col;
  update();
}

void WaveView::setThirdChannelColor(QColor &col) {
  curveColorsV[2]=col;
  update();
}

void WaveView::setFourthChannelColor(QColor &col) {
  curveColorsV[3]=col;
  update();
}

void WaveView::setFifthChannelColor(QColor &col) {
  curveColorsV[4]=col;
  update();
}

void WaveView::setSixthChannelColor(QColor &col) {
  curveColorsV[5]=col;
  update();
}

void WaveView::setSeventhChannelColor(QColor &col) {
  curveColorsV[6]=col;
  update();
}

void WaveView::setEighthChannelColor(QColor &col) {
  curveColorsV[7]=col;
  update();
}

void WaveView::setChannelColor(int channel,QColor &col) {
  if ( (channel<1) || (channel>WAVEVIEW_MAXCURVES) ) return;
  curveColorsV[channel-1]=col;
  update();
}  

long WaveView::startMark(void) const {
  return(startMarkSample);
}

long WaveView::endMark(void) const {
  return(endMarkSample);
}

long WaveView::playMark(void) const {
  return(playMarkSample);
}

bool WaveView::autoZoom(void) const {
  return(autoYZoom);
}

bool WaveView::followPlay(void) const {
  return(autoFollow);
}

bool WaveView::timeScale(void) const {
  return(showTimescale);
}

bool WaveView::timeRange(void) const {
  return(showTimerange);
}

bool WaveView::volumeRange(void) const {
  return(showVolumerange);
}

double WaveView::zoom(void) const {
  return(zoomY);
}

QColor WaveView::startMarkColor() const {
  return(startMarkColorV);
}

QColor WaveView::endMarkColor() const {
  return(endMarkColorV);
}

QColor WaveView::playingColor() const {
  return(playingColorV);
}

QColor WaveView::axisColor() const {
  return(axisColorV);
}

QColor WaveView::backColor() const {
  return(backColorV);
}

QColor WaveView::selectColor() const {
  return(selectColorV);
}

QColor WaveView::scaleColor() const {
  return(scaleColorV);
}

QColor WaveView::firstChannelColor() const {
  return(curveColorsV[0]);
}

QColor WaveView::secondChannelColor() const {
  return(curveColorsV[1]);
}

QColor WaveView::thirdChannelColor() const {
  return(curveColorsV[2]);
}

QColor WaveView::fourthChannelColor() const {
  return(curveColorsV[3]);
} 

QColor WaveView::fifthChannelColor() const {
  return(curveColorsV[4]);
}

QColor WaveView::sixthChannelColor() const {
  return(curveColorsV[5]);
}

QColor WaveView::seventhChannelColor() const {
  return(curveColorsV[6]);
}

QColor WaveView::eighthChannelColor() const {
  return(curveColorsV[7]);
}

QColor WaveView::channelColor(int chan) const {
  if ( (chan<1) || (chan>WAVEVIEW_MAXCURVES) ) return(curveColorsV[0]);
  return(curveColorsV[chan-1]);
}

QSize WaveView::sizeHint(void) const {
  return QSize(200,100);
}

QSizePolicy WaveView::sizePolicy(void) const {
  return QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
}

void WaveView::resizeEvent(QResizeEvent *) {
  updateGeometry();
}

void WaveView::updateGeometry(void) {
  double displaySecs;
  int i;

  widgetHeight=height();
  widgetWidth=width();

  reservedTop=showTimescale?fontMetrics().height()*2:0;
  reservedBottom=showTimerange?4:0;
  reservedLeft=showVolumerange?4:0;

  fontBase=fontMetrics().ascent();

  realHeight=widgetHeight-reservedTop-reservedBottom;
  realWidth=widgetWidth-reservedLeft;

  if ( (maximumY<=minimumY) || (realHeight<=0) ) {
    scaleY=0;
  } else {
    scaleY=(double)realHeight/(double)(maximumY-minimumY);
  }
  offsetY=maximumY*scaleY+reservedTop;

  if ( (displayMax<=displayMin) || (realWidth<=0) ) {
    scaleX=0;
  } else {
    scaleX=(double)realWidth/(double)(displayMax-displayMin);
  }
  
  if (realWidth<=0) {
    steppingX=0;
  } else {
    steppingX=(double)(displayMax-displayMin)/(double)realWidth; 
  }

  spanX=(int)(steppingX)-1;
  if (spanX<0) spanX=0;
  flushCache();
  setupCache();
  if (autoYZoom) zoomY=1;

  displaySecs=(double)(displayMax-displayMin)/(double)sampleRate;

  for (i=0;waveview_scale_label[i]!=0;++i) {
    if (displaySecs/waveview_scale_label[i]>5) break;
    labelScale=waveview_scale_label[i];
    mainScale=waveview_scale_main[i];
    subScale=waveview_scale_sub[i];
    labelPrec=waveview_scale_prec[i];    
  }

  update();
}


void WaveView::fillAverage(int curve,int x) {
  long long pcummulate=0;
  long long ncummulate=0;
  long data,sx,ex;
  long int ps=0,ns=0,i;

  sx=(long int) (displayMin+steppingX*x);;
  ex=spanX;
  if (sx+ex>displayMax) {
    ex=displayMax-sx;
  }
 
  for (i=sx;i<=(sx+ex);++i) {
    data=getData(curve,i);
    if (data<0) {
      ncummulate+=data;
      ++ns;
    } else {
      pcummulate+=data;
      ++ps;
    }
  } 
  cachePositive[curve][x]=(ps==0?0:pcummulate/ps);
  cacheNegative[curve][x]=(ns==0?0:ncummulate/ns);
  if (cachePositive[curve][x]>maxSample) {
    maxSample=cachePositive[curve][x];
  }
  if (cacheNegative[curve][x]<minSample) {
    minSample=cacheNegative[curve][x];
  }
}

long WaveView::averagePosData(int curve,int x) {
  if ( (curve<0) || (curve>WAVEVIEW_MAXCURVES) ) return(0);
  if (cachePositive[curve]==0) return(0);
  if (cachedWidth<=x) return(0);
  return(cachePositive[curve][x]);
}

long WaveView::averageNegData(int curve,int x) {
  if ( (curve<0) || (curve>WAVEVIEW_MAXCURVES) ) return(0);
  if (cacheNegative[curve]==0) return(0);
  if (cachedWidth<=x) return(0);
  return(cacheNegative[curve][x]);
}


long WaveView::getData(int curve,long x) {
  return(waveData->getData(curve,x));
}

void WaveView::flushCache(void) {
  int i;
  
  for (i=0;i<WAVEVIEW_MAXCURVES;++i) {
    if (cachePositive[i]!=0) {
      delete [] cachePositive[i];
      cachePositive[i]=0;
    }
    if (cacheNegative[i]!=0) {
      delete [] cacheNegative[i];
      cacheNegative[i]=0;
    }
  }
  cachedWidth=0;
  maxSample=0;
  minSample=0;
}

void WaveView::setupCache(void) {
  int i;

  for (i=0;i<curves;++i) {
    cachePositive[i]=new long[realWidth];
    cacheNegative[i]=new long[realWidth];
  }
  cachedWidth=0;
}

void WaveView::fillCache(int x) {
  int i;
  for (i=0;i<curves;++i) {
    fillAverage(i,x);
  }
  if (cachedWidth<=(x)) cachedWidth=(x+1);
}


int WaveView::coordX(long int sample) {
  int resu;
  if ( (sample<displayMin) || (sample>displayMax) ) return(-1);
  resu=(int) ((sample-displayMin)*scaleX);
  if (resu>=realWidth) return(reservedLeft+realWidth-1);
  return(reservedLeft+resu);
}

int WaveView::coordY(int value) {
  int resu;
  resu=(int)(offsetY-value*scaleY*zoomY);
  if ( (resu<reservedTop) || (resu>=(realHeight+reservedTop)) ) {
    return(resu<reservedTop?reservedTop:realHeight+reservedTop-1);
  }
  return(resu);
}

long int WaveView::sampleX(int x) {
  int resu;
  resu= (long int) (displayMin+steppingX*(x-reservedLeft));
  return(resu);
}

void WaveView::drawPartial(QPainter *painter,int x) {
  int i,markx,markx2;

  if ( (x<reservedLeft) || (x>=(realWidth+reservedLeft)) ) {
    return; 
  }

  if ( startMarkSample!=-1) {
    markx=coordX(startMarkSample);
    if (markx==x) {
      painter->setPen(QPen(startMarkColorV));
      painter->drawLine( x,coordY(maximumY),
                    x,coordY(minimumY));
    }
  }
  if ( endMarkSample!=-1) {
    markx=coordX(endMarkSample);
    if (markx==x) {
      painter->setPen(QPen(endMarkColorV));
      painter->drawLine( x,coordY(maximumY),
                    x,coordY(minimumY));
    }
  }

  if ( playMarkSample!=-1) {
    markx=coordX(playMarkSample);
    if (markx==x) {
      painter->setPen(QPen(playingColorV));
      painter->drawLine( x,coordY(maximumY),
                    x,coordY(minimumY));
    }
  }

  if (selectStart!=-1) {
    markx=coordX(selectStart);
    markx2=coordX(selectEnd);
    if ( ( (markx!=-1) && (markx2!=-1) &&
           (markx<=x) && (markx2>=x) ) ||
         ( (markx==-1) && (markx2!=-1) &&
           (markx2<=x) )               ||
         ( (markx!=-1) && (markx2==-1) &&
           (markx>=x) )  ) {
      painter->setPen(QPen(selectColorV));
      painter->drawLine( x,coordY(maximumY),
                    x,coordY(minimumY));
    } 

  }

  for (i=0;i<curves;++i) {
    painter->setPen(QPen(curveColorsV[i]));
    painter->drawLine( x,coordY(averagePosData(i,x-reservedLeft)),
                  x,coordY(averageNegData(i,x-reservedLeft)));
  }

}

void WaveView::paintEvent(QPaintEvent *event) {
  QRect innerRect(reservedLeft,reservedTop,realWidth,realHeight);

  QPainter paint(this);

  if (event->erased()) {
    paintType=Repaint_Normal;
  }

  if ( (paintType==Repaint_Normal) && (cachedWidth==0) ) {
    paint.fillRect(innerRect,QBrush(backColorV)); 
    paintType=Repaint_Scanning;
    drawTimescale(&paint);
  }

  switch (paintType) {
    case Repaint_Normal:
      {
        int i,sx,ex;
        QRect paintRect;

        paintRect=innerRect&event->rect();        
        paint.fillRect(paintRect,QBrush(backColorV)); 
        drawTimescale(&paint);

        sx=paintRect.x();
        ex=paintRect.width()-1+sx;
        if (cachedWidth<=(ex-reservedLeft)) {
          ex=cachedWidth-1+reservedLeft;
        }

        for (i=sx;i<=ex;++i) {
          drawPartial(&paint,i);  
        }
      }
      break;
    case Repaint_Scanning:
      if (cachedWidth<realWidth) {
        fillCache(cachedWidth);
        drawPartial(&paint,cachedWidth-1+reservedLeft);
        if (cachedWidth<realWidth) {
          myTimer->start(0,true);
        } else {
          if (autoYZoom) {
            double zfp,zfn;
            zfp= (maxSample==0?maximumY: (double)maximumY/(double)maxSample);
            zfn= (minSample==0?-minimumY: (double)minimumY/(double)minSample);
            if (zfp>zfn) {
              zoomY=zfn;
            } else {
              zoomY=zfp;
            }
            update();
          }
        }
      } 
      break;
  }

  paint.end();
  paintType=Repaint_Normal;
  return;
}

void WaveView::drawTimescale(QPainter *painter) {
  double currentPos,divider;
  double currentMins,currentSecs;
  int mins;
  int xc;
  QString s1,s2;

  if (showTimescale) {
    painter->setPen(QPen(foregroundColor()));
    painter->setFont(font());

    currentPos=displayMin;
    divider=ceil(currentPos/subScale/sampleRate);
    currentPos=divider*subScale*sampleRate;
    while (currentPos<=displayMax) {
      xc=coordX((long)currentPos);
      painter->drawLine( xc,reservedTop*3/4,
                      xc,reservedTop-1);
      currentPos+=subScale*sampleRate;
    }    

    currentPos=displayMin;
    divider=ceil(currentPos/mainScale/sampleRate);
    currentPos=divider*mainScale*sampleRate;
    while (currentPos<=displayMax) {
     xc=coordX((long)currentPos);
       painter->drawLine( xc,reservedTop*5/8,
                      xc,reservedTop-1);
      currentPos+=mainScale*sampleRate;
    }    

    currentPos=displayMin;
    divider=ceil(currentPos/labelScale/sampleRate);
    currentPos=divider*labelScale*sampleRate;
    while (currentPos<=displayMax) {
      xc=coordX((long)currentPos);
      currentSecs=currentPos/sampleRate;
      currentMins=floor(currentSecs/60);
      currentSecs-=60*currentMins;
      mins=(int) currentMins;
      s1.setNum(mins);
      s2.setNum(currentSecs,'f',labelPrec);
      s1+=QChar(':');
      if (currentSecs<10) s1+=QChar('0');
      s1+=s2;
      painter->drawText(xc,fontBase,s1);
      currentPos+=labelScale*sampleRate;
    }    
  }
  if ( (showTimerange) && ((maximumX-minimumX)>0) ) {
    double xscale;
    int x1,x2;
    QRect paintRect;
 
    xscale=(double) realWidth/(double) (maximumX-minimumX);
    x1=(int) (xscale*(double)displayMin);
    x2=(int) (xscale*(double)displayMax);
    if (x2>=realWidth) x2=realWidth-1;
    paintRect.setCoords(reservedLeft,widgetHeight-reservedBottom,
                        realWidth+reservedLeft-1,widgetHeight-1); 
    painter->fillRect(paintRect,QBrush(backColorV));   
    paintRect.setCoords(reservedLeft+x1,widgetHeight-reservedBottom,
                        reservedLeft+x2,widgetHeight-1); 
    painter->fillRect(paintRect,QBrush(scaleColorV));   
  }

  if ( (showVolumerange) && ((realHeight)>0) ) {
    int y1,y2;
    QRect paintRect;

    y2=(int)(offsetY-minimumY*scaleY/zoomY); 
    y1=(int)(offsetY-maximumY*scaleY/zoomY); 
    if (y2>=(realHeight+reservedTop)) y2=realHeight-1+reservedTop;
    if (y1<reservedTop) y1=0;
    paintRect.setCoords(0,reservedTop,
                        reservedLeft-1,reservedTop+realHeight-1); 
    painter->fillRect(paintRect,QBrush(backColorV));   
    paintRect.setCoords(0,y1,reservedLeft-1,y2); 
    painter->fillRect(paintRect,QBrush(scaleColorV));   
  }
}

void WaveView::timerEvent(void) {
  if (cachedWidth!=0) {
    paintType=Repaint_Scanning;
    repaint(false);
  }
}

void WaveView::mousePressEvent(QMouseEvent *event) {
  long startX,stopX,playX;
  int moveTolerance=3;

  mouseState=Mouse_Clicking;
  switch (event->button()) {
    case LeftButton:
      mouseButton=Button_Left;
      break;
    case RightButton:
      mouseButton=Button_Right;
      break;
    case MidButton:
      mouseButton=Button_Mid;
      break;
    default:
      mouseState=No_Action;
      return;
  }
  mouseX=event->x();
  mouseY=event->y();
  pressX=mouseX;
  pressY=mouseY;

  startX=coordX(startMarkSample);
  stopX=coordX(endMarkSample);
  playX=coordX(playMarkSample);

  if ( (playX!=-1) && ( mouseX<(playX+moveTolerance)) && 
       (mouseX>(playX-moveTolerance)) ) {
    mouseState=Mouse_MoveMark;    
    mouseMove=PlayMark;
  }

  if ( (startX!=-1) && ( mouseX<(startX+moveTolerance)) && 
       (mouseX>(startX-moveTolerance)) ) {
    mouseState=Mouse_MoveMark;    
    mouseMove=StartMark;
  }

  if ( (stopX!=-1) && ( mouseX<(stopX+moveTolerance)) && 
       (mouseX>(stopX-moveTolerance)) ) {
    mouseState=Mouse_MoveMark;    
    mouseMove=StopMark;
  }
  
  handleMouseAction();
}

void WaveView::mouseReleaseEvent(QMouseEvent *) {
  handleMouseAction();
  switch (mouseState) {
    case Mouse_Clicking:
      switch (mouseButton) {
        case Button_Left: 
          setPlayMark(sampleX(pressX));
          playChanged(playMarkSample);
          break;
        case Button_Right:
          setVisibleRange(minimumX,maximumX);
          break;
        default:
          break;
      } 
      break;

    case Mouse_Dragging:
      switch (mouseButton) {
        case Button_Left: 
          setStartMark(selectStart);
          setEndMark(selectEnd);
          startChanged(startMarkSample);
          endChanged(endMarkSample);
          break;
        case Button_Right: 
          setVisibleRange(selectStart,selectEnd);
          break;
        default:
          break;
      } 
      setSelectRegion();
      break;
    case Mouse_MoveMark:
      handleMoveMark();
      break;
    default:
      break;
  }
  mouseState=No_Action;
}


void WaveView::mouseMoveEvent(QMouseEvent *event) {
  mouseX=event->x();
  mouseY=event->y();
  handleMouseAction();
}

void WaveView::wheelEvent(QWheelEvent *event) {
  if (event->delta()>0) {
    zoomOut(event->delta()/10);    
  } else {
    zoomIn(-(event->delta()/10));
  }
}

void WaveView::handleMouseAction(void) {
  if (mouseState!=Mouse_MoveMark) {
    if ( (mouseX>(pressX+3)) || (mouseX<(pressX-3))) {
      mouseState=Mouse_Dragging;
    } else {
      mouseState=Mouse_Clicking;
      setSelectRegion();
    }
  }

  if ( (mouseState==Mouse_Dragging) && (mouseX>reservedLeft) && (mouseX<(realWidth+reservedLeft)) ) {
    if (mouseX>pressX) {
      setSelectRegion(sampleX(pressX),sampleX(mouseX));
    } else {
      setSelectRegion(sampleX(mouseX),sampleX(pressX));
    }
  }

  if ( mouseState==Mouse_MoveMark)  { 
    handleMoveMark();
  }
}

void WaveView::handleMoveMark() {
  long int sampleMouse;

  sampleMouse=sampleX(mouseX);
  switch (mouseMove) {
    case StartMark:
      if (sampleMouse!=startMarkSample) {
        setStartMark(sampleMouse);        
        emit startChanged(startMarkSample);
      }
      break;
    case StopMark:
      if (sampleMouse!=endMarkSample) {
        setEndMark(sampleMouse);
        emit endChanged(endMarkSample);
      }
      break;
    case PlayMark:
      if (sampleMouse!=playMarkSample) {
        setPlayMark(sampleMouse);
        emit playChanged(playMarkSample);
      }
      break;
  }
}

void WaveView::setFont(const QFont &fnt) {
  QWidget::setFont(fnt);
  updateGeometry();
}
