/*
	miscutils.h - Misc functions
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MISCUTILS_H__
#define __MISCUTILS_H__

#include <qdatetime.h>

#include <kprocess.h>

/** @short Misc functions. */
class MiscUtils
{
public:
	/**
	 * Closes the CD-ROM tray (if this option is enabled).
	 */
	static void closeCDTray();

	/**
	 * Displays a custom message.
	 * @param text A text to display
	 * @param testCommand A command to execute (used internally in settings dialog to test entered command)
	 */
	static void customMessage(const QString &text, const QString &testCommand = QString::null);

	/**
	 * Converts seconds to time string.
	 * @param secs A number of seconds to convert
	 * @param format A date/time format
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static QString formatDateTime(const int secs, const QString &format = QString::null);

	/**
	 * Converts seconds to time string.
	 * @param dt A date/time to convert
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static QString formatDateTime(const QDateTime &dt);

	/**
	 * Returns @c true if function (@p key) is restricted (KDE Kiosk).
	 * The key value is read from the configuration file
	 * (usually /usr/share/config/kshutdownrc).
	 * See Handbook (F1) for details.
	 */
	static bool isRestricted(const QString &key);

	/**
	 * Visual and sound notifications.
	 * @param secs A number of seconds to action
	 */
	static void notifyUser(const int secs);

	/**
	 * Sets fixed/minimum size.
	 * @param w A widget
	 */
	static void pack(QWidget *w);

	/**
	 * A sound or visual notification.
	 * @param name An event name
	 * @param text A notification text (visible in the passive message)
	 * @code
	 * playSound("kshutdown-one", "1 second before action");
	 * @endcode
	 */
	static void playSound(const QString &name, const QString &text);

	/**
	 * Executes a command.
	 * @param command A command to execute
	 * @return @c true if successfull; otherwise @c false
	 */
	static bool runCommand(const QString &command);

	/**
	 * Executes a shell command before the specified action.
	 * @param configEntry A configuration entry (e.g. <i>"Shut Down"</i>)
	 */
	static void runCommandBeforeAction(const QString &configEntry);

	/**
	 * Executes a shell command.
	 * @param command A shell command to execute
	 * @param mode A run mode (@c KProcess::DontCare by default)
	 * @param pause A timeout in seconds (no timeout by default)
	 * @return @c true if successfull; otherwise @c false
	 */
	static bool runShellCommand(const QString &command, const KProcess::RunMode mode = KProcess::DontCare, const int pause = -1);

	/**
	 * Sets "tool tip text" and "what's this text" of @p widget to @p value.
	 */
	static void setHint(QWidget *widget, const QString &value);

	/**
	 * Displays a "KRun" error message in a passive message.
	 * @param command A command that failed
	 */
	static void showRunErrorMessage(const QString &command);

	/**
	 * Displays an information if KShutDown is in the <b>Test Mode</b>.
	 * @param message A message to display
	 */
	static void showTestMessage(const QString &message);
};

#endif // __MISCUTILS_H__
