#ifndef _KVI_FRAME_H_INCLUDED_
#define _KVI_FRAME_H_INCLUDED_

// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qptrlist.h>

#include "kvi_dns.h"
#include "kvi_irc_socket.h"
#include "kvi_mainwindow.h"
#include "kvi_settings.h"
#include "kvi_string.h"

class QRect;
class QSplitter;

class KviAccel;
class KviChannel;
class KviConfig;
class KviConsole;
class KviDccChat;
class KviDccManager;
class KviDccSend;
class KviDccVoice;
class KviDirectoryBrowser;
class KviDirectoryBrowserWidget;
class KviHelpWindow;
class KviIrcSocket;
class KviIrcUserList;
class KviIrcView;
class KviLinksWindow;
class KviListWindow;
class KviMainMenu;
class KviMainToolBar;
class KviMdiButton;
class KviMdiManager;
class KviMdiToolBar;
class KviPopupMenu;
class KviQuery;
class KviServerParser;
class KviStatusBar;
class KviSysTrayBar;
class KviSysTrayIoLed;
class KviSysTrayOnLineTimer;
class KviTaskBar;
class KviTaskBarButton;
class KviToolBar;
class KviUserParser;
class KviUserToolBar;
class KviUserWindow;
class KviWindow;

typedef struct KviGlobalStuff
{
	KviStr            szCurrentNick;
	KviStr            szCurrentUserName;
	KviStr            szCurrentRealName;
	KviStr            szCurrentServerHost;
	KviStr            szCurrentServerPort;
	KviIpAddresses    szCurrentServerAddresses;
	KviStr            szCurrentServerIpAddress;
	KviStr            szCurrentNetwork;
	KviStr            szCurrentPassword;
	KviStr            szCurrentMaskFromServer;
	KviStr            szLocalHostIp;            // Looked up at connection time
	KviStr            szLocalHostName;          // Received from server in our mask ASAP
	KviStr            szCurrentProxyHost;
	KviIpAddresses    szCurrentProxyAddresses;
	KviStr            szCurrentProxyIpAddress;
	KviStr            szCurrentProxyPort;
	KviStr            szCurrentProxyPassword;
	KviStr            szCurrentProxyUsername;
	KviStr            szConnectServerHost;
	KviStr            szConnectServerPort;
	KviStr            szConnectServerIp;
	KviStr            szConnectNetwork;
	KviStr            szConnectPassword;
	KviStr            szConnectProxyHost;
	KviStr            szConnectProxyIp;
	KviStr            szConnectProxyPort;
	KviStr            szConnectProxyPassword;
	KviStr            szConnectProxyUsername;
	KviStr            szNotifyListString;

	bool              bServerSupportsWatchList;
	bool              bServerSupportsEMode;       // True if the server supports MODE <channe> e BanException list
	bool              bUserHostInProgress;
	bool              bSentQuit;

	KviStr            szCurrentUserMode;

	QString           szCurrentSelectionText;
	QString           szCurrentSelectionToken;
	KviWindow        *pCurrentSelectionWindow;

	KviWindow        *pCurrentActiveWindow;
	QPtrList<KviStr> *pDnsWhoisPending;
	QPtrList<KviStr> *pMultimediaFileOffers;

#ifdef COMPILE_NEED_IPV6
	bool              bIPv6Mode;
	bool              bCIPv6Mode;
#endif
};

class KviFrame : public KviMainWindow
{
	friend class KviApp;
	friend class KviWindow;
	friend class KviTaskBar;
	friend class KviIrcSocket;
	friend class KviIrcView;
	friend class KviServerParser;
	friend class KviUserParser;
	friend class KviConsole;
	Q_OBJECT
public:
	KviFrame();
	~KviFrame();
public:
	KviMainToolBar            *m_pToolBar;
	KviMdiToolBar             *m_pMdiToolBar;
	KviMainMenu               *m_pMainMenu;
	KviStatusBar              *m_pStatusBar;
	KviTaskBar                *m_pTaskBar;
	KviSysTrayBar             *m_pSysTrayBar;
	QSplitter                 *m_pSplitter;     // Central widget
	KviMdiManager             *m_pMdi;          // MDI manager: child of the splitter
	QPtrList<KviWindow>       *m_pWinList;      // Main list of windows
	KviConsole                *m_pConsole;      // Console window (always there)
	KviIrcSocket              *m_pSocket;       // IRC socket
	KviDccManager             *m_pDccManager;
	bool                       m_bInSDIMode;

	enum KviIrcConnectionState { Ready, WaitingForDns, WaitingForConnect, LoggingIn, Connected, WaitingForReconnect };
	KviIrcConnectionState      m_state;
	KviDns                    *m_pDns;    // DNS resolver... usually 0, if not resolving any host

	KviGlobalStuff             m_global;
	KviServerParser           *m_pServerParser;
	KviUserParser             *m_pUserParser;
	KviIrcUserList            *m_pUserList;
	int                        m_iUserListUpdateTimer;
	int                        m_iNotifyListUpdateTimer;
	KviPopupMenu              *m_pRecentServersPopup;
	KviPopupMenu              *m_pRecentChannelsPopup;
	KviPopupMenu              *m_pRecentNicknamesPopup;
	QTimer                    *m_pReconnectTimer;
	QTimer                    *m_pMultimediaFileOfferTimer;
	KviUserToolBar            *m_pUserToolBar;
	KviSysTrayIoLed           *m_pSysTrayIoLed;
	KviSysTrayOnLineTimer     *m_pSysTrayOnLineTimer;
	KviMdiButton              *m_pSDICloseButton;
	KviMdiButton              *m_pSDIRestoreButton;
	KviMdiButton              *m_pSDIMinimizeButton;
	KviDirectoryBrowserWidget *m_pDirBrowser;
	KviHelpWindow             *m_pHelpBrowser;
	KviAccel                  *m_pAccel;
	QPtrList<KviStr>          *m_pChannelsBeforeDisconnect;
protected:
	bool                       m_userIdle;
	QTimer                    *m_pIdleTimer;
public:
	bool userIdle() const;
	void stopIdleTimer();
	void restartIdleTimer();
	void textInput(const QString *);      // KviInput uses this

	// Window management
	bool dirBrowserOpen();
	void    addWindow(KviWindow *pWnd, bool bShow = true);
	void attachWindow(KviWindow *pWnd, bool bShow = true, bool overrideGeometry = true, QRect *r = 0);
	void detachWindow(KviWindow *pWnd);
	void closeWindow(KviWindow *pWnd);

	KviWindow  *activeWindow();
	KviPopupMenu *taskBarPopup(KviWindow *pWnd, bool bIncludeWindowPopup = false);
	KviPopupMenu *windowPopup(KviIrcView *pView, KviWindow *pWnd, bool bIncludeTaskbarPopup = true);

	void forceUpdateToAllMdiChildren();
	void applyMdiOptions();
	void applyOptions();

	KviWindow      *findWindow(const char *caption);
	KviChannel     *findChannel(const char *chanName);
	KviLinksWindow *getLinksWindow();
	KviListWindow  *getListWindow();
	KviQuery       *findQuery(const char *queryName);
	KviUserWindow  *findUserWindow(const char *name);
	KviDccSend     *findWaitingDccSend(unsigned short uPort, const char *nick);
	KviDccSend     *findListeningDccSend(unsigned short uPort, const char *nick);

	bool windowExists(KviWindow *pWnd);
	void setActiveWindow(KviWindow *pWnd);

	KviChannel          *createChannel(const char *name);
	KviQuery            *createQuery(const char *name, bool isRemote);
	KviQuery            *raiseOrCreateQuery(const char *name);
	KviLinksWindow      *createLinksWindow();
	KviListWindow       *createListWindow();
	KviDccChat          *createDccChat(const char *name);
	KviDccSend          *createDccSend(const char *name);
	KviDccVoice         *createDccVoice(const char *name);
	KviUserWindow       *createUserWindow(const char *name);
	KviDirectoryBrowser *createDirectoryBrowser(const char *dirname);

	void userModeChange(const char *umode);
	void outputPrivmsg(
		KviWindow *wnd, int type, const char *talker, const char *talker_mask, const char *msg, bool isPrivmsg = true
	);
	void updateCaption();
	void startUpdatingUserList();
	void stopUpdatingUserList();
	bool isUpdatingUserList();
	void doAsyncServerCommandCall();
	void startWatchList();
	void startNotifyList();
	void startNotifyOrWatchList();
	void stopNotifyList();
	void stopWatchList();
	void stopNotifyOrWatchList();
	void updateNotifyOrWatchList();
	void clearConsoleListBox();
	void switchWindows(bool bRight);

	void fillRecentServersPopup();
	void fillRecentNicknamesPopup();
	void fillRecentChannelsPopup();
	void fillRecentPopup(KviPopupMenu *popup, KviStr &data);

	void addRecentServer(KviStr &serverName, KviStr &port);
	void addRecentChannel(const char *chan);
	void addRecentNickname(KviStr &nickname);
	bool addRecentEntry(KviStr &list, KviStr &entry);

	void actionForCurrentlySelectedTextToken(const char *action);
	void updateUserToolBar();
	void addMultimediaFileOffer(const KviStr &offer);
	void findMultimediaFileOffer(KviStr &retPath, KviStr &filename);

	bool isLoggingIn();
	bool isConnected();
	KviIrcConnectionState state();

	void helpForObject(QObject *);
	void requestHelpOn(const char *topic);
	void requestHelpSearchOn(const char *topic);

public slots:
	void closeDirBrowser();
	void closeHelpBrowser();

	void slot_connectToServer();
	void slot_toggleStatusBar();
	void slot_toggleTaskBar();
	void slot_toggleMdiToolBar();
	void slot_toggleToolBar();
	void slot_toggleSysTrayBar();
	void slot_requestLinks();
	void slot_whatIsThisRequest();
	void slot_showScriptCenter();
	void slot_toggleUserToolBar();
	void slot_lineUpToolBarsNormal();
	void slot_lineUpToolBarsCompact();

	void connectToRecentServer(int id);
	void joinRecentChannel(int id);
	void setRecentNickname(int id);

	// DNS slot, server name resolved
	void serverNameResolved(KviDnsData *);
	void  proxyNameResolved(KviDnsData *);

	void copyCurrentSelectionTextToClipboard();
	void doHelpOnCurrentlySelectedTextToken();
	void doHelpSearchOnCurrentlySelectedTextToken();

	void whoisCurrentlySelectedToken();
	void queryCurrentlySelectedToken();
	void hostCurrentlySelectedToken();
	void dnsCurrentlySelectedToken();
	void joinCurrentlySelectedToken();
	void topicCurrentlySelectedToken();
	void dccChatCurrentlySelectedToken();
	void execCurrentlySelectedText();

	void activateListWindow();
	void partChannel(KviWindow *);
	void asyncServerCommand();
	void reconnectToServer();
	void cleanupMultimediaFileOfferList();
	void mdiManagerSDIModeChange(bool bEntering);
	void toggleDirBrowser();
	void toggleHelpBrowser();
protected:
	bool isValidStringIp(const char *szIp);
	void applyToolbarOptions();
	bool checkHighlight(const char *msg);
	void setState(KviIrcConnectionState state);
	void haveServerIp();
	void haveProxyIp();
	int  tryServerConnect(KviProxyStruct *);
	void connectToServer();
	void disconnectFromServer();
	void connectionAttemptFailed(int errNum);
	void abortDns();
	void abortConnection();
	void abortReconnect();
	void resolveLocalHost();
	void getArtificialNickname(KviStr &nickToChange);
	bool findCurrentServerToContact();
	bool findNextServerToContact();
	bool findCurrentProxyHost();
	void lookupServerHostname();
	void lookupProxyHostname();
	void handleConnectedToIrcServer();
	void handleDisconnectedFromServer();
	void loginToIrcServer();
	void loginNickRefused();
	void ircLoginSuccessful(KviStr &szAcceptedNickname);

	// Socket events
	void socketEvent(int sockEventId);
	virtual void   closeEvent(QCloseEvent *);
	virtual void focusInEvent(QFocusEvent *);
	virtual void   timerEvent(QTimerEvent *);
	virtual bool focusNextPrevChild(bool next);
	void resetGlobals();
	void resetProxyGlobals();
	void childWindowCloseRequest(KviWindow *pWnd);
	void taskbarButtonLeftClicked(KviWindow *pWnd, KviTaskBarButton *btn);
	void taskbarButtonRightClicked(KviWindow *pWnd, KviTaskBarButton *btn);
	void ircViewTextSelected(KviIrcView *pView, KviWindow *pWnd, const char *text);
	void windowPopupRequested(KviIrcView *pView, KviWindow *pWnd);
	void resizeEvent(QResizeEvent *);
	void adjustSDIButtons();

	// Creation
	void createSplitter();
	void createConsole();
	void createMainMenu();
	void createToolBar();
	void createMdiToolBar();
	void createTaskBar();
	void createStatusBar();
	void createMdiManager();
	void createSDIModeButtons();
	void createUserToolBar();
	void createSysTrayBar();

	void saveToolBarPositions();
	void restoreToolBarPositions();

	void saveToolBarState(KviToolBar *, KviConfig *);
	void restoreToolBarState(KviToolBar *, KviConfig *, Dock defDock, int defIndex, bool bDefNewline, int defExtra);

	KviAccel *installAccelerators(QWidget *);

protected slots:
	void accelActivated(int id);
	void triggerOnIdleEvent();
};

#endif // _KVI_FRAME_H_INCLUDED_
