/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qglobal.h>
#include <qdir.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qspinbox.h>
#include <qclipboard.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kstdaction.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kglobalsettings.h>
#include <kprinter.h>
#include <kstatusbar.h>
#include <klineeditdlg.h>
#include "kxstitch.h"
#include "kxstitchview.h"
#include "kxstitchdoc.h"
#include "editview.h"
#include "flosspalette.h"
#include "paletteview.h"
#include "flossscheme.h"
#include "preview.h"
#include "patterncanvas.h"
#include "printdlgpage.h"
#include "configuration.h"
#include "configurationdialog.h"
#include "calibratedialog.h"

#define ID_STATUS_MSG 1
#define ID_LOCATION 2
#define ID_SCALE_FORMAT 3
#define ID_STITCH_MASK 4
#define ID_COLOR_MASK 5
#define ID_BACKSTITCH_MASK 6
#define ID_KNOTS_MASK 7

KXStitchApplication::KXStitchApplication()
  : KApplication(),
    SchemeManager()
{
}

KXStitchApp::KXStitchApp(QWidget* , const char* name)
  : KMainWindow(0, name)
{
  m_config=kapp->config();

  initStatusBar();
  initActions();
  readOptions();
  initDocument();
  initView();

  connect(m_view->m_editor,SIGNAL(locationOrSize(QPoint)),this,SLOT(slotLocationOrSize(QPoint)));
  connect(m_view->m_editor,SIGNAL(scaleFormat(QString)),this,SLOT(slotScaleFormat(QString)));
  connect(statusBar(),SIGNAL(pressed(int)),this,SLOT(slotStatusBarClicked(int)));
  connect(m_view->m_editor,SIGNAL(stitchMask(bool)),this,SLOT(slotStitchMask(bool)));
  connect(m_view->m_editor,SIGNAL(colorMask(bool)),this,SLOT(slotColorMask(bool)));
  connect(m_view->m_editor,SIGNAL(excludeBackstitches(bool)),this,SLOT(slotExcludeBackstitches(bool)));
  connect(m_view->m_editor,SIGNAL(excludeKnots(bool)),this,SLOT(slotExcludeKnots(bool)));
  connect(m_view->m_palette,SIGNAL(needEditViewReset()),m_view->m_editor->viewport(),SLOT(repaint()));

  m_view->configure();

  createGUI();
}

KXStitchApp::~KXStitchApp()
{
  // nothing to do here
}

void KXStitchApp::initActions()
{
//  KIconLoader *loader = KGlobal::iconLoader();

  KStdAction::openNew(this, SLOT(slotFileNew()), actionCollection());
  KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());
  m_fileOpenRecent = KStdAction::openRecent(this, SLOT(slotFileOpenRecent(const KURL&)), actionCollection());
  KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
  KStdAction::saveAs(this, SLOT(slotFileSaveAs()), actionCollection());
  KStdAction::close(this, SLOT(slotFileClose()), actionCollection());
  KStdAction::print(this, SLOT(slotFilePrint()), actionCollection());
  new KAction(i18n("Insert Background Image..."),0,this,SLOT(slotBackgroundImage()),actionCollection(),"background_image");
  new KAction(i18n("Clear Background Image"),0,this,SLOT(slotClearBackgroundImage()),actionCollection(),"clear_background");
  actionCollection()->action("clear_background")->setEnabled(false);
  new KAction(i18n("File Properties..."), 0, this, SLOT(slotFileProperties()), actionCollection(), "file_properties");
  new KAction(i18n("Import Image File..."), 0, this, SLOT(slotFileImport()), actionCollection(), "file_import");
  new KAction(i18n("Scan Image..."), 0, this, SLOT(slotFileScan()), actionCollection(), "file_scan");
  KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
  connect(kapp, SIGNAL(lastWindowClosed()), kapp, SLOT(quit()));
//  m_viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
  KMainWindow::setStandardToolBarMenuEnabled(true);
//  m_viewStitchBar = new KToggleAction(i18n("Show Stitch Toolbar"),0,this,SLOT(slotViewStitchBar()),actionCollection(),"viewstitchbar");
//  m_viewToolToolBar = new KToggleAction(i18n("Show Tool Toolbar"),0,this,SLOT(slotViewToolToolBar()),actionCollection(),"viewtooltoolbar");
  m_viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());

  new KAction(i18n("Calibrate colors..."), 0, this, SLOT(slotCalibrate()),actionCollection(),"calibrate");
  new KAction(i18n("Configure KXStitch..."),0,this,SLOT(slotSettingsConfigure()),actionCollection(),"configure");
}

void KXStitchApp::slotCalibrate()
{
  CalibrateDialog *dlg = new CalibrateDialog(this);
  if (dlg->exec() == QDialog::Accepted)
  {
    // request the document to refresh its colors based on the calibrated ones
    m_doc->renderPixmap();
    m_view->resetView();
  }
}

void KXStitchApp::initStatusBar()
{
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
  statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft);
  statusBar()->insertFixedItem("",ID_LOCATION,true);
  statusBar()->setItemFixed(ID_LOCATION,70);
  statusBar()->insertFixedItem("CM",ID_SCALE_FORMAT,true); // largest string
  statusBar()->changeItem("ST",ID_SCALE_FORMAT);
  statusBar()->insertFixedItem("S",ID_STITCH_MASK,true);
  statusBar()->changeItem("",ID_STITCH_MASK);
  statusBar()->insertFixedItem("C",ID_COLOR_MASK,true);
  statusBar()->changeItem("",ID_COLOR_MASK);
  statusBar()->insertFixedItem("B",ID_BACKSTITCH_MASK,true);
  statusBar()->changeItem("",ID_BACKSTITCH_MASK);
  statusBar()->insertFixedItem("K",ID_KNOTS_MASK,true);
  statusBar()->changeItem("",ID_KNOTS_MASK);
}

void KXStitchApp::initDocument()
{
  m_doc = new KXStitchDoc(this);
  m_doc->newDocument();
}

void KXStitchApp::initView()
{
  m_view = new KXStitchView(this);
  m_doc->addView(m_view);
  setCentralWidget(m_view);
  setCaption(m_doc->URL().fileName(),false);
}

void KXStitchApp::openDocumentFile(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));

  m_doc->openDocument(url);
  if (!url.isEmpty())
    m_fileOpenRecent->addURL( url );
  slotClearBackgroundImage();
  m_view->resetView();
  setCaption(m_doc->URL().fileName(),false);
  slotStatusMsg(i18n("Ready."));
}

KXStitchDoc *KXStitchApp::getDocument() const
{
  return m_doc;
}

void KXStitchApp::saveOptions()
{
  m_config->setGroup("General Options");
  m_config->writeEntry("Geometry", size());
  m_config->writeEntry("ParentSplitter", m_view->sizes());
  QSplitter *splitter = (QSplitter *)(m_view->child("ChildSplitter",0));
  m_config->writeEntry("ChildSplitter", splitter->sizes());
//  m_config->writeEntry("Show Toolbar", m_viewToolBar->isChecked());
//  m_config->writeEntry("Show StitchBar", m_viewStitchBar->isChecked());
//  m_config->writeEntry("Show ToolToolBar", m_viewToolToolBar->isChecked());
  m_config->writeEntry("Show Statusbar",m_viewStatusBar->isChecked());
//  m_config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
//  m_config->writeEntry("StitchToolBarPos", (int) toolBar("StitchToolBar")->barPos());
//  m_config->writeEntry("ToolToolBarPos", (int) toolBar("ToolsToolbar")->barPos());
  m_fileOpenRecent->saveEntries(m_config,"Recent Files");
}

void KXStitchApp::readOptions()
{
  m_config->setGroup("General Options");

  // bar status settings
//  bool bViewToolbar = m_config->readBoolEntry("Show Toolbar", true);
//  m_viewToolBar->setChecked(bViewToolbar);
//  slotViewToolBar();

//  bool bViewStitchBar = m_config->readBoolEntry("Show StitchBar", true);
//  m_viewStitchBar->setChecked(bViewStitchBar);
//  slotViewStitchBar();

//  bool bViewToolToolBar = m_config->readBoolEntry("Show ToolToolBar", true);
//  m_viewToolToolBar->setChecked(bViewToolToolBar);
//  slotViewToolToolBar();

  bool bViewStatusbar = m_config->readBoolEntry("Show Statusbar", true);
  m_viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();

  // bar position settings
//  KToolBar::BarPosition toolBarPos;
//  toolBarPos=(KToolBar::BarPosition) m_config->readNumEntry("ToolBarPos", KToolBar::Top);
//  toolBar("mainToolBar")->setBarPos(toolBarPos);
//  toolBarPos=(KToolBar::BarPosition) m_config->readNumEntry("StitchToolBarPos", KToolBar::Top);
//  toolBar("StitchToolbar")->setBarPos(toolBarPos);
//  toolBarPos=(KToolBar::BarPosition) m_config->readNumEntry("ToolToolBarPos", KToolBar::Top);
//  toolBar("ToolsToolbar")->setBarPos(toolBarPos);

  // initialize the recent file list
  m_fileOpenRecent->loadEntries(m_config,"Recent Files");
  QSize size=m_config->readSizeEntry("Geometry");
  if(size.isEmpty())
  {
    size = kapp->desktop()->size();
    size*=2;
    size/=3;
  }
  resize(size);
}

void KXStitchApp::saveProperties(KConfig *_cfg)
{
  if(m_doc->URL().fileName()!=i18n("Untitled") && !m_doc->isModified()){
    // saving to tempfile not necessary
  }
  else
  {
    KURL url=m_doc->URL();
    _cfg->writeEntry("filename", url.url());
    _cfg->writeEntry("modified", m_doc->isModified());
    QString tempname = kapp->tempSaveName(url.url());
    QString tempurl= KURL::encode_string(tempname);
    KURL _url(tempurl);
    m_doc->saveDocument(_url);
  }
}


void KXStitchApp::readProperties(KConfig* _cfg){
  QString filename = _cfg->readEntry("filename", "");
  KURL url(filename);
  bool modified = _cfg->readBoolEntry("modified", false);
  if(modified)
  {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);
    KURL _url(tempname);

    if(canRecover)
    {
      m_doc->openDocument(_url);
      m_doc->setModified();
      setCaption(m_doc->URL().fileName(),true);
      QFile::remove(tempname);
    }
  }
  else
  {
    if(!filename.isEmpty())
    {
      m_doc->openDocument(url);
      setCaption(m_doc->URL().fileName(),false);
    }
  }
}

bool KXStitchApp::queryClose()
{
  return m_doc->saveModified();
}

bool KXStitchApp::queryExit()
{
  saveOptions();
  return true;
}

void KXStitchApp::repaint()
{
  m_view->m_palette->repaint();
  m_view->m_preview->viewport()->repaint(false);
  m_view->m_editor->viewport()->repaint();
}

void KXStitchApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));

  KXStitchApp *new_window= new KXStitchApp();
  new_window->show();

  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating new document..."));

  if(!m_doc->saveModified())
  {
    // here saving wasn't successful
  }
  else
  {
    if (m_doc->newDocument())
      setCaption(m_doc->URL().fileName(), false);
  }
  slotClearBackgroundImage();
  m_view->resetView();
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotBackgroundImage()
{
  KURL url;

  url = KFileDialog::getImageOpenURL(QString::null,this,i18n("Background Image"));
  if (!url.path().isNull())
  {
    QImage image;
    image.load(url.path());
    m_view->m_editor->setBackgroundImage(image);
    actionCollection()->action("clear_background")->setEnabled(true);
  }
}

void KXStitchApp::slotClearBackgroundImage()
{
  m_view->m_editor->clearBackgroundImage();
  actionCollection()->action("clear_background")->setEnabled(false);
}

void KXStitchApp::slotFileProperties()
{
  m_doc->fileProperties();
  m_view->resetView();
}

void KXStitchApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));

  if(!m_doc->saveModified())
  {
    // here saving wasn't successful
  }
  else
  {
    KURL url=KFileDialog::getOpenURL(QString("::%1").arg(KGlobalSettings::documentPath()),
        i18n("*|All files"), this, i18n("Open File..."));
    if(!url.isEmpty())
    {
      m_doc->openDocument(url);
      m_doc->setURL(url);
      setCaption(m_doc->URL().fileName(), false);
      m_fileOpenRecent->addURL( url );
    }
  }
  slotClearBackgroundImage();
  m_view->resetView();
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileOpenRecent(const KURL& url)
{
  slotStatusMsg(i18n("Opening file..."));

  if(!m_doc->saveModified())
  {
    // here saving wasn't successful
  }
  else
  {
    m_doc->openDocument(url);
    m_doc->setURL(url);
    setCaption(m_doc->URL().fileName(), false);
  }
  slotClearBackgroundImage();
  m_view->resetView();
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileSave()
{
  slotStatusMsg(i18n("Saving file..."));

  if (m_doc->URL().fileName() == i18n("Untitled"))
    slotFileSaveAs();
  else
    m_doc->saveDocument(m_doc->URL());

  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileSaveAs()
{
  slotStatusMsg(i18n("Saving file with a new filename..."));

  KURL url=KFileDialog::getSaveURL(QString("::%1").arg(KGlobalSettings::documentPath()),
        i18n("*|All files"), this, i18n("Save as..."));
  if(!url.isEmpty())
  {
    m_doc->saveDocument(url);
    m_doc->setURL(url);
    m_fileOpenRecent->addURL(url);
    setCaption(m_doc->URL().fileName(),m_doc->isModified());
  }
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));

  close();

  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));

  KPrinter printer(true);
  PrintDlgPage *dlgPage = new PrintDlgPage(&printer, QSize(m_doc->canvas()->patternWidth(),m_doc->canvas()->patternHeight()));
  printer.addDialogPage(dlgPage);

  if (printer.setup(this))
  {
    m_view->print(&printer);
  }

  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileImport()
{
  slotStatusMsg(i18n("Importing image..."));

  if(!m_doc->saveModified())
  {
    // here saving wasn't successful
  }
  else
  {
    slotStatusMsg(i18n("Importing..."));
    m_doc->importImage();
  }
  setCaption(m_doc->URL().fileName(), false);
  slotClearBackgroundImage();
  m_view->resetView();
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileScan()
{
  slotStatusMsg(i18n("Scanning image..."));

  if (!m_doc->saveModified())
  {
    // saving not successful
  }
  else
  {
    m_doc->scanImage();
  }
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  saveOptions();
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  kapp->closeAllWindows();
}

void KXStitchApp::slotViewToolBar()
{
/*
  slotStatusMsg(i18n("Toggling toolbar..."));
  if(!m_viewToolBar->isChecked())
  {
    toolBar("mainToolBar")->hide();
  }
  else
  {
    toolBar("mainToolBar")->show();
  }
  slotStatusMsg(i18n("Ready."));
*/
}

void KXStitchApp::slotViewStitchBar()
{
  slotStatusMsg(i18n("Toggling Stitch toolbar..."));
  if(!m_viewStitchBar->isChecked())
  {
    toolBar("StitchToolBar")->hide();
  }
  else
  {
    toolBar("StitchToolBar")->show();
  }
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotViewToolToolBar()
{
  slotStatusMsg(i18n("Toggling tool toolbar..."));
  if(!m_viewToolToolBar->isChecked())
  {
    toolBar("ToolsToolbar")->hide();
  }
  else
  {
    toolBar("ToolsToolbar")->show();
  }
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  if(!m_viewStatusBar->isChecked())
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }
  slotStatusMsg(i18n("Ready."));
}

void KXStitchApp::slotStatusMsg(const QString &text)
{
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}

void KXStitchApp::slotLocationOrSize(QPoint location)
{
  if (location != QPoint(-1,-1))
    statusBar()->changeItem(QString("%1,%2").arg(location.x()).arg(location.y()),ID_LOCATION);
  else
    statusBar()->changeItem("",ID_LOCATION);
}

void KXStitchApp::slotScaleFormat(QString format)
{
  statusBar()->changeItem(format,ID_SCALE_FORMAT);
}

void KXStitchApp::slotStitchMask(bool enable)
{
  statusBar()->changeItem(enable?"S":"",ID_STITCH_MASK);
}

void KXStitchApp::slotColorMask(bool enable)
{
  statusBar()->changeItem(enable?"C":"",ID_COLOR_MASK);
}

void KXStitchApp::slotExcludeBackstitches(bool enable)
{
  statusBar()->changeItem(enable?"B":"",ID_BACKSTITCH_MASK);
}

void KXStitchApp::slotExcludeKnots(bool enable)
{
  statusBar()->changeItem(enable?"K":"",ID_KNOTS_MASK);
}

void KXStitchApp::slotStatusBarClicked(int item)
{
  if (item == ID_COLOR_MASK)
    actionCollection()->action("ColorMask")->activate();
  if (item == ID_STITCH_MASK)
    actionCollection()->action("StitchMask")->activate();
  if (item == ID_BACKSTITCH_MASK)
    actionCollection()->action("ExcludeBackstitches")->activate();
  if (item == ID_KNOTS_MASK)
    actionCollection()->action("ExcludeKnots")->activate();
}

void KXStitchApp::slotSettingsConfigure()
{
  ConfigurationDialog *dialog = new ConfigurationDialog(this);
  if (dialog->exec() == QDialog::Accepted)
    m_view->configure();
}
