// LabPlot : Worksheet.cc

#include <stdlib.h>
#include <iostream>
#include <math.h>

#include <qnamespace.h>
#include <qpushbutton.h>
#include <qstring.h>
#include <qtextcodec.h>
#include <qfiledialog.h>
#include <qpaintdevicemetrics.h>
#include <qcursor.h>
#include <qpicture.h>
#include <qinputdialog.h>
#include <qworkspace.h>
#include <qstringlist.h>
#include <klocale.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kprocess.h>

#include "Worksheet.h"
#include "GraphListDialog.h"
#include "TitleDialog.h"
#include "LegendDialog.h"
#include "AxesDialog.h"
#include "ObjectDialog.h"
#include "PlotSettingsDialog.h"
#include "WorksheetDialog.h"
#include "Plot2DSimple.h"
#include "Plot2DSurface.h"
#include "Plot3D.h"
#include "PlotPie.h"
#include "PlotPolar.h"
#include "PlotTernary.h"
#include "PlotQWT3D.h"

#ifdef HAVE_UNDO
UNDO *wundo;
#endif

using namespace std;

//! main worksheet class. (WNoAutoErase needed for flicker free redrawing)
Worksheet::Worksheet(QWidget *p, MainWin *mw, const char *name)
#if QT_VERSION > 0x030102	// TODO : check this
	:QWidget(p,name,Qt::WNoAutoErase), mw(mw)
#else
	:QWidget(p,name), mw(mw)
#endif
{
	kdDebug()<<"Worksheet()"<<endl;
	widgettype=WWORKSHEET;

	KConfig *config = mw->Config();
	config->setGroup( "Worksheet" );
	QString t = i18n("Worksheet");
	title = config->readEntry("Title",t);

	// check if name already used
	QWidgetList list = mw->getWorkspace()->windowList();
	int sheets = mw->NrWorksheets()+mw->NrSpreadsheets();
	bool found_name=false;
	for(int i=0;i<sheets;i++) {
		if(list.at(i)->caption() == title )
			found_name=true;
	}

	if(found_name)
		title += QString(" ") + QString::number(sheets);
	title_enabled = config->readBoolEntry("TitleEnabled",true);

	setCaption(title);

#if QT_VERSION > 0x030007
	timestamp = QDateTime::currentDateTime(Qt::LocalTime);
#else
	timestamp = QDateTime::currentDateTime();
#endif
	timestamp_enabled = config->readBoolEntry("TimeStampEnabled",true);

	printer = new KPrinter;

	// TODO : hardcoded :-(
	setMinimumSize(300,200);
	int w = config->readNumEntry("Width",600);
	int h = config->readNumEntry("Height",400);
	pixmap = new QPixmap(w,h);
	resize(w,h);

	background = config->readColorEntry("Background", &Qt::white);
	for(int i=0;i<NR_OBJECTS;i++) {
		label[i] = new Label();
		line[i] = new Line();
		rect[i] = new Rect();
		ellipse[i] = new Ellipse();
		image[i] = new LImage();
	}

	X = width(); Y = height();

	setMouseTracking(true);
	
	New();
	
	show();
	
	kdDebug()<<"Worksheet() DONE"<<endl;
}

#ifdef HAVE_UNDO
UNDO *Worksheet::getUndo() { return wundo; }
void Worksheet::setUndo(UNDO *u) { wundo=u; }
#endif

void* Worksheet::operator new(size_t size) {
	kdDebug()<<"NEW WORKSHEET"<<endl;
#ifdef HAVE_UNDO
	wundo = undo_new();
	undo_set_memory_limit(wundo,100000);
	void *p = undo_malloc(wundo, size);
#else
	void *p = malloc(size);
#endif
	return p;
}

void Worksheet::operator delete (void *p) {
#ifdef HAVE_UNDO
	undo_free(wundo,p);
#else
	free(p);
#endif
}

QStringList Worksheet::Info() {
	kdDebug()<<"Worksheet::Info()"<<endl;
	QStringList s;

	s<<title;
	s<<timestamp.toString();
	s<<"API:"<<QString::number(api);
	s<<"NR_PLOTS:"<<QString::number(nr_plots);

	return s;
}

void Worksheet::closeNow() {
	delete this;
	
	mw->updateSheetList();
}

void Worksheet::closeEvent(QCloseEvent *e) {
	kdDebug()<<"Worksheet::closeEvent()"<<endl;
	
	KConfig *config = mw->Config();
	config->setGroup( "General Options" );
	if (config->readBoolEntry("AskCloseSheet",true)) {
		int ok = KMessageBox::warningContinueCancel(this, i18n("Do you really want to close the worksheet ?"));
		if(ok == KMessageBox::Cancel)
			return;
	}

	closeNow();
	e->accept();
}

void Worksheet::New() {
	kdDebug()<<"Worksheet::New()"<<endl;

	for (int i=0;i<NR_PLOTS;i++)
		plot[i]=0;
	api=0;
	nr_plots=0;

	ax=ay=anumber=bx=by=lx=ly=tx=ty=tlx=tly=0;
	labelx=labely=linesx=linesy=lineex=lineey=rectx=recty=0;
	ellipsex=ellipsey=imagex=imagey=0;
	tmagx=tmagy=tmagx2=tmagy2=0;
	tpanx=tpany=0.0;
	tdatamodex=0;
	object_index=0;
	tmp_object_index=0;
	moving_cornerF1 = moving_cornerF2 = false;
	moving_cornerB1 = moving_cornerB2 = false;
	moving_borderx1 = moving_borderx2 = false;
	moving_bordery1 = moving_bordery2 = false;
	moving_center = false;

	updatePixmap();
	kdDebug()<<"Worksheet::New() DONE"<<endl;
}

void Worksheet::deletePlot(int item) {
	kdDebug()<<"Worksheet::deletePlot() : Plot "<<item<<" of "<<nr_plots<<endl;
	if(item>(int)nr_plots)
		return;
	
	// reorder plots
	if (item<(int)nr_plots-1)
		plot[item]=plot[nr_plots-1];
	
	// hack for qwt 3d plot
	if(plot[nr_plots-1]->Type() == PQWT3D)
		plot[nr_plots-1]->draw(0,0,0);
	
	plot[nr_plots-1]=0;
	nr_plots--;
	
	// change api
	api=0;
	
	updatePixmap();
}

void Worksheet::deleteActivePlot() {
	kdDebug()<<"Worksheet::deleteActivePlot() : active plot "<<api<<" of "<<nr_plots<<endl;
	
	// reorder plots
	if (api<nr_plots-1)
		plot[api]=plot[nr_plots-1];

	// hack for qwt 3d plot
	if(plot[nr_plots-1]->Type() == PQWT3D)
		plot[nr_plots-1]->draw(0,0,0);
	
	plot[nr_plots-1]=0;
	nr_plots--;

	// change api
	api=0;

	kdDebug()<<"	NEW api "<<api<<endl;
	kdDebug()<<"	NEW nr_plots "<<nr_plots<<endl;
	
	updatePixmap();
}

Plot *Worksheet::newPlot(PType type, bool insert) {
	kdDebug()<<"Worksheet::newPlot() : type="<<type<<endl;
	Plot *newplot=0;
	switch(type) {
	case P2D:
		newplot = new Plot2DSimple(this);break;
	case PSURFACE:
		newplot = new Plot2DSurface(this);break;
	case P3D:
		newplot = new Plot3D(this);break;
	case PPIE:
		newplot = new PlotPie(this);break;
	case PPOLAR:
		newplot = new PlotPolar(this);break;
	case PTERNARY:
		newplot = new PlotTernary(this);break;
	case PQWT3D:
		newplot = new PlotQWT3D(this);break;
	default: break;
	}
	newplot->setType(type);

	if(insert) {
		api=nr_plots;
		plot[nr_plots]=newplot;
		nr_plots++;
		updatePixmap();
	}
	
	return newplot;
}

Plot* Worksheet::clonePlot(int item,bool insert) {
	Plot *p = getPlot(item);
	Plot *newp = newPlot(p->Type(),insert);
//	Plot *newp = getPlot(NrPlots()-1);
		
	// copy ranges
	newp->setRanges(p->Ranges());
	newp->setActRanges(p->ActRanges());
		
	// TODO : copy all settings
	newp->setRegion(p->Region());
	newp->setRegionEnabled(p->regionEnabled());
	newp->setTransparent(p->Transparent());
	newp->setClipOffset(p->ClipOffset());

	// copy all axes ?
/*	int nr_axes=0;
	switch (p->Type()) {
	case P2D: case PSURFACE : nr_axes=4; break;
	case P3D: nr_axes=12; break;
	case PPOLAR: nr_axes=2; break;
	case PTERNARY: nr_axes=1; break;
	default:break;
	}
	for(int i=0;i<nr_axes;i++) {
		newp->enableBorder(i,p->BorderEnabled(i));
		newp->enableGrid(2*i,p->GridEnabled(2*i));	// major
		newp->enableGrid(2*i+1,p->GridEnabled(2*i+1));	// minor
	}
*/
	newp->setBaselineEnabled(p->BaselineEnabled());
	newp->setXBaselineEnabled(p->XBaselineEnabled());
	newp->setBaseline(p->Baseline());
	newp->setXBaseline(p->XBaseline());
	newp->setMarksEnabled(p->marksEnabled());
	newp->setMarkX(p->markX());
	newp->setMarkY(p->markY());
	newp->setBackground(p->Background());
	newp->setGraphBackground(p->graphBackground());

	// clone all graphs
	GraphList *gl = p->getGraphList();
	GraphList *newgl = newp->getGraphList();
	for(unsigned int item=0;item<gl->Number();item++) {
		// clone label
		Label *l = new Label();
		*l = *(gl->getGraph(item)->getLabel());
		
		switch(gl->getStruct(item)) {
		case GRAPH2D: {
			Graph2D g = *(gl->getGraph2D(item));
			g.setLabel(l);
			newgl->addGraph2D(&g);
			}; break;
		case GRAPH3D: {
			Graph3D g = *(gl->getGraph3D(item));
			g.setLabel(l);
			newgl->addGraph3D(&g);
		      }; break;
		case GRAPH4D: {
			Graph4D g = *(gl->getGraph4D(item));
			g.setLabel(l);
			newgl->addGraph4D(&g);
		      }; break;
		case GRAPHM: {
			GraphM g = *(gl->getGraphM(item));
			g.setLabel(l);
			newgl->addGraphM(&g);
		     }; break;
		case GRAPHIMAGE: {
			GraphIMAGE g = *(gl->getGraphIMAGE(item));
			g.setLabel(l);
			newgl->addGraphIMAGE(&g);
			 }; break;
		case GRAPHL: {
			GraphL g = *(gl->getGraphL(item));
			g.setLabel(l);
			newgl->addGraphL(&g);
			}; break;
		default: break;
		}
	}
	
	if(insert) updatePixmap();
#ifdef HAVE_UNDO
	undo_snapshot(wundo);
#endif

	return newp;
}

void Worksheet::resizeEvent( QResizeEvent *e) {
	kdDebug()<<"Worksheet::resizeEvent()"<<endl;

	X = width(); Y = height();
	this->resize(e->size());
	updatePixmap();
#ifdef HAVE_UNDO
	undo_snapshot(wundo);
#endif
} 

//! update off screen pixmap (double buffering)
void Worksheet::updatePixmap() {
	kdDebug()<<"Worksheet::updatePixmap()"<<endl;
	kdDebug()<<"	nr_plots="<<nr_plots<<endl;
	if(pixmap && pixmap->size() != size())
		pixmap->resize(size());

	QPainter p(pixmap, this);
	Draw(&p,X,Y);

	// defining_maglens
	if(mw->defining_maglens==2 && tmagx2>0 && tmagx2-tmagx>0) {
		p.setBrush(Qt::NoBrush);
		
		p.setPen(Qt::black);
		p.drawRect(tmagx,tmagy,tmagx2-tmagx,tmagy2-tmagy);
		p.setPen(Qt::NoPen);
	}
	
	// defining_mask
	if(mw->defining_mask == 2 && tmaskx2>0 && tmaskx2-tmaskx>0) {
		p.setBrush(Qt::NoBrush);
		
		p.setPen(Qt::black);
		p.drawRect(tmaskx,tmasky,(tmaskx2-tmaskx),(tmasky2-tmasky));
		p.setPen(Qt::NoPen);
	}

	// draw plot numbers and rects
	for (unsigned int i=0;i<nr_plots;i++) {
		if(plot[i]==0) continue;
		int x = (int)(plot[i]->Position().X()*X)+16*i;
		int y = (int)(plot[i]->Position().Y()*Y);
		p.setPen(Qt::black);
		p.drawRect(x+5,y+5,15,15);
		if (i==api)
			p.fillRect(x+6,y+6,13,13,Qt::green);
		p.setFont(QFont("Adobe Times",8));
		p.drawText(x+5+4,y+17,QString::number(i));
		p.setPen(Qt::NoPen);
	}

	// datamode
	if(mw->dataMode()) {
		p.setPen(Qt::red);
		int x = tdatamodex, y = 0;
		// TODO : which plot
		if(plot[0]) {
			Point p = plot[0]->dataValue(x/(double)X);
			x = (int) (p.X()*X);
			y = (int) (p.Y()*Y)-1;
		}
		p.drawLine(x-6,y,x-2,y);
		p.drawLine(x+6,y,x+2,y);
		p.drawLine(x,y+6,x,y+2);
		p.drawLine(x,y-6,x,y-2);
		p.setPen(Qt::NoPen);
	}

	update();
	if(mw && nr_plots>0) mw->updateSheetList();	//needed when opening a project file
	kdDebug()<<"Worksheet::updatePixmap() DONE"<<endl;
}

// move cursor in datamode if keyboard-datamode used
void Worksheet::keyPressEvent(QKeyEvent *e) {
	mw->Config()->setGroup( "General Options" );
	if(!mw->Config()->readBoolEntry("KeyboardDataMode",false))
		return;

	int key = e->key();
	bool dm = mw->dataMode();
	//TODO : switch to next point
	switch (key){
	case Qt::Key_Left : if(dm) tdatamodex -= 10 ; break;
	case Qt::Key_Right : if(dm) tdatamodex += 10 ; break;
	default : break;
	}

	updatePixmap();
}

void Worksheet::paintEvent(QPaintEvent *event) {
////	kdDebug()<<"Worksheet::paintEvent()"<<endl;
	QMemArray<QRect> rects = event->region().rects();
	for (int i = 0; i < (int)rects.size(); ++i)
		bitBlt(this, rects[i].topLeft(), pixmap, rects[i]);
}

void Worksheet::mouseDoubleClickEvent(QMouseEvent *e) {
	kdDebug()<<"Worksheet::mouseDoubleClickEvent : plot = "<<api<<endl;
	
	// dont crash if moving over empty worksheet
	if (plot[api]==0)
		return;
	
	int x=e->x(), y=e->y();
	PType type = plot[api]->Type();
	Point pos = plot[api]->Position();
	Point size = plot[api]->Size();
	int xmin=(int)(X*(pos.X()+plot[api]->P1().X()*size.X()));
	int xmax=(int)(X*(pos.X()+plot[api]->P2().X()*size.X()));
	int ymin=(int)(Y*(pos.Y()+plot[api]->P1().Y()*size.Y()));
	int ymax=(int)(Y*(pos.Y()+plot[api]->P2().Y()*size.Y()));
	kdDebug()<<"	xmin/xmax ymin/ymax = "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;

	for (unsigned int i=0;i<nr_plots;i++) {
		if(plot[i]==0) continue;
		int redrectx = (int)(plot[i]->Position().X()*X)+5+16*i;
		int redrecty = (int)(plot[i]->Position().Y()*Y)+5;
		if(x>redrectx && x <redrectx+15 && y>redrecty && y<redrecty+15) {
			api=i;
			if(mw->plotsettings_dialog == 0)
				mw->plotsettings_dialog = new PlotSettingsDialog(mw,"PlotSettingsDialog");
			mw->plotsettings_dialog->show();
			return;
		}
	}

	if (type == P2D || type == PSURFACE || type == PPIE || 
		type == PPOLAR || type == PTERNARY) {
		if (plot[api]->Title()->inside(x,y,pos,size,X,Y)) {
			if(mw->title_dialog == 0)
				mw->title_dialog = new TitleDialog(mw,0);
			mw->title_dialog->show();
			return;
		}
		if (plot[api]->insideLegend(x,y)) {
			if(mw->legend_dialog == 0)
				mw->legend_dialog = new LegendDialog(mw,0);
			mw->legend_dialog->show();
			return;
		}
		for (int i=0;i<NR_OBJECTS;i++) {
			if (label[i]->inside(x,y,pos,size,X,Y)) {
				(new ObjectDialog(mw,0,1,0))->show();
				return;
			}
			if (line[i]->inside(x,y,X,Y)) {
				(new ObjectDialog(mw,0,0,0))->show();
				return;
			}
			if (rect[i]->inside(x,y,X,Y)) {
				(new ObjectDialog(mw,0,2,0))->show();
				return;
			}
			if (ellipse[i]->inside(x,y,X,Y)) {
				(new ObjectDialog(mw,0,3,0))->show();
				return;
			}
			if (image[i]->inside(x,y,X,Y)) {
				(new ObjectDialog(mw,0,4,0))->show();
				return;
			}
		}
		if (x>xmin && x<xmax && y>ymin && y<ymax) {
			(new GraphListDialog(mw,0))->show();
			return;
		}
		else {
			// pie plot has no axes
			if(type == PPOLAR || type == PTERNARY) {
				if(mw->axes_dialog == 0)
					mw->axes_dialog = new AxesDialog(mw,0,0);
				mw->axes_dialog->show();
				mw->updateAxesDialog(0);
				return;	
			}
			else if(type !=PPIE) {
				if(plot[api]->inside(x/(double)X,y/(double)Y)) {
					if(y>ymax) {
						if(mw->axes_dialog == 0)
							mw->axes_dialog = new AxesDialog(mw,0,0);
						mw->axes_dialog->show();
						mw->updateAxesDialog(0);
						return;
					}
					else if(x<xmin) {
						if(mw->axes_dialog == 0)
							mw->axes_dialog = new AxesDialog(mw,0,1);
						mw->axes_dialog->show();
						mw->updateAxesDialog(1);
						return;
					}
					else if(x>xmax) {
						if(mw->axes_dialog == 0)
							mw->axes_dialog = new AxesDialog(mw,0,2);
						mw->axes_dialog->show();
						mw->updateAxesDialog(2);
						return;
					}
					else if(y<ymin) {
						if(mw->axes_dialog == 0)
							mw->axes_dialog = new AxesDialog(mw,0,3);
						mw->axes_dialog->show();
						mw->updateAxesDialog(3);
						return;
					}
				}
			}
			
		}
		// default
		(new WorksheetDialog(mw,0))->show();
	}
	else if (type == P3D) {
		int x1,x2,y1,y2;
		// d = (y1(x2-x3)+y2(x3-x1)+y3(x1-x2))/(y2-y1)
		x1=xmin; x2=(xmax+xmin)/2; y1=ymax; y2=(ymax+ymin)/2;
		int dy = (y1*(x2-x)+y2*(x-x1)+y*(x1-x2))/(y2-y1);
		x1=xmax; x2=(3*xmax-xmin)/2; y1=ymax; y2=(ymax+ymin)/2;
		int dy2 = (y1*(x2-x)+y2*(x-x1)+y*(x1-x2))/(y2-y1);
		x1=xmax; x2=(3*xmax-xmin)/2; y1=ymin; y2=ymin-(ymax-ymin)/2;
		int dy3 = (y1*(x2-x)+y2*(x-x1)+y*(x1-x2))/(y2-y1);
		x1=xmin; x2=(xmax+xmin)/2; y1=ymin; y2=ymin-(ymax-ymin)/2;
		int dy4 = (y1*(x2-x)+y2*(x-x1)+y*(x1-x2))/(y2-y1);

		if (plot[api]->insideLegend(x,y)) {
			if(mw->legend_dialog == 0)
				mw->legend_dialog = new LegendDialog(mw,0);
			mw->legend_dialog->show();
		}
		else if (plot[api]->Title()->inside(x,y,pos,size,X,Y)) {
			if(mw->title_dialog == 0)
				mw->title_dialog = new TitleDialog(mw,0);
			mw->title_dialog->show();
		}
		else if (x>(xmax+xmin)/2 && x<(3*xmax-xmin)/2 && y>=(ymax+ymin)/2 && y<(ymax+ymin)/2+20) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,0);	// x
			mw->axes_dialog->show();
			mw->updateAxesDialog(0);
		}
		else if (x>xmin && x<xmax && y>=ymax && y<ymax+20) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,3);	// x2
			mw->axes_dialog->show();
			mw->updateAxesDialog(3);
		}
		else if (x>xmin && x<xmax && y>ymin-20 && y<=ymin) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,6);	// x3
			mw->axes_dialog->show();
			mw->updateAxesDialog(6);
		}
		else if (x>(xmax+xmin)/2 && x<(3*xmax-xmin)/2 && y>ymin-(ymax-ymin)/2-20 && y<=ymin-(ymax-ymin)/2) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,9);	// x4
			mw->axes_dialog->show();
			mw->updateAxesDialog(9);
		}
		else if (x>(xmin+xmax)/2-20 && x<=(xmin+xmax)/2 && y>ymin-(ymax-ymin)/2 && y<(ymax+ymin)/2) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,2);	// z
			mw->axes_dialog->show();
			mw->updateAxesDialog(2);
		}
		else if (x>xmin-20 && x<=xmin && y>ymin && y<ymax) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,5);	// z2
			mw->axes_dialog->show();
			mw->updateAxesDialog(5);
		}
		else if (x>=xmax && x<xmax+20 && y>ymin && y<ymax) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,8);	// z3
			mw->axes_dialog->show();
			mw->updateAxesDialog(8);
		}
		else if (x>=(3*xmax-xmin)/2 && x<(3*xmax-xmin)/2+20 && y>ymin-(ymax-ymin)/2 && y<(ymax+ymin)/2) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,11);	// z4
			mw->axes_dialog->show();
			mw->updateAxesDialog(11);
		}
		else if (dy<=0 && dy >-20 ) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,1);	// y
			mw->axes_dialog->show();
			mw->updateAxesDialog(1);
		}
		else if (dy2<20 && dy2 >=0 ) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,4);	// y2
			mw->axes_dialog->show();
			mw->updateAxesDialog(4);
		}
		else if (dy3<20 && dy3 >=0 ) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,7);	// y3
			mw->axes_dialog->show();
			mw->updateAxesDialog(7);
		}
		else if (dy4<=0 && dy4 >-20 ) {
			if (mw->axes_dialog == 0)
				mw->axes_dialog = new AxesDialog(mw,0,10);	// y4
			mw->axes_dialog->show();
			mw->updateAxesDialog(10);
		}
		else {
			if (plot[api]->inside(x,y))
				(new GraphListDialog(mw,0))->show();
			else
				(new WorksheetDialog(mw,0))->show();
		}
	}
}

void Worksheet::mousePressEvent(QMouseEvent *e) {
	kdDebug()<<"Worksheet::mousePressEvent()"<<endl;
	int x=e->x(), y=e->y();
	int tmpapi=-1;

	// dont crash if clicking on empty worksheet
	if (plot[api]==0)
		return;

	Point pos = plot[api]->Position();
	Point size = plot[api]->Size();
	LRange *range = plot[api]->ActRanges();
	PType type = plot[api]->Type();
	TScale xscale=LINEAR,yscale=LINEAR;
	
	if(type != PPIE) {
		xscale = plot[api]->getAxis(0)->Scale();
		yscale = plot[api]->getAxis(1)->Scale();
	}

	int xmin=(int)(X*(pos.X()+plot[api]->P1().X()*size.X()));
	int xmax=(int)(X*(pos.X()+plot[api]->P2().X()*size.X()));
	int ymin=(int)(Y*(pos.Y()+plot[api]->P1().Y()*size.Y()));
	int ymax=(int)(Y*(pos.Y()+plot[api]->P2().Y()*size.Y()));

	// if right mouse reset maglens
	if (e->button()==Qt::RightButton) {
		mw->defining_maglens=0;
		mw->defining_mask=0;
		setCursor(Qt::ArrowCursor);
	}
		
	// magnifying lens
	if(mw->defining_maglens == 1) {
		tmagx=x; tmagy=y;
		mw->defining_maglens=2;
	}
	else if(mw->defining_maglens == 2) {		// set actrange of active plot to selected area
		double minx = range[0].rMin(), maxx=range[0].rMax();
		double miny = range[1].rMin(), maxy=range[1].rMax();

		double x1=0, x2=0, y1=0, y2=0;
		switch(xscale) {
		case LINEAR:
			x1 = minx + (tmagx-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = minx + (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			break;
		case LOG10:
			x1 = pow(10,log10(minx)+(double)(tmagx-xmin)/(xmax-xmin)*log10(maxx/minx));
			x2 = pow(10,log10(minx)+(double)(x-xmin)/(xmax-xmin)*log10(maxx/minx));
			break;
		case LOG2:
			x1 = pow(2,log2(minx)+(double)(tmagx-xmin)/(xmax-xmin)*log2(maxx/minx));
			x2 = pow(2,log2(minx)+(double)(x-xmin)/(xmax-xmin)*log2(maxx/minx));
			break;
		case LN:
			x1 = pow(M_E,log(minx)+(double)(tmagx-xmin)/(xmax-xmin)*log(maxx/minx));
			x2 = pow(M_E,log(minx)+(double)(x-xmin)/(xmax-xmin)*log(maxx/minx));
			break;
		case SQRT:
			//TODO
			x1 = minx + (tmagx-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = minx + (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			break;
		case SX2:
			//TODO
			break;
		}
		switch(yscale) {
		case LINEAR:
			y1 = miny + (ymax-y)/(double)(ymax-ymin)*(maxy-miny);
			y2 = miny + (ymax-tmagy)/(double)(ymax-ymin)*(maxy-miny);
			break;
		case LOG10:
			y1 = pow(10,log10(miny)+(double)(ymax-y)/(ymax-ymin)*log10(maxy/miny));
			y2 = pow(10,log10(miny)+(double)(ymax-tmagy)/(ymax-ymin)*log10(maxy/miny));
			break;
		case LOG2:
			y1 = pow(2,log2(miny)+(double)(ymax-y)/(ymax-ymin)*log2(maxy/miny));
			y2 = pow(2,log2(miny)+(double)(ymax-tmagy)/(ymax-ymin)*log2(maxy/miny));
			break;
		case LN:
			y1 = pow(M_E,log(miny)+(double)(ymax-y)/(ymax-ymin)*log(maxy/miny));
			y2 = pow(M_E,log(miny)+(double)(ymax-tmagy)/(ymax-ymin)*log(maxy/miny));
			break;
		case SQRT:
			//TODO
			y1 = miny + (ymax-y)/(double)(ymax-ymin)*(maxy-miny);
			y2 = miny + (ymax-tmagy)/(double)(ymax-ymin)*(maxy-miny);
			break;
		case SX2:
			//TODO
			break;
		}

		kdDebug()<<"MAG LENS : x1/x2 | y1/y2 = "<<x1<<'/'<<x2<<" | "<<y1<<'/'<<y2<<endl;

		plot[api]->setXRange(x1,x2);
		plot[api]->setYRange(y1,y2);
		// TODO : 3D -> set ZRange ? (not good !)
			
		mw->defining_maglens=0;
		tmagx=tmagx2=tmagy=tmagy2=0;
		setCursor(Qt::ArrowCursor);
		updatePixmap();
	}

	// mask data
	if(mw->defining_mask == 1) {
		tmaskx=x; tmasky=y;
		mw->defining_mask=2;
	}
	else if(mw->defining_mask == 2) {		// mask all data inside rect
		double minx = range[0].rMin(), maxx=range[0].rMax();
		double miny = range[1].rMin(), maxy=range[1].rMax();

		// see maglens
		double x1=0, x2=0, y1=0, y2=0;
		switch(xscale) {
		case LINEAR:
			x1 = minx + (double)(tmaskx-xmin)/(xmax-xmin)*(maxx-minx);
			x2 = minx + (double)(x-xmin)/(xmax-xmin)*(maxx-minx);
			break;
		case LOG10:
			x1 = pow(10,log10(minx)+(double)(tmaskx-xmin)/(xmax-xmin)*log10(maxx/minx));
			x2 = pow(10,log10(minx)+(double)(x-xmin)/(xmax-xmin)*log10(maxx/minx));
			break;
		case LOG2:
			x1 = pow(2,log2(minx)+(double)(tmaskx-xmin)/(xmax-xmin)*log2(maxx/minx));
			x2 = pow(2,log2(minx)+(double)(x-xmin)/(xmax-xmin)*log2(maxx/minx));
			break;
		case LN:
			x1 = pow(M_E,log(minx)+(double)(tmaskx-xmin)/(xmax-xmin)*log(maxx/minx));
			x2 = pow(M_E,log(minx)+(double)(x-xmin)/(xmax-xmin)*log(maxx/minx));
			break;
		case SQRT:
			//TODO
			x1 = minx + (tmaskx-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = minx + (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			break;
		case SX2:
			//TODO
			break;
		}
		switch(yscale) {
		case LINEAR:
			y1 = miny + (double)(ymax-y)/(ymax-ymin)*(maxy-miny);
			y2 = miny + (double)(ymax-tmasky)/(ymax-ymin)*(maxy-miny);
			break;
		case LOG10:
			y1 = pow(10,log10(miny)+(double)(ymax-y)/(ymax-ymin)*log10(maxy/miny));
			y2 = pow(10,log10(miny)+(double)(ymax-tmasky)/(ymax-ymin)*log10(maxy/miny));
			break;
		case LOG2:
			y1 = pow(2,log2(miny)+(double)(ymax-y)/(ymax-ymin)*log2(maxy/miny));
			y2 = pow(2,log2(miny)+(double)(ymax-tmasky)/(ymax-ymin)*log2(maxy/miny));
			break;
		case LN:
			y1 = pow(M_E,log(miny)+(double)(ymax-y)/(ymax-ymin)*log(maxy/miny));
			y2 = pow(M_E,log(miny)+(double)(ymax-tmasky)/(ymax-ymin)*log(maxy/miny));
			break;
		case SQRT:
			//TODO
			x1 = minx + (tmaskx-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = minx + (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			break;
		case SX2:
			//TODO
			break;
		}

		kdDebug()<<"MASK : x1/x2 | y1/y2 = "<<x1<<'/'<<x2<<" | "<<y1<<'/'<<y2<<endl;

		GraphList *gl = plot[api]->getGraphList();
		for (unsigned int i=0;i<gl->Number();i++) {
			// check all graphs for points inside [x1:x2] [y1;y2]
			// TODO : 3D,4D,M,---
			GRAPHType s = gl->getStruct(i);
			switch(s) {
			case GRAPH2D: {
				Graph2D *g = gl->getGraph2D(i);
				Point *d = g->Data();
				
				for (int j=0;j<g->Number();j++) {
					if(d[j].X()>x1 && d[j].X()<x2 && d[j].Y()>y1 && d[j].Y() < y2) {
						d[j].setMasked(true);
					}
				}
				};break;
			case GRAPH3D: {
				Graph3D *g = gl->getGraph3D(i);
				Point3D *d = g->Data();
				
				for (int j=0;j<g->Number();j++) {
					if(d[j].X()>x1 && d[j].X()<x2 && d[j].Y()>y1 && d[j].Y() < y2) {
						d[j].setMasked(true);
					}
				}
				};break;
			case GRAPH4D: {
				Graph4D *g = gl->getGraph4D(i);
				Point4D *d = g->Data();
				
				for (int j=0;j<g->Number();j++) {
					if(d[j].X()>x1 && d[j].X()<x2 && d[j].Y()>y1 && d[j].Y() < y2) {
						d[j].setMasked(true);
					}
				}
				};break;
			default: break;
			}
		}

		mw->defining_mask=0;
		tmaskx=tmaskx2=tmasky=tmasky2=0;
		setCursor(Qt::ArrowCursor);
		updatePixmap();
	}
	
	// pan zoom
	if(mw->defining_panzoom==1) {
		setCursor(QCursor(Qt::PointingHandCursor));

		double minx = range[0].rMin(), maxx=range[0].rMax();
		double miny = range[1].rMin(), maxy=range[1].rMax();
		// see maglens
		switch(xscale) {
		case LINEAR:	tpanx = minx+(x-xmin)/(double)(xmax-xmin)*(maxx-minx); break; 
		case LOG10:	tpanx = pow(10,log10(minx)+(double)(x-xmin)/(xmax-xmin)*log10(maxx/minx)); break;
		case LOG2:	tpanx = pow(2,log2(minx)+(double)(x-xmin)/(xmax-xmin)*log2(maxx/minx)); break;
		case LN:		tpanx = pow(M_E,log(minx)+(double)(x-xmin)/(xmax-xmin)*log(maxx/minx)); break;
		case SQRT:
			//TODO
			tpanx = minx+(x-xmin)/(double)(xmax-xmin)*(maxx-minx); 
			break;
		case SX2:
			//TODO
			break;
		}
		switch(yscale) {
		case LINEAR: 	tpany = miny+(ymax-y)/(double)(ymax-ymin)*(maxy-miny); break;
		case LOG10: 	tpany = pow(10,log10(miny)+(double)(ymax-y)/(ymax-ymin)*log10(maxy/miny)); break;
		case LOG2: 	tpany = pow(2,log2(miny)+(double)(ymax-y)/(ymax-ymin)*log2(maxy/miny)); break;
		case LN: 		tpany = pow(M_E,log(miny)+(double)(ymax-y)/(ymax-ymin)*log(maxy/miny)); break;
		case SQRT:
			//TODO
			tpany = miny+(ymax-y)/(double)(ymax-ymin)*(maxy-miny); 
			break;
		case SX2:
			//TODO
			break;
		}
		
		kdDebug()<<"	PAN ZOOM x/y"<<tpanx<<' '<<tpany<<endl;

		mw->defining_panzoom=2;
	}
	
	// if click inside plot rect set api
	for (unsigned int i=0;i<nr_plots;i++) {
		if(plot[i]==0) continue;
		int redrectx = (int)(plot[i]->Position().X()*X)+5+16*i;
		int redrecty = (int)(plot[i]->Position().Y()*Y)+5;
		if(x>redrectx && x <redrectx+15 && y>redrecty && y<redrecty+15) {
			tmpapi = i;
		}
	}
	// if click inside plot set api
	if(tmpapi==-1) {
		for (unsigned int i=0;i<nr_plots;i++) {
			if(plot[i]==0) continue;
			if (plot[i]->inside(x/(double)X,y/(double)Y))
				tmpapi = i;
		}
		// overlayed plots : use api
		if (plot[api]->inside(x/(double)X,y/(double)Y))
				tmpapi = api;
	}
	if(tmpapi!=-1)
		api=tmpapi;

	kdDebug()<<"	nr_plots = "<<nr_plots<<endl;
	kdDebug()<<"	API = "<<api<<endl;

	if (plot[api]!=0) {
		type = plot[api]->Type();

		if (plot[api]->insideF1Corner(x/(double)X,y/(double)Y))
			moving_cornerF1 = true;
		if (plot[api]->insideF2Corner(x/(double)X,y/(double)Y))
			moving_cornerF2 = true;
		if (plot[api]->insideB1Corner(x/(double)X,y/(double)Y))
			moving_cornerB1 = true;
		if (plot[api]->insideB2Corner(x/(double)X,y/(double)Y))
			moving_cornerB2 = true;
		else if (plot[api]->insideX1Border(x/(double)X,y/(double)Y))
			moving_borderx1 = true;
		else if (plot[api]->insideX2Border(x/(double)X,y/(double)Y))
			moving_borderx2 = true;
		else if (plot[api]->insideY1Border(x/(double)X,y/(double)Y))
			moving_bordery1 = true;
		else if (plot[api]->insideY2Border(x/(double)X,y/(double)Y))
			moving_bordery2 = true;
		else if (plot[api]->insideCenter(x/(double)X,y/(double)Y))
			moving_center = true;
	}

	if(mw->defining_baseline == true) {	// position
		double yvalue=YCoordinate(y,ymin,ymax);
		plot[api]->setBaselineEnabled(true);
		plot[api]->setBaseline(yvalue);
		repaint();
		//mw->defining_baseline=false;
	}
	if(mw->defining_region == 1) {	// left position
		double xvalue = XCoordinate(x,xmin,xmax);
		plot[api]->setRegionMin(xvalue);
		mw->message(i18n("Define Region : click on right position"));
		mw->defining_region = 2;
		return;
	}
	else if (mw->defining_region == 2) {	// right position
		double xvalue = XCoordinate(x,xmin,xmax);
		plot[api]->enableRegion();
		plot[api]->setRegionMax(xvalue);
		mw->defining_region = 0;
		updatePixmap();
		return;
	}
	if(mw->defining_line == 1) {	// first position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
//		kdDebug()<<"DEFINE LINE : START x="<<xvalue<<" y="<<yvalue<<endl;
		while (line[tmp_object_index]->startPoint().X() != line[tmp_object_index]->endPoint().X())
			tmp_object_index++;
		line[tmp_object_index]->setStartPoint(xvalue,yvalue);
		mw->message(i18n("Define Line : click on second position"));
		mw->defining_line=2;
		return;
	}
	else if (mw->defining_line == 2) {	// second position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
//		kdDebug()<<"DEFINE LINE : END x="<<xvalue<<" y="<<yvalue<<endl;
		line[tmp_object_index]->setEndPoint(xvalue,yvalue);
		mw->defining_line=0;
		tmp_object_index=0;
		updatePixmap();
		return;
	}
	if(mw->defining_label == true) {	// position
		kdDebug()<<"defining label .."<<endl;
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		kdDebug()<<" label["<<tmp_object_index<<"]="<<label[tmp_object_index]->Title()<<endl;
		while (! label[tmp_object_index]->Title().isEmpty())
			tmp_object_index++;
		label[tmp_object_index]->setPosition(xvalue,yvalue);
		mw->defining_label=0;
		(new ObjectDialog(mw,"ObjectDialog",1,tmp_object_index))->show();
		return;
	}

	if(mw->defining_rect == 1) {	// first position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		while (rect[tmp_object_index]->endPoint().X() != 0)
			tmp_object_index++;
		rect[tmp_object_index]->setStartPoint(xvalue,yvalue);
		mw->message(i18n("Define Rectangle : click on lower right corner"));
		mw->defining_rect=2;
		return;
	}
	else if (mw->defining_rect == 2) {	// second position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		Point start = rect[tmp_object_index]->startPoint();
		rect[tmp_object_index]->setEndPoint(xvalue-start.X(),yvalue-start.Y());
		mw->defining_rect=0;
		tmp_object_index=0;
		updatePixmap();
		return;
	}
	if(mw->defining_ellipse == 1) {	// first position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		while (ellipse[tmp_object_index]->endPoint().X() != 0)
			tmp_object_index++;
		ellipse[tmp_object_index]->setStartPoint(xvalue,yvalue);
		mw->message(i18n("Define Ellipse : click on lower right corner"));
		mw->defining_ellipse=2;
		return;
	}
	else if (mw->defining_ellipse == 2) {	// second position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		Point start = ellipse[tmp_object_index]->startPoint();
		ellipse[tmp_object_index]->setEndPoint(xvalue-start.X(),yvalue-start.Y());
		mw->defining_ellipse=0;
		tmp_object_index=0;
		updatePixmap();
		return;
	}
	if(mw->defining_image == true) {	// position
		double xvalue=x/(double)X;
		double yvalue=y/(double)Y;
		while (image[tmp_object_index]->Pos().X() != 0)
			tmp_object_index++;
		image[tmp_object_index]->setPos(xvalue,yvalue);
		mw->defining_image=0;
		(new ObjectDialog(mw,"ObjectDialog",4,tmp_object_index))->show();
		return;
	}

	double tmpx=0, tmpy=0;
	if (plot != 0) {
		tmpx = X*size.X();
		tmpy = Y*size.Y();
	}

	// if inside legend, take it
	if (plot[api] && plot[api]->insideLegend(x,y)) {
		lx = x - (int) (plot[api]->getLegend()->X()*tmpx);
		ly = y - (int) (plot[api]->getLegend()->Y()*tmpy);
	}

	// if inside title, take it
	Label *title = 0;
	if (plot[api]!=0)
		title = plot[api]->Title();

	if(title && title->inside(x,y,pos,size,X,Y)) {
		kdDebug()<<"inside TITLE"<<endl;
		tx = (int)(x - title->X()*tmpx);
		ty = (int)(y - title->Y()*tmpy);
		kdDebug()<<" TX/TY : "<<tx<<' '<<ty<<endl;
	}

//	kdDebug()<<"after title/legend check"<<endl;

	for (int i=0;i<NR_OBJECTS;i++) {
		if(label[i]->inside(x,y,pos,size,X,Y)) {
			labelx = (int)(x - label[i]->X()*X);
			labely = (int)(y - label[i]->Y()*Y);
			object_index=i;
		}
		else if(line[i]->inside(x,y,X,Y)) {
			linesx = (int)(x - line[i]->startPoint().X()*X);
			linesy = (int)(y - line[i]->startPoint().Y()*Y);
			lineex = (int)(x - line[i]->endPoint().X()*X);
			lineey = (int)(y - line[i]->endPoint().Y()*Y);
			object_index=i;
		}
		else if(rect[i]->inside(x,y,X,Y)) {
			rectx = (int)(x - rect[i]->startPoint().X()*X);
			recty = (int)(y - rect[i]->startPoint().Y()*Y);
 			object_index=i;
		}
		else if(ellipse[i]->inside(x,y,X,Y)) {
			ellipsex = (int)(x - ellipse[i]->startPoint().X()*X);
			ellipsey = (int)(y - ellipse[i]->startPoint().Y()*Y);
			object_index=i;
		}
		else if(image[i]->inside(x,y,X,Y)) {
			imagex = (int)(x - image[i]->Pos().X()*X);
			imagey = (int)(y - image[i]->Pos().Y()*Y);
			object_index=i;
		}
	}

	// if inside axis take the axes label
	if (type == P2D || type == PSURFACE) {
		for (int i=0;i<4;i++) {
			Label *label = 0;
			if(plot[api])
				label = plot[api]->getAxis(i)->getLabel();

			if (i==0 || i==3) {	// x,x2
				if (label && label->inside(x,y,pos,size,X,Y)) {
					kdDebug()<<"INSIDE X/X2 AXIS"<<endl;
					ax = (int)(x - label->X()*tmpx);
					ay = (int)(y - label->Y()*tmpy);
					anumber = i;
				}
			}
			else {	// y,y2
				if (label && label->insideY(x,y,pos,size,X,Y)) {
					kdDebug()<<"INSIDE Y/Y2 AXIS"<<endl;
					ax = (int)(x - label->X()*tmpx);
					ay = (int)(y - label->Y()*tmpy);
					anumber = i;
				}
			}
		}
	}
	else if (type == P3D) {
		kdDebug()<<"type = P3D"<<endl;
		for (int i=0;i<12;i++) {
			Label *label = plot[api]->getAxis(i)->getLabel();
			bool inside=false;

			if (i==0 || i==3 || i==6 || i==9) {		// z,z2,z3,z4
				if (label->inside(x,y,pos,size,X,Y))
					inside =true;
			}
			else if (i==1 || i==4 || i==7 || i==10)	{	// y,y2,y3,y4
				if (label->insideZ(x,y,pos,size,X,Y))
					inside =true;
			}
			else {						// z,z2,z3,z4
				if (label->insideY(x,y,pos,size,X,Y))
					inside =true;
			}
			if (inside) {
				ax = (int)(x - label->X()*tmpx);
				ay = (int)(y - label->Y()*tmpy);
				anumber = i;
			}
		}
	}

	// if near border of axes (+/- 10) take it
	kdDebug()<<" CHECK BORDER : x="<<x<<" / xmin="<<xmin<<endl;
	if (type == P2D || type == PSURFACE) {
		if (x>xmin-5 && x<xmin+5 && y>ymin && y<ymax) { // xmin
			kdDebug()<<"NEAR XMIN"<<endl;
			bx=x-xmin;
			by=y-ymin;
			anumber = 1;
		}
		else if (x>xmax-5 && x<xmax+5 && y>ymin && y<ymax) { // xmax
			kdDebug()<<"NEAR XMAX"<<endl;
			bx=x-xmax;
			by=y-ymin;
			anumber = 2;
		}
		else if (x>xmin && x<xmax && y>ymin-5 && y<ymin+5) { // ymin
			kdDebug()<<"NEAR YMIN"<<endl;
			bx=x-xmin;
			by=y-ymin;
			anumber = 3;
		}
		else if (x>xmin && x<xmax && y>ymax-5 && y<ymax+5) { // ymax
			if (type == PSURFACE)	// why ?
				bx=x-xmin;
			else
				bx=1;	// must be != 0
			by=y-ymax;
			anumber = 0;
			kdDebug()<<"NEAR YMAX bx/by = "<<bx<<' '<<by<<endl;
		}
	}
	else if (type == P3D) {
		if (x>xmin &&  x<xmax && y>=ymax && y<ymax+20) {
			bx=x-xmin;
			by=y-ymax;
			anumber = 0;
		}
		if (x>xmin-20 &&  x<=xmin && y>ymin && y<ymax) {
			bx=x-xmin;
			by=y-ymax;
			anumber = 1;
		}
	}
	updatePixmap();	// update api mark
	kdDebug()<<"Worksheet::mousePressEvent DONE"<<endl;
}

void Worksheet::mouseMoveEvent(QMouseEvent *e) {
	int x=e->x(), y=e->y();

	// dont crash if moving over empty worksheet
	if (plot[api]==0)
		return;

	Point pos = plot[api]->Position(), size = plot[api]->Size();
	PType type = plot[api]->Type();

	TScale xscale=LINEAR, yscale=LINEAR;
	if(type != PPIE) {
		xscale=plot[api]->getAxis(0)->Scale();
		yscale=plot[api]->getAxis(1)->Scale();
	}

	double xmin=X*(pos.X()+plot[api]->P1().X()*size.X());
	double xmax=X*(pos.X()+plot[api]->P2().X()*size.X());
	double ymin=Y*(pos.Y()+plot[api]->P1().Y()*size.Y());
	double ymax=Y*(pos.Y()+plot[api]->P2().Y()*size.Y());

	if(mw->dataMode()) {
		tdatamodex=x;
		updatePixmap();
		return;
	}

	if(mw->defining_maglens==2) {
		tmagx2=x; tmagy2=y;
		updatePixmap();
	}
	else if (mw->defining_panzoom==2) {	// constantly set new actual ranges
		LRange *actrange = plot[api]->ActRanges();
		LRange *range = plot[api]->Ranges();

		kdDebug()<<"PAN ZOOM : tpanx/tpany = "<<tpanx<<' '<<tpany<<endl;

		double minx = actrange[0].rMin(), maxx=actrange[0].rMax();
		double miny = actrange[1].rMin(), maxy=actrange[1].rMax();
		double minrx = range[0].rMin(), maxrx=range[0].rMax();
		double minry = range[1].rMin(), maxry=range[1].rMax();

		// check values
		checkRanges(xscale,&minrx,&maxrx);
		checkRanges(yscale,&minry,&maxry);

		// see maglens
		double x1=0, x2=0, y1=0, y2=0; 
		switch(xscale) {
		case LINEAR:
			x1 = tpanx - (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = x1+(maxx-minx);
			break;
		case LOG10:
			x1 = pow(10,log10(tpanx)-(double)(x-xmin)/(xmax-xmin)*log10(maxx/minx));
			x2 = x1*pow(10,log10(maxrx/minrx));
			break;
		case LOG2:
			x1 = pow(2,log2(tpanx)-(double)(x-xmin)/(xmax-xmin)*log2(maxx/minx));
			x2 = x1*pow(2,log2(maxrx/minrx));
			break;
		case LN:
			x1 = pow(M_E,log(tpanx)-(double)(x-xmin)/(xmax-xmin)*log(maxx/minx));
			x2 = x1*pow(M_E,log(maxrx/minrx));
			break;
		case SQRT:
			//TODO
			x1 = tpanx - (x-xmin)/(double)(xmax-xmin)*(maxx-minx);
			x2 = x1+(maxx-minx);
			break;
		case SX2:
			//TODO
			break;
		}
		switch(yscale) {
		case LINEAR:
			y1 = tpany - (ymax-y)/(double)(ymax-ymin)*(maxy-miny);
			y2 = y1+(maxy-miny);
			break;
		case LOG10:
			y1 = pow(10,log10(tpany)-(double)(ymax-y)/(ymax-ymin)*log10(maxy/miny));
			y2 = y1*pow(10,log10(maxry/minry));
			break;
		case LOG2:
			y1 = pow(2,log2(tpany)-(double)(ymax-y)/(ymax-ymin)*log2(maxy/miny));
			y2 = y1*pow(2,log2(maxry/minry));
			break;
		case LN:
			y1 = pow(M_E,log(tpany)-(double)(ymax-y)/(ymax-ymin)*log(maxy/miny));
			y2 = y1*pow(M_E,log(maxry/minry));
			break;
		case SQRT:
			//TODO
			y1 = tpany - (ymax-y)/(double)(ymax-ymin)*(maxy-miny);
			y2 = y1+(maxy-miny);
		case SX2:
			//TODO
			break;
		}

		kdDebug()<<"	PAN ZOOM : x1/x2 "<<x1<<' '<<x2<<endl;
		plot[api]->setXRange(x1,x2);
		plot[api]->setYRange(y1,y2);
		updatePixmap();
	}

	if(mw->defining_mask==2) {
		tmaskx2=x; tmasky2=y;
		updatePixmap();
	}
			
	if (mw->defining_maglens>0 || mw->defining_panzoom>0 || mw->defining_mask>0) {
		// leave cursor as is
	}
	else if (plot[api]->insideF1Corner(x/(double)X,y/(double)Y) ||plot[api]->insideF2Corner(x/(double)X,y/(double)Y) )
		setCursor(QCursor(Qt::SizeFDiagCursor));
	else if (plot[api]->insideB1Corner(x/(double)X,y/(double)Y) || plot[api]->insideB2Corner(x/(double)X,y/(double)Y))
		setCursor(QCursor(Qt::SizeBDiagCursor));
	else if (plot[api]->insideX1Border(x/(double)X,y/(double)Y) || plot[api]->insideX2Border(x/(double)X,y/(double)Y))
		setCursor(QCursor(Qt::SizeVerCursor));
	else if (plot[api]->insideY1Border(x/(double)X,y/(double)Y) || plot[api]->insideY2Border(x/(double)X,y/(double)Y))
		setCursor(QCursor(Qt::SizeHorCursor));
	else if (plot[api]->insideCenter(x/(double)X,y/(double)Y))
		setCursor(QCursor(Qt::SizeAllCursor));
	else
		setCursor(Qt::ArrowCursor);

	if(moving_cornerF1) {
		plot[api]->setSize(pos.X()+size.X()-x/(double)X,pos.Y()+size.Y()-y/(double)Y);
		plot[api]->setPosition(x/(double)X,y/(double)Y);
		repaint();
	}
	else if(moving_cornerF2) {
		plot[api]->setSize(x/(double)X-pos.X(),y/(double)Y-pos.Y());
		repaint();
	}
	else if(moving_cornerB2) {
		plot[api]->setSize(x/(double)X-pos.X(),size.Y()+pos.Y()-y/(double)Y);
		plot[api]->setPosition(pos.X(),y/(double)Y);
		repaint();
	}
	else if(moving_cornerB1) {
		plot[api]->setSize(size.X()+pos.X()-x/(double)X,y/(double)Y-pos.Y());
		plot[api]->setPosition(x/(double)X,pos.Y());
		repaint();
	}
	else if (moving_borderx1) {
		plot[api]->setSize(size.X(),y/(double)Y-pos.Y());
		repaint();
	}
	else if (moving_borderx2) {
		kdDebug()<<"	MOVING Border x2 of PLOT"<<endl;
		plot[api]->setSize(size.X(),size.Y()+pos.Y()-y/(double)Y);
		plot[api]->setPosition(pos.X(),y/(double)Y);
		repaint();
	}
	else if (moving_bordery1) {
		plot[api]->setSize(pos.X()+size.X()-x/(double)X,size.Y());
		plot[api]->setPosition(x/(double)X,pos.Y());
		repaint();
	}
	else if (moving_bordery2) {
		plot[api]->setSize(x/(double)X-pos.X(),size.Y());
		repaint();
	}
	else if (moving_center) {
		plot[api]->setPosition(x/(double)X-size.X()/2.0,y/(double)Y-size.Y()/2.0);
		repaint();
	}

	if(mw->defining_region != 0)
		return;
	if(mw->defining_line != 0)
		return;
	if(mw->defining_label == true)
		return;
	if(mw->defining_rect != 0)
		return;
	if(mw->defining_ellipse != 0)
		return;
	if(mw->defining_image == true)
		return;

	if(mw->defining_baseline == true) {	// position
		double yvalue=YCoordinate(y,ymin,ymax);
		plot[api]->setBaseline(yvalue);
		repaint();
		return;
	}
	
	double tmpx = X*plot[api]->Size().X();
	double tmpy = Y*plot[api]->Size().Y();

	if (lx > 0) {		// only if legend is dragged
		plot[api]->getLegend()->setPosition((x-lx)/tmpx,(y-ly)/tmpy);
		updatePixmap();
	}
	else if (tx > 0) {		// only if title is dragged
		Label *title = plot[api]->Title();
		title->setPosition((x-tx)/tmpx,(y-ty)/tmpy);
		updatePixmap();
	}
	else if (labelx > 0) {
		label[object_index]->setPosition((x-labelx)/(double)X,(y-labely)/(double)Y);
		updatePixmap();
	}
	else if (abs(linesx) > 0) {
		line[object_index]->setEndPoint((x-lineex)/(double)X,(y-lineey)/(double)Y);
		line[object_index]->setStartPoint((x-linesx)/(double)X,(y-linesy)/(double)Y);
		updatePixmap();
	}
	else if (rectx > 0) {
		rect[object_index]->setStartPoint((x-rectx)/(double)X,(y-recty)/(double)Y);
		updatePixmap();
	}
	else if (ellipsex > 0) {
		ellipse[object_index]->setStartPoint((x-ellipsex)/(double)X,(y-ellipsey)/(double)Y);
		updatePixmap();
	}
	else if (imagex > 0) {
		image[object_index]->setPos((x-imagex)/(double)X,(y-imagey)/(double)Y);
		updatePixmap();
	}
	else if (ax!=0) {
		Label *label = plot[api]->getAxis(anumber)->getLabel();
		label->setPosition((x-ax)/tmpx,(y-ay)/tmpy);
		updatePixmap();
	}
	else if (bx!=0) {
		if (type == P2D || type == PSURFACE) {
			Label *label = plot[api]->getAxis(anumber)->getLabel();
			if (anumber==0) {	// x
				int pos=by+y;
				label->setPosition(label->X(),label->Y()+(pos-ymax)/(double)Y);
				plot[api]->setYMax(pos,Y);
			}
			else if (anumber==1) {	// y
				label->setPosition(label->X()+(x-xmin+bx)/(double)X,label->Y());
				plot[api]->setXMin(bx+x,X);
			}
			else if (anumber==2) {	// y2
				label->setPosition(label->X()+(x-xmax+bx)/(double)X,label->Y());
				plot[api]->setXMax(bx+x,X);
			}
			else if (anumber==3) {	// x2
				label->setPosition(label->X(),label->Y()+(y-ymin+by)/(double)Y);
				plot[api]->setYMin(by+y,Y);
			}
		}
		else if (type == P3D) {
			if (anumber==0) {	// x
				Label *label = plot[api]->getAxis(3)->getLabel();
				label->setPosition(label->X(),label->Y()+(y-ymax+by)/(double)Y);
				plot[api]->setYMax(by+y,Y);
			}
			if (anumber==1) {	// z
				Label *label = plot[api]->getAxis(5)->getLabel();
				label->setPosition(label->X()+(x-xmin+bx)/(double)X,label->Y());
				plot[api]->setXMin(bx+x,X);
			}
		}

		updatePixmap();
	}
	
	if (plot[api]->insidePlottingArea(x/(double)X,y/(double)Y)) {
		double xvalue=0, yvalue=0;
		// TODO : ternary & polar coordinates
		// TODO : format for coordinates ?
		if (type != PPIE && type != PTERNARY && type != PPOLAR) {
			xvalue = XCoordinate(x,xmin,xmax);
			yvalue=YCoordinate(y,ymin,ymax);
		}
		mw->message("( " + QString::number(xvalue) + " / " + QString::number(yvalue) + " )");
	}
	else if (mw->defining_maglens==1 || mw->defining_mask==1)
			mw->message(i18n("click on first position"));
	else if (mw->defining_maglens==2 || mw->defining_mask==2)
			mw->message(i18n("click on second position"));
	else
		mw->message("");
}

double Worksheet::XCoordinate(double x, double xmin, double xmax) {
	LRange* actrange = plot[api]->ActRanges();
	double xvalue=0;
	
	TScale xscale = LINEAR;
	if(plot[api] && plot[api]->getAxis(0))
		xscale = plot[api]->getAxis(0)->Scale();
	double minx = actrange[0].rMin(), maxx = actrange[0].rMax();
		
	//TODO
	if (xscale == LINEAR)
		xvalue = minx+(maxx-minx)*(x-xmin)/(xmax-xmin);
	else if (xscale == LOG10)
		xvalue = pow(10,log10(minx)+(double)(x-xmin)/(xmax-xmin)*log10(maxx/minx));
	else if (xscale == LOG2)
		xvalue = pow(2,log2(minx)+(double)(x-xmin)/(xmax-xmin)*log2(maxx/minx));
	else if (xscale == LN)
		xvalue = pow(M_E,log(minx)+(double)(x-xmin)/(xmax-xmin)*log(maxx/minx));
	else if (xscale == SQRT)
		// TODO : check this
		//xvalue = pow(sqrt(minx)+(double)(x-xmin)/(xmax-xmin)*sqrt(maxx-minx),2);
		xvalue = minx+(double)(x-xmin)/(xmax-xmin)*(maxx-minx);

	return xvalue;
}

double Worksheet::YCoordinate(double y, double ymin, double ymax) {
	LRange* actrange = plot[api]->ActRanges();
	double yvalue=0;
	TScale yscale = LINEAR;
	if(plot[api] && plot[api]->getAxis(1))
		yscale = plot[api]->getAxis(1)->Scale();
	double miny = actrange[1].rMin(), maxy = actrange[1].rMax();
	
	//TODO
	if (yscale == LINEAR)
		yvalue = miny+(maxy-miny)*(double)(y-ymax)/(ymin-ymax);
	else if (yscale == LOG10)
		yvalue = pow(10,log10(miny)+(double)(ymax-y)/(ymax-ymin)*log10(maxy/miny));
	else if (yscale == LOG2)
		yvalue = pow(2,log2(miny)+(double)(ymax-y)/(ymax-ymin)*log2(maxy/miny));
	else if (yscale == LN)
		yvalue = pow(M_E,log(miny)+(double)(ymax-y)/(ymax-ymin)*log(maxy/miny));
	else if (yscale == SQRT)
		// TODO : check this
		//yvalue = pow(sqrt(miny)+(double)(y-ymin)/(ymax-ymin)*sqrt(maxy-miny),2);
		yvalue = miny+(maxy-miny)*(double)(y-ymax)/(ymin-ymax);

	return yvalue;
}

void Worksheet::mouseReleaseEvent(QMouseEvent *) {
	kdDebug()<<"Worksheet::mouseReleaseEvent()"<<endl;
	lx = ly = ax = ay = tx = ty = bx = by = 0;
	labelx=labely=linesx=linesy=lineex=lineey=rectx=recty=0;
	ellipsex=ellipsey=imagex=imagey=0;
	object_index=0;
	anumber = 0;
	moving_cornerF1 = moving_cornerF2 = false;
	moving_cornerB1 = moving_cornerB2 = false;
	moving_borderx1 = moving_borderx2 = false;
	moving_bordery1 = moving_bordery2 = false;
	moving_center=0;

	if (mw->defining_panzoom==2) {
		// TODO : release range 
		setCursor(Qt::ArrowCursor);
		mw->defining_panzoom=0;
		repaint();
	}
	
	mw->defining_baseline=false;
	mw->updateLegendDialog();
	mw->updateTitleDialog();
	mw->updateAxesDialog();
	mw->updatePlotSettingsDialog();
}

void Worksheet::Print(QString fn) {
	kdDebug()<<"Worksheet::Print() : filename = "<<fn<<endl;

	setupPrinter(printer,fn);

	// ok ?
	if (fn == QString("out.ps")) {
		printer->setup(this);
		fn = printer->outputFileName();
	}

	QPainter p(printer);
	QPaintDeviceMetrics metrics( printer );

	int w = metrics.width();
	int h = metrics.height();
	kdDebug()<<"	Metrics = "<<w<<' '<<h<<endl;

	// keep original size
	KConfig *config = mw->Config();
	config->setGroup( "KPrinter Settings" );
	if (config->readBoolEntry("UseOriginalSize",true)) {
		w=X;
		h=Y;
	}

	if(plot[api]->Type() == PQWT3D) {
		kdDebug()<<"	PRINTING QWT3D plot"<<endl;
		((PlotQWT3D*) plot[api])->Export(&p,fn,"PS",w,h);
		plot[api]->draw(&p,X,Y);		// update QWT3D plot
	}
	else
		Draw(&p,w,h);

	kdDebug()<<"Worksheet::Print() done"<<endl;
}

void Worksheet::ExportSVG(QString filename) {
	kdDebug()<<"Worksheet::ExportSVG() : filename="<<filename<<endl;
	if(filename.isEmpty()) {
		QString tmpname = mw->Filename().replace( QRegExp(QString(".lpl.*$")), ".svg" );
		if (tmpname.isEmpty())
			tmpname=QString("out.svg");
		filename = QFileDialog::getSaveFileName(tmpname,i18n("SVG Image (*.svg)"),this);

		if(filename.isEmpty())	// "Cancel"
			return;

		if(filename.find(QRegExp("\\.svg"))==-1)
			filename.append(".svg");
	 }
	 
	// check if svg file exists
	if ( QFile::exists(filename) ) {
		int answer = KMessageBox::warningYesNoCancel( this, 
			i18n( "Overwrite\n\'%1\'?" ).arg( filename ), i18n("Export to SVG"));
		if (answer != KMessageBox::Yes)
			return;
	}

	QPicture pic;
	QPainter p;
	p.begin( &pic ); 		// paint in picture
	Draw(&p,X,Y);			// draw it
	p.end();		 	// painting done
	pic.save(filename,"svg");
}

void Worksheet::ExportPIC(QString filename) {
	kdDebug()<<"Worksheet::ExportPIC() : filename="<<filename<<endl;
	if(filename.isEmpty()) {
		QString tmpname = mw->Filename().replace( QRegExp(QString(".lpl.*$")), ".pic" );
		if (tmpname.isEmpty())
			tmpname=QString("out.pic");
		filename = QFileDialog::getSaveFileName(tmpname,i18n("PIC File (*.pic)"),this);

		if(filename.isEmpty())	// "Cancel"
			return;

		if(filename.find(QRegExp("\\.pic"))==-1)
			filename.append(".pic");
	}
			
	// check if pic file exists
	if ( QFile::exists(filename) ) {
		int answer = KMessageBox::warningYesNoCancel( this, 
			i18n( "Overwrite\n\'%1\'?" ).arg( filename ), i18n("Export to PIC"));
		if (answer != KMessageBox::Yes)
			return;
	}
	
	QPicture pic;
	QPainter p;
	p.begin( &pic ); 		// paint in picture
	Draw(&p,X,Y);			// draw it
	p.end();			// painting done
	pic.save(filename);
}

void Worksheet::ExportPS(QString fn) {
	kdDebug()<<"Worksheet::ExportPS() : filename="<<fn<<endl;
	if(fn.isEmpty()) {
		QString tmpname = mw->Filename().replace( QRegExp(QString(".lpl.*$")), ".ps" );
		if (tmpname.isEmpty())
			tmpname=QString("out.ps");
		fn = QFileDialog::getSaveFileName(tmpname,i18n("Postscript (*.ps)"),this);

		if(fn.isEmpty())	// "Cancel"
			return;

		if(fn.find(QRegExp("\\.ps"))==-1)	// append extension
			fn.append(".ps");
	}
		
	// check if file exists
	if ( QFile::exists(fn) ) {
		int answer = KMessageBox::warningYesNoCancel( this, i18n( "Overwrite\n\'%1\'?" ).arg( fn ), i18n("Export to PS"));
		if (answer != KMessageBox::Yes)
			return;
	}

	// use normal KPrinter instead of -sDEVICE=pswrite
	Print(fn);
}

void Worksheet::ExportEPS(QString fn) {
	kdDebug()<<"Worksheet::ExportEPS() : filename="<<fn<<endl;
	if(fn.isEmpty()) {
		// use project filename with ".eps" as default
		QString tmpname = mw->Filename().replace( QRegExp(QString(".lpl.*$")), ".eps" );
		if (tmpname.isEmpty())
			tmpname=QString("out.eps");
		fn = QFileDialog::getSaveFileName(tmpname,i18n("Encapsulated Postscript (*.eps)"),this);

		if(fn.isEmpty())	// "Cancel"
			return;

		if(fn.find(QRegExp("\\.eps"))==-1)
			fn.append(".eps");
	}
		
	// check if file exists
	if ( QFile::exists(fn) ) {
		int answer = KMessageBox::warningYesNoCancel( this, i18n( "Overwrite\n\'%1\'?" ).arg( fn ), i18n("Export to EPS"));
		if (answer != KMessageBox::Yes)
			return;
	}
	
	KConfig *config = mw->Config();
	config->setGroup( "KPrinter Settings" );

	if (config->readBoolEntry("UsePS2EPS",true)) {
		kdDebug()<<"Using ps2eps to create EPS"<<endl;
		KTempFile *tmpfile = new KTempFile(QString::null,".ps");
		Print(tmpfile->name());

		KProcess *proc = new KProcess;
		*proc << "ps2epsi";
		*proc << tmpfile->name()<<fn;
		delete tmpfile;

		if( proc->start(KProcess::Block) == false)
			kdDebug()<<"COULD NOT FIND ps2epsi! Continue with normal export."<<endl;
		else
			return;
	}
	
	// use a clean KPrinter object
	kdDebug()<<"Using ghostscript to create EPS"<<endl;
	KPrinter printer(false);
	setupPrinter(&printer,fn);

	QPainter p(&printer);
	QPaintDeviceMetrics metrics( &printer );

	int w = metrics.width();
	int h = metrics.height();
	kdDebug()<<"Metrics = "<<w<<' '<<h<<endl;
	printer.setPrintProgram("gs -sDEVICE=epswrite -dBATCH -dNOPAUSE -sOutputFile=%out %in" );

	// keep original size
	if (config->readBoolEntry("UseOriginalSize",true)) {
		w=X;
		h=Y;
	}

	if(plot[api]->Type() == PQWT3D) {
		kdDebug()<<"	PRINTING QWT3D plot"<<endl;
		((PlotQWT3D*) plot[api])->Export(&p,fn,"EPS",w,h);
		plot[api]->draw(&p,X,Y);		// update QWT3D plot
	}
	else
		Draw(&p,w,h);
}

void Worksheet::ExportPDF(QString fn) {
	// TODO : use print like ps ?
	kdDebug()<<"Worksheet::ExportPDF() : filename="<<fn<<endl;
	if(fn.isEmpty()) {
		QString tmpname = mw->Filename().replace( QRegExp(QString(".lpl.*$")), ".pdf" );
		if (tmpname.isEmpty())
			tmpname=QString("out.pdf");
		fn = QFileDialog::getSaveFileName(tmpname,i18n("Portable Document Format (*.pdf)"),this);

		if(fn.isEmpty())	// "Cancel"
			return;

		if(fn.find(QRegExp("\\.pdf"))==-1)
			fn.append(".pdf");
	}
	
	// check if file exists
	if ( QFile::exists(fn) ) {
		int answer = KMessageBox::warningYesNoCancel( this, i18n( "Overwrite\n\'%1\'?" ).arg( fn), i18n("Export to PDF"));
		if (answer != KMessageBox::Yes)
			return;
	}

	KConfig *config = mw->Config();
	config->setGroup( "KPrinter Settings" );
	if (config->readBoolEntry("UsePS2PDF",true)) {
		kdDebug()<<"Using ps2pdf to create PDF"<<endl;
		KTempFile *tmpfile = new KTempFile(QString::null,".ps");
		Print(tmpfile->name());

		KProcess *proc = new KProcess;
		*proc << "ps2pdf";
		*proc << tmpfile->name()<<fn;
		delete tmpfile;

		if( proc->start(KProcess::Block) == false)
			kdDebug()<<"COULD NOT FIND ps2pdf! Continue with normal export."<<endl;
		else
			return;
	}
	
	// use a clean KPrinter object
	kdDebug()<<"Using ghostscript to create PDF"<<endl;
	KPrinter printer(false);
	setupPrinter(&printer,fn);

	QPainter p(&printer);
	QPaintDeviceMetrics metrics( &printer );

	int w = metrics.width();
	int h = metrics.height();
	kdDebug()<<"Metrics = "<<w<<' '<<h<<endl;

	// keep original size
	if (config->readBoolEntry("UseOriginalSize",true)) {
		w=X;
		h=Y;
	}

	printer.setPrintProgram( "gs -sDEVICE=pdfwrite -sPAPERSIZE=%psl -dBATCH -dNOPAUSE -sOutputFile=%out %in" );

	if(plot[api]->Type() == PQWT3D) {
		kdDebug()<<"	PRINTING QWT3D plot"<<endl;
		((PlotQWT3D*) plot[api])->Export(&p,fn,"PDF",w,h);
		plot[api]->draw(&p,X,Y);		// update QWT3D plot
	}
	else
		Draw(&p,w,h);
}

void Worksheet::setupPrinter(KPrinter *pr, QString fn) {
	kdDebug()<<"Worksheet::setupPrinter()"<<endl;
	KConfig *config = mw->Config();
	config->setGroup( "KPrinter Settings" );

	pr->setCreator(QString("LabPlot ")+LVERSION);
	pr->setOutputToFile( true );
	pr->setOutputFileName( fn );
	pr->setPageSize((KPrinter::PageSize)config->readNumEntry("PageSize",9));
	if (config->readNumEntry("Orientation",1))
		pr->setOrientation(KPrinter::Landscape);
	else
		pr->setOrientation(KPrinter::Portrait);

	if (config->readBoolEntry("ColorMode",true))
		pr->setColorMode(KPrinter::Color);
	else
		pr->setColorMode(KPrinter::GrayScale);
}

// wrapper functions for QSA
Plot2DSimple* Worksheet::get2DPlot(int i) { 
	if(plot[i]->Type() == P2D) 
		return (Plot2DSimple *) plot[i]; 
	return 0;
}

Plot2DSurface* Worksheet::get2DSurfacePlot(int i) { 
	if(plot[i]->Type() == PSURFACE) 
		return (Plot2DSurface *) plot[i]; 
	return 0;
}

Plot3D* Worksheet::get3DPlot(int i) { 
	if(plot[i]->Type() == P3D) 
		return (Plot3D *) plot[i]; 
	return 0;
}

PlotQWT3D* Worksheet::getQWT3DPlot(int i) { 
	if(plot[i]->Type() == PQWT3D) 
		return (PlotQWT3D *) plot[i]; 
	return 0;
}

PlotPie* Worksheet::getPiePlot(int i) { 
	if(plot[i]->Type() == PPIE) 
		return (PlotPie *) plot[i]; 
	return 0;
}

PlotPolar* Worksheet::getPolarPlot(int i) { 
	if(plot[i]->Type() == PPOLAR) 
		return (PlotPolar *) plot[i]; 
	return 0;
}

PlotTernary* Worksheet::getTernaryPlot(int i) { 
	if(plot[i]->Type() == PTERNARY) 
		return (PlotTernary *) plot[i]; 
	return 0;
}

void Worksheet::Draw( QPainter *p, int width, int height ) {
	kdDebug()<<"Worksheet:Draw()"<<endl;
	
	// draw background color
	p->setBrush(background);
	p->setPen(NoPen);
	p->drawRect(0,0,width,height);
	//kdDebug()<<"	Worksheet	width = "<<X<<" / height = "<<Y<<endl;
	//kdDebug()<<"	drawing with	width = "<<width<<" / height = "<<height<<endl;

	kdDebug()<<"	Worksheet:Draw() drawing "<<nr_plots<<" plots"<<endl;
	kdDebug()<<"		API = "<<api<<endl;
	for (unsigned int i=0;i<nr_plots;i++) {
		kdDebug()<<"	Worksheet:Draw() drawing plot "<<i+1<<endl;
		if(plot[i]==0) {
			kdDebug()<<"	PLOT "<<i<<" is 0"<<endl;
			continue;
		}
		kdDebug()<<"		Type = "<<plot[i]->Type()<<endl;
		if(plot[i]!=0 && i!=api)
			plot[i]->draw(p,width,height);
	}
	// draw api plot at last
	if(nr_plots>0 && plot[api]!=0)
		plot[api]->draw(p,width,height);

	kdDebug()<<"	Worksheet:Draw() drawing objects"<<endl;
	for (int i=0;i<NR_OBJECTS;i++) {
		//kdDebug()<<"DRAW OBJECTS "<<i<<" : width = "<<width<<" height="<<height<<endl;
		line[i]->draw(p,width,height);
		label[i]->draw(this,p,Point(0,0),Point(1,1),width,height,0);
		rect[i]->draw(p,width,height);
		ellipse[i]->draw(p,width,height);
		image[i]->draw(p,(double)width,(double)height);
	}
	p->setPen(NoPen);

	// draw title and timestamp with standard font at the end
	kdDebug()<<"	Worksheet:Draw() drawing title and timestamp"<<endl;
	kdDebug()<<"	title = "<<title<<endl;
	QFont font = mw->defaultFont();
	font.setPointSize((int)(0.5*font.pointSize()));
	p->setFont(font);

	if(title_enabled)
		p->drawText(10,20,title);

	if(timestamp_enabled)
		p->drawText(10,height-10,timestamp.toString(Qt::TextDate));
	kdDebug()<<"Worksheet:Draw() DONE"<<endl;
}

void Worksheet::Export(QString fn, QString format) {
	kdDebug()<<"Worksheet:Export() : filename="<<fn<<" format="<<format<<endl;
	QPixmap pm(X,Y);
	QPainter p;

	pm.fill( background );
	p.begin(&pm);
	Draw(&p,X,Y);

	bool ok;
	int quality = QInputDialog::getInteger(i18n("Export to image"),i18n("Quality (0-100)"),75,1,100,1,&ok);
	if(!ok) return;

	if(plot[api]->Type() == PQWT3D) {
		//TODO : quality
		((PlotQWT3D *) plot[api])->Export(&p,fn,format,X,Y);
	}
	else {
		// TODO : options for eps format?
		pm.save(fn,format,quality);
	}
	p.end();
#ifdef HAVE_UNDO
	undo_snapshot(wundo);
#endif
}

QDomElement Worksheet::saveXML(QDomDocument doc) {
	QDomElement wstag = doc.createElement( "Worksheet" );
	wstag.setAttribute("api",QString::number(api));
	wstag.setAttribute("nr_plots",QString::number(nr_plots));
	
	QDomElement tag = doc.createElement( "Position" );
	tag.setAttribute("x",QString::number(parentWidget()->pos().x()));
	tag.setAttribute("y",QString::number(parentWidget()->pos().y()));
    	wstag.appendChild( tag );
	tag = doc.createElement( "Size" );
	tag.setAttribute("width",QString::number(X));
	tag.setAttribute("height",QString::number(Y));
   	wstag.appendChild( tag );
	tag = doc.createElement( "Title" );
   	wstag.appendChild( tag );
  	QDomText t = doc.createTextNode( title );
    	tag.appendChild( t );
	tag = doc.createElement( "TitleEnabled" );
   	wstag.appendChild( tag );
  	t = doc.createTextNode( QString::number(title_enabled) );
    	tag.appendChild( t );
	tag = doc.createElement( "Background" );
   	wstag.appendChild( tag );
  	t = doc.createTextNode( background.name() );
    	tag.appendChild( t );
	tag = doc.createElement( "Timestamp" );
   	wstag.appendChild( tag );
#if QT_VERSION > 0x030007
  	t = doc.createTextNode( QString::number(timestamp.toTime_t()) );
#else
	QDateTime stoneage(QDate(1970,1,1));
  	t = doc.createTextNode( QString::number(stoneage.secsTo(timestamp)) );
#endif
    	tag.appendChild( t );
	tag = doc.createElement( "TimestampEnabled" );
   	wstag.appendChild( tag );
  	t = doc.createTextNode( QString::number(timestamp_enabled) );
    	tag.appendChild( t );

	// drawing objects
	for(int i=0;i<NR_OBJECTS;i++) {
		// only save used objects
		if(! label[i]->Title().isEmpty()) {
			tag = label[i]->saveXML(doc);
    			wstag.appendChild(tag);
		}
		Point start = line[i]->startPoint() ,end = line[i]->endPoint();
		if(fabs(start.X()-end.X())>1.0e-6 || fabs(start.Y()-end.Y())>1.0e-6) {
			tag = line[i]->saveXML(doc);
    			wstag.appendChild(tag);
		}
		start = rect[i]->startPoint();
		end = rect[i]->endPoint();
		if(fabs(start.X()-end.X())>1.0e-6 || fabs(start.Y()-end.Y())>1.0e-6) {
			tag = rect[i]->saveXML(doc);
    			wstag.appendChild(tag);
		}
		start = rect[i]->startPoint();
		end = rect[i]->endPoint();
		if(fabs(start.X()-end.X())>1.0e-6 || fabs(start.Y()-end.Y())>1.0e-6) {
			tag = ellipse[i]->saveXML(doc);
    			wstag.appendChild(tag);
		}
		if(! image[i]->Name().isEmpty()) {
			tag = image[i]->saveXML(doc);
    			wstag.appendChild(tag);
		}
	}

	// all plots
	for (unsigned int i=0;i<nr_plots;i++) {
		tag = plot[i]->savePlotXML(doc);
    		wstag.appendChild(tag);
	}

	return wstag;
}

void Worksheet::save(QTextStream *t) {
	kdDebug()<<"Worksheet:Save()"<<endl;

	*t<<X<<' '<<Y<<endl;

	// NEW : worksheet settings
	*t<<title<<endl;
	*t<<title_enabled<<endl;
	*t<<background.name()<<endl;
#if QT_VERSION > 0x030007
	*t<<timestamp.toTime_t()<<endl;
#else
	QDateTime stoneage(QDate(1970,1,1));
	*t<<stoneage.secsTo(timestamp)<<endl;
#endif
	*t<<timestamp_enabled<<endl;
	*t<<api<<' '<<nr_plots<<endl;

	// "drawing objects"
	for(int i=0;i<NR_OBJECTS;i++) {
		label[i]->save(t);
		line[i]->save(t);
		rect[i]->save(t);
		ellipse[i]->save(t);
		image[i]->save(t);
	}

	// save all plots
	for (unsigned int i=0;i<nr_plots;i++) {
		*t<<plot[i]->Type()<<endl;
		plot[i]->save(t);
	}
}

void Worksheet::openXML(QDomNode node) {
	int nr_plots=0;
	int nr_label=0, nr_line=0, nr_rect=0, nr_ellipse=0, nr_image=0;
	while(!node.isNull()) {
		QDomElement e = node.toElement();
//		kdDebug()<<"WS TAG = "<<e.tagName()<<endl;
//		kdDebug()<<"WS TEXT = "<<e.text()<<endl;

		if(e.tagName() == "Position") {
			int px = e.attribute("x").toInt();
			int py = e.attribute("y").toInt();
			parentWidget()->move(QPoint(px,py));
			if(px<0 || py<0) {	// fullscreen
				mw->getWorkspace()->tile();
				showMaximized();
			}
		}
		else if(e.tagName() == "Size") {
			X = e.attribute("width").toInt();
			Y = e.attribute("height").toInt();
			resize(X,Y);
			kdDebug()<<"Resizing to "<<X<<' '<<Y<<endl;
		}
		else if(e.tagName() == "Title")
			title = e.text();
		else if(e.tagName() == "TitleEnabled")
			title_enabled = (bool) e.text().toInt();
		else if(e.tagName() == "Background")
			background = QColor(e.text());
		else if(e.tagName() == "Timestamp")
			timestamp.setTime_t(e.text().toInt());
		else if(e.tagName() == "TimestampEnabled")
			timestamp_enabled = (bool) e.text().toInt();
	
		else if(e.tagName() == "Label")
			label[nr_label++]->openXML(e.firstChild());		
		else if(e.tagName() == "Line")
			line[nr_line++]->openXML(e.firstChild());		
		else if(e.tagName() == "Rect")
			rect[nr_rect++]->openXML(e.firstChild());		
		else if(e.tagName() == "Ellipse")
			ellipse[nr_ellipse++]->openXML(e.firstChild());		
		else if(e.tagName() == "Image")
			image[nr_image++]->openXML(e.firstChild());		
		
		else if(e.tagName() == "Plot") {
			newPlot((PType) e.attribute("type").toInt());
			plot[nr_plots++]->openPlotXML(e.firstChild());
		}

		node = node.nextSibling();
	}
}

//! open worksheet : type : old type from version <11
void Worksheet::open(QTextStream *t,int version,PType oldtype) {
	kdDebug()<<"Worksheet::open() : version = "<<version<<endl;
	QString tmp;
	int tmpint, type, tmpnr_plots;

	*t>>X>>Y;

	kdDebug()<<"Dimension "<<X<<' '<<Y<<endl;
	resize(X,Y);

	// NEW : worksheet settings
	if(version > 10) {
		t->readLine();
		title = t->readLine();
		*t>>tmpint;
		title_enabled = (bool)tmpint;
		*t>>tmp;
		background = QColor(tmp);
		t->readLine();
		if(version > 14) {
			*t>>tmpint;
			timestamp.setTime_t(tmpint);
		}
		else {
			tmp = t->readLine();
			kdDebug()<<" timestring "<<tmp<<endl;
			timestamp = QDateTime::fromString(tmp);
		}
		*t>>tmpint;
		timestamp_enabled = (bool)tmpint;
		*t>>api>>tmpnr_plots;

		kdDebug()<<"Title : "<<title<<endl;
		setCaption(title);
		kdDebug()<<"Title enabled : "<<title_enabled<<endl;
		kdDebug()<<"background : "<<background.name()<<endl;
		kdDebug()<<"Timestamp : "<<timestamp.toString()<<endl;
		kdDebug()<<"Timestamp enabled : "<<timestamp_enabled<<endl;
		kdDebug()<<"API/NR_PLOTS = "<<api<<' '<<tmpnr_plots<<endl;
	}

	// drawing objects
	int nr_objects=0;
	if (version > 7)
		nr_objects = 10;
	if (version > 21)
		nr_objects = 100;
	
	for(int i=0;i<nr_objects;i++) {
		label[i]->open(t,version);
		line[i]->open(t,version);
		rect[i]->open(t,version);
		ellipse[i]->open(t,version);
		image[i]->open(t,version);
	}

	// open all plots
	if(version<11) {
		type=oldtype;
		api=0;
		tmpnr_plots=1;
	}

	kdDebug()<<"API = "<<api<<" / NR_PLOTS = "<<nr_plots<<endl;
	for (int i=0;i<tmpnr_plots;i++) {
		if (version>10)
			*t>>type;
		kdDebug()<<" opening PLOT "<<i<<" / Type = "<<type<<endl;
		newPlot((PType)type);
		plot[i]->open(t, version);
	}

	redraw();	// needed for empty plot
}

void Worksheet::addGraph2D(Graph2D *g, PType type) {
	kdDebug()<<"Worksheet::addGraph2D() (Worksheet "<<caption()<<")"<<endl;
	kdDebug()<<"	api = "<<api<<" of "<<nr_plots<<" / type = "<<type<<endl;
	kdDebug()<<"	Graph2D Name = "<<g->Name()<<endl;

	if(plot[api] !=0 && plot[api]->Type() == PQWT3D ) {
		KMessageBox::error(this,i18n("QWT 3D plot only supports one plot per worksheet!"));
		return;
	}

	// add new plot if type mismatch
	if(plot[api] !=0 && plot[api]->Type() != type)
		newPlot(type);
		
	// add to plot
	if(plot[api] == 0)
		newPlot(type);

	plot[api]->getGraphList()->addGraph2D(g);

	resetRanges();

	// set actrange for new plots
	LRange *actrange = plot[api]->ActRanges();
	if (actrange[0].rMax()-actrange[0].rMin() == 0)
		plot[api]->setActRanges(plot[api]->Ranges());

	updatePixmap();
}

void Worksheet::addGraph3D(Graph3D *g, PType type) {
	kdDebug()<<"Worksheet::addGraph3D() : type = "<<type<<endl;
	kdDebug()<<"	g->Number() = "<<g->Number()<<endl;

	// add new plot if type mismatch
	if(plot[api] !=0 && plot[api]->Type() != type) {
		kdDebug()<<"	add new plot"<<endl;
		newPlot(type);
	}
		
	// add to plot
	if(plot[api] != 0) {
		kdDebug()<<"	add to active plot"<<endl;
		plot[api]->getGraphList()->addGraph3D(g);
	}
	else {
		kdDebug()<<"	no plot found"<<endl;
		// TODO : what if 2derr ?
		newPlot(type);
		plot[nr_plots-1]->getGraphList()->addGraph3D(g);
	}

	resetRanges();

	LRange *actrange = plot[api]->ActRanges();
	if (actrange[0].rMax()-actrange[0].rMin() == 0)
		plot[api]->setActRanges(plot[api]->Ranges());

	updatePixmap();
}

void Worksheet::addGraphM(GraphM *g, PType type) {
	kdDebug()<<"Worksheet::addGraphM() : type = "<<type<<endl;
	bool resizing=false;

	if (plot[api]==0 || plot[api]->Type() != type) {
		kdDebug()<<"	opening new plot"<<endl;
		switch (type) {
		case PSURFACE : {
			Plot2DSurface *newplot = new Plot2DSurface(this);
			newplot->getGraphList()->addGraphM(g);
			newplot->setSize(width()/(double)X,height()/(double)Y);
			plot[api] = newplot;
			plot[api]->setType(PSURFACE);
			resizing = true;	// resize later
			}; break;
		case P3D : {
			Plot3D *newplot = new Plot3D(this);
			newplot->getGraphList()->addGraphM(g);
			newplot->setSize(width()/(double)X,height()/(double)Y);
			plot[api] = newplot;
			plot[api]->setType(P3D);
			resizing = true;	// resize later
			}; break;
		case PQWT3D: {
			PlotQWT3D *newplot = new PlotQWT3D(this);
			newplot->getGraphList()->addGraphM(g);
			newplot->draw(0,X,Y);		// update QWT3D plot
			plot[api] = newplot;
			plot[api]->setType(PQWT3D);
			resizing = true;
			}; break;
		 default : break;
		 }
		nr_plots++;
	}
	else {
		kdDebug()<<"	using old plot"<<endl;
		switch (type) {
		case PSURFACE : case P3D : {
			plot[api]->getGraphList()->addGraphM(g);
			resizing = true;	// resize later
			}; break;
		case PQWT3D: {
			plot[api]->getGraphList()->addGraphM(g);
			plot[api]->draw(0,X,Y);		// update QWT3D plot
			}; break;
		default: break;
		}
	}
	
	resetRanges();

	LRange *actrange = plot[api]->ActRanges();
	kdDebug()<<"	ACTRANGE x/y/z = "<<actrange[0].rMin()<<'/'<<actrange[0].rMax()<<' '<<actrange[1].rMin()<<'/'<<actrange[1].rMax()<<' '<<actrange[2].rMin()<<'/'<<actrange[2].rMax()<<endl;
	if (actrange[0].rMax()-actrange[0].rMin() == 0) {
		kdDebug()<<"	Calling setActRanges()"<<endl;
		plot[api]->setActRanges(plot[api]->Ranges());
	}

	updatePixmap();
}

void Worksheet::addGraph4D(Graph4D *g) {
	kdDebug()<<"Worksheet::addGraph4D()"<<endl;
	kdDebug()<<"	g->Number() = "<<g->Number()<<endl;
	PType type = P2D;
	
	// add new plot if type mismatch
	if(plot[api] !=0 && plot[api]->Type() != type)
		newPlot(type);
	
	// add to plot
	if(plot[api] != 0) {
		plot[api]->getGraphList()->addGraph4D(g);
	}
	else {
		newPlot(type);
		plot[api]->getGraphList()->addGraph4D(g);
	}

	resetRanges();

	LRange *actrange = plot[api]->ActRanges();
	if (actrange[0].rMax()-actrange[0].rMin() == 0)
		plot[api]->setActRanges(plot[api]->Ranges());

	updatePixmap();
}

void Worksheet::addGraphIMAGE(GraphIMAGE *g) {
	kdDebug()<<"Worksheet::addGraphIMAGE()"<<endl;
	kdDebug()<<"	g->Number() = "<<g->Number()<<endl;
	PType type = PSURFACE;
	
	// add new plot if type mismatch
	if(plot[api] !=0 && plot[api]->Type() != type)
		newPlot(type);
	
	kdDebug()<<"OK"<<endl;
	// add to plot
	if(plot[api] != 0) {
		plot[api]->getGraphList()->addGraphIMAGE(g);
	kdDebug()<<"OK"<<endl;
	}
	else {
		newPlot(type);
	kdDebug()<<"OK"<<endl;
		plot[api]->getGraphList()->addGraphIMAGE(g);
	}

	kdDebug()<<"OK"<<endl;
	resetRanges();

	kdDebug()<<"OK"<<endl;
	LRange *actrange = plot[api]->ActRanges();
	if (actrange[0].rMax()-actrange[0].rMin() == 0)
		plot[api]->setActRanges(plot[api]->Ranges());

	kdDebug()<<"OK"<<endl;
	updatePixmap();
	kdDebug()<<"Worksheet::addGraphIMAGE() DONE"<<endl;
}

void Worksheet::addGraphL(GraphL *g, PType type) {
	kdDebug()<<"Worksheet::addGraphL() (Worksheet "<<caption()<<")"<<endl;
	kdDebug()<<"	api = "<<api<<" of "<<nr_plots<<" / type = "<<type<<endl;
	kdDebug()<<"	GraphL Name = "<<g->Name()<<endl;

	if(plot[api] !=0 && plot[api]->Type() == PQWT3D ) {
		KMessageBox::error(this,i18n("QWT 3D plot only supports one plot per worksheet!"));
		return;
	}

	// add new plot if type mismatch
	if(plot[api] !=0 && plot[api]->Type() != type)
		newPlot(type);
		
	// ad to plot
	if(plot[api] != 0) {
		plot[api]->getGraphList()->addGraphL(g);
	}
	else {
		newPlot(type);
		
		plot[api]->getGraphList()->addGraphL(g);
	}

	resetRanges();

	// set actrange for new plots
	LRange *actrange = plot[api]->ActRanges();
	if (actrange[0].rMax()-actrange[0].rMin() == 0)
		plot[api]->setActRanges(plot[api]->Ranges());

	updatePixmap();
}

// reset xrange, yrange, zrange
void Worksheet::resetRanges() {
	kdDebug()<<"Worksheet::resetRanges()"<<endl;
	double xmin=0,xmax=1,ymin=0,ymax=1,zmin=0,zmax=1;
	GraphList *graphlist = plot[api]->getGraphList();
	for (unsigned int i=0;i<graphlist->Number();i++) {
		if(!graphlist->getGraph(i)->isShown())
			continue;	// dont use hidden graphs
			
		GRAPHType s = graphlist->getStruct(i);

		kdDebug()<<"	Graph "<<i<<" / Type = "<<s<<endl;

		LRange xrange,yrange,zrange;
		if(s == GRAPH2D) {
			Graph2D *g = graphlist->getGraph2D(i);
			xrange = g->Range(0);
			yrange = g->Range(1);
		}
		else if(s == GRAPH3D) {
			Graph3D *g = graphlist->getGraph3D(i);
			PType t = plot[api]->Type();

			xrange = g->Range(0);
			if(t == P2D)	// x-y-dy
				yrange = g->ErrorDYRange();
			else
				yrange = g->Range(1);
			zrange = g->Range(2);
		}
		else if(s == GRAPHM) {
			GraphM *g = graphlist->getGraphM(i);
			xrange = g->Range(0);
			yrange = g->Range(1);
			zrange = g->Range(2);
		}
		else if(s == GRAPH4D) {
			Graph4D *g = graphlist->getGraph4D(i);
			xrange = g->ErrorDXRange();
			yrange = g->ErrorDYRange();
		}
		else if(s == GRAPHIMAGE) {
			GraphIMAGE *g = graphlist->getGraphIMAGE(i);
			xrange = g->Range(0);
			yrange = g->Range(1);
		}
		if (i==0) {	// first graph
			xmin=xrange.rMin();
			xmax=xrange.rMax();
			ymin=yrange.rMin();
			ymax=yrange.rMax();
			
			if (s == GRAPH3D || s == GRAPHM) {
				zmin=zrange.rMin();
				zmax=zrange.rMax();
			}
		}
		else {
			xrange.rMin()<xmin?xmin=xrange.rMin():0;
			xrange.rMax()>xmax?xmax=xrange.rMax():0;
			yrange.rMin()<ymin?ymin=yrange.rMin():0;
			yrange.rMax()>ymax?ymax=yrange.rMax():0;

			if (s == GRAPH3D || s == GRAPHM) {
				zrange.rMin()<zmin?zmin=zrange.rMin():0;
				zrange.rMax()>zmax?zmax=zrange.rMax():0;
			}
		}
	}
	kdDebug()<<"	xmin/xmax "<<xmin<<' '<<xmax<<endl;
	kdDebug()<<"	ymin/ymax "<<ymin<<' '<<ymax<<endl;
	kdDebug()<<"	zmin/zmax "<<zmin<<' '<<zmax<<endl;

	LRange range[3];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);

	plot[api]->setRanges(range);

	mw->setModified();
}

// check values of scale
void Worksheet::checkRanges(TScale scale, double *min, double *max) {
	int range = 100;
	switch(scale) {
	case LOG10: case LOG2 : case LN:
		if(*min<=0)
			*min=fabs(*max)/range;
		if(*max<=0)
			*max=(*min)*range;
		break;
	case SQRT:
		if(*min<0)
			*min=0;
		if(*max<0)
			*max=1;
		break;
	default: break;
	}
}

// calculate part for range for shift/scale
double Worksheet::partRanges(TScale scale, double min, double max) {
	double r=0;
	
	switch(scale) {
	case LINEAR: 	r=fabs(max-min)/4.0; break;
	case LOG10:	r=pow(10,log10(max/min)/4.0); break;
	case LOG2:	r=pow(2,log2(max/min)/4.0); break;
	case LN:		r=pow(M_E,log(max/min)/4.0); break;
	case SQRT:
		//TODO
		r=fabs(max-min)/4.0;
		break;
	case SX2:
		//TODO
		break;
	}
	
	return r;
}
