/* 
   Copyright (c) 1991-1999 Thomas T. Wetmore IV

   Permission is hereby granted, free of charge, to any person
   obtaining a copy of this software and associated documentation
   files (the "Software"), to deal in the Software without
   restriction, including without limitation the rights to use, copy,
   modify, merge, publish, distribute, sublicense, and/or sell copies
   of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be
   included in all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
   NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
   BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
   ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
*/
/* modified 05 Jan 2000 by Paul B. McBride (pmcbride@tiac.net) */
/*=============================================================
 * delete.c -- Removes person and family records from database 
 * Copyright(c) 1992-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93    2.3.5 - 15 Aug 93
 *   3.0.0 - 30 Jun 94    3.0.2 - 10 Dec 94
 *   3.0.3 - 21 Jan 96
 *===========================================================*/

#include "sys_inc.h"
#include "llstdlib.h"
#include "table.h"
#include "translat.h"
#include "gedcom.h"
#include "indiseq.h"
#include "liflines.h"
#include "feedback.h"

#include "llinesi.h"

extern STRING idpdel, cfpdel, cffdel;
extern STRING idfrmv, idfrsp, idfrch;
extern STRING idcrmv, ntchld, ntprnt, idsrmv, idsrmf, normls, cfcrmv;
extern STRING okcrmv, ntsinf, ntcinf, cfsrmv, oksrmv, ronlye, idcrmf;

/*=====================================================
 * choose_and_remove_family -- Choose & delete a family
 *  (remove all members, and delete F record)
 *===================================================*/
void
choose_and_remove_family (void)
{
	NODE fam, node, indi;
	INDISEQ spseq, chseq;
	STRING tag, key;
	char confirm[512], members[64];
	STRING cfptr=confirm;
	INT cflen=sizeof(confirm);

	fam = ask_for_fam_by_key(idfrmv, idfrsp, idfrch);
	if (!fam)
		return;

	/* get list of spouses & children */
	spseq = create_indiseq_null(); /* spouses */
	chseq = create_indiseq_null(); /* children */
	for (node=nchild(fam); node; node = nsibling(node)) {
		tag = ntag(node);
		if (eqstr("HUSB", tag) || eqstr("WIFE", tag)) {
			key = strsave(rmvat(nval(node)));
			append_indiseq_null(spseq, key, NULL, TRUE, TRUE);
		} else if (eqstr("CHIL", tag)) {
			key = strsave(rmvat(nval(node)));
			append_indiseq_null(chseq, key, NULL, TRUE, TRUE);
		}
	}

	/* build confirm string */
	sprintf(members, " (%s: %d spouse(s), %d child(ren))", 
		fam_to_key(fam), ISize(spseq), ISize(chseq));
	llstrcatn(&cfptr, cffdel, &cflen);
	llstrcatn(&cfptr, members, &cflen);

	if (ask_yes_or_no(confirm)) {

		if (ISize(spseq)+ISize(chseq) == 0) {
			/* handle empty family */
			remove_empty_fam(fam);
		}
		else {
			/* the last remove command will delete the family */
			FORINDISEQ(spseq, el, num)
				indi = key_to_indi(skey(el));
				remove_spouse(indi, fam);
			ENDINDISEQ

			FORINDISEQ(chseq, el, num)
				indi = key_to_indi(skey(el));
				remove_child(indi, fam);
			ENDINDISEQ
		}
	}
	
	remove_indiseq(spseq);
	remove_indiseq(chseq);
}
/*================================================================
 * delete_indi -- Delete person and links; if this leaves families
 *   with no links, remove them
 *  indi:  [in]  person to remove - (if null, will ask for person)
 *  conf:  [in]  have user confirm ?
 *==============================================================*/
void
delete_indi (NODE indi, BOOLEAN conf)
{
	/* prompt if needed */
	if (!indi && !(indi = ask_for_indi_old(idpdel, NOCONFIRM, DOASK1)))
		return;
	/* confirm if caller desired */
	if (conf && !ask_yes_or_no(cfpdel)) return;

	/* alright, we finished the UI, so delegate to the internal workhorse */
	remove_indi(indi);
}
/*===========================================
 * choose_and_remove_spouse -- Remove spouse 
 *  from family (prompting for spouse and/or family
 *  if NULL passed)
 *  nolast: don't remove last member of family?
 *=========================================*/
BOOLEAN
choose_and_remove_spouse (NODE indi, NODE fam, BOOLEAN nolast)
{
	if (readonly) {
		message(ronlye);
		return FALSE;
	}

/* Identify spouse to remove */
	if (!indi) indi = ask_for_indi_old(idsrmv, NOCONFIRM, NOASK1);
	if (!indi) return FALSE;
	if (!FAMS(indi)) {
		message(ntprnt);
		return FALSE;
	}

/* Identify family to remove spouse from */
	if (!fam) fam = choose_family(indi, "e", idsrmf, TRUE);
	if (!fam) return FALSE;
	if (nolast && num_fam_xrefs(fam) < 2) {
		message(normls);
		return FALSE;
	}
	if (!ask_yes_or_no(cfsrmv)) return FALSE;

	/* call internal workhorse remove_spouse() to do the actual removal */
	if (!remove_spouse(indi, fam)) {
		message(ntsinf);
		return FALSE;
	}
	message(oksrmv);
	return TRUE;
}
/*===========================================
 * choose_and_remove_child -- Remove child
 *  from family (prompting for child and/or family
 *  if NULL passed)
 *  nolast: don't remove last member of family?
 *=========================================*/
BOOLEAN
choose_and_remove_child (NODE indi, NODE fam, BOOLEAN nolast)
{
	if (readonly) {
		message(ronlye);
		return FALSE;
	}
		
/* Identify child and check for FAMC nodes */
	if (!indi) indi = ask_for_indi_old(idcrmv, NOCONFIRM, NOASK1);
	if (!indi) return FALSE;
	if (!FAMC(indi)) {
		message(ntchld);
		return FALSE;
	}

/* Identify family to remove child from */
	if (!fam) fam = choose_family(indi, "e", idcrmf, FALSE);
	if (!fam) return FALSE;
	if (nolast && num_fam_xrefs(fam) < 2) {
		message(normls);
		return FALSE;
	}
	if (!ask_yes_or_no(cfcrmv)) return TRUE;

	if (!remove_child(indi, fam)) {
		message(ntcinf);
		return FALSE;
	}

	message(okcrmv);
	return TRUE;
}
