/****************************************************************************
**
** $Id: lincvs.cpp,v 1.124 2001/11/04 13:01:54 joseh Exp $
**
** Copyright (C) 2001 The LinCVS development team.
**    Tilo Riemer <riemer@lincvs.org>
**    Falk Brettschneider <gigafalk@yahoo.com>
**    Wim Delvaux <wim.delvaux@chello.be>
**    Jose Hernandez <joseh@tesco.net>
**    Helmut Koll <HelmutKoll@web.de>
**    Sven Trogisch <trogisch@iapp.de>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
*****************************************************************************/

#include "config.h"
#include "ac_system_defs.h"

#include <unistd.h>

//----------------------------------------------------------------------------

#include <qapplication.h>
#include <qtextcodec.h>
#include <qtabwidget.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qmultilineedit.h>
#include <qpopupmenu.h>
#include <qinputdialog.h>
#include <qkeycode.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qaccel.h>
#include <qsplitter.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qstatusbar.h>
#include <qregexp.h> 
#include <qfont.h>
#include <qtooltip.h>


#include "lincvs.h"
#include "dialogs.h"
#include "cfg.h"
#include "PixmapCache.h"
#include "colortab.h"
#include "cvslistview.h"
#include "noncvslistview.h"
#include "CmdThread.h"
#include "directory.h"
#include "DiffDialogImpl.h"
#include "LogDialogImpl.h"
#include "AboutDialogImpl.h"
#include "AnnotateDialogImpl.h"
#include "CommitDialogImpl.h"
#include "HistoryDialogImpl.h"
#include "ProfilesDialogImpl.h"
#include "PropertiesDialogImpl.h"
#include "SettingsDialogImpl.h"
#include "TagDialogImpl.h"
#include "AnnotateGrepLineDialogImpl.h"
#include "AnnotateGrepLinesDialogImpl.h"
#include "MergeDialog.h"


Config *conf;


ApplicationWindow::ApplicationWindow()
	: KMainWindow(0, "LinCVS main window")
{
   createTmpDir();

   sshAgentIsRunning = false;
   
   m_coImportLastAccessMethod = -1;
   QString tmpCvsRsh = getenv("CVS_RSH");

   m_bExternalDiffIsRunning = false;
   m_curDiffFileName = "";      //used for external diff
   m_curTmpDiffFileName = "";   //used for external diff

   
   m_curCommentFileName = "";
   
/*! get config file */
   const char filename[] = "lincvsrc"; // becomes -> .lincvsrc
   conf = new Config(&filename[0],
                     &m_coImportUser,
                     &m_coImportServer, 
                     &m_coImportRepository,
                     &m_coLinCVSProfile);

   if (bUseSmallIcons) setIcon (findEmbeddedPixmap ("lincvs-16"));
   else setIcon (findEmbeddedPixmap ("lincvs-32"));

#ifndef KDE2
   //workaround for qt 2.2.4: we must call setstyle before creating any toolbar
   SettingsDialogImpl::setStyle(CVSSTYLE);
#endif

   globalListViewsEnabled = false;
   
   m_pCoImportThread = 0;
   m_curPic = 0;
   m_pCurCvsDir = 0;
   if (cvsRsh.length() == 0) cvsRsh = tmpCvsRsh;   //use CVS_RSH
   if (cvsRsh.length() == 0) cvsRsh = "rsh";   //if cvsRsh is empty use rsh now
   
   m_uninterruptible = false;
   globalStopAction = false;
   m_showWarning = true;
   m_cvsThreadIsRunning = false;
   m_coThreadIsRunning = false;
   m_importThreadIsRunning = false;
   
   StatTimerEnable = true;
   initMenuAndToolbar();
   initComponents();
   globalStatusBar = m_pStatusBarLabel;
   
   m_pStatusBarLabel->setText( tr("Scan Project Directories"));

/*! show help screen after installation */
   if (!conf->fileFound)  help();

   QTimer::singleShot(100, this, SLOT(initialDirScanning()));

	QFont font;
	font.setFamily(CVSFONT.NAME);
	font.setPointSize(CVSFONT.POINTSIZE);
	font.setWeight(CVSFONT.WEIGHT);
	font.setItalic(CVSFONT.ITALIC);
	font.setCharSet(QFont::charSetForLocale());
	QApplication::setFont(font, true);
}

//----------------------------------------------------------------------------

ApplicationWindow::~ApplicationWindow()
{
   GEOM_X = geometry().x();
   GEOM_Y = geometry().y();
   GEOM_WIDTH  = geometry().width();
   GEOM_HEIGHT = geometry().height();

   SPLITTER_POS_H = m_pWorkBenchPane->width();
   SPLITTER_POS_V = m_pWorkFilePane->height();

   WORKBENCH_COLUMN_WIDTH = m_pWorkBenchTree->columnWidth(0);

   CVSLISTVIEW_TABWIDTH_NAME = m_pCvsFileList->columnWidth(0);
   CVSLISTVIEW_TABWIDTH_REV = m_pCvsFileList->columnWidth(1);
   CVSLISTVIEW_TABWIDTH_STICKY = m_pCvsFileList->columnWidth(2);
   CVSLISTVIEW_TABWIDTH_STATE = m_pCvsFileList->columnWidth(3);
   CVSLISTVIEW_TABWIDTH_COMMIT = m_pCvsFileList->columnWidth(4);
   CVSLISTVIEW_TABWIDTH_MODLOCAL = m_pCvsFileList->columnWidth(5);
   CVSLISTVIEW_TABWIDTH_MODUTC = m_pCvsFileList->columnWidth(6);
   NONCVSLISTVIEW_TABWIDTH_NAME = m_pNonCvsFileList->columnWidth(0);
   NONCVSLISTVIEW_TABWIDTH_MODLOCAL = m_pNonCvsFileList->columnWidth(1);
   NONCVSLISTVIEW_TABWIDTH_MODUTC = m_pNonCvsFileList->columnWidth(2);

   writeCfg();
   delete m_pCommitDlg;

   //remove any temporary file
   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);
   if (m_curTmpDiffFileName.length()) QFile::remove(m_curTmpDiffFileName);

   //remove all tmp files from external diff
   QDir d(tmpDir);
   d.setFilter( QDir::Files | QDir::NoSymLinks );

   const QStringList list = d.entryList();
   int i;  
   for (i = 0; i < list.count(); i++) {
       if (list[i].find("tmp.") == 0) {//found temp file
	   QFile tmpFile(tmpDir + "/" + list[i]);
	   cout << "remove " << tmpFile.name() << "\n";
	   tmpFile.remove();
       }
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::initialDirScanning()
{
   for(unsigned int i = 0; i < m_projectNameList.count(); i++) {
     if(globalStopAction){
         m_pStatusBarLabel->setText( tr("Scanning aborted") );
         QStringList tmpList;
         unsigned int j;
         for(j = i; j < m_projectNameList.count(); j++) {
            tmpList.append(m_projectNameList[j]);
         }
         for(j = 0; j < tmpList.count(); j++) {
            m_projectNameList.remove(tmpList[j]);
            conf->rmItem("/WORKBENCH", (const char*)tmpList[j],  m_projectNameList);
         }

         i = m_projectNameList.count();
      }
      else {
         addProject(m_projectNameList[i]);
      }
   }
   
   m_lastOpenedDir = QDir::homeDirPath();
   
   m_pStatusBarLabel->setText(tr("Ready"));
   globalStopAction = false;
   globalListViewsEnabled = true;
}

//----------------------------------------------------------------------------

void ApplicationWindow::initMenuAndToolbar()
{
	// toolbar
	m_pTools = new cvsToolbar (this);

	// menu
	QPopupMenu *project = new QPopupMenu(this);
	menuBar()->insertItem( tr("&Project"), project);
	project->insertItem( tr("&Add to workbench..."), this, SLOT(addProjectSlot()), Key_F5);
	project->insertItem( tr("&Checkout..."), this, SLOT(checkOut()), Key_F3);
	project->insertItem( tr("&Import..."), this, SLOT(import()), Key_F4);
	project->insertSeparator();
	project->insertItem(tr("&Exit"), qApp, SLOT(closeAllWindows()), CTRL+Key_Q);
   
	QPopupMenu *options = new QPopupMenu(this);
	menuBar()->insertItem(tr("&Options"), options);
	options->insertItem(tr("&Settings..."), this, SLOT(AppSettings()), CTRL+Key_S);
	options->insertSeparator();

	QPopupMenu * statUpd = new QPopupMenu( this );
	statUpd->setCheckable( TRUE );
	options->insertItem(tr("Status &Update"), statUpd);
	statUpd->insertItem(tr("Enable"), m_pTools, 
			    SLOT(setStatTimerEnable()), CTRL+Key_E);
	statUpd->insertItem(tr("Disable"), m_pTools, 
			    SLOT(setStatTimerDisable()), CTRL+Key_D);
	statUpd->insertItem(tr("Immediate"), this, 
			    SLOT(checkStatus()), CTRL+Key_U);

	options->insertSeparator();
	options->insertItem( tr("&Profiles..."), this, SLOT(profiles()), CTRL+Key_P);

   //this one is the last menu!
   QPopupMenu * help = new QPopupMenu( this );
   menuBar()->insertSeparator();
   menuBar()->insertItem( tr("&Help"), help );

   help->insertItem( tr("&About"), this, SLOT(about()));
   help->insertItem( tr("Help"), this, SLOT(help()), Key_F1);
   help->insertSeparator();
   help->insertItem( tr("About&Qt"), this, SLOT(aboutQt()));


   // StatusBar
   m_pStatusBarLabel = new QLabel(statusBar());
   m_pProgressWidget = new QLabel(statusBar());
   m_pStatusBarLabel->setMinimumHeight(22);
   m_pProgressWidget->setFixedSize(32, 16);
   statusBar()->addWidget(m_pStatusBarLabel, 1, true);
   statusBar()->addWidget(m_pProgressWidget, 0, true);

   for (int i=0; i<NUM_ANIM_PICS; i++)
	   m_progressIcon[i] = findEmbeddedPixmap (QString().sprintf("Connect32x16-%02d", i));

   m_pProgressWidget->setPixmap(m_progressIcon[0]);
}

//----------------------------------------------------------------------------

void ApplicationWindow::initComponents()
{
   resize( 600, 400 );

   //create menu for cvsdirlistview
   QPopupMenu *menu = new QPopupMenu(this);

   menu->insertItem (findEmbeddedPixmap ("FolderStatus16x16"),
		     tr("Status"), this, SLOT(statusDir()),
		     0, DP_Status);

   menu->insertItem (findEmbeddedPixmap ("FolderStatus16x16"),
		     tr("Query Update"), this, SLOT(queryUpdateDir()), 
		     0, DP_QueryUpdate);

   menu->insertItem (tr("History..."), this, SLOT(historyDir()), 
		     0, DP_History);

   menu->insertSeparator();

   menu->insertItem (findEmbeddedPixmap ("Update16x16"),
		     tr("Update"), this, SLOT(updateDir()), 
		     0, DP_Update);

   menu->insertItem (findEmbeddedPixmap ("Commit16x16"),
		     tr("Commit..."), this, SLOT(commitDir()), 
		     0, DP_Commit);

   menu->insertItem (findEmbeddedPixmap ("Tag16x16"),
		     tr("Tag..."), this, SLOT(tagDir()), 
		     0, DP_Tag);
   menu->insertSeparator();
   menu->insertItem(tr("Login"), this, SLOT(loginDir()), 
   		0, DP_Login);
   menu->insertSeparator();

   menu->insertItem (findEmbeddedPixmap ("Refresh16x16"),
		     tr("Reread Directory"), this, SLOT(rereadProject()), 
		     0, DP_ReRead);
   menu->insertItem (findEmbeddedPixmap ("Delete16x16"),
		     tr("Remove From Workbench"), this, SLOT(removeProjectSlot()), 
		     0, DP_RemoveFromWB );
   menu->insertItem(findEmbeddedPixmap ("Enable16x16"),
		    tr("Enable Directory"), this, SLOT(enableProjectSlot()), 
		    0, DP_Enable );
   menu->insertItem(findEmbeddedPixmap ("Disable16x16"),
		    tr("Disable Directory"), this, SLOT(disableProjectSlot()), 
		    0, DP_Disable );
   menu->insertSeparator();
   menu->insertItem(tr("Properties..."), this, SLOT(showDirProperties()), 
   		0, DP_Properties );

   menu->insertSeparator();
   { QPopupMenu * RemoveMenu = new QPopupMenu(this);
     menu->insertItem(tr("Remove"), RemoveMenu, DP_RemoveDirectory);
     RemoveMenu->insertItem(tr("From repository"), 
        this, SLOT(removeDirFromRep()) );
     RemoveMenu->insertItem(tr("From repository and disk"), 
        this, SLOT(removeDirFromRepAndDisk()) );
   }

   menu->insertSeparator();
   menu->insertItem(tr("Update Other Revisions..."), this, SLOT(updateRevDir()), 
		    0, DP_RevisionUpdate);

   /**************************************************************
    * Add workbench pane to the main window
    **************************************************************/
   QSplitter *m_pHorzSplitter = new QSplitter (QSplitter::Horizontal, this);
   m_pWorkBenchPane = new QWidget (m_pHorzSplitter, "WorkBenchPane");
   m_pWorkBenchTree = new CvsDirListView (menu, m_pWorkBenchPane);
   m_pWorkBenchTree->setRootIsDecorated (true);
   m_pWorkBenchTree->addColumn (tr ("Workbench"));
   m_pWorkBenchTree->setColumnWidth (0,WORKBENCH_COLUMN_WIDTH);
   m_pTools->addWhatsThisText (m_pWorkBenchTree, m_pTools->wBench);
   setCentralWidget (m_pHorzSplitter);
   QVBoxLayout *CvsDirListViewLayout = new QVBoxLayout (m_pWorkBenchPane); 
   CvsDirListViewLayout->setResizeMode (QLayout::FreeResize);
   CvsDirListViewLayout->setSpacing (6);
   CvsDirListViewLayout->setMargin (6);
   CvsDirListViewLayout->addWidget (m_pWorkBenchTree);
   QValueList <int> HorzSplitterPos;
   HorzSplitterPos.append (SPLITTER_POS_H);
   m_pHorzSplitter->setResizeMode (m_pWorkBenchPane, QSplitter::KeepSize);
   m_pHorzSplitter->setSizes (HorzSplitterPos);

   //create menu for cvsfilelistview
   menu = new QPopupMenu(this);

   menu->insertItem (findEmbeddedPixmap ("FileStatus16x16"),
		     tr("Status"), this, SLOT(statusFile()));

   menu->insertItem (findEmbeddedPixmap ("FileStatus16x16"),
		     tr("Query Update"), this, SLOT(queryUpdateFile()));

   menu->insertItem(tr("Rename"), this, SLOT(renameFile()));
   
   menu->insertItem (findEmbeddedPixmap ("LogTree16x16"),
		     tr("Log tree"), this, SLOT(logFile()));
   menu->insertItem (tr("Diff side-by-side"),
		     this, SLOT(diffFileSideBySide()));
   menu->insertItem (tr("Diff to console"),
		     this, SLOT(diffFile()));
   
   menu->insertItem(tr("History"), this, SLOT(historyFile()));

   menu->insertSeparator();

   /* In future, some of these menu options could be moved to
    * the main toolbar to reduce the size of the context menu.
    * In that case we will need to enable and disable the tool
    * buttons as appropriate.
    */
   menu->insertItem (findEmbeddedPixmap ("FileModified16x16"),
   		     tr("Edit"), this, SLOT(cvsEditFile()));
   menu->insertItem (findEmbeddedPixmap ("FileUnchanged16x16"),
   		     tr("Unedit"), this, SLOT(cvsUneditFile()));
   menu->insertItem (findEmbeddedPixmap ("Update16x16"),
		     tr("Update"), this, SLOT(updateFile()));
   menu->insertItem (findEmbeddedPixmap ("Commit16x16"),
		     tr("Commit..."), this, SLOT(commitFile()));
   menu->insertItem (findEmbeddedPixmap ("Tag16x16"),
		     tr("Tag..."), this, SLOT(tagFile()));
   menu->insertSeparator();

   QPopupMenu * RemoveMenu = new QPopupMenu(this);

   RemoveMenu->insertItem(tr("From repository"), 
                            	this, SLOT(removeFileFromRep()) );
   RemoveMenu->insertItem(findEmbeddedPixmap ("Delete16x16"),
			  tr("From repository and disk"), 
			  this, SLOT(removeFile()));
   RemoveMenu->insertItem(tr("From disk"), 
                              this, SLOT(removeFileFromDisk()) );                              

   menu->insertItem (tr("Remove"), RemoveMenu );
   menu->insertSeparator();
   menu->insertItem(tr("Update Other Revisions..."),
		    this, SLOT(updateRevFile()));

   /**************************************************************
    * Create the work file pane and tab widget on the main window
    **************************************************************/
   QSplitter *m_pVertSplitter = new QSplitter(QSplitter::Vertical, m_pHorzSplitter);
   m_pWorkFilePane = new QWidget (m_pVertSplitter, "WorkFilePane");
   m_pFileViews = new ColorTabWidget (m_pWorkFilePane);
   QVBoxLayout *pWorkFilePaneLayout = new QVBoxLayout (m_pWorkFilePane); 
   pWorkFilePaneLayout->setSpacing (6);
   pWorkFilePaneLayout->setMargin (6);
   pWorkFilePaneLayout->addWidget (m_pFileViews);
   m_pTools->addWhatsThisText(m_pFileViews, m_pTools->tabFileList);
   QValueList <int> VertSplitterPos;
   VertSplitterPos.append (SPLITTER_POS_V);
   m_pVertSplitter->setResizeMode (m_pWorkFilePane, QSplitter::KeepSize);   
   m_pVertSplitter->setSizes (VertSplitterPos);

   /**************************************************************
    * Add cvs file view page to the main window
    **************************************************************/
   QWidget *pCvsListViewPage = new QWidget (m_pFileViews, "CvsListViewPage");
   m_pCvsFileList = new CvsListView (menu, pCvsListViewPage);
   QVBoxLayout *pCvsListViewLayout = new QVBoxLayout (pCvsListViewPage); 
   pCvsListViewLayout->setSpacing (6);
   pCvsListViewLayout->setMargin (6);
   pCvsListViewLayout->addWidget (m_pCvsFileList);
   m_pFileViews->addTab (pCvsListViewPage, new ColorTab (tr("Cvs files")));
   m_pTools->addWhatsThisText (m_pCvsFileList, m_pTools->cvsFile);
   connect (m_pCvsFileList, SIGNAL(doubleClickedFile (QString)),
   	this, SLOT(editFile (QString)));

   /**************************************************************
    * Add non-cvs file view page to the main window
    **************************************************************/
   QWidget *pNonCvsListViewPage = new QWidget(m_pFileViews, "NonCvsListViewPage");
   m_pNonCvsFileList = new NonCvsListView(pNonCvsListViewPage);
   QVBoxLayout *pNonCvsListViewLayout = new QVBoxLayout (pNonCvsListViewPage); 
   pNonCvsListViewLayout->setSpacing (6);
   pNonCvsListViewLayout->setMargin (6);
   pNonCvsListViewLayout->addWidget (m_pNonCvsFileList);
   m_pFileViews->addTab (pNonCvsListViewPage, new ColorTab (tr ("Non-Cvs files")));

   m_pTools->addWhatsThisText(m_pNonCvsFileList, m_pTools->nonCvsFile);
   connect( m_pNonCvsFileList, SIGNAL( itemCount( int ) ),
   	this, SLOT( updateColor( int ) ) );
   connect( m_pNonCvsFileList, SIGNAL( addFiles() ),
   	this, SLOT( addSelectedFiles() ) );
   connect( m_pNonCvsFileList, SIGNAL( addBinFiles() ),
   	this, SLOT( addSelectedBinFiles() ) );
   connect (m_pNonCvsFileList, SIGNAL (deleteNonCvsFiles ()),
	    this, SLOT (removeNonCvsFileFromDisk ()));
   connect( m_pNonCvsFileList, SIGNAL( doubleClickedFile( QString ) ),
   	this, SLOT( editFile( QString ) ) );

   /**************************************************************
    * Create the message pane widget on the main window
    **************************************************************/
   QWidget *pMessagePane = new QWidget (m_pVertSplitter, "MessagePane" );
   m_pMessages = new QMultiLineEdit (pMessagePane);
   QVBoxLayout *pMessagePaneLayout = new QVBoxLayout (pMessagePane); 
   pMessagePaneLayout->setSpacing (6);
   pMessagePaneLayout->setMargin (6);
   pMessagePaneLayout->addWidget (m_pMessages);
   m_pMessages->setReadOnly (true);
   m_pTools->addWhatsThisText(m_pMessages, m_pTools->status);
   
   /**************************************************************
    * Create modal instance commitDlg
    **************************************************************/
   m_pCommitDlg = new CommitDialogImpl (this, 0, true);
   
   connect(&m_timer, SIGNAL(timeout()), this, SLOT(checkStatus()));
   connect(&m_checkCvsResultTimer, SIGNAL(timeout()), this, SLOT(checkCvsCallResult()));
   connect(&m_progressTimer, SIGNAL(timeout()), this, SLOT(progress()));
   connect(&m_checkCoResultTimer, SIGNAL(timeout()), this, SLOT(checkCoResult()));
   connect(&m_checkImportResultTimer, SIGNAL(timeout()), this, SLOT(checkImportResult()));
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkOut()
{
   CheckOutAndImportDlg *dlg = new CheckOutAndImportDlg(&m_coImportLastAccessMethod,
                                                        &m_coImportUser,
                                                        &m_coImportModule,
                                                        &m_coImportServer,
                                                        &m_coImportRepository,
                                                        &m_coImportLocalDir,
                                                        &m_coLinCVSProfile,
                                                        &m_coImportVendorTag,
                                                        &m_coImportReleaseTag,
                                                        &m_coImportLastComment,
                                                        &m_tagList, 
                                                        false, tr("Checkout Module"), 
                                                        this);   //modal
   if(dlg->exec()){//ok
      QString command;
      QString cvsRoot;
      
      QDir localDir(dlg->localDir());
      if(!localDir.exists()) {
         if(!localDir.mkdir(dlg->localDir())) {
            QMessageBox::warning(0, tr("Warning"), tr("Couldnt create local dir ") 
                                 + dlg->localDir() + ".", 0);
            dlg->adaptStatePointer();   
            delete dlg;
            return;
         }
      }

      m_pStatusBarLabel->setText("");

      m_coImportLocalDirectory = localDir.absPath();

      m_coModule = dlg->module();
      
      bool usingSsh = false;
      command = "";   //initialize

      command += "cd ";
      //command += m_coImportLocalDirectory + " && ";
      QString coImportLocalDirectory = m_coImportLocalDirectory;
      command += coImportLocalDirectory.replace( QRegExp(" "), "\\ " ) + " && ";

      QString repository = dlg->repository();
      removeDoubleSlashes(repository);

      // Remove trainling "/" from the path.
      repository.replace (QRegExp ("/+$"), "");
      
      if(dlg->mode() == MODE_PSERVER){//password server
         //set CVSROOT for pserver mode
         cvsRoot = ":pserver:" + dlg->user() + "@" + dlg->server() + ":";

         if(!isInCvsPass(cvsRoot + repository)){
            login(cvsRoot + repository);
            qApp->processEvents(100);
            sleep(1);
            qApp->processEvents(100);
            
            if(!isInCvsPass(cvsRoot + repository)) {
               QMessageBox::warning(0, tr("Warning"), tr("Login failed."), 0);
               dlg->adaptStatePointer();   
               delete dlg;
               return;
            }
         }
      } else if(dlg->mode() == MODE_RSH) {//remote shell access
         cvsRsh = dlg->cvsRsh();
         if(dlg->rshMode() == RSH_EXT) {
            cvsRoot = ":ext:";
            cvsRoot += dlg->user() + "@" + dlg->server() + ":";

            if (cvsRsh.find("ssh") == 0) {
               if (!sshAgentIsRunning) startSshAgent();
               usingSsh = true;
               cvsRoot += repository;
               command += "set-ssh-env.sh \"env CVS_RSH=" 
		   + cvsRsh + " cvs -z" + COMPRESSION + " -d "
                         + cvsRoot + " co -P ";
               if(!dlg->cvsParameter().isNull()) {
                  command += dlg->cvsParameter() + " ";
               }
               command += m_coModule + "\"";
               m_pMessages->clear();
               m_pMessages->append(command + "\n");
               m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 
            } else command += "env CVS_RSH=" + cvsRsh + " ";
         } else {
            cvsRoot = ":server:";
            cvsRoot += dlg->user() + "@" + dlg->server() + ":";
         }
      }

         
      if (!usingSsh) {
         //set CVSROOT
         cvsRoot += repository;
 
	 command += "cvs -z" + QString().setNum(COMPRESSION) + " -d ";
         command += cvsRoot;
         command += " co -P ";
         if(!dlg->cvsParameter().isNull()) {
            command += dlg->cvsParameter() + " ";
         }
         command += m_coModule;
         m_pMessages->clear();
         m_pMessages->append(command + "\n");
         m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 
         command += " 2>&1";
      }
   
      centralWidget()->setCursor(waitCursor);
      m_pStatusBarLabel->setText(tr("Please wait, Im working ..."));
      m_timer.stop();   
      m_progressTimer.start(PROGRESSINTERVALL);
      globalListViewsEnabled = false;
      globalStopAction = false;
      m_uninterruptible = true;
      m_pTools->m_pStopAction->setEnabled(true);
      menuBar()->setEnabled(false);
      m_coThreadIsRunning = true;

      m_pCoImportThread = new CmdThread(command);
      m_pCoImportThread->start();

      m_checkCoResultTimer.start(CVSCALLINTERVALL);
   }

   dlg->adaptStatePointer();   
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkCoResult()
{
   bool res;
   //laeuft der thread noch?
   if (m_pCoImportThread->finished()) res = false;
   else res = true;

   unsigned int len = m_pCoImportThread->m_outputQueue.count();
   unsigned int i;
   QString tmp;
   
   if (len) {//nicht leer
      for(i = 0; i < len; i++) tmp += m_pCoImportThread->m_outputQueue[i] + "\n";

      tmp.truncate(tmp.length() - 1);   //cut last newline
      m_pMessages->append(tmp);   //append haengtn "\n" an
      m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 

      QStringList::Iterator it = m_pCoImportThread->m_outputQueue.begin();

      m_pCoImportThread->lockOutputQueue ();
      for (i = 0; i < len; i++) it = m_pCoImportThread->m_outputQueue.remove(it);
      m_pCoImportThread->unlockOutputQueue ();
   }

   if (!res && !globalStopAction) finishCheckOut();
}

//----------------------------------------------------------------------------

void ApplicationWindow::finishCheckOut()
{
   //alten zustand wiederherstellen
   m_checkCoResultTimer.stop();
   m_progressTimer.stop();
   m_coThreadIsRunning = false;
   globalStopAction = false;

   delete m_pCoImportThread;
   m_pCoImportThread = 0;

   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);

   //cvs dir is already in workbench?
   bool isInWorkBench = false;
   Directory *myChild = (Directory*)(m_pWorkBenchTree->firstChild());
   while( myChild && !isInWorkBench) {
      if(m_coImportLocalDirectory.find(myChild->fullName()) == 0) {
         isInWorkBench = true;
      }
      else {
         myChild = (Directory*)(myChild->nextSibling());
      }
   }
         
   if(isInWorkBench) {
      myChild->addDir(m_coImportLocalDirectory + "/" + m_coModule);
      myChild->setAllToUpToDate();
      m_pTools->m_pStopAction->setEnabled(false);
      if(globalStopAction) m_pStatusBarLabel->setText(tr("Adding aborted"));
      else m_pStatusBarLabel->setText(tr("Ready"));
         
      QMessageBox::information (0, tr("Checkout Info"), tr("This project is now in workbench."), 0);
   }
   else {
      QDialog *dlgQuestion = new QDialog(0, 0, true);
      dlgQuestion->setCaption("Checkout Info");

      QBoxLayout *topLayer = new QVBoxLayout(dlgQuestion, 5);
      QLabel *label = new QLabel(tr("Add this project to workbench?"), dlgQuestion);
      topLayer->addWidget(label);
      topLayer->addSpacing(15);

      //create background widget and layer for "ok" and "cancel"
      QWidget *w = new QWidget(dlgQuestion);
      topLayer->addWidget(w);
      QBoxLayout *buttonLayer = new QHBoxLayout(w);
      buttonLayer->addStretch();
   
      //ok and cancel
      QPushButton *ok = new QPushButton(w);
      ok->setText(tr("Yes"));
      buttonLayer->addWidget(ok);
      buttonLayer->addStretch();
      QPushButton *cancel = new QPushButton(w);
      cancel->setText(tr("No"));
      buttonLayer->addWidget(cancel);
      buttonLayer->addStretch();

      connect(ok, SIGNAL(clicked()), dlgQuestion, SLOT(accept()));
      connect(cancel, SIGNAL(clicked()), dlgQuestion, SLOT(reject()));
         
      if(dlgQuestion->exec()) {
         Directory * item = new Directory(
         	m_pWorkBenchTree, 
         	m_coImportLocalDirectory, 
         	m_pCvsFileList,
         	m_pNonCvsFileList );
         item->setAllToUpToDate();
         m_projectNameList.append(item->fullName());   //evtl. hier links aufloesen
         conf->addItem("/WORKBENCH", item->fullName());
         m_lastOpenedDir = item->fullName();
         m_lastOpenedDir.truncate(m_lastOpenedDir.findRev("/"));
         m_pTools->m_pStopAction->setEnabled(false);
         if(globalStopAction) m_pStatusBarLabel->setText(tr("Adding aborted"));
         else m_pStatusBarLabel->setText(tr("Ready"));
         item->setOpen(true);
      }
      
      delete dlgQuestion;
   }
   
   m_pMessages->setCursorPosition(0, 0); 

   m_pTools->m_pStopAction->setEnabled(false);
   menuBar()->setEnabled(true);
   m_uninterruptible = false;
   globalListViewsEnabled = true;
   globalStopAction = false;
      
   centralWidget()->unsetCursor();
   startCheckStat();   
}

//----------------------------------------------------------------------------

void ApplicationWindow::import()
{
   CheckOutAndImportDlg *dlg = new CheckOutAndImportDlg(&m_coImportLastAccessMethod,
                                                        &m_coImportUser,
                                                        &m_coImportModule,
                                                        &m_coImportServer,
                                                        &m_coImportRepository,
                                                        &m_coImportLocalDir,
                                                        &m_coLinCVSProfile,
                                                        &m_coImportVendorTag,
                                                        &m_coImportReleaseTag,
                                                        &m_coImportLastComment,
                                                        &m_tagList, 
                                                        true, tr("Import Module"), 
                                                        this);   //modal
   if(dlg->exec()){//ok
      QString command;
      QString cvsRoot;
      
      QDir localDir(dlg->localDir());
      if(!localDir.exists()) {
         QMessageBox::warning(0, tr("Warning"), tr("Directory ") + dlg->localDir() 
                              + tr(" doesnt exist."), 0, 1);
         dlg->adaptStatePointer();   
         delete dlg;
         return;
      }

      m_pStatusBarLabel->setText("");

      m_coImportLocalDirectory = localDir.absPath();

      bool usingSsh = false;
      command = "";   //initialize

      command += "cd ";
//      command += m_coImportLocalDirectory + " && ";
      QString coImportLocalDirectory = m_coImportLocalDirectory;
      command += coImportLocalDirectory.replace( QRegExp(" "), "\\ " ) + " && ";

      QString repository = dlg->repository();
      removeDoubleSlashes(repository);

      // Remove trainling "/" from the path.
      repository.replace (QRegExp ("/+$"), "");
      
      //alas, no option -F supported by cvs
      m_curCommentFileName = "";
      
      if(dlg->mode() == MODE_PSERVER){//password server
         //set CVSROOT for pserver mode
         cvsRoot = ":pserver:" + dlg->user() + "@" + dlg->server() + ":";
         
         if(!isInCvsPass(cvsRoot + repository)){
            login(cvsRoot + repository);
            qApp->processEvents(100);
            sleep(1);
            qApp->processEvents(100);
            
            if(!isInCvsPass(cvsRoot + repository)) {
               QMessageBox::warning(0, tr("Warning"), tr("Login failed."), 0);
               dlg->adaptStatePointer();   
               delete dlg;
               return;
            }
         }
      }   
      else
      if(dlg->mode() == MODE_RSH) {//remote shell access
         cvsRsh = dlg->cvsRsh();
         if(dlg->rshMode() == RSH_EXT) {
            cvsRoot = ":ext:";
            cvsRoot += dlg->user() + "@" + dlg->server() + ":";

            if (cvsRsh.find("ssh") == 0) {
               if (!sshAgentIsRunning) startSshAgent();
               cvsRoot += repository;
               usingSsh = true;
               
               if (m_curCommentFileName.length()) {
                  command += "set-ssh-env.sh \"env CVS_RSH=" 
		      + cvsRsh + " cvs -z" + COMPRESSION + " -d "
                            + cvsRoot;
                  command += " import -F " + m_curCommentFileName + " ";
                  command += dlg->module() + " " + dlg->tag1() + " " + dlg->tag2() + "\"";
               } else {//fallback to option -m
                  command += "set-ssh-commit-env.sh \"env CVS_RSH=" 
		      + cvsRsh + " cvs -z" + COMPRESSION + " -d "
                            + cvsRoot;
                  command += " import -m\" ";
                  command += "\"" + dlg->comment() + "\" ";
                  command += "\"" + dlg->module() + " " + dlg->tag1() + " " + dlg->tag2() + "\"";
               }
               
               m_pMessages->clear();
               m_pMessages->append(command + "\n");
               m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 
            } else command += "env CVS_RSH=" + cvsRsh + " ";
         } else {
            cvsRoot = ":server:";
            cvsRoot += dlg->user() + "@" + dlg->server() + ":";
         }
      }
         
      if(!usingSsh) {
         //set CVSROOT
         cvsRoot += repository;
 
	 command += "cvs -z" + QString().setNum(COMPRESSION) + " -d ";
         command += cvsRoot;

         if (m_curCommentFileName.length()) {
            command += " import -F " + m_curCommentFileName + " ";
         } else {//fallback to option -m
            command += " import -m ";
            command += "\"" + dlg->comment() + "\" ";
         }
         
         command += dlg->module() + " " + dlg->tag1() + " " + dlg->tag2();
         m_pMessages->clear();
         m_pMessages->append(command + "\n");
         m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 
         command += " 2>&1";
      }
      
      centralWidget()->setCursor(waitCursor);
      m_pStatusBarLabel->setText(tr("Please wait, Im working ..."));
      m_timer.stop();   
      m_progressTimer.start(PROGRESSINTERVALL);
      globalListViewsEnabled = false;
      globalStopAction = false;
      m_uninterruptible = true;
      m_pTools->m_pStopAction->setEnabled(true);
      menuBar()->setEnabled(false);
      m_importThreadIsRunning = true;

      m_pCoImportThread = new CmdThread(command);
      m_pCoImportThread->start();

      m_checkImportResultTimer.start(CVSCALLINTERVALL);
   }
   
   dlg->adaptStatePointer();   
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkImportResult()
{
   bool res;
   //laeuft der thread noch?
   if (m_pCoImportThread->finished()) res = false;
   else res = true;

   unsigned int len = m_pCoImportThread->m_outputQueue.count();
   unsigned int i;
   QString tmp;
   
   if (len) {//nicht leer
      for(i = 0; i < len; i++) tmp += m_pCoImportThread->m_outputQueue[i] + "\n";

      tmp.truncate(tmp.length() - 1);   //cut last newline
      m_pMessages->append(tmp);   //append haengtn "\n" an
      m_pMessages->setCursorPosition(m_pMessages->numLines(), 0); 

      QStringList::Iterator it = m_pCoImportThread->m_outputQueue.begin();

      m_pCoImportThread->lockOutputQueue ();
      for (i = 0; i < len; i++) it = m_pCoImportThread->m_outputQueue.remove(it);
      m_pCoImportThread->unlockOutputQueue ();
   }

   if (!res && !globalStopAction) finishImport();
}

//----------------------------------------------------------------------------

void ApplicationWindow::finishImport()
{
   //alten zustand wiederherstellen
   delete m_pCoImportThread;
   m_pCoImportThread = 0;

   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);
   
   m_checkImportResultTimer.stop();
   m_progressTimer.stop();
   m_importThreadIsRunning = false;
   globalStopAction = false;
   m_uninterruptible = false;
   globalListViewsEnabled = true;

   m_pTools->m_pStopAction->setEnabled(false);
   menuBar()->setEnabled(true);
      
   centralWidget()->unsetCursor();
   startCheckStat();   

   m_pStatusBarLabel->setText(tr("Ready"));
}

//----------------------------------------------------------------------------

void ApplicationWindow::addProjectSlot()
{
   globalListViewsEnabled = false;
   addProject();
   globalListViewsEnabled = true;
}

//----------------------------------------------------------------------------

void ApplicationWindow::addProject(const char * fileName, bool expand)
{
   bool addNewProject = false;

   QString fn = fileName;
   if(fn.isEmpty()){//popup dialog
      addNewProject = true;

      /* Using the Qt standard file dialog preserves the look and feel
       * of the application across platforms.
       */
      fn = QFileDialog::getExistingDirectory (QString::null, this, NULL,
					      tr("Add directory to workbench"),
					      true);

      // Remove trainling "/" from the path.
      fn.replace (QRegExp ("/+$"), "");
   }
   
   if ( !fn.isEmpty() ){
      if(addNewProject && m_projectNameList.findIndex(fn) > -1){
         QMessageBox::warning(0, tr("Warning"), tr("This project is already in workbench."), 0);
         statusBar()->message( tr("Adding aborted"), 4000 );
      }
      else {
         m_pStatusBarLabel->setText("");
         m_uninterruptible = true;
         m_pTools->m_pStopAction->setEnabled(true);
         menuBar()->setEnabled(false);
//         m_pWorkBenchTree->setEnabled(false);
//         m_pCvsFileList->setEnabled(false);

         centralWidget()->setCursor(waitCursor);
         Directory * item = new Directory (
           m_pWorkBenchTree, 
           fn, 
           m_pCvsFileList, 
           m_pNonCvsFileList );
         if(addNewProject){
           m_projectNameList.append(item->fullName());   //evtl. hier links aufloesen
           conf->addItem("/WORKBENCH", item->fullName());
         }
         m_lastOpenedDir = item->fullName();
         m_lastOpenedDir.truncate(m_lastOpenedDir.findRev("/"));
         centralWidget()->unsetCursor();
         m_pTools->m_pStopAction->setEnabled(false);
         if(addNewProject){
            if(globalStopAction){
               m_pStatusBarLabel->setText(tr("Adding aborted"));
            }
            else {
               m_pStatusBarLabel->setText(tr("Ready"));
            }
            globalStopAction = false;
            item->setOpen(true);
         }
         else {
            if(expand) item->setOpen(true);
         }
         menuBar()->setEnabled(true);
//         m_pWorkBenchTree->setEnabled(true);
//         m_pCvsFileList->setEnabled(true);
         m_uninterruptible = false;
         
         startCheckStat();
      }
   }
   else
      m_pStatusBarLabel->setText(tr("Adding aborted"));
}

//----------------------------------------------------------------------------

void ApplicationWindow::removeProjectSlot()
{
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();

   if(!dir) return;

   /*! remove from disabled list if applicable */
   enableProject();
 

   bool dirRemoved = false;
   for(unsigned int i = 0; i < m_projectNameList.count(); i++) {
      if(m_projectNameList[i].compare(dir->fullName()) == 0) {
         m_projectNameList.remove(dir->fullName());
         delete dir;
         /*! remove it from cfg file */
         conf->rmItem("/WORKBENCH", dir->fullName(),  m_projectNameList);
         i = m_projectNameList.count();
         dirRemoved = true;
      }
   }
      
   if(!dirRemoved)
      QMessageBox::information(0, tr("Information"), 
                               tr("This Directory is not a top level dir."), 0);
}

//----------------------------------------------------------------------------



/*---------------------------------------------------------------------------*/
/*!
\fn			void ApplicationWindow::disableProjectSlot(void)
\brief		Disable a dir at top level form workbench.
          Dir is still listed but not further updated and scanned until enabled
          again. Entry is inserted in file list in cfg file section DISABLED.


<BR><HR>*/
/*---------------------------------------------------------------------------*/

void ApplicationWindow::disableProjectSlot(void)
{
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!dir) return;

   m_projectDisabledList.append (dir->fullName());
   conf->addItem("/DISABLED", dir->fullName());
   rereadProject();
}


/*---------------------------------------------------------------------------*/
/*!
\fn			void ApplicationWindow::enableProjectSlot()
\brief		Reread project after reenable.

<BR><HR>*/
/*---------------------------------------------------------------------------*/

void ApplicationWindow::enableProjectSlot()
{
  enableProject();
  rereadProject();  
}


/*---------------------------------------------------------------------------*/
/*!
\fn			void ApplicationWindow::enableProject()
\brief		Reenable a dir at top level form workbench.
          Dir is fully accessible again. Entry is removed from file list 
          in cfg file section DISABLED.

<BR><HR>*/
/*---------------------------------------------------------------------------*/

void ApplicationWindow::enableProject(void)
{
  Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
  if(dir) {
    for(unsigned int i = 0; i < m_projectDisabledList.count(); i++) {
      if(m_projectDisabledList[i].compare(dir->fullName()) == 0) {
        m_projectDisabledList.remove(dir->fullName());
        conf->rmItem("/DISABLED", dir->fullName(), m_projectDisabledList);
        i = m_projectDisabledList.count();
      }
    }
  }
  else
    return;
}

/*---------------------------------------------------------------------------*/

void ApplicationWindow::performRereadProject( QString dirName ) {
      statusBar()->clear();
      globalListViewsEnabled = false;
      addProject(dirName, true);      
      m_projectNameList.append(dirName);
      conf->addItem("/WORKBENCH", dirName);
      m_pStatusBarLabel->setText( tr("Ready"));
      globalListViewsEnabled = true;
}

//----------------------------------------------------------------------------

void ApplicationWindow::rereadProject() {

   // jump to TOP of this directory -> bit better
   Directory *dir = ((Directory *)m_pWorkBenchTree->currentItem() );

   if(!dir) return;

   rereadProjectOfDir( dir );
}

//----------------------------------------------------------------------------

void ApplicationWindow::showDirProperties()
{
   Directory *item = (Directory *)m_pWorkBenchTree->currentItem();
   if(!item) return;

   PropertiesDialogImpl Properties (this, item);
   Properties.exec();
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkStatus()
{
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();

   if(!dir) return;

   if(dir->entriesFileModified()){
      dir->activateItem();
      return;
   }
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(m_pCvsFileList->itemRect(myChild).isValid()) {
         dir->checkStatus(myChild);   
      }
      myChild = myChild->nextSibling();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::statusDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QString command = "status -v";

   // in fact should probably also reset root under which this entry
   // is attached ... (we should bubble up to the parent with depth=0)
   m_pCurCvsDir->resetState( );

   startCvsCall(command, STATUS);
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateAny() {
   // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         updateFile();
         return;
      }
      myChild = myChild->itemBelow();
   }
   
   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      updateDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::mergeAny() {
   // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         mergeFile();
         return;
      }
      myChild = myChild->itemBelow();
   }
   
   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      mergeDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::queryUpdateDir()
{
    updateDir(true);
}
//----------------------------------------------------------------------------

void ApplicationWindow::updateDir()
{
    updateDir(false);
}
//----------------------------------------------------------------------------

void ApplicationWindow::updateDir(bool query)
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QString command = "-z" + QString().setNum(COMPRESSION) +
       (query ? " -n " : "") + " update -P -d";

   startCvsCall(command, UPDATE_DIR);
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateRevDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   TagUpdateDlg *dlg = new TagUpdateDlg(&m_tagList, tr("Update Other Revision"),
                                        this);
                                             
   if(dlg->exec()) {
      QString command = "-z" + QString().setNum(COMPRESSION) + " update -P -d";
      if(!dlg->cvsParameter().isNull()) {
         command += " " + dlg->cvsParameter();
      }
      startCvsCall(command, UPDATE_DIR);
   }
   
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::mergeDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   MergeDialog dlg(this, "MergeBranches", true);
   if (dlg.exec()) {

      QString command = "-z" + QString().setNum(COMPRESSION) + " update -P -d -j ";
      command += dlg.m_pFirstTag->currentText();
      if (dlg.m_bUseSecondTag->isChecked()) command += " -j " + dlg.m_pSecondTag->currentText();

      startCvsCall(command, UPDATE_DIR);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::commitAny() {
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         commitFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      commitDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::commitDir()
{
   if (m_pCommitDlg->exec()) {//ok
      m_pStatusBarLabel->setText("");
      m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
      m_pCurCvsDir->resetState();
      if(!m_pCurCvsDir) return;

      m_curCommentFileName = writeTmpFile(m_pCommitDlg->m_pComment->text());

      QString lang = QString(QTextCodec::locale()).left(2);
      QString command;

      if (m_curCommentFileName.length())
      {
	  command = "-z" + QString().setNum(COMPRESSION) +
	      " commit -F " + m_curCommentFileName;
      }
      else
      {
	  // fallback to option -m
	  command = "-z" + QString().setNum(COMPRESSION) +
	      " commit -m \"" + m_pCommitDlg->m_pComment->text() + "\"";

         if (lang == "ja"){
             QTextCodec *codec = QTextCodec::codecForLocale();
             command += codec->fromUnicode(m_pCommitDlg->m_pComment->text());
         }
         else
             command += m_pCommitDlg->m_pComment->text();
         command += "\"";
      }
      
      startCvsCall(command, COMMIT);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateAny() 
{
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      annotateDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLine() 
{
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateGrepLineFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      annotateGrepLineDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLines() 
{
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateGrepLinesFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( (Directory *)m_pWorkBenchTree->currentItem() ) {
      annotateGrepLinesDir();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QString command = "annotate";
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLineDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   AnnotateGrepLineDialogImpl dlg(this, "AnnotateGrepLine", true);
   if (dlg.exec()) {
       m_pCurCvsDir->SetAnnotateSearchLine(dlg.m_SearchLine->currentText(),
					   dlg.m_bCaseSensitive->isChecked());

       QString command = "annotate";
       startCvsCall(command, ANNOTATE_GREP_LINE);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLinesDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   AnnotateGrepLinesDialogImpl dlg(this, "AnnotateGrepLines", true);
   if (dlg.exec()) {
       QStringList searchLines;
       QListViewItem *myChild = dlg.m_SearchLines->firstChild();
       while(myChild) {
	   searchLines.append(myChild->text(0));
	   myChild = myChild->nextSibling();
       }

       m_pCurCvsDir->SetAnnotateSearchLines(searchLines,
					    dlg.m_bCaseSensitive->isChecked());

       QString command = "annotate";
       startCvsCall(command, ANNOTATE_GREP_LINES);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::tagDir()
{
	m_pStatusBarLabel->setText("");
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
	if(!m_pCurCvsDir) return;

	TagDialogImpl *dlg = new TagDialogImpl(this, tr("Tag a Directory"));
	dlg->initTagList(&m_tagList); 
	int result = dlg->exec();

	if ((result != 0) && (m_tagList.isEmpty()))
	{
		QMessageBox::warning(0, tr("Warning"), tr("You must input a Tag"), 0);
	}
	else
	{
		QString command;

		switch (static_cast<TagDialogImpl::DIALOG_RET_CODE_E>(result))
		{
		case TagDialogImpl::TAG_REGULAR:
			command = "tag " + m_tagList[0];
			break;

		case TagDialogImpl::TAG_BRANCH:
			command = "tag -b " + m_tagList[0];
			break;

		case TagDialogImpl::TAG_DELETE:
			command = "tag -d " + m_tagList[0];
			break;
		}

		if (!command.isEmpty())
			startCvsCall(command);
	}
   
	delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::logDir()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QString command = "log";
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::historyDir()
{
   m_pStatusBarLabel->setText ("");

   if ((m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem()) == 0)
	   return;

   HistoryDialogImpl *pHistDlg = new HistoryDialogImpl (this);
   if (pHistDlg->parseHistory (m_pCurCvsDir->fullName(), "", m_pCurCvsDir->connectMethod()))
	   pHistDlg->show();
   else
	   delete pHistDlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::startCvsCall(QString& command, 
	int evaluateCommand, QStringList *updateFileList, int Synchronous )
{
   centralWidget()->setCursor(waitCursor);
   m_pStatusBarLabel->setText(tr("Please wait, Im working ..."));

   m_timer.stop();   
   m_progressTimer.start(PROGRESSINTERVALL);
   
   globalStopAction = false;

   if (m_pCurCvsDir->startCvsCall(
   	command, m_pMessages, evaluateCommand, updateFileList, Synchronous )) {
      m_cvsThreadIsRunning = true;
      globalListViewsEnabled = false;
      m_uninterruptible = true;
      m_pTools->m_pStopAction->setEnabled(true);
      menuBar()->setEnabled(false);

      m_checkCvsResultTimer.start(CVSCALLINTERVALL);
   } else {//loginfehler
      startCheckStat();   
      m_progressTimer.stop();
      centralWidget()->unsetCursor();
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::checkCvsCallResult()
{
   bool res = m_pCurCvsDir->checkCvsCallResult(m_pMessages);
   if (!res && !globalStopAction) setIdle();
}

//----------------------------------------------------------------------------

void ApplicationWindow::setIdle()
{
   //alten zustand wiederherstellen
   m_checkCvsResultTimer.stop();
   m_progressTimer.stop();
   
   //remove temporary file
   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);
   m_curCommentFileName = "";
   
   m_cvsThreadIsRunning = false;
   m_uninterruptible = false;
   m_pTools->m_pStopAction->setEnabled(false);
   menuBar()->setEnabled(true);
   globalStopAction = false;
   m_pCurCvsDir->deleteThreadAndActivateIfNecessary();
   m_pStatusBarLabel->setText(tr("Ready"));
   m_pMessages->setCursorPosition(0, 0); 

   globalListViewsEnabled = true;
      
   checkStatus();   //cvs view aktualisieren
   centralWidget()->unsetCursor();
   startCheckStat();

   if (m_bExternalDiffIsRunning) {//external diff
       m_bExternalDiffIsRunning = false;
       if (fork() == 0) {
	   execlp (externalDiffProgram, externalDiffProgram, 
		   m_curTmpDiffFileName.latin1(), 
		   m_curDiffFileName.latin1(), 
		   0);
       }

       m_curDiffFileName = "";
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::loginDir()
{
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!dir) return;

   dir->loginOk(true);
}

//----------------------------------------------------------------------------

void ApplicationWindow::editFile( QString FN) {

   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!dir) return;

   QString FFN;
   FFN = dir->fullName() + "/" + FN;
   FFN.replace( QRegExp(" "), "\\ " );
   QString Cmd = EDITOR + " " + FFN + " 2>&1 &";
   //Cmd = Cmd.arg(FFN);
cout << Cmd << "\n";   
   system( Cmd );
}

//----------------------------------------------------------------------------

void ApplicationWindow::addFiles()
{
/*
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!dir) return;

   QStringList s;
   QString f = QFileDialog::getOpenFileName(dir->fullName(), QString::null, this);
   if (!f.isEmpty()) {
      s.append(f);
   }

   if(!s.isEmpty()){
      statusBar()->clear();
      dir->addFiles(m_pMessages, &s);
      m_pStatusBarLabel->setText(tr("Ready"));
   }
*/   
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateColor( int ct ) {
   if( ct == 0 ) {
      m_pFileViews->getColorTabBar()->colorTab( 1 )->resetColor( );
   } else {
      QColor C( 255, 0, 0 );
      m_pFileViews->getColorTabBar()->colorTab( 1 )->setColor( C );
   }
   m_pFileViews->getColorTabBar()->repaint( TRUE );
}

//----------------------------------------------------------------------------

void ApplicationWindow::addSelectedFiles()
{
   addSelectedFiles(false);
}

//----------------------------------------------------------------------------

void ApplicationWindow::addSelectedBinFiles()
{   addSelectedFiles(true);
}

//----------------------------------------------------------------------------

void ApplicationWindow::addSelectedFiles(bool bAsBinary) {

	int NeedToReread = 0;
	// in what directory are the files
	Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();
	if(!dir) return;

	QStringList s;
	QString St;
	QString FirstDir;
	QFileInfo FI;
	QListViewItem * It = m_pNonCvsFileList->firstChild();
	while( It ) {
	  if( It->isSelected() ) {
	    St = dir->fullName()+"/"+It->text(0);
	    s.append( St );
	    if( ! NeedToReread ) {
	      // check if St is directory 
	      FI.setFile( St );
	      if( FI.isDir() ) {
		FirstDir = St;
		NeedToReread = 1;
	      }
	    }
	  }
	  It = It->itemBelow();
	}

	if(!s.isEmpty()){
	  statusBar()->clear();
	  dir->addFiles(m_pMessages, &s, bAsBinary);
	  m_pStatusBarLabel->setText(tr("Ready"));
	}

	if( NeedToReread ) {
	  dir = rereadProjectOfDir( dir );

	  if( ! FirstDir.isEmpty() ) {
	    // make directory active so that content of directory is shown
	    dir = dir->searchDirOfPath( FirstDir );
	    if( dir ) {
	      m_pWorkBenchTree->setSelected( dir, TRUE );
	      dir->activateItem();
	    }
	  }
	}
}

Directory * ApplicationWindow::rereadProjectOfDir( Directory * dir ) {

	// have added directory
	int setToSelected = dir->isSelected();
	QString dirToOpenAfterRescan = dir->fullName();

	// move to Top of this project
	dir = dir->topDir();

	// remove from project list
	QString dirName = dir->fullName();
	bool dirRemoved = false;
	for(unsigned int i = 0; i < m_projectNameList.count(); i++) {
	  if(m_projectNameList[i].compare(dirName) == 0) {
	     m_projectNameList.remove(dirName);
	     conf->rmItem("/WORKBENCH", dirName, m_projectNameList);
	     delete dir;
	     i = m_projectNameList.count();
	     dirRemoved = true;
	  }
	}
	  
	if(!dirRemoved)
	  QMessageBox::information(0, tr("Information"), 
			tr("This Directory is not a top level dir."), 0);
	else {
	  // add project again
	  performRereadProject( dirName );

	  // reopen tree
	  dir = (Directory *)m_pWorkBenchTree->find( dirToOpenAfterRescan );
	  if( dir )  {
	    if( setToSelected ) {
	      m_pWorkBenchTree->setSelected( dir, TRUE );
	      dir->activateItem();
	    }
	    dir->setAllOpen();
	  }
	}

	return dir;
}

//----------------------------------------------------------------------------

void ApplicationWindow::queryUpdateFile()
{
    updateFile(true);
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateFile()
{
    updateFile(false);
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateFile(bool query)
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QStringList s;
   QString fileName;
   QString tmp = "";

   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         fileName = myChild->text(0).stripWhiteSpace();
         fileName.replace( QRegExp(" "), "\\ " );
         tmp += " " + fileName;
         s.append(fileName);
      }
      myChild = myChild->nextSibling();
   }

   QString queryString = (query ? " -n " : "" );
   QString command = "-z" + QString().setNum(COMPRESSION) +
       queryString + " update " + tmp;
   startCvsCall(command, UPDATE_FILES, &s);
}

//----------------------------------------------------------------------------

void ApplicationWindow::mergeFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   MergeDialog dlg(this, "MergeBranches", true);
   if (dlg.exec()) {
       QStringList s;
       QString fileName;
       QString tmp = "";

       QListViewItem *myChild = m_pCvsFileList->firstChild();
       while(myChild) {
	   if(myChild->isSelected()) {
	       fileName = myChild->text(0).stripWhiteSpace();
	       fileName.replace( QRegExp(" "), "\\ " );
	       tmp += " " + fileName;
	       s.append(fileName);
	   }
	   myChild = myChild->nextSibling();
       }

       QString command = "-z" + QString().setNum(COMPRESSION) + " update -P -d -j ";
       command += dlg.m_pFirstTag->currentText();
       if (dlg.m_bUseSecondTag->isChecked()) command += " -j " + dlg.m_pSecondTag->currentText();
       command += tmp;

       startCvsCall(command, UPDATE_FILES, &s);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::watchAdd()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "watch add" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::watchRemove()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "watch remove" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::watchOn()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "watch on" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::watchOff()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "watch off" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::cvsEditFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "edit" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::cvsUneditFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "unedit" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::updateRevFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   TagUpdateDlg *dlg = new TagUpdateDlg(&m_tagList, tr("Update Other Revision"),
                                        this);

   if(dlg->exec()) {
      QStringList s;
      QString fileName;
      QString tmp = "";

      QListViewItem *myChild = m_pCvsFileList->firstChild();
      while(myChild) {
         if(myChild->isSelected()) {
            fileName = myChild->text(0).stripWhiteSpace();
            fileName.replace( QRegExp(" "), "\\ " );
            tmp += " " + fileName;
            s.append(fileName);
         }
         myChild = myChild->nextSibling();
      }

      QString command =  "-z" + QString().setNum(COMPRESSION) + " update ";
      if(!dlg->cvsParameter().isNull()) {
         command += " " + dlg->cvsParameter();
      }
      command += tmp;
      startCvsCall(command, UPDATE_FILES, &s);
   }
   
   delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::statusFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QString command = "status -v" + getNamesOfSelectedFiles();
   startCvsCall(command, STATUS);
}

//----------------------------------------------------------------------------

void ApplicationWindow::renameFile() {

	m_pStatusBarLabel->setText("");
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();

	QString fname =  getNameOfFirstSelectedFile();

	if( fname.isEmpty() ) {
	 QMessageBox::warning(0, tr("Warning"), tr("You must select one file"), 0);
	 return;
	}

	if(!m_pCurCvsDir) return;

	QString NewName = QInputDialog::getText ( 
	        "Rename", "New name : ", fname ) ;

        if( ! NewName.isEmpty() ) {
	   QString command = "mv " + 
	        m_pCurCvsDir->fullName() + "/" + fname.stripWhiteSpace()  
	        + " " + 
		m_pCurCvsDir->fullName() + "/" + NewName.stripWhiteSpace();

	   if( system( command.ascii() ) < 0 ) {
	     QMessageBox::critical(0, tr("Rename"), tr("Cannot rename file"), 0);
	     return;
	   }
	   command = "remove " + fname;
	   startCvsCall(command, STATUS);
	   command = "add " + NewName;
	   startCvsCall(command, STATUS);
        }
}

//----------------------------------------------------------------------------

void ApplicationWindow::diffFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "diff" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::diffFileSideBySide()
{
    m_bExternalDiffIsRunning = false;

    QString fname =  getNameOfFirstSelectedFile();

    if( fname.isEmpty() ) {
	QMessageBox::warning(0, tr("Warning"), tr("You must select one file"), 0);
	return;
    }

    m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
    if(!m_pCurCvsDir) return;

    m_pStatusBarLabel->setText("");

    if (bUseExternalDiffForSideBySide) {
       if (m_curTmpDiffFileName.length()) QFile::remove(m_curTmpDiffFileName);

       m_curDiffFileName = m_pCurCvsDir->fullName() + "/" +  fname.stripWhiteSpace();
       m_curTmpDiffFileName = createTempFile() + "-" + fname.stripWhiteSpace();

	QString command = "-Q update -p" + fname + " > " + m_curTmpDiffFileName;
	m_bExternalDiffIsRunning = true;
	startCvsCall(command);
    } else {	      
	DiffDialogImpl *l = new DiffDialogImpl(this);
	if(l->parseCvsDiff( m_pCurCvsDir->fullName().replace( QRegExp(" "), "\\ " ), fname, 
			    m_pCurCvsDir->connectMethod(), "", "" )) {
	    l->show();
	} else delete l;
    }
}

//----------------------------------------------------------------------------

void ApplicationWindow::commitFile()
{
   if(m_pCommitDlg->exec()) {//ok
      m_pStatusBarLabel->setText("");
      m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
      if(!m_pCurCvsDir) return;

      m_curCommentFileName = writeTmpFile(m_pCommitDlg->m_pComment->text());
      
     	QString lang = QString(QTextCodec::locale()).left(2);
      QString command;
      if (m_curCommentFileName.length())
      {
	  command = "-z" + QString().setNum(COMPRESSION) +
	      " commit -F " + m_curCommentFileName;
      }
      else
      {
 	  // fallback to option -m
	  command = "-z" + QString().setNum(COMPRESSION) +
	      " commit -m \"" + m_pCommitDlg->m_pComment->text() + "\"";
         if (lang == "ja"){
             QTextCodec *codec = QTextCodec::codecForLocale();
             command += codec->fromUnicode(m_pCommitDlg->m_pComment->text());
         }
         else
             command += m_pCommitDlg->m_pComment->text();
         command += "\"";
      }
      
      command += getNamesOfSelectedFiles();

      startCvsCall(command, COMMIT);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::tagFile()
{
	m_pStatusBarLabel->setText("");
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
	if(!m_pCurCvsDir) return;

	TagDialogImpl *dlg = new TagDialogImpl(this, tr("Tag File(s)"));
	dlg->initTagList (&m_tagList);
	int result = dlg->exec();

	if ((result != 0) && (m_tagList.isEmpty()))
	{
		QMessageBox::warning(0, tr("Warning"), tr("You must input a Tag"), 0);
	}
	else
	{
		QString command;

		switch (static_cast<TagDialogImpl::DIALOG_RET_CODE_E>(result))
		{
		case TagDialogImpl::TAG_REGULAR:
			command = "tag " + m_tagList[0] + getNamesOfSelectedFiles();
			break;

		case TagDialogImpl::TAG_BRANCH:
			command = "tag -b " + m_tagList[0] + getNamesOfSelectedFiles();
			break;

		case TagDialogImpl::TAG_DELETE:
			command = "tag -d " + m_tagList[0] + getNamesOfSelectedFiles();
			break;
		}

		if (!command.isEmpty())
			startCvsCall(command);
	}
   
	delete dlg;
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "annotate" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLineFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   AnnotateGrepLineDialogImpl dlg(this, "AnnotateGrepLine", true);
   if (dlg.exec()) {
       m_pCurCvsDir->SetAnnotateSearchLine(dlg.m_SearchLine->currentText(),
					   dlg.m_bCaseSensitive->isChecked());

       QString command = "annotate" + getNamesOfSelectedFiles();
       startCvsCall(command, ANNOTATE_GREP_LINE);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::annotateGrepLinesFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   AnnotateGrepLinesDialogImpl dlg(this, "AnnotateGrepLines", true);
   if (dlg.exec()) {
       QStringList searchLines;
       QListViewItem *myChild = dlg.m_SearchLines->firstChild();
       while(myChild) {
	   searchLines.append(myChild->text(0));
	   myChild = myChild->nextSibling();
       }

       m_pCurCvsDir->SetAnnotateSearchLines(searchLines,
					    dlg.m_bCaseSensitive->isChecked());

       QString command = "annotate" + getNamesOfSelectedFiles();
       startCvsCall(command, ANNOTATE_GREP_LINES);
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::logFile()
{
   QString Fname;
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;

   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected())  {
         Fname = myChild->text(0).stripWhiteSpace();
         Fname.replace( QRegExp(" "), "\\ " );
         break;
      }
      myChild = myChild->nextSibling();
   }

   if( ! Fname.isEmpty() ) {
     m_pCurCvsDir->loginOk(false);
     LogDialogImpl * DD = new LogDialogImpl ( this );
      if( DD->parseCvsLog( m_pCurCvsDir->fullName().replace( QRegExp(" "), "\\ " ), 
                           Fname , m_pCurCvsDir->connectMethod() )) {
         DD->show();
      } else delete DD;
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::historyFile()
{
   m_pStatusBarLabel->setText("");
   m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
   if(!m_pCurCvsDir) return;
      
   QString command = "history" + getNamesOfSelectedFiles();
   startCvsCall(command);
}

//----------------------------------------------------------------------------

void ApplicationWindow::removeFile()
{
	m_pStatusBarLabel->setText("");
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
	if(!m_pCurCvsDir) return;

	if(!m_pCurCvsDir->loginOk()) return;

	doRemoveFileFromRepOrDisk( 1, 1 );
}

//----------------------------------------------------------------------------

void ApplicationWindow::removeFileFromRep() {
	// backup file
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();
	if(!m_pCurCvsDir) return;

	if(!m_pCurCvsDir->loginOk()) return;

	doRemoveFileFromRepOrDisk( 1, 0 );
}

//----------------------------------------------------------------------------

void ApplicationWindow::removeFileFromDisk() {

       m_pStatusBarLabel->setText("");

       m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();

       if(!m_pCurCvsDir) return;

       doRemoveFileFromRepOrDisk( 0, 1 );
}


void ApplicationWindow::removeNonCvsFileFromDisk() {

       m_pStatusBarLabel->setText("");

       m_pCurCvsDir = (Directory *)m_pWorkBenchTree->currentItem();

       if(!m_pCurCvsDir) return;

       doRemoveFileFromRepOrDisk( 0, 1, true );
}


//----------------------------------------------------------------------------

void ApplicationWindow::doRemoveFileFromRepOrDisk( 
	int RmFromRep, int RmFromDisk, bool bNonCvsFile)
{
   QString tmp = "";
   QStringList AllFiles;
   QString fileName;
   QString CurDir = m_pCurCvsDir->fullName() + "/";

	 if( RmFromDisk ) {
	   if( m_showWarning) {
	     //create dialog  
	     WarningDlg *dlg = new WarningDlg(
	        tr("Warning"), 
	        tr("This command will remove file(s) from disk!"), this);
	     
	     if(!dlg->exec()) {
		delete dlg;
		centralWidget()->unsetCursor();
		return;
	     }
	     m_showWarning = dlg->showWarningNextTime();
	     delete dlg;
	   }
	 }

	 // we are sure we can delete files from disk if needed
	 
	 centralWidget()->setCursor(waitCursor);

	 QListViewItem *myChild = (bNonCvsFile) ?
				   m_pNonCvsFileList->firstChild() :
				   m_pCvsFileList->firstChild();

   while(myChild) {
      if(myChild->isSelected()) {
         fileName = myChild->text(0).stripWhiteSpace();
         QString quotedFileName = fileName;
         quotedFileName.replace( QRegExp(" "), "\\ " );
         QFile f( CurDir + fileName);
         
         if(f.exists()) {
            if( RmFromDisk ) {
               f.remove();   //spaeter auf erfolg testen
            } else {
               // make backup
               QString S = "__" + fileName;
               AllFiles.append( S );
               S = CurDir + S;
               // rename file
               rename( f.name(), S );
            }
         }
         tmp += "  " + quotedFileName;
      }
      myChild = myChild->nextSibling();
   }


	 if( RmFromRep ) {
	   // remove from repository
	   if( ! tmp.isEmpty() ) {
	     QString command = "remove" + tmp;
	     // synchronous call ->because of rename
	     startCvsCall(command, ACTIVATING_IS_NECESSARY, 0, 1);
	   }

      
	   if( ! RmFromDisk ) {
	     // restor backup files 
	     for( QStringList::Iterator it = AllFiles.begin();
		    it != AllFiles.end(); ++ it ) {
		QString S1 = CurDir + (*it);
		QString S2 = CurDir + (*it).mid(2);
		// remove __
		rename( S1, S2 );
	     }
	   }
	 }

	 if (bNonCvsFile)
	 {
		 // Remove deleted list items from the non-cvs file listview.
		 QListViewItemIterator it(m_pNonCvsFileList);

		 while (it.current())
		 {
			 /* Whenever we remove an item from a list view,
			  * the iterator is automatically moved to the next
			  * valid list item so we do need no move the
			  * iterator foward after a deletion.
			  */
			 if (it.current()->isSelected())
				 delete it.current();
			 else
				 ++it;
		 }
	 }

	 centralWidget()->unsetCursor();
}

//----------------------------------------------------------------------------

QString ApplicationWindow::getNamesOfSelectedFiles()
{
   QString tmp = "";
    
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) tmp += " " + myChild->text(0).stripWhiteSpace().replace( QRegExp(" "), "\\ " );
      myChild = myChild->nextSibling();
   }
   
   return tmp;
}

//----------------------------------------------------------------------------

QString ApplicationWindow::getNameOfFirstSelectedFile()
{
   QString tmp = "";
    
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
        tmp += " " + myChild->text(0).stripWhiteSpace().replace( QRegExp(" "), "\\ " );
        break;
      }
      myChild = myChild->nextSibling();
   }
   
   return tmp;
}

//----------------------------------------------------------------------------

void ApplicationWindow::stopCurAction()
{
   globalStopAction = true;
   
   if(m_cvsThreadIsRunning) {//nur fuer cvs thread
      m_cvsThreadIsRunning = false;   //"prellen" verhindern
      QTimer::singleShot(1000, this, SLOT(killCvsThread()));
   }

   if(m_coThreadIsRunning) {//nur fuer cvs thread
      m_coThreadIsRunning = false;   //"prellen" verhindern
      QTimer::singleShot(1000, this, SLOT(killCoThread()));
   }

   if(m_importThreadIsRunning) {//nur fuer cvs thread
      m_importThreadIsRunning = false;   //"prellen" verhindern
      QTimer::singleShot(1000, this, SLOT(killImportThread()));
   }
}

//----------------------------------------------------------------------------

void ApplicationWindow::killCoThread()
{
   m_pCoImportThread->exit();
   checkCoResult();   //queue leeren
   finishCheckOut();
}

//----------------------------------------------------------------------------

void ApplicationWindow::killImportThread()
{
   m_pCoImportThread->exit();
   checkImportResult();   //queue leeren
   finishImport();
}

//----------------------------------------------------------------------------

void ApplicationWindow::killCvsThread()
{
   m_pCurCvsDir->killCvsThread();
   m_pCurCvsDir->checkCvsCallResult(m_pMessages);   //queue leeren
   setIdle();
}


void ApplicationWindow::startCheckStat()
{
  if (StatTimerEnable) m_timer.start(CHECKSTATUSINTERVALL);   
}

//----------------------------------------------------------------------------

void ApplicationWindow::profiles()
{
    ProfilesDialogImpl *profileDlg =
	new ProfilesDialogImpl(&m_coImportUser, &m_coImportServer,
			       &m_coImportRepository, &m_coLinCVSProfile);
    profileDlg->exec();
    delete profileDlg;
}


void ApplicationWindow::AppSettings()
{
	SettingsDialogImpl *sd = new SettingsDialogImpl(this);
	sd->exec();
	delete sd;
}


void ApplicationWindow::progress()
{
	++m_curPic %= NUM_ANIM_PICS;
	m_pProgressWidget->setPixmap(m_progressIcon[m_curPic]);
}


void ApplicationWindow::closeEvent( QCloseEvent* e )
{
//  writeCfg();

#ifdef LINCVS_APP
    if (bAskForQuit) {
	if(!m_uninterruptible){
	    if ( QMessageBox::information( this, tr("LinCVS"), 
					   tr("Are you sure you want to exit LinCVS?"), 
					   tr("&Exit"), tr("&Cancel")
					   ) == 0 ) {
		QString pid;
		pid.setNum(sshPid);
		if (sshAgentIsRunning) system("kill " + pid);
		e->accept();
	    } else {
		e->ignore();
	    }
	} else {
	    e->ignore();
	}
    } else {
	e->accept();
    }

#else
   if(!m_uninterruptible) {
      QString pid;
      pid.setNum(sshPid);
      if (sshAgentIsRunning) system("kill " + pid);
      e->accept();
   } else e->ignore();
#endif
}

/*---------------------------------------------------------------------------*/
/*!
\fn			void ApplicationWindow::writeCfg()
\brief		Writes the cfg file back. Old file will be overwritten. Uses the 
            library AppConf.

\param		ApplicationWindow::writeCfg		
<BR><HR>*/
/*---------------------------------------------------------------------------*/

void ApplicationWindow::writeCfg()
{
  conf->putCfgFile();
}

//----------------------------------------------------------------------------

void ApplicationWindow::about()
{
    QString versionString = "Version ";
    versionString += VERSION;
    AboutDialogImpl AboutDlg(this, versionString);
    AboutDlg.exec();
}

//----------------------------------------------------------------------------

void ApplicationWindow::help()
{
   QDialog*dlg = new QDialog(this, 0, true);   //modal
   dlg->setCaption(tr("Help"));
   
   QBoxLayout *layer = new QVBoxLayout(dlg, 5);
   QMultiLineEdit*e = new QMultiLineEdit(dlg);
   e->setReadOnly(true);
   layer->addWidget(e);

   QWidget *backGround = new QWidget(dlg);
   layer->addWidget(backGround);

   QBoxLayout *buttonLayer = new QHBoxLayout(backGround, 3);
   buttonLayer->addStretch();
   
   QPushButton *ok = new QPushButton(backGround);
   ok->setText(tr("OK"));
   buttonLayer->addWidget(ok);
   buttonLayer->addStretch();
   
   connect(ok, SIGNAL(clicked()), dlg, SLOT(accept()));
   dlg->resize(600, 300);

   QFile f;
   QString line, filename, dataDir;
   dataDir = LC_DATA_DIR;
   int startPos, endPos;
   if ((startPos = dataDir.find("${")) > -1) {
       if ((endPos = dataDir.find("}")) > startPos) {
	   dataDir.replace(startPos, endPos + 1 - startPos, LC_PREFIX_DIR);
       }   
   }

   filename = dataDir + "/lincvs/README";
   f.setName(filename);
   
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      while(!textStream.atEnd()) {
         e->insertLine(textStream.readLine());
      }
      f.close();

      dlg->exec();
      delete dlg;
   } else {
       QString error = tr("File not found: ");
       QMessageBox::warning(this, tr("LinCVS error "), error + filename );
   }
}


void ApplicationWindow::aboutQt()
{
   QMessageBox::aboutQt( this, "Qt..." );
}

//----------------------------------------------------------------------------

void ApplicationWindow::removeDirFromRep() {
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();

   if(!dir) return;

   if(!dir->loginOk()) return;

   dir->removeFromRep();

   rereadProjectOfDir( dir->topDir() );
}

void ApplicationWindow::removeDirFromRepAndDisk() {
   Directory *dir = (Directory *)m_pWorkBenchTree->currentItem();

   if(!dir) return;

   if(!dir->loginOk()) return;

   dir->removeFromRepAndDisk();

   rereadProjectOfDir( dir->topDir() );
}


// Local Variables:
// c-basic-offset: 4
// End:
