/* $Id: util-lang.c,v 1.14 2005/03/28 16:24:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "util-lang.h"

#define UTIL_LANG_DATA "lingLanguages"
#define UTIL_LANG_CONFIG "lingConfig"
enum
{
     UTIL_LANG_USED, 
     UTIL_LANG_LEARN_FROM,
     UTIL_LANG_TRANSLATION,
     UTIL_LANG_LANGUAGE,
     UTIL_LANG_COLUMNS
};

static GtkListStore* util_create_language_list (learnLanguage *which,
                                                lingConfig *config);
static void toggle_main_language (GtkCellRendererToggle *cell, gchar *path,
                                  GtkTreeView *tree);
static void toggle_use_language (GtkCellRendererToggle *cell, gchar *path,
                                 GtkTreeView *tree);
static GtkListStore* util_create_single_language_list (lingchar *which,
                                                       lingchar **languages,
                                                       lingConfig *config);
static void toggle_single_language (GtkCellRendererToggle *cell, gchar *path,
                                    GtkTreeView *tree);

static GtkListStore*
util_create_language_list (learnLanguage *which, lingConfig *config)
{
     GtkListStore *list;
     GtkTreeIter iter;
     learnLanguage *lang = which;
     lingchar *language = NULL;

     list = gtk_list_store_new (UTIL_LANG_COLUMNS,
                                G_TYPE_BOOLEAN,
                                G_TYPE_BOOLEAN,
                                G_TYPE_STRING,
                                G_TYPE_STRING);

     g_object_set_data (G_OBJECT (list), UTIL_LANG_DATA, which);
     
     while (lang && !lang->main)
          lang = lang->next;

     debug ("Adding languages...\n");

     while (which)
     {
          if (lang)
               language = ling_lang_get_lang_translation (config,
                                                          which->language,
                                                          lang->language);

          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              UTIL_LANG_USED, which->used,
                              UTIL_LANG_LEARN_FROM, which->main,
                              UTIL_LANG_LANGUAGE, which->language,
                              UTIL_LANG_TRANSLATION, (language) ? language : 
                              which->language,
                              -1);
          if (language)
               free (language);
          which = which->next;
     }
     return list;
}

static void
toggle_main_language (GtkCellRendererToggle *cell, gchar *path,
                      GtkTreeView *tree)
{
     learnLanguage *languages = NULL;
     gchar *language = NULL;
     GtkTreeIter iter;
     GtkListStore *list;
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     lingConfig *cfg = g_object_get_data (G_OBJECT (tree), UTIL_LANG_CONFIG);

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          languages = g_object_get_data (G_OBJECT (model), UTIL_LANG_DATA);
          gtk_tree_model_get (GTK_TREE_MODEL (model), &iter,
                              UTIL_LANG_LANGUAGE, &language,
                              -1);
          learn_language_change_main_language (languages, language);

          g_free (language);
          list = util_create_language_list (languages, cfg);
          gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
          g_object_unref (G_OBJECT (list));
     }
     return;
}

static void
toggle_use_language (GtkCellRendererToggle *cell, gchar *path,
                     GtkTreeView *tree)
{
     learnLanguage *languages = NULL;
     gboolean use = FALSE;
     gboolean main_l = FALSE;
     gchar *language = NULL;
     GtkTreeIter iter;
     GtkListStore *list;
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     lingConfig *cfg = g_object_get_data (G_OBJECT (tree), UTIL_LANG_CONFIG);

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          languages = g_object_get_data (G_OBJECT (model), UTIL_LANG_DATA);
          gtk_tree_model_get (model, &iter,
                              UTIL_LANG_USED, &use,
                              UTIL_LANG_LEARN_FROM, &main_l,
                              UTIL_LANG_LANGUAGE, &language,
                              -1);
          /* if it is the main language, which shouldbe switched,
           * warn the user
           */
          if (main_l)
          {
               error_warning (_("No main language selected!"),
                              _("You have to select a main language."),
                              ERR_NOT_AVAILABLE);
               g_free (language);
               return;
          }

          learn_language_use_language (languages, language, !use);
          
          g_free (language);
          list = util_create_language_list (languages, cfg);
          gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
          g_object_unref (G_OBJECT (list));
     }
     return;
}

/* used by the editor preferences */
static GtkListStore*
util_create_single_language_list (lingchar *which, lingchar **languages,
                                  lingConfig *config)
{
     GtkListStore *list;
     GtkTreeIter iter;
     gint i = 0;
     lingchar *language = NULL;
     gboolean used = FALSE;

     list = gtk_list_store_new (3, G_TYPE_BOOLEAN, G_TYPE_STRING,
                                G_TYPE_STRING);
     g_object_set_data (G_OBJECT (list), UTIL_LANG_DATA, languages);

     debug ("Adding languages for editor...\n");

     while (languages[i])
     {
          if (which)
          {
               language = ling_lang_get_lang_translation (config, languages[i],
                                                          which);
               used = !g_utf8_collate (which, languages[i]);
          }

          gtk_list_store_append (list, &iter);
          gtk_list_store_set (list, &iter,
                              0, used,
                              1, (language) ? language : languages[i],
                              2, languages[i],
                              -1);
          ling_free (language);
          i++;
     }
     return list;
};

/* used by the editor preferences */
static void
toggle_single_language (GtkCellRendererToggle *cell, gchar *path,
                        GtkTreeView *tree)
{
     GtkTreeIter iter;
     GtkListStore *list;
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     lingConfig *cfg = g_object_get_data (G_OBJECT (tree), UTIL_LANG_CONFIG);
     lingchar **val = g_object_get_data (G_OBJECT (tree), "save_in_var");
     lingchar **lang = NULL;
     lingchar *language = NULL;
     gint i = 0;
     gboolean use = FALSE;

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          lang = g_object_get_data (G_OBJECT (model), UTIL_LANG_DATA);
          gtk_tree_model_get (model, &iter, 0, &use, 2, &language, -1);

          list = util_create_single_language_list (language, lang, cfg);
          
          /* save in preference value */
          while (lang[i])
          {
               if (g_utf8_collate (language, lang[i]) == 0)
               {
                    *val = lang[i];
                    break;
               }
               i++;
          }
          g_free (language);

          gtk_tree_view_set_model (tree, GTK_TREE_MODEL (list));
          g_object_unref (G_OBJECT (list));
     }
     return;

}

GtkWidget*
util_lang_create_language_tree_view (lingGtkPrefs *settings,
                                     learnLanguage *which,
                                     const gchar *title[3])
{
     GtkWidget *scr_win;
     GtkWidget *view_list;
     GtkListStore *list;
     GtkCellRenderer *renderer;

     list = util_create_language_list (which, settings->prefs->config);
     view_list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_set_data (G_OBJECT (view_list), UTIL_LANG_CONFIG,
                        settings->prefs->config);
     g_object_unref (G_OBJECT (list));

     renderer = gtk_cell_renderer_toggle_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  UTIL_LANG_USED, title[0],
                                                  renderer, "active",
                                                  UTIL_LANG_USED, NULL);
     g_signal_connect (G_OBJECT (renderer), "toggled",
                       G_CALLBACK (toggle_use_language), view_list);

     renderer = gtk_cell_renderer_toggle_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  UTIL_LANG_LEARN_FROM,
                                                  title[1],
                                                  renderer, "active",
                                                  UTIL_LANG_LEARN_FROM, NULL);
     g_signal_connect (G_OBJECT (renderer), "toggled",
                       G_CALLBACK (toggle_main_language), view_list); 

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  UTIL_LANG_TRANSLATION,
                                                  title[2],
                                                  renderer, "text",
                                                  UTIL_LANG_TRANSLATION, NULL);

     scr_win = hig_scrolled_window_new ();
     gtk_container_add (GTK_CONTAINER (scr_win), view_list);
     return scr_win;
}

/* used by the editor preferences */
GtkWidget*
util_lang_create_language_single_selection (lingGtkPrefs *settings)
{
     GtkWidget *scr_win;
     GtkWidget *view_list;
     GtkListStore *list;
     GtkCellRenderer *renderer;

     list = util_create_single_language_list (settings->edit_lang,
                                              settings->prefs->languages,
                                              settings->prefs->config);

     view_list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_set_data (G_OBJECT (view_list), UTIL_LANG_CONFIG,
                        settings->prefs->config);
     g_object_set_data (G_OBJECT (view_list), "save_in_var",
                        &settings->edit_lang);
     g_object_unref (G_OBJECT (list));

     renderer = gtk_cell_renderer_toggle_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  0, _("Use"),
                                                  renderer, "active",
                                                  0, NULL);
     g_signal_connect (G_OBJECT (renderer), "toggled",
                       G_CALLBACK (toggle_single_language), view_list);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_list),
                                                  1, _("Language"),
                                                  renderer, "text",
                                                  1, NULL);

     scr_win = hig_scrolled_window_new ();
     gtk_container_add (GTK_CONTAINER (scr_win), view_list);
     return scr_win;
}
