/*
 * $Id: volume.c,v 1.18 2001/12/06 01:14:52 antona Exp $
 *
 * volume.c - NTFS volume handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "types.h"
#include "list.h"
#include "volume.h"
#include "support.h"
#include "bootsect.h"
#include "disk_io.h"
#include "endians.h"
#include "attrib.h"
#include "mft.h"

/**
 * ntfs_sync_volume - sync a mounted ntfs volume to physical storage
 * @vol:	ntfs volume to sync
 *
 * Sync a mounted ntfs volumes to physical storage.
 *
 * Return zero on success and -1 if one or more mft records and/or
 * non-resident attributes were	busy, and could not be committed to disk.
 */
int ntfs_sync_volume(ntfs_volume *vol)
{
	struct list_head *tmp;
	int s = 0;

	list_for_each(tmp, &vol->dirty_attrs) {
		if (!ntfs_flush_attr(list_entry(tmp, attr, a_dirty_list)) || s)
			continue;
		s = -1;
	}
	list_for_each(tmp, &vol->dirty_mft_entries) {
		if (!ntfs_flush_mft_entry(list_entry(tmp, mft_entry,
							m_dirty_list)) || s)
			continue;
		s = -1;
	}
	return s;
}

static __inline__ ntfs_volume *__allocate_ntfs_volume()
{
	ntfs_volume *v = (ntfs_volume *)malloc(sizeof(ntfs_volume));
	if (!v)
		return NULL;
	memset(v, 0, sizeof(ntfs_volume));
	INIT_LIST_HEAD(&v->open_files);
	INIT_LIST_HEAD(&v->mft_entries);
	INIT_LIST_HEAD(&v->dirty_mft_entries);
	INIT_LIST_HEAD(&v->dirty_attrs);
	return v;
}

static __inline__ int __free_ntfs_volume(ntfs_volume *v)
{
	if (v->nr_open_files || v->nr_mft_entries || v->nr_dirty_attrs)
		return -1;
	if (v->dev_name)
		free(v->dev_name);
	if (v->vol_name)
		free(v->vol_name);
	if (v->lcn_bitmap)
		free(v->lcn_bitmap);
	if (v->mft_bitmap)
		free(v->mft_bitmap);
	if (v->mft_runlist)
		free(v->mft_runlist);
	if (v->upcase)
		free(v->upcase);
	free(v);
	return 0;
}

/**
 * __init_ntfs_volume - initialize an ntfs_volume
 * @v:		ntfs_volume to initialize
 *
 * Before we can use any of the libntfs higher level functions to open files,
 * load mft records and such like, we need to load the system files but these
 * are files. Catch 22!
 *
 * So we have to cheat, i.e. we setup a minumum set of fields in @v to more or
 * less faked values, just to allow us to use the library functions to open the
 * system files. Perhaps not the most beautiful solution but IMO the most
 * elegant one. Otherwise we would have to keep around special functions to work
 * without all the information required or we would need to do everything
 * manually (as it used to be in the early days of Linux-NTFS).
 *
 * Using a fake-it approach means we still use the same code for everything,
 * even for meta data, which is way cool. It does mean we have some uglyness in
 * the mount code path but hey, it's just the mount... All the real fun comes
 * after mounting anyway.
 *
 * Once we have opened the system files we do the proper initialisation of @v,
 * disposing off the faked info as we go along.
 */
int __init_ntfs_volume(ntfs_volume *v)
{
	return 0;
}

ntfs_volume *ntfs_mount(const char *name, unsigned long rwflag)
{
	const char *OK = "OK";
	const char *FAILED = "FAILED";
	ntfs_volume *vol = NULL;
	NTFS_BOOT_SECTOR *bs = NULL;
	MFT_RECORD *mb = NULL;
	ATTR_RECORD *a;
	VOLUME_INFORMATION *vinf;
	__u32 u;
	__s64 l;
	ssize_t br;
	__u8 sectors_per_cluster, bits;
	__s8 c;
	attr_search_context ctx;
	MFT_REF mref;
	int err, ro = rwflag & MS_RDONLY;

	if (!name) {
		errno = EINVAL;
		return NULL;
	}
	/* Allocate the volume structure. */
	vol = __allocate_ntfs_volume();
	if (!vol) {
		perror("Error allocating memory for ntfs volume structure");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Make a copy of the partition name. */
	if (!(vol->dev_name = strdup(name))) {
		perror("Error allocating memory for partition name");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Allocate the boot sector structure. */
	if (!(bs = (NTFS_BOOT_SECTOR *)malloc(sizeof(NTFS_BOOT_SECTOR)))) {
		puts(FAILED);
		perror("Error allocating memory for bootsector");
		errno = ENOMEM;
		goto error_exit;
	}
	printf("Reading bootsector... ");
	if ((vol->fd = open(name, ro ? O_RDONLY: O_RDWR)) == -1) {
		int eo = errno;
		puts(FAILED);
		perror("Error opening partition file");
		errno = eo;
		goto error_exit;
	}
	/* Now read the bootsector. */
	br = ntfs_pread(vol->fd, bs, sizeof(NTFS_BOOT_SECTOR), 1, 0);
	if (br != 1) {
		int eo = errno;
		puts(FAILED);
#define ESTR "Error reading bootsector"
		if (br == -1) {
			perror(ESTR);
			errno = eo;
		} else if (!br) {
			fprintf(stderr, "Error: partition is smaller than " \
					"bootsector size. Weird!\n");
			errno = EINVAL;
		} else {
			fprintf(stderr, ESTR ": unknown error\n");
			errno = EINVAL;
		}
#undef ESTR
		goto error_exit;
	}
	puts(OK);
#ifdef DEBUG
	if (!is_boot_sector_ntfs(bs, 0)) {
#else
	if (!is_boot_sector_ntfs(bs, 1)) {
#endif
		fprintf(stderr, "Error: %s is not a valid NTFS partition!\n",
				name);
		errno = EINVAL;
		goto error_exit;
	}
	/* The bounds checks on mft_lcn and mft_mirr_lcn (i.e. them being
	 * below or equal the number_of_clusters) really belong in the
	 * is_boot_sector_ntfs but in this way we can just do this once. */
	sectors_per_cluster = bs->bpb.sectors_per_cluster;
#ifdef DEBUG
	printf("NumberOfSectors = %Li\n", sle64_to_cpu(bs->number_of_sectors));
	printf("SectorsPerCluster = 0x%x\n", sectors_per_cluster);
#endif
	if (sectors_per_cluster & sectors_per_cluster - 1) {
		fprintf(stderr, "Error: %s is not a valid NTFS partition! "
				"sectors_per_cluster is not a power of 2.\n",
				name);
		errno = EINVAL;
		goto error_exit;
	}
	vol->number_of_clusters = sle64_to_cpu(bs->number_of_sectors) >>
						ffs(sectors_per_cluster) - 1;

	vol->mft_lcn = sle64_to_cpu(bs->mft_lcn);
	vol->mftmirr_lcn = sle64_to_cpu(bs->mftmirr_lcn);
#ifdef DEBUG
	printf("MFT LCN = %Li\n", vol->mft_lcn);
	printf("MFTMirr LCN = 0x%Li\n", vol->mftmirr_lcn);
#endif
	if ((vol->mft_lcn > vol->number_of_clusters) ||
	    (vol->mftmirr_lcn > vol->number_of_clusters)) {
		fprintf(stderr, "Error: %s is not a valid NTFS partition! "
				"($Mft LCN or\n$MftMirr LCN is greater than "
				"the number of clusters!\n", name);
		errno = EINVAL;
		goto error_exit;
	}
	vol->cluster_size = sectors_per_cluster *
					le16_to_cpu(bs->bpb.bytes_per_sector);
	if (vol->cluster_size & vol->cluster_size - 1) {
		fprintf(stderr, "Error: %s is not a valid NTFS partition! "
				"cluster_size is not a power of 2.\n", name);
		errno = EINVAL;
		goto error_exit;
	}
	vol->cluster_size_bits = ffs(vol->cluster_size) - 1;
	/*
	 * Need to get the clusters per mft record and handle it if it is
	 * negative. Then calculate the mft_record_size. A value of 0x80 is
	 * illegal, thus signed char is actually ok!
	 */
	c = bs->clusters_per_mft_record;
#ifdef DEBUG
	printf("ClusterSize = 0x%x\n", vol->cluster_size);
	printf("ClusterSizeBits = %u\n", vol->cluster_size_bits);
	printf("ClustersPerMftRecord = 0x%x\n", c);
#endif
	/*
	 * When clusters_per_mft_record is negative, it means that it is to
	 * be taken to be the negative base 2 logarithm of the mft_record_size
	 * min bytes. Then:
	 *   	 mft_record_size = 2^(-clusters_per_mft_record) bytes.
	 */
	if (c < 0)
		vol->mft_record_size = 1 << -c;
	else
		vol->mft_record_size = vol->cluster_size * c;
	if (vol->mft_record_size & vol->mft_record_size - 1) {
		fprintf(stderr, "Error: %s is not a valid NTFS partition! "
				"mft_record_size is not a power of 2.\n", name);
		errno = EINVAL;
		goto error_exit;
	}
	vol->mft_record_size_bits = ffs(vol->mft_record_size) - 1;
	vol->max_open_files = 10000;
#ifdef DEBUG
	printf("MftRecordSize = 0x%x\n", vol->mft_record_size);
	printf("MftRecordSizeBits = %u\n", vol->mft_record_size_bits);
	printf("max_open_files set to %u\n", vol->max_open_files);
#endif
	/* Start with $Mft. */
	printf("Loading $Mft... ");
	if (!(mb = (MFT_RECORD *)malloc(vol->mft_record_size))) {
		puts(FAILED);
		perror("Error allocating memory for $Mft");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Can't use any of the higher level functions yet! */
	br = mst_pread(vol->fd, (char *)mb, vol->mft_record_size, 1, 
					vol->mft_lcn << vol->cluster_size_bits);
	if (br != 1) {
		puts(FAILED);
#define ESTR "Error reading $Mft"
		if (br == -1)
			perror(ESTR);
		else if (br < 4)
			fprintf(stderr, "Error: $Mft LCN is outside of "
					"the partition?!?\n");
		else
	                fprintf(stderr, ESTR ": unknown error\n");
#undef ESTR
		errno = EIO;
		goto error_exit;
	}
	if (is_baad_record(mb)) {
		puts(FAILED);
		fprintf(stderr, "Error: Incomplete multi sector "
		                "transfer detected in $Mft.\nCannot "
		                "handle this yet. )-:\n");
		errno = EIO;
		goto error_exit;
	}
	if (!is_mft_recordp(mb)) {
		puts(FAILED);
		fprintf(stderr, "Error: Invalid mft record for $Mft.\n"
				"Cannot handle this yet. )-:\n");
		errno = EIO;
		goto error_exit;
	}
	/* Find the bitmap attribute. */
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = mb;
	ctx.attr = (ATTR_RECORD*)((char*)mb + le16_to_cpu(mb->attrs_offset));
	if (p2n(ctx.attr) < p2n(mb) ||
	    (char*)ctx.attr > (char*)mb + vol->mft_record_size) {
		puts(FAILED);
		fprintf(stderr, "Error: corrupt mft record for $Mft.\n"
				"Cannot handle this yet. )-:\n");
		errno = EIO;
		goto error_exit;
	}
	ctx.is_first = TRUE;
	if (!find_attr($BITMAP, NULL, 0, 0, NULL, 0, NULL, 0, &ctx)) {
		puts(FAILED);
		fprintf(stderr, "$bitmap attribute not found in $Mft?!?\n");
		errno = EIO;
		goto error_exit;
	}
	a = ctx.attr;
	/* Get attribute value size and allocate a big enough buffer. */
	l = get_attribute_value_length(a);
	if (!l) {
		puts(OK);
		puts("Error: $bitmap in $Mft has zero length! T'is weird!");
		errno = EIO;
		goto error_exit;
	}
	vol->mft_bitmap = (__u8*)malloc(l);
	if (!vol->mft_bitmap) {
		puts(FAILED);
		puts("Not enough memory to load bitmap attribute from $Mft.");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Read in the bitmap attribute value into the buffer. */
	if (l != get_attribute_value(vol, mb, a, vol->mft_bitmap)) {
		puts(FAILED);
		puts("Amount of data read does not correspond to expected "
		     "length!");
		errno = EIO;
		goto error_exit;
	}
#ifdef DEBUG
	{ int _i_;
	printf("$MFT $BITMAP =\n");
	for (_i_ = 0; _i_ < l; _i_++)
		printf("0x%s%x%s", vol->mft_bitmap[_i_] < 0x10 ? "0" : "",
				vol->mft_bitmap[_i_], _i_ % 8 ? ", " : "\n");
	printf("\n");
	}
#endif
	/* Find the $DATA attribute in $Mft. */
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = mb;
	ctx.attr = (ATTR_RECORD*)((char*)mb + le16_to_cpu(mb->attrs_offset));
	ctx.is_first = TRUE;
	if (!find_attr($DATA, NULL, 0, 0, NULL, 0, NULL, 0, &ctx)) {
		puts(FAILED);
		fprintf(stderr, "$DATA attribute not found in $Mft?!?\n");
		errno = EIO;
		goto error_exit;
	}
	a = ctx.attr;
	/* Determine the number of mft records in $Mft. */
	vol->number_of_mft_records = a->data_size >> vol->mft_record_size_bits;
	/* The $DATA attribute of the $Mft has to be non-resident. */
	if (!a->non_resident) {
		puts(FAILED);
		fprintf(stderr, "$Mft $DATA attribute is resident!?!\n");
		errno = EINVAL;
		goto error_exit;
	}
	/* Get the run list. */
	vol->mft_runlist = decompress_run_list(a);
	if (!vol->mft_runlist) {
		puts(FAILED);
		fprintf(stderr, "Error decompressing run list from $Mft $DATA"
			        " attribute.\n");
		errno = EINVAL;
		goto error_exit;
	}
	puts(OK);
	/* Done with the $Mft mft record. */
	free(mb);
	mb = NULL;
	/* Now load the bitmap from $Bitmap. */
	printf("Loading $Bitmap... ");
	mref = (MFT_REF)FILE_$Bitmap;
	if (err = __read_file_record(vol, &mref, &mb, NULL)) {
		puts(FAILED);
		errno = -err;
		goto error_exit;
	}
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = mb;
	/* Find the bitmap (it is in the $DATA attribute). */
	if (!find_first_attr($DATA, NULL, 0, 0, NULL, 0, NULL, 0, &ctx)) {
		puts(FAILED);
		fprintf(stderr, "bitmap attribute not found in $Bitmap?!?\n");
		errno = EIO;
		goto error_exit;
	}
	a = ctx.attr;
	/* Get attribute value size and allocate a big enough buffer.*/
	l = get_attribute_value_length(a);
	if (!l) {
		puts(OK);
		puts("Error: bitmap in $Bitmap has zero length! T'is weird!");
		errno = EIO;
		goto error_exit;
	}
	vol->lcn_bitmap = (__u8*)malloc(l);
	if (!vol->lcn_bitmap) {
		puts(FAILED);
		puts("Not enough memory to load $Bitmap.");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Read in the bitmap attribute value into the buffer. */
	if (l != get_attribute_value(vol, mb, a, vol->lcn_bitmap)) {
		puts(FAILED);
		puts("Amount of data read does not correspond to expected "
		     "length!");
		errno = EIO;
		goto error_exit;
	}
	/* Done with the $BitMap mft record. */
	puts(OK);
	free(mb);
	mb = NULL;
	/* Now load the upcase table from $UpCase. */
	printf("Loading $UpCase... ");
	mref = (MFT_REF)FILE_$UpCase;
	if (err = __read_file_record(vol, &mref, &mb, NULL)) {
		puts(FAILED);
		errno = -err;
		goto error_exit;
	}     
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = mb;
	/* Find the bitmap (it is in the $DATA attribute). */
	if (!find_first_attr($DATA, NULL, 0, 0, NULL, 0, NULL, 0, &ctx)) {
		puts(FAILED);
		fprintf(stderr, "$DATA attribute not found in $UpCase?!?\n");
		errno = EIO;
		goto error_exit;
	}
	a = ctx.attr;
	/* Get attribute value size and allocate a big enough buffer.*/
	l = get_attribute_value_length(a);
	if (!l) {
		puts(FAILED);
		puts("Error: $DATA in $UpCase has zero length! T'is weird!");
		errno = EIO;
		goto error_exit;
	}
	/* Note: Normally, the upcase table has a length equal to 65536
	 * 2-byte Unicode characters but allow for different cases, so no
	 * checks done. Just check we don't overflow 32-bits worth of Unicode
	 * characters. */
	if (l & ~0x1ffffffffULL) {
		puts(FAILED);
		puts("Error: Upcase table is too big (max 32-bit allowed).");
		errno = EINVAL;
		goto error_exit;
	}
	vol->upcase_len = l >> 1;
	vol->upcase = (uchar_t*)malloc(l);
	if (!vol->upcase) {
		puts(FAILED);
		puts("Not enough memory to load $UpCase.");
		errno = ENOMEM;
		goto error_exit;
	}
	/* Read in the $DATA attribute value into the buffer. */
	if (l != get_attribute_value(vol, mb, a, (__u8*)vol->upcase)) {
		puts(FAILED);
		puts("Amount of data read does not correspond to expected "
		     "length!");
		errno = EIO;
		goto error_exit;
	}
	/* Done with the $UpCase mft record. */
	puts(OK);
	free(mb);
	mb = NULL;
	/* Now load $Volume and set the version information and flags in the
	 * vol structure accordingly. */
	printf("Loading $Volume... ");
	mref = (MFT_REF)FILE_$Volume;
	if (err = __read_file_record(vol, &mref, &mb, NULL)) {
		puts(FAILED);
		errno = -err;
		goto error_exit;
	}
	memset(&ctx, 0, sizeof(attr_search_context));
	ctx.mrec = mb;
	/* Find the $VOLUME_INFORMATION attribute. */
	if (!find_first_attr($VOLUME_INFORMATION, NULL, 0, 0, NULL, 0, NULL, 0,
				&ctx)) {
		puts(FAILED);
		fprintf(stderr, "$VOLUME_INFORMATION attribute not found in "
				"$Volume?!?\n");
		errno = EIO;
		goto error_exit;
	}
	a = ctx.attr;
	/* Has to be resident. */
	if (a->non_resident) {
		fprintf(stderr, "Error: Attribute $VOLUME_INFORMATION must " \
				"be resident (and it isn't)!\n");
		errno = EIO;
		goto error_exit;
	}
	/* Get a pointer to the value of the attribute. */
	vinf = (VOLUME_INFORMATION*)(le16_to_cpu(a->value_offset) + (char*)a);
	/* Sanity checks. */
	if ((char*)vinf + le32_to_cpu(a->value_length) > 
				le16_to_cpu(mb->bytes_in_use) + (char*)mb ||
				le16_to_cpu(a->value_offset) + 
				le32_to_cpu(a->value_length) >
						le32_to_cpu(a->length)) {
		fprintf(stderr, "Error: Attribute $VOLUME_INFORMATION in " \
				"$Volume is corrupt!\n");
		errno = EIO;
		goto error_exit;
	}
	/* Setup vol from the volume information attribute value. */
	vol->major_ver = vinf->major_ver;
	vol->minor_ver = vinf->minor_ver;
	/* Do not use le16_to_cpu() macro here as our VOLUME_FLAGS are
	   defined using cpu_to_le16() macro and hence are consistent. */
	vol->flags = vinf->flags;
	/* Phew! Done. */

	/* FIXME: Need to initialise vol->vol_name as well. */

	puts(OK);
	free(mb);
	mb = NULL;

	/* FIXME: Need to deal with FILE_$AttrDef. */

	return vol;

error_exit:
	{
		int eo = errno;
		
		if (bs)
			free(bs);
		if (mb)
			free(mb);
		if (vol) {
			if (vol->fd)
				close(vol->fd);
			if (vol->dev_name)
				free(vol->dev_name);
			if (vol->vol_name)
				free(vol->vol_name);
			if (vol->lcn_bitmap)
				free(vol->lcn_bitmap);
			if (vol->mft_bitmap)
				free(vol->mft_bitmap);
			if (vol->mft_runlist)
				free(vol->mft_runlist);
			if (vol->upcase)
				free(vol->upcase);
			free(vol);
		}
		errno = eo;
	}
	return NULL;
}

BOOL ntfs_umount(ntfs_volume *vol, const int force)
{
	if (!vol) {
		errno = EINVAL;
		return FALSE;
	}
	/* FIXME: Do the cleanup. */
	if (vol->fd)
		close(vol->fd);
	if (vol->dev_name)
		free(vol->dev_name);
	if (vol->vol_name)
		free(vol->vol_name);
	if (vol->lcn_bitmap)
		free(vol->lcn_bitmap);
	if (vol->mft_bitmap)
		free(vol->mft_bitmap);
	if (vol->mft_runlist)
		free(vol->mft_runlist);
	if (vol->upcase)
		free(vol->upcase);
	free(vol);
	return TRUE;
}

