/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT44App.h"

#include <wx/config.h>
#include <wx/ffile.h>

IMPLEMENT_APP(CJT44App)

#include "common/Exception.h"
#include "common/Log.h"

#include "jt44/JT44Defs.h"

CJT44App::CJT44App() :
wxApp(),
m_frame(NULL),
m_controller(NULL),
m_messages(NULL),
m_file(NULL),
m_saving(false),
m_recording(false),
m_txEnable(DEFAULT_TX_ENABLED)
{
}

CJT44App::~CJT44App()
{
}

bool CJT44App::OnInit()
{
	SetVendorName(JT44_VENDOR_NAME);
	SetAppName(JT44_APPL_NAME);

	wxConfigBase* config = wxConfigBase::Get();
	config->SetRecordDefaults();

	wxLog* log = new CLog(JT44_LOGFILE_NAME);
	wxLog::SetActiveTarget(log);

	createController();
	::wxSleep(1);

	createFileReader();

	readRegistry();

	m_frame = new CJT44Frame();
	m_frame->Show();

	SetTopWindow(m_frame);

	return true;
}

int CJT44App::OnExit()
{
	wxASSERT(m_controller != NULL);

	m_controller->Delete();
	m_file->Delete();

	delete wxConfigBase::Set(NULL);

	return 0;
}

#ifdef __WXDEBUG__
void CJT44App::OnAssert(const wxChar* file, int line, const wxChar* cond, const wxChar* msg)
{
	::wxLogFatalError(wxT("Assertion failed on line %d in file %s: %s %s"), line, file, cond, msg);
}
#endif

void CJT44App::receiveCardMessage(CJT44Message* message) const
{
	wxASSERT(m_frame != NULL);
	wxASSERT(message != NULL);

	m_frame->showCardMessage(message);
}

void CJT44App::receiveFileMessage(CJT44Message* message) const
{
	wxASSERT(m_frame != NULL);
	wxASSERT(message != NULL);

	m_frame->showFileMessage(message);
}

void CJT44App::logMessage(const CJT44Message& message) const
{
	if (m_saving) {
		wxString     id = message.getId();
		int     quality = message.getQuality();
		double       Dt = message.getDt();
		int          DF = message.getDF();
		double strength = message.getStrength();

		wxString   text = message.getText();
		wxString oeText = message.getOddEvenText();
		wxString l4Text = message.getLast4Text();

		wxFFile* file = new wxFFile(JT44_MESSAGES_NAME, "a");

		if (!file->IsOpened()) {
			wxString text;
			text.Printf(wxT("Cannot open save text file: %s"), JT44_MESSAGES_NAME.c_str());
			error(text);

			delete file;
			return;
		}

		wxString line;
		line.Printf(wxT("%s  %d  %.0f  %.1f  %d  %s  %s  %s\n"), id.c_str(), quality, strength - 0.5, Dt, DF, text.c_str(), oeText.c_str(), l4Text.c_str());

		file->Write(line);
		file->Close();

		delete file;
	}
}

void CJT44App::showCardLevels(CJT44Levels* levels) const
{
	wxASSERT(levels != NULL);
	wxASSERT(m_frame != NULL);

	m_frame->showCardLevels(levels);
}

void CJT44App::showFileLevels(CJT44Levels* levels) const
{
	wxASSERT(levels != NULL);
	wxASSERT(m_frame != NULL);

	m_frame->showFileLevels(levels);
}

void CJT44App::error(const wxString& text) const
{
	wxASSERT(m_frame != NULL);

	m_frame->error(text);
}

void CJT44App::getPersonal(wxString& callsign, wxString& locator) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString callsignKey = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_CALLSIGN;
	wxString locatorKey  = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_LOCATOR;

	profile->Read(callsignKey, &callsign, DEFAULT_CALLSIGN);
	profile->Read(locatorKey,  &locator,  DEFAULT_LOCATOR);
}

void CJT44App::setPersonal(const wxString& callsign, const wxString& locator) const
{
	CText::setLocalCallsign(callsign);
	CText::setLocalLocator(locator);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString callsignKey = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_CALLSIGN;
	wxString locatorKey  = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_LOCATOR;

	profile->Write(callsignKey, callsign);
	profile->Write(locatorKey,  locator);
	profile->Flush();
}

int CJT44App::getBand() const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_BAND;

	long band;
	profile->Read(key, &band, DEFAULT_BAND);

	return int(band);
}

void CJT44App::setBand(int band) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_BAND;

	profile->Write(key, long(band));
	profile->Flush();
}

void CJT44App::setRemote(const wxString& callsign, const wxString& report)
{
	if (callsign.IsEmpty())
		CText::setRemoteCallsign(wxEmptyString);
	else
		CText::setRemoteCallsign(callsign);

	if (report.IsEmpty())
		CText::setLocalReport(wxEmptyString);
	else
		CText::setLocalReport(report);
}

bool CJT44App::getTXFirst() const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_TRANSMIT + wxT("/") + KEY_FIRST;

	bool txFirst;
	profile->Read(key, &txFirst, DEFAULT_TX_FIRST);

	return txFirst;
}

void CJT44App::setTXFirst(bool txFirst) const
{
	wxASSERT(m_controller != NULL);

	m_controller->setTXFirst(txFirst);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_TRANSMIT + wxT("/") + KEY_FIRST;

	profile->Write(key, txFirst);
	profile->Flush();
}

bool CJT44App::getTXEnable() const
{
	return m_txEnable;
}

void CJT44App::setTXEnable(bool txEnable)
{
	wxASSERT(m_controller != NULL);

	m_controller->setTXEnable(txEnable);

	m_txEnable = txEnable;
}

int CJT44App::getMinSync() const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_RECEIVE + wxT("/") + KEY_MIN_SYNC;

	long minSync;
	profile->Read(key, &minSync, DEFAULT_MIN_SYNC);

	return int(minSync);
}

void CJT44App::setMinSync(int minSync) const
{
	wxASSERT(m_controller != NULL);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_RECEIVE + wxT("/") + KEY_MIN_SYNC;

	profile->Write(key, minSync);
	profile->Flush();
}

wxString CJT44App::getReplacedMessage(int n) const
{
	wxASSERT(n >= 0 && n < JT44_MAX_MESSAGES);
	wxASSERT(m_messages[n] != NULL);

	return m_messages[n]->getReplacedText(JT44_MESSAGE_LENGTH);
}

wxString CJT44App::getOriginalMessage(int n) const
{
	wxASSERT(n >= 0 && n < JT44_MAX_MESSAGES);
	wxASSERT(m_messages[n] != NULL);

	return m_messages[n]->getOriginalText();
}

void CJT44App::getMessage(int n, wxString& message) const
{
	wxASSERT(n >= 0 && n < JT44_MAX_MESSAGES);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString number;
	number.Printf(wxT("%d"), n);

	wxString key = wxT("/") + SECTION_MESSAGES + wxT("/") + KEY_MESSAGE + number;
	wxString defMessage;

	switch (n) {
		case 0:
			defMessage = DEFAULT_MESSAGE_0;
			break;
		case 1:
			defMessage = DEFAULT_MESSAGE_1;
			break;
		case 2:
			defMessage = DEFAULT_MESSAGE_2;
			break;
		case 3:
			defMessage = DEFAULT_MESSAGE_3;
			break;
		case 4:
			defMessage = DEFAULT_MESSAGE_4;
			break;
		case 5:
			defMessage = DEFAULT_MESSAGE_5;
			break;
		default:
			defMessage = wxEmptyString;
			break;
	}

	profile->Read(key, &message, defMessage);
}

void CJT44App::setMessage(int n, const wxString& message)
{
	wxASSERT(n >= 0 && n < JT44_MAX_MESSAGES);

	if (m_messages[n] != NULL)
		delete m_messages[n];

	m_messages[n] = new CText(message);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString number;
	number.Printf(wxT("%d"), n);

	wxString key = wxT("/") + SECTION_MESSAGES + wxT("/") + KEY_MESSAGE + number;

	profile->Write(key, message);
	profile->Flush();
}

void CJT44App::getPTT(wxString& type, wxString& device) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey   = wxT("/") + SECTION_PTT + wxT("/") + KEY_TYPE;
	wxString deviceKey = wxT("/") + SECTION_PTT + wxT("/") + KEY_DEVICE;

	profile->Read(typeKey,   &type,   DEFAULT_PTT_TYPE);
	profile->Read(deviceKey, &device, DEFAULT_PTT_DEVICE);
}

void CJT44App::setPTT(const wxString& type, const wxString& device) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey   = wxT("/") + SECTION_PTT + wxT("/") + KEY_TYPE;
	wxString deviceKey = wxT("/") + SECTION_PTT + wxT("/") + KEY_DEVICE;

	profile->Write(typeKey,   type);
	profile->Write(deviceKey, device);
	profile->Flush();
}

void CJT44App::getSound(wxString& type, wxString& fileName, int& bits) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey     = wxT("/") + SECTION_SOUND + wxT("/") + KEY_TYPE;
	wxString fileNameKey = wxT("/") + SECTION_SOUND + wxT("/") + KEY_FILENAME;
	wxString bitsKey     = wxT("/") + SECTION_SOUND + wxT("/") + KEY_BITS;

	long longBits;

	profile->Read(typeKey,     &type,     DEFAULT_SOUND_TYPE);
	profile->Read(fileNameKey, &fileName, DEFAULT_SOUND_FILENAME);
	profile->Read(bitsKey,     &longBits, DEFAULT_SOUND_BITS);

	bits = int(longBits);
}

void CJT44App::setSound(const wxString& type, const wxString& fileName, int bits) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey     = wxT("/") + SECTION_SOUND + wxT("/") + KEY_TYPE;
	wxString fileNameKey = wxT("/") + SECTION_SOUND + wxT("/") + KEY_FILENAME;
	wxString bitsKey     = wxT("/") + SECTION_SOUND + wxT("/") + KEY_BITS;

	long longBits = long(bits);

	profile->Write(typeKey,     type);
	profile->Write(fileNameKey, fileName);
	profile->Write(bitsKey,     longBits);
	profile->Flush();
}

void CJT44App::setSaving(bool saving)
{
	m_saving = saving;
}

void CJT44App::deleteMessages() const
{
	::wxRemoveFile(JT44_MESSAGES_NAME);
}

void CJT44App::setRecording(bool recording)
{
	wxASSERT(m_controller != NULL);

	m_recording = recording;

	m_controller->setRXRecording(m_recording);
}

void CJT44App::sendMessage(const wxString& text)
{
	wxASSERT(m_controller != NULL);

	m_controller->sendMessage(text);
}

void CJT44App::readFile(const wxString& fileName)
{
	wxASSERT(m_file != NULL);

	if (!m_file->IsPaused()) {
		error(wxT("Another WAV file is currently being processed"));
		return;
	}

	try {
		m_file->setFileName(fileName);

		m_file->Resume();
		m_file->Pause();
        }
	catch (CException& ex) {
		error(ex.getMessage());
		return;
	}
}

void CJT44App::createController()
{
	try {
		wxString type, fileName;
		int bits;

		getSound(type, fileName, bits);
		CSoundDev* soundDev = CSoundDev::createSoundDev(type, fileName, JT44_SAMPLE_RATE, bits);

		getPTT(type, fileName);
		CPTTPort* pttPort = CPTTPort::createPTTPort(type, fileName);

		m_controller = new CJT44Controller(soundDev, pttPort);

		m_controller->Create();
		m_controller->Run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "JT44: an error has occured when creating the sound device or the PTT port: %s\n", ex.getMessage().c_str());
		throw;
	}
}

void CJT44App::createFileReader()
{
	m_file = new CJT44FileReceive();

	m_file->Create();
	m_file->Run();
	m_file->Pause();
}

void CJT44App::readRegistry()
{
	wxASSERT(m_controller != NULL);

	wxString callsign, locator;
	getPersonal(callsign, locator);
	CText::setLocalCallsign(callsign);
	CText::setLocalLocator(locator);

	bool txEnable = getTXEnable();
	m_controller->setTXEnable(txEnable);

	bool txFirst = getTXFirst();
	m_controller->setTXFirst(txFirst);

	m_messages = new CText*[JT44_MAX_MESSAGES];

	for (int i = 0; i < JT44_MAX_MESSAGES; i++) {
		wxString message;
		getMessage(i, message);
		m_messages[i] = new CText(message);
	}

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	profile->Flush();
}
