###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###

import gtk
import gobject
import xml.sax.saxutils
from datetime import datetime
import pango

from db_manager import DBManager
from misc_widget import MultiDragTreeview
from misc_widget import get_scrolled_window
from view_song import TreeviewSong
from podcast_manager import PodcastManager
import config
import utils

WATCHER_TIME = 500

class PodcastBrowserView(gtk.VBox):
    def __init__(self,media_organizer):
        gtk.VBox.__init__(self,False,6)

        self.pane = gtk.VPaned()

        self.media_organizer = media_organizer
        self.feed_view = MultiDragTreeview()
        self.song_view = PodcastSongview(media_organizer,self)

        self.feed_model = gtk.ListStore(object,gobject.TYPE_STRING,gobject.TYPE_INT,gobject.TYPE_STRING)

        self.feed_view.append_column(gtk.TreeViewColumn(_("Feed"),gtk.CellRendererText(),markup=1))
        self.feed_view.append_column(gtk.TreeViewColumn(_("Pod."),gtk.CellRendererText(),text=2))
        self.feed_view.get_column(0).set_expand(True)
        self.feed_view.get_column(0).set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        self.feed_view.get_column(1).get_cell_renderers()[0].set_property("xalign",1)
        self.feed_view.set_model(self.feed_model)

        self.feed_view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)

        self.feed_view.get_selection().connect("changed", self.on_row_selected)
        self.feed_view.connect("row-activated", self.on_row_song_activated)
        targets = [("text/uri-list", 0, 0)]
        self.feed_view.drag_source_set(
            gtk.gdk.BUTTON1_MASK, targets,
            gtk.gdk.ACTION_COPY)
        self.feed_view.connect("drag-data-get", self.on_drag_data_get)
        self.feed_view.connect("popup-menu",self.__popup_menu)
        self.connect("key-press-event", self.on_key_press)

        self.pane.pack1(get_scrolled_window(self.feed_view),False,True)
        self.pane.pack2(get_scrolled_window(self.song_view),True,False)
        self.pane.compute_position(-1,200,200)
        self.pane.set_position(200)

        self.download_progress = gtk.ProgressBar()
        self.label_download = gtk.Label()
        self.label_download.set_alignment(0,0.5)
        bbox = gtk.HBox(False,12)
        bbox.pack_start(gtk.Button(stock=gtk.STOCK_REFRESH),False,False)
        bbox.get_children()[0].connect("clicked",self.refresh_all)
        if False:
            bbox.pack_start(gtk.Button(stock=gtk.STOCK_ADD),False,False)
            bbox.get_children()[1].connect("clicked",PodcastManager.big_test)

        bbox.pack_start(self.label_download,False,False)
        bbox.pack_start(self.download_progress,True,True)

        self.pack_start(self.pane,True,True)
        self.pack_start(bbox,False,False)

        PodcastManager.connect("podcast-updated",self.podcast_updated_cb)
        self.songs_cache = None
        self.in_update = False
        self.selected = []
        self.__sig=None
        self.__id_download_podcast_watcher = None
        self.__id_download_feed_watcher = None
        self.menu = self.get_menu()

    def on_key_press(self, widget, event):
        if event.keyval == gtk.gdk.keyval_from_name("Delete"):
            self.remove_selected()
            return True

    def podcast_updated_cb(self,widget,list_song):
        for song in list_song:
            if song in self.songs_cache:
                self.songs_cache[self.songs_cache.index(song)] = song
            else:
                self.songs_cache.append(song)
                PodcastManager.download_podcast(song)
                if self.__id_download_podcast_watcher!=None: gobject.source_remove(self.__id_download_podcast_watcher)
                self.__id_download_podcast_watcher = gobject.timeout_add(WATCHER_TIME,self.download_podcast_watcher)
        self.fill(self.songs_cache)

    def refresh_cache(self,DB=DBManager):
        """
        Warning DB parameter is only for boost startup
        If more than one DB used :
         you will have a sqlite thread problem
         OR a risk to desyncronize file with treeview and loose information
        """
        self.songs_cache = DB.get_podcasts()
        for song in self.songs_cache:
            PodcastManager.download_podcast(song)
        if self.__id_download_podcast_watcher!=None: gobject.source_remove(self.__id_download_podcast_watcher)
        self.__id_download_podcast_watcher = gobject.timeout_add(WATCHER_TIME,self.download_podcast_watcher)

    def download_podcast_watcher(self):
        elapsed = len(PodcastManager.podcast_uri_pool)
        total = len(self.songs_cache)
        current = total - elapsed
        self.label_download.set_text("%d/%d"%(current,total)+" "+_("Podcast downloaded"))
        ratio = PodcastManager.current_podcast_ratio
        if ratio>1:ratio=1
        self.download_progress.set_text("%d%%"%(ratio*100))
        self.download_progress.set_fraction(ratio)
        if elapsed==0:
            self.download_progress.set_text("100%")
            self.download_progress.set_fraction(1.0)
            return False
        else:
            return True

    def download_feed_watcher(self):
        elapsed = len(PodcastManager.feed_task_pool)
        total = len(self.feed_model)-1
        current = total - elapsed
        self.label_download.set_text("%d/%d"%(current,total)+" "+_("Feeds refreshed"))
        if total==0:
            ratio=1
        else:
            ratio = float(current)/float(total)
        if ratio>1:ratio=1
        self.download_progress.set_text("%d%%"%(ratio*100))
        self.download_progress.set_fraction(ratio)
        if elapsed==0:
            self.download_progress.set_text("100%")
            self.download_progress.set_fraction(1.0)
            return False
        else:
            return True

    def fill(self,songs):
        values = {}

        self.feed_view.set_model(None)
        self.feed_model.clear()
        for song in songs:
            values.setdefault("<###ALL###>",{"<###songs###>":set(),"<###feed_uri###>":None})
            values["<###ALL###>"]["<###songs###>"].add(song)
            values.setdefault(song.get_property("podcast_feed_title"),{"<###songs###>":set(),"<###feed_uri###>":song.get_property("podcast_feed_uri")})
            values[song.get_property("podcast_feed_title")]["<###songs###>"].add(song)

        if values.has_key("<###ALL###>"):
            all = values["<###ALL###>"]
            del values["<###ALL###>"]
        else: all = {"<###songs###>":set(),"<###feed_uri###>":None}


        keys = values.keys()
        keys.sort()
        def get_tuple(name,value):
            return (value["<###songs###>"],name,len(value["<###songs###>"]),value["<###feed_uri###>"])

        iter_to_select = []
        for key in keys:
            iter = self.feed_model.append(get_tuple(utils.xmlescape(key),values[key]))
            if key in self.selected:
                iter_to_select.append(iter)

        iter_all = self.feed_model.insert(0,get_tuple("<b>"+utils.xmlescape(_("All"))+"</b>",all))

        self.feed_view.set_model(self.feed_model)

        if len(iter_to_select)==0:
            self.feed_view.get_selection().select_iter(iter_all)
        else:
            first = True
            for iter in iter_to_select:
                if first:
                    self.feed_view.scroll_to_cell(self.feed_model.get_path(iter))
                    first=False
                self.feed_view.get_selection().select_iter(iter)


    def populate(self):
        self.fill(self.songs_cache)
        return

    def __popup_menu(self,widget):
        self.menu.popup(None,None,None,0,gtk.get_current_event_time())

    def play_selected(self,*data):
        model, rows = self.feed_view.get_selection().get_selected_rows()
        if len(rows)<0:
            songs = self.songs_cache
        else:
            setsongs = list([model[row[0]][0] for row in rows])
            songs = []
            for setsong in setsongs:
                songs.extend(list(setsong))

        play = True
        for song in songs:
            self.media_organizer.player.playlist.add_song(song,False,play)
            play = False

    def enqueue_selected(self,*data):
        model, rows = self.feed_view.get_selection().get_selected_rows()
        if len(rows)<0:
            songs = self.songs_cache
        else:
            setsongs = list([model[row[0]][0] for row in rows])
            songs = []
            for setsong in setsongs:
                songs.extend(list(setsong))

        for song in songs:
            self.media_organizer.player.playlist.add_song(song,False,False)

    def on_row_song_activated(self, treeview, path, view_column):
        self.play_selected()

    def refresh_selected(self,*param):
        if self.songs_cache==None:
            return
        model, rows = self.feed_view.get_selection().get_selected_rows()
        if len(rows) < 1:
            return
        for row in rows:
            if model[row][3]!=None:
                PodcastManager.download_podcast_feed(model[row][3])
        if self.__id_download_feed_watcher!=None: gobject.source_remove(self.__id_download_feed_watcher)
        self.__id_download_feed_watcher = gobject.timeout_add(WATCHER_TIME,self.download_feed_watcher)

    def refresh_all(self,*param):
        if self.songs_cache==None:
            return
        for row in self.feed_model:
            if row[3]!=None:
                PodcastManager.download_podcast_feed(row[3])
        if self.__id_download_feed_watcher!=None: gobject.source_remove(self.__id_download_feed_watcher)
        self.__id_download_feed_watcher = gobject.timeout_add(WATCHER_TIME,self.download_feed_watcher)


    def remove_selected(self,*data):
        model, rows = self.feed_view.get_selection().get_selected_rows()
        if len(rows) < 1 or  model[rows[0]][3]==None:
            return
        for row in rows:
            PodcastManager.remove_podcast_feed(model[row][3])
            for song in model[row][0]:
                if song in self.songs_cache:
                    del self.songs_cache[self.songs_cache.index(song)]
                    DBManager.delete_song(song)
                    PodcastManager.remove_podcast(song)

        #self.refresh_cache()
        self.fill(self.songs_cache)

    def on_row_selected(self,treeselection):
        model,rows = treeselection.get_selected_rows()
        if len(rows)<=0:
            return
            songs = self.songs_cache
        else:
            setsongs = list([model[row[0]][0] for row in rows])
            del self.selected
            self.selected = list([model[row[0]][1] for row in rows])

            songs = []
            for setsong in setsongs:
                songs.extend(list(setsong))

        songs.sort()
        self.song_view.fill(songs)

    def on_drag_data_get(self,treeview, context, selection, info, timestamp):
        model,rows = treeview.get_selection().get_selected_rows()
        if len(rows)<0:
            return
        else:
            setsongs = list([model[row[0]][0] for row in rows])
            songs = []
            for setsong in setsongs:
                songs.extend(list(setsong))

        list_uri = list([ utils.convert_to_uri(song.get_property("uri")) for song in songs])
        self.get_toplevel().last_drag_drop = songs
        selection.set_uris(list_uri)

    def get_menu(self):

        menu_ui = """
            <ui>
                <popup name="SongMenu">
                  <menuitem action="Play"/>
                  <menuitem action="Queue"/>
                  <menuitem action="Delete"/>
                  <menuitem action="Refresh"/>
                  <menuitem action="RefreshAll"/>
                </popup>
            </ui>
            """

        self.uimanager = gtk.UIManager()
        self.uimanager.add_ui_from_string(menu_ui)
        accelgroup = self.uimanager.get_accel_group()

        actiongroup = gtk.ActionGroup('Listen')
        actiongroup.add_actions([('Play', gtk.STOCK_MEDIA_PLAY, _('_Play'), None,
                                         _('Play this song'), self.play_selected)])
        actiongroup.add_actions([('Queue', gtk.STOCK_ADD, _('_Queue'), None,
                                         _('Queue this song'), self.enqueue_selected)])
        actiongroup.add_actions([('Delete', gtk.STOCK_DELETE, _('_Remove from library'), None,
                                         _('Remove this song from the library'), self.remove_selected)])
        actiongroup.add_actions([('Refresh', gtk.STOCK_REFRESH, _('_Refresh feed'), None,
                                         '', self.refresh_selected)])
        actiongroup.add_actions([('RefreshAll', gtk.STOCK_REFRESH, _('_Refresh all feed'), None,
                                         '', self.refresh_all)])

        self.uimanager.insert_action_group(actiongroup, 1)
        #gobject.idle_add(self.get_toplevel().add_accel_group,accelgroup)

        return self.uimanager.get_widget("/SongMenu")




class PodcastSongview(TreeviewSong):
    def __init__(self,media_organizer,browser):
        self.browser = browser
        self.media_organizer = media_organizer
        allow_column={"title":True,"album":False,"artist":False,"track":True,"duration":True}
        TreeviewSong.__init__(self,media_organizer,allow_column)

        """ WARNING track is date for podcast see song.py """
        self.get_column(0).set_title(_("Date"))
        self.set_model(self.model_song)


    def get_tag_by_title(self,title):
        if title == _("Album"): tag = "album"
        elif title == _("Length"): tag = "duration"
        elif title ==  _("Artist"): tag = "artist"
        elif title ==  _("Title"): tag = "title"
        elif title == _("Date"): tag = "tracknr"
        return tag

    def remove_selected(self,*data):
        model, rows = self.get_selection().get_selected_rows()
        if len(rows) < 1:
            return
        rows.reverse()

        for row in rows:
            song = model[row][0]
            self.browser.songs_cache = filter(lambda s: s!=song ,self.browser.songs_cache)
            PodcastManager.remove_podcast(song)
            DBManager.delete_song(song)


        self.browser.fill(self.browser.songs_cache)


    def get_menu_ui(self):
        return """
            <ui>
                <popup name="SongMenu">
                  <menuitem action="Play"/>
                  <menuitem action="Queue"/>
                  <menuitem action="DeleteLibrary"/>
                </popup>
            </ui>
            """
