# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import gtk.glade
import gobject

import threading

import utils
from config import config
from song import Song

from xdg_support import get_xdg_data_file
from library import ListenDB
from widget.dialog import WindowError

GEOMETRY = {}
GEOMETRY["advanced"] = (800, 600)
GEOMETRY["simple"] = (500, 350)

MUSICBRAINZ = False
try: 
    import musicbrainz2 #@UnusedImport
    from tunepimp import tunepimp
    dir(tunepimp.tunepimp).index("setMusicDNSClientId")
except: pass
else: MUSICBRAINZ = True

EDITABLE_TAG = """
title genre artist album
#track date #disc
"""


class SongEditor(object):
    def __init__(self, songs):
        
        #Build copy of song
        self.new_songs = [Song(s) for s in songs]
        
        self.changed = []
        
        self.xml = gtk.glade.XML(get_xdg_data_file("trackedit.glade") , domain="listen")
        gw = self.xml.get_widget
        self.win = gw("WindowTrackInfo")
        
        self.win.set_transient_for(utils.get_main_window())

        single_mode = len(songs) == 1
        if single_mode:
            gw("MultiTrackHeader").hide_all()
            gw("MultiTrackHeader").set_no_show_all(False)
            
        else:
            next = gw("Next")
            next.connect("clicked", self.next)
            previous = gw("Previous")
            previous.connect("clicked", self.previous)
        
        self.pos = -1   
        self.threads_completion = []
        for name in ["Artist", "Album", "Genre", "Date", "Title"]:
            entry = gw(name)
            btn = gw(name + "Sync")
            if name == "Date":
                translate_name = _("Year")
            else:
                translate_name = _(name)
            btn.set_tooltip_text(_("Set all %s to this value") % translate_name)
            if single_mode:
                btn.hide()
            else:
                btn.connect("clicked", self.sync_text, name)
            entry.connect("changed", self.on_text_edit, name)
            
            tag = name.lower()
            if tag in ["album", "artist", "genre"]:
                model_completition = gtk.ListStore(gobject.TYPE_STRING)
                entry_completition = gtk.EntryCompletion()
                entry_completition.set_popup_completion(True)
                entry_completition.set_inline_completion(True)
                entry_completition.set_text_column(0)
                thread = threading.Thread(target=self.thread_populate_completition, args=(entry, tag, model_completition, entry_completition))
            
                self.threads_completion.append(thread)
        for name in ["#Track"]:
            spin = gw(name)
            spin.connect("value-changed", self.on_spin_edit, name)
            btn = gw(name + "Sync")
            btn.set_tooltip_text(_("Set all %s to this value") % _("Track Number"))
            if single_mode:
                btn.hide()
            else:
                btn.connect("clicked", self.sync_spin, name)
                
        for thread in self.threads_completion:
            thread.start()
        
        btn = gw("#TrackIterator")
        btn.set_tooltip_text(_("Set all Track Number Automaticaly"))
        if single_mode:
            btn.hide()
        else:
            btn.connect("clicked", self.on_iter_track)
        if MUSICBRAINZ:
            btn = gw("MuzicBrainz")
            btn.connect("clicked", self.on_musicbrainz)
            
            btn = gw("MuzicBrainzSync")
            btn.set_tooltip_text(_("Fetch informations for all track"))
            if single_mode:
                btn.hide()
            else:
                btn.connect("clicked", self.sync_musicbrainz)
        else:
            gw("MuzicBrainz").hide()
            gw("MuzicBrainzSync").hide()
            parent = gw("MuzicBrainz").get_parent()
            space = gtk.Fixed()
            space.show_all()
            space.set_size_request(160, -1)
            
            parent.pack_start(space, True, True)
        
        self.move_to(0)
        
        
        progress = gw("Progress")
        progress.hide()
        self.cancel_action = None
        
        self.win.connect("destroy", self.quit)
        self.win.resize(int(config.get("song_editor", "width")), int(config.get("song_editor", "height")))
        self.win.show()
        gw("CancelButton").connect("clicked", self.cancel)
        gw("SaveButton").connect("clicked", self.save)
        
    def quit(self, *args, **kargs):
            self.win.destroy()    
        
    def cancel(self, *args, **kargs):
        if self.cancel_action:
            self.cancel_action()
        else:
            self.quit()
        
    def save(self, *args, **kargs):
        self.status(False)
        
        total = len(self.changed)
        progress = self.xml.get_widget("Progress")
        progress.show()
        progress.set_fraction(0)
        progress.set_text("0/%d" % total)
        error = ""
        for i, song in enumerate(self.changed):
            tags_modifiable = {}
            s = ListenDB.get_song(song.get("uri"))
            [ tags_modifiable.update({key:value}) for key , value in song.items() if key in EDITABLE_TAG.split() and s.get(key) != value]

            #print song is s
            if tags_modifiable:
                if  not ListenDB.set_property(s, tags_modifiable, write_to_file=True):
                    error += s.get_path() + "\n"
                    try: error += s.last_error + "\n"
                    except: pass

            progress.set_fraction(round(round(i) / total, 2))
            progress.set_text("%d/%d" % (i, total))
            while gtk.events_pending():gtk.main_iteration()
        
        progress.hide()
        self.status(True)
        size = self.win.get_size()
        config.set("song_editor", "width", str(size[0]))
        config.set("song_editor", "height", str(size[1]))
        if error != "":
            WindowError(_("Can't write metadata for:"), error)
        else:
            self.quit()
            
        
            
    def status(self, status):
        b = self.xml.get_widget("InfoBox")
        b.set_sensitive(status)
        b = self.xml.get_widget("MultiTrackHeader")
        b.set_sensitive(status)
        self.xml.get_widget("CancelButton").set_sensitive(status)
        self.xml.get_widget("SaveButton").set_sensitive(status)

    def sync_musicbrainz(self, w):
        self.status(False)    
        self.xml.get_widget("CancelButton").set_sensitive(True)
        
        total = len(self.new_songs)
        progress = self.xml.get_widget("Progress")
        progress.show()
        progress.set_fraction(0)
        progress.set_text("0/%d" % total)
        
        self.thread_run = True
        cond = threading.Condition()
        def update(i):  
            self.set_to_update([self.new_songs[i]])
            progress.set_fraction(round(round(i + 1) / total, 2))
            progress.set_text("%d/%d" % (i + 1, total))
            self.move_to(i)  
            
        def thread(songs):
            i = 0
            cond.acquire()
            while self.thread_run:
                cond.release()
                s = songs.pop(0)
                s.read_from_musicbrainz()    
                gobject.idle_add(update, i)
                i += 1
                if len(songs) == 0:
                    gobject.idle_add(cancel)
                    cond.acquire()
                    break
                cond.acquire()
            cond.release()
        
        t = threading.Thread(target=thread, args=(self.new_songs[:],))
        t.setDaemon(True)
        
        def cancel():
            cond.acquire()
            self.thread_run = False
            cond.release()
            t.join()
            progress.hide()    
            self.cancel_action = None
            self.status(True)
            
        self.cancel_action = cancel
        
        t.start()
    
    def set_to_update(self, songs):
        for song in songs:
            if song not in self.changed:
                self.changed.append(song)
            
    def on_musicbrainz(self, w):
        self.status(False)    
        s = self.new_songs[self.pos]
        s.read_from_musicbrainz()
        self.fill_song(s)
        self.set_to_update([s])
        self.status(True)
        
    def on_iter_track(self, w):
        for i, s in enumerate(self.new_songs):
            s["#track"] = i + 1
        self.set_to_update(self.new_songs)
        self.move_to(self.pos)
        
    def on_text_edit(self, w, name):
        value = self.xml.get_widget(name).get_text().strip()
        if value.strip() != "":
            self.new_songs[self.pos][name.lower()] = value
        else:
            try: del(self.new_songs[self.pos][name.lower()])
            except KeyError: pass
        self.set_to_update([self.new_songs[self.pos]])

               
    def sync_text(self, w, name): 
        value = self.xml.get_widget(name).get_text().strip()
        for s in self.new_songs:
            if value.strip() != "":
                s[name.lower()] = value
            else:
                try: del(s[name.lower()])
                except KeyError: pass
        self.set_to_update(self.new_songs)
        
    def on_spin_edit(self, w, name):
        value = int(self.xml.get_widget(name).get_value())
        if value != 0:
            self.new_songs[self.pos][name.lower()] = value
        else:
            try: del(self.new_songs[self.pos][name.lower()])
            except KeyError:pass  
        self.set_to_update([self.new_songs[self.pos]])      
             
    def sync_spin(self, w, name):
        value = int(self.xml.get_widget(name).get_value())
        for s in self.new_songs:
            if value == 0:
                try: del(s[name.lower()])
                except KeyError:pass
            else:
                s[name.lower()] = value
        self.set_to_update(self.new_songs)      
            
        
    def next(self, *args, **kargs):
        self.move_to(self.pos + 1)
        
    def previous(self, *args, **kargs):
        self.move_to(self.pos - 1)
        
    def move_to(self, pos):
        self.pos = pos
        gw = self.xml.get_widget
        self.fill_song(self.new_songs[self.pos]) 
        if 0 == self.pos:
            gw("Previous").set_sensitive(False)
        else:
            gw("Previous").set_sensitive(True)
        if len(self.new_songs) - 1 == self.pos:
            gw("Next").set_sensitive(False)
        else:
            gw("Next").set_sensitive(True)
        self.fill_song(self.new_songs[self.pos]) 
        self.update_title()    
        
        
    def update_title(self):
        title = self.xml.get_widget("TitleLabel")
        title.set_markup("<big><b>Edit track %d on %d</b></big>" % (self.pos + 1, len(self.new_songs)))
    
  
    
    def fill_song(self, song):
        gw = self.xml.get_widget
        
        self.set_text(song, "Artist", "artist")
        self.set_text(song, "Album", "album")
        self.set_text(song, "Genre", "genre")
        self.set_text(song, "Date", "date")
        self.set_text(song, "Title", "title")
        
        value = song.get("#track")
        if not value: value = 0
        gw("#Track").set_value(value)
        
        gw("Uri").set_text(song.get_str("uri"))
        gw("FileType").set_text(song.get_ext()[1:])
        gw("LastPlayedLabel").set_markup("<small><i>" + song.get_str("#lastplayed") + "</i></small>")
        gw("PlayCountLabel").set_markup("<small><i>" + song.get_str("#playcount") + "</i></small>")
        gw("DurationLabel").set_markup("<small><i>" + song.get_str("#duration") + "</i></small>")
        gw("AddedLabel").set_markup("<small><i>" + song.get_str("#added") + "</i></small>")
        gw("BitRate").set_text(song.get_str("#bitrate"))

        
        
    def set_text(self, song, name, tag):
        text = song.get(tag)
        if not text: text = ""
        entry = self.xml.get_widget(name)
        entry.set_text(text)
    
    def thread_populate_completition(self, entry, key, model_completition, entry_completition):
        #Get a db descripteur only for read
        entries = set()
        [ entries.add(s.get_str(key)) for s in ListenDB.get_songs("local") ]

        for value in entries:
            model_completition.append((value,))
        gobject.idle_add(entry_completition.set_model, model_completition)
        gobject.idle_add(entry.set_completion, entry_completition)
        

