/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <ctype.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "support.h"
#include "callbacks.h"
#include "chat.h"
#include "server.h"
#include "scheme.h"
#include "commands.h"
#include "log.h"
#include "handler.h"
#include "hotlist.h"
#include "string_list.h"

#define MAX_LINE_LENGTH  150

char *get_word(GtkText * text, int deselect ATTR_UNUSED)
{
  guint start_pos;
  guint end_pos;
  GtkEditable *editable;
  char *str;
  static char result[1024];
  guint cursor;
  char left, right;

  editable = GTK_EDITABLE(text);

  left = right = 0;
  cursor = global.popup_row;
  while (1) {
    if (cursor < 1)
      break;
    if (GTK_TEXT_INDEX(text, cursor - 1) == '<') {
      left = 2;
    }
    if (GTK_TEXT_INDEX(text, cursor - 1) == '[') {
      left = 1;
    }
    if (GTK_TEXT_INDEX(text, cursor - 1) == '{') {
      left = 2;
    }
    if (GTK_TEXT_INDEX(text, cursor - 1) == '\n')
      break;
    if (GTK_TEXT_INDEX(text, cursor - 1) == ' ')
      break;
    cursor--;
  }
  start_pos = cursor;
  while (1) {
    if (cursor >= gtk_text_get_length(text))
      break;
    if (GTK_TEXT_INDEX(text, cursor) == '>') {
      right = 2;
    }
    if (GTK_TEXT_INDEX(text, cursor) == ']') {
      right = 1;
    }
    if (GTK_TEXT_INDEX(text, cursor) == '}') {
      right = 2;
    }
    if (GTK_TEXT_INDEX(text, cursor) == '\n')
      break;
    if (GTK_TEXT_INDEX(text, cursor) == ' ')
      break;
    cursor++;
  }
  end_pos = cursor;

  if (!left && !right)
    return NULL;
  if ((left == 1) && (right == 1))
    return NULL;
  if (left == right) {
    start_pos++;
    end_pos--;
  } else {
    if (left > right) {
      start_pos++;
    } else {
      end_pos--;
    }
  }

  str = gtk_editable_get_chars(GTK_EDITABLE(text), start_pos, end_pos);
  /*
     if (deselect)
     gtk_editable_delete_selection(GTK_EDITABLE(text));
     else
     gtk_editable_select_region(GTK_EDITABLE(text), start_pos, end_pos);
   */
  *result = 0;
  if (str && (strlen(str) < 1024)) strcpy(result, str);
  g_free(str);  // using g_free() as is was allocated with g_strdup();

  return result;
}


char *get_popup_user(int mode)
{
  char *user = NULL;
  transfer_t *transfer;
  file_t *file;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;
  static char result[1024];
  socket_t *socket;
  user_t *userinfo;
  resume_t *resume;
  access_t *access;
  subscription_t* sub;

  if ((mode == M_ONLINE) || (mode == M_HOTLIST)) {
    gtk_clist_get_pixtext(GTK_CLIST(global.popup_list),
			  global.popup_row, 0,
			  &user, &space, &pixmap, &bitmap);
  } else if (mode == M_SEARCH) {
    file = (file_t *) gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
					     global.popup_row);
    if (!file)
      return NULL;
    user = file->user;
  } else if (mode == M_TRANSFER) {
    socket =
	(socket_t *) gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
					    global.popup_row);
    transfer = (transfer_t *) (socket->data);
    if (!transfer)
      return NULL;
    user = transfer->user_info->user;
  } else if ((mode == M_GLOBAL) || (mode == M_STRING_LIST)) {
    gtk_clist_get_text(GTK_CLIST(global.popup_list),
		       global.popup_row, 0, &user);
  } else if (mode == M_WHOIS) {
    user = GTK_LABEL(global.popup_list)->label;
  } else if (mode == M_TEXT) {
    user = get_word(GTK_TEXT(global.popup_list), 1);
  } else if (mode == M_IGNORE) {
    gtk_clist_get_text(GTK_CLIST(global.popup_list),
		       global.popup_row, 0, &user);
  } else if (mode == M_BROWSE) {
    userinfo =
	gtk_object_get_data(GTK_OBJECT(global.popup_list), "userinfo");
    user = userinfo->username;
  } else if (mode == M_RESUME) {
    resume =
	(resume_t *) gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
					    global.popup_row);
    if (resume->socket) {
      transfer = resume->socket->data;
      user = transfer->user_info->user;
    } else {
      user = resume->user;
    }
  } else if (mode == M_ACCESS) {
    access = (access_t *) gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
						 global.popup_row);
    user = access->name;
  } else if (mode == M_SUBSCRIPTION) {
    sub = gtk_clist_get_row_data(GTK_CLIST(global.popup_list),
				 global.popup_row);
    if (sub) user = sub->user;
  }

  if (user) {
    strcpy(result, user);
    return result;
  } else return NULL;
}

GtkWidget *create_online_popup(void)
{
  GtkWidget *popup;
  GtkWidget *item;
  GtkWidget *separator;
  GtkAccelGroup *popup_accels;

  if (global.popup_row >= 0) {
    popup = create_user_popup(M_ONLINE);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
  } else {
    popup = gtk_menu_new();
    gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
    popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));
  }

  item = gtk_menu_item_new_with_label(_("Refresh List"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_refresh_list_activate), NULL);

  separator = gtk_menu_item_new();
  gtk_widget_show(separator);
  gtk_container_add(GTK_CONTAINER(popup), separator);
  gtk_widget_set_sensitive(separator, FALSE);

  item = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

void line_insert_lf(char *line)
{
  char *last_ws = line;
  char *temp = line;
  int length;

  if (!line)
    return;
  length = strlen(line);
  if (length == 0)
    return;
  if (line[length - 1] == '\n') {
    line[length - 1] = 0;
    length--;
  }
  if (length <= MAX_LINE_LENGTH)
    return;

  while (line) {
    while (temp - line <= MAX_LINE_LENGTH) {
      last_ws = temp;
      temp = strpbrk(temp + 1, " \n\t");
      if (!temp)
	break;
      if (*temp == '\n') {
	last_ws = temp;
	break;
      }
    }
    if ((last_ws - line > 0) && (temp)) {
      *last_ws = '\n';
      line = temp = last_ws + 1;
    } else {
      break;
    }
  }
}

void refresh_channels()
{
  GList *dlist;
  chat_page_t *page;

  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = (chat_page_t *) (dlist->data);
    if ((page->type == P_PUBLIC) && page->active)
      send_command(CMD_CLIENT_NAMES_LIST, page->name);
  }
}

void chat_print(char *color, char *text)
{
  if (!text)
    return;
  chat_print_colored(global.current_page, M_PUBLIC, color, text);
  return;
}

void chat_print_ln(char *color, char *text)
{
  if (text)
    chat_print_colored(global.current_page, M_PUBLIC, color, text);
  chat_print_colored(global.current_page, M_PUBLIC, color, "\n");
  return;
}


int scroll_mode(GtkWidget * text)
{
  int value = GTK_ADJUSTMENT(GTK_TEXT(text)->vadj)->value;
  int page_size = GTK_ADJUSTMENT(GTK_TEXT(text)->vadj)->page_size;
  int upper = GTK_ADJUSTMENT(GTK_TEXT(text)->vadj)->upper;

  if (value + page_size < upper)
    return 0;
  else
    return 1;
}

void snap_to_bottom(GtkWidget * text)
{
  GtkAdjustment *vertical_adjustment;

  vertical_adjustment = GTK_ADJUSTMENT(GTK_TEXT(text)->vadj);
  gtk_adjustment_set_value(vertical_adjustment,
			   vertical_adjustment->upper
			   - vertical_adjustment->lower
			   - vertical_adjustment->page_size);
}

int chat_page_thaw(gpointer data)
{
  chat_page_t *page = (chat_page_t *) data;

  if (page->header && (page->type == P_PUBLIC))
    gtk_text_thaw(GTK_TEXT(page->header));
  if (page->wallop)
    gtk_text_thaw(GTK_TEXT(page->wallop));
  if (page->main)
    gtk_text_thaw(GTK_TEXT(page->main));

  if (page->timeout >= 0) {
    gtk_timeout_remove(page->timeout);
    page->timeout = -1;
  }

  if (page->snap & 1)
    snap_to_bottom(page->header);
  if (page->snap & 2)
    snap_to_bottom(page->wallop);
  if (page->snap & 4)
    snap_to_bottom(page->main);

  page->snap = 0;
  page->modified = 0;


  return 1;
}

void chat_page_freeze(chat_page_t * page, int message_type)
{

  if ((message_type == M_TOPIC) && ((page->modified & 1) == 0) &&
      scroll_mode(page->header))
    page->snap |= 1;
  if ((message_type == M_OP) && ((page->modified & 2) == 0) &&
      scroll_mode(page->wallop))
    page->snap |= 2;
  if ((message_type == M_PUBLIC) && ((page->modified & 4) == 0) &&
      scroll_mode(page->main))
    page->snap |= 4;

  if (page->timeout >= 0) {	// update l_freeze
    gtk_timeout_remove(page->timeout);
  } else {			// first l_freeze
    if (page->header && (page->type == P_PUBLIC))
      gtk_text_freeze(GTK_TEXT(page->header));
    if (page->wallop)
      gtk_text_freeze(GTK_TEXT(page->wallop));
    if (page->main)
      gtk_text_freeze(GTK_TEXT(page->main));
  }

  page->timeout = gtk_timeout_add(100, chat_page_thaw, page);
}

GdkColor *get_nick_color(char *nick)
{
  static GdkColor fore;
  char *hex = "3456789abcdef";
  char ncolor[8] = "#000000";
  int i1;
  unsigned long sum = 0;
  char *pos;

  if (!nick)
    return &fore;

  pos = nick;
  while (*pos) {
    if (sum)
      sum *= *pos;
    else
      sum = *pos;
    pos++;
  }
  for (i1 = 1; i1 < 7; i1++) {
    ncolor[i1] = hex[sum % 13];
    sum /= 13;
  }

  gdk_color_parse(ncolor, &fore);

  return &fore;
}

void chat_print_channel(chat_page_t * page, int message_type,
			char *color, char *text)
{
  style_t *style;
  GtkWidget *temp;
  GdkColor *mcolor;

  if (page->type == P_FIX) {
    page = chat_page_search("MOTD", P_OTHER);
    if (!page)
      page = create_other_page("MOTD", "MOTD");
  }
  if (!page)
    return;

  if (message_type == M_TOPIC)
    temp = page->header;
  else if (message_type == M_OP)
    temp = page->wallop;
  else
    temp = page->main;

  if (!color) {
    style = style_get(global.scheme, "user");
    mcolor = get_nick_color(NULL);
  } else if (!strcmp(color, "nick")) {
    style = style_get(global.scheme, "user");
    mcolor = get_nick_color(text);
  } else {
    style = style_get(global.scheme, color);
    mcolor = style->fore;
  }

  chat_page_freeze(page, message_type);
  gtk_text_insert(GTK_TEXT(temp), style->font,
		  mcolor, style->back, text, strlen(text));

  highlight(page, 1);

  return;
}

void chat_print_channel_color(chat_page_t * page, int message_type,
			      char *color, GdkColor * fore,
			      GdkColor * back, char *text)
{
  style_t *style;
  GtkWidget *temp;
  GdkColor *f;
  GdkColor *b;

  if (page->type == P_FIX) {
    page = chat_page_search("MOTD", P_OTHER);
    if (!page)
      page = create_other_page("MOTD", "MOTD");
  }
  if (!page)
    return;

  if (message_type == M_TOPIC)
    temp = page->header;
  else if (message_type == M_OP)
    temp = page->wallop;
  else
    temp = page->main;

  if (!color) {
    style = style_get(global.scheme, "user");
    f = get_nick_color(NULL);
  } else if (!strcmp(color, "nick")) {
    style = style_get(global.scheme, "user");
    f = get_nick_color(text);
  } else {
    style = style_get(global.scheme, color);
    f = style->fore;
  }

  if (fore)
    f = fore;

  if (back)
    b = back;
  else
    b = style->back;

  chat_page_freeze(page, message_type);
  gtk_text_insert(GTK_TEXT(temp), style->font, f, b, text, strlen(text));

  highlight(page, 1);

  return;
}

void chat_print_colored(chat_page_t * page, int message_type,
			char *base_color, char *text)
{
  GdkColor *fgcolor = NULL;
  GdkColor *bgcolor = NULL;
  char *pos2;
  int len;
  char temp_str[2048];
  int index;

  while (1) {
    pos2 = search_highlight_string(text, &len);
    if (!pos2)
      break;

    strncpy(temp_str, text, pos2 - text);
    temp_str[pos2 - text] = 0;
    chat_print_channel_color(page, message_type, base_color,
			     fgcolor, bgcolor, temp_str);

    if ((pos2[0] == 0x03)) {
      if ((pos2[1] == 0) || (pos2[2] == 0) || (pos2[3] == 0))
	return;
      if (pos2[1] == 0x0f) {
	fgcolor = NULL;
	bgcolor = NULL;
	len = 2;
      } else if (pos2[1] == ',') {
	index = ColorTable2(pos2[3] - '0');
	if (index < 0)
	  index = -index;
	bgcolor = &global.color_table[index];
	len = 4;
      } else if (pos2[1] != '-') {
	index = ColorTable(pos2[1] - '0', pos2[2] - '0');
	if (index < 0)
	  index = -index;
	fgcolor = &global.color_table[index];
	len = 3;
      }
    } else {
      strncpy(temp_str, pos2, len);
      temp_str[len] = 0;
      chat_print_channel(page, message_type, "highlight", temp_str);
    }

    text = pos2 + len;
  }
  chat_print_channel_color(page, message_type, base_color, fgcolor,
			   bgcolor, text);

  return;
}

void join_channel(char *name)
{
  send_command(CMD_CLIENT_JOIN, name);
}

void on_hide_show_clicked(GtkButton * button, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *temp;
  GtkPaned *paned;
  int pos;

  temp = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(button), "con_widget"));

  paned = GTK_PANED(global.current_page->toplevel);

  if (GTK_WIDGET_VISIBLE(temp)) {
    pos = (int) gtk_object_get_data(GTK_OBJECT(button), "pan_pos");
    gtk_paned_set_position(paned, pos);
  } else {
    gtk_object_set_data(GTK_OBJECT(button), "pan_pos",
			(gpointer) (paned->child1_size));
    gtk_paned_set_position(paned,
			   paned->child1_size + temp->allocation.width);
  }
}

void create_private_page(char *name)
{
  GtkWidget *scrolledwindow27;
  GtkWidget *text4;
  GtkWidget *tab_label;
  GtkWidget *label;
  GtkNotebook *notebook;
  chat_page_t *page;
  hot_t *hot;

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));

  switch_to_page(1);
  page = chat_page_search(name, P_PRIVATE);
  if (page) {
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			  gtk_notebook_page_num(GTK_NOTEBOOK(notebook), page->toplevel));
    return;
  }

  scrolledwindow27 = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow27);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow27),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  text4 = gtk_text_new(NULL, NULL);
  gtk_signal_connect_after(GTK_OBJECT(text4), "button_press_event",
			   GTK_SIGNAL_FUNC(on_text_button_press_event),
			   NULL);

  gtk_widget_set_style(text4, global.style[2]);

  gtk_widget_show(text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add(GTK_CONTAINER(scrolledwindow27), text4);

  tab_label = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (tab_label);
  gtk_widget_show (tab_label);

  label = gtk_label_new(name);
  gtk_widget_show(label);
  gtk_box_pack_start (GTK_BOX (tab_label), label, TRUE, TRUE, 0);

  gtk_notebook_append_page(notebook, scrolledwindow27, tab_label);
  page = chat_page_new(name, P_PRIVATE);
  page->header = NULL;
  page->main = text4;
  page->wallop = NULL;
  page->mlabel = tab_label;
  page->tlabel = label;
  page->toplevel = scrolledwindow27;
  gtk_object_set_data(GTK_OBJECT(tab_label), "page", page);
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
					      scrolledwindow27));

  hot = hotlist_search_user(name);
  if (hot) {
    if (hot->online == HOT_OFFLINE)
      chat_page_set_active(page, 0);
  } else {
    //    chat_page_set_active(page, 0);    
    hotlist_add(name, NULL, 0);
  }
}

void chat_print_time_stamp(chat_page_t * page, int message_type)
{
  char stime[200] = "";

  if (global.options.timestamps == TIME_NONE)
    return;
  chat_print_channel(page, message_type, "message",
		     current_time(stime, global.options.timestamps));
}

void cmd_public_message_no_log(char *data, int message_type)
{
  char *channel;
  char *user;
  chat_page_t *page;
  char *message;
  user_timestamp_t *stamp;

  channel = arg(data, 0);
  if (!channel)
    return;
  user = arg(NULL, 0);
  if (!user)
    return;
  message = arg(NULL, 1);

  if (global.options.public_ignore &&
      string_list_search(LIST_IGNORE, user))
    return;

  stamp = timestamp_search(global.appearance, user);
  if (!stamp) {
    stamp = timestamp_new(user);
    global.appearance = g_list_append(global.appearance, stamp);
  } else
    timestamp_touch(stamp);

  // searching any type, as this function is also used for
  // private pages
  page = chat_page_search(channel, -1);
  if (!page)
    return;

  chat_print_time_stamp(page, message_type);
  chat_print_channel(page, message_type, "user", "<");

  if (!l_strcasecmp(SERVER->nick, user))
    chat_print_channel(page, message_type, "yourself", user);
  else if (global.options.colored_nicks)
    chat_print_channel(page, message_type, "nick", user);
  else if (string_list_search(LIST_FRIEND, user))
    chat_print_channel(page, message_type, "friend", user);
  else
    chat_print_channel(page, message_type, "user", user);
  chat_print_channel(page, message_type, "user", "> ");

  if (message)
    chat_print_colored(page, message_type, "text", message);
  chat_print_colored(page, message_type, "text", "\n");
}

void send_afk(char *user, afk_t * afk)
{
  static time_t old_time = 0;
  char *message;
  int d, h, m;
  char s1[1024];
  char s2[1024];
  char s3[1024];
  time_t tim;

  // do not send AFK message to enemies
  if (string_list_search(LIST_FRIEND, user))
    return;

  if (!(afk->message[0]))
    return;

  tim = global.current_time;
  // only send one AFK message per second to avoid
  // automatic message loops
  if (tim <= old_time + 1)
    return;
  old_time = tim;

  tim -= afk->since;		// seconds
  tim /= 60;			// minutes
  m = tim % 60;			// %minutes
  tim /= 60;			// hours
  h = tim % 24;			// %hours
  d = tim / 24;			// days
  if (d)
    sprintf(s1, "%d days ", d);
  else
    *s1 = 0;
  if (h || d)
    sprintf(s2, "%d hours ", h);
  else
    *s2 = 0;
  if (m || h || d)
    sprintf(s3, "%d minutes", m);
  else
    sprintf(s3, "%d minutes", 0);
  message = l_strdup_printf("AFK: %s [%s%s%s]", afk->message, s1, s2, s3);
  send_private(user, message, 1);
  l_free(message);
}

int check_channel_wallop(char *data)
{
  char *pos;
  char *command;
  char *nick;
  char *message;
  char *channel;
  chat_page_t *page;

  pos = strchr(data, ' ');
  if (!pos)
    return 0;
  if (strncmp(pos, " [ops/", 5))
    return 0;

  nick = arg(data, 0);
  pos = arg(NULL, 0);
  message = arg(NULL, 1);
  channel = strchr(pos, '/') + 1;
  pos = strchr(channel, ']');
  *pos = 0;

  page = chat_page_search(channel, P_PUBLIC);
  if (!page)
    return 1;

  if (global.options.piping & PIPE_CHWALLOP) {
    command = l_strdup_printf("%s %s %s", channel, nick, message);
    cmd_public_message_no_log(command, M_OP);
    l_free(command);
  } else {
    int old_stamp = global.options.timestamps;
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print_channel(page, M_PUBLIC, "message", _("(Channel op) "));
    global.options.timestamps = TIME_NONE;
    command = l_strdup_printf("%s %s %s", channel, nick, message);
    cmd_public_message_no_log(command, M_PUBLIC);
    l_free(command);
    global.options.timestamps = old_stamp;
  }
  log(channel, LOG_CHANNEL, "<%s> [Wallop] %s\n", nick, message);

  return 1;
}

int check_server_join_quit(char *data)
{
  char *temp;
  char *server;
  char *str1;
  char *str2;
  int mode = 0;

  temp = l_strdup(data);
  str1 = arg(temp, 0);
  server = arg(NULL, 0);
  str2 = arg(NULL, 1);

  if (!str1)
    goto done_false;
  if (!server)
    goto done_false;
  if (!str2)
    goto done_false;

  if (l_strcasecmp(str1, "Server"))
    goto done_false;
  if (!strncmp(str2, "has quit", 8))
    mode = 1;
  if (!strncmp(str2, "has joined", 10))
    mode = 2;
  if (mode == 0)
    goto done_false;
  if (mode == 2) {
    lopster_links(NULL);
    goto done_true;
  }
  // remove the server (mode 1)
  client_message(_("Removing"), "[%s]", server);
  if (link_remove(global.links, server))
  {
    client_message("Huh", _("Local server has quit?"));
  }

done_true:
  l_free(temp);
  return 1;

done_false:
  l_free(temp);
  return 0;
}

int check_server_pong(char *data)
{
  char *server;
  char *temp;
  char *ping;
  link_t *link;

  if (strncmp("Pong from server", data, 16))
    return 0;

  temp = l_strdup(data + 17);
  server = arg(temp, 0);
  ping = arg(NULL, 0);
  if (ping)
    ping++;
  else
    return 1;

  if ((link = link_search_rec(global.links, server)) == NULL)
     return 1;

  if (link->ping)
    l_free(link->ping);
  link->ping = l_strdup(ping);
  return 1;
}

/*
void check_get_motd(char* data) { // for macachu
  char* mode;

  if (SERVER->network == N_UNKNOWN) {
    if (get_version(data)) {
      if (global.usermode) {
	mode = make_string_from_list(global.usermode, " ");
	send_command(CMD_CLIENT_USER_MODE, "NONE");
	send_command(CMD_CLIENT_USER_MODE, mode);
      }
      lopster_links(NULL);
    }
    update_user_stats();
    setup_sensitive(0);
  }
  if (!global.links) {
    get_server(data);
  }
}
*/

void check_op(char *data)
{
  char *pos1;
  char *pos2;

  pos1 = strchr(data, ' ');
  if (!pos1) return;
  pos1++;
  if (!strncmp(pos1, "opped you on channel ", 21)) {
    opchannel_add(pos1+21);
    return;
  }
  if (!strncmp(pos1, "deopped you on channel ", 23)) {
    opchannel_remove(pos1+23);
    return;
  }
  pos2 = strchr(pos1, ' ');
  if (!pos2) return;
  pos2++;
  if (!strncmp(pos2, "set you as operator on channel ", 31)) {
    opchannel_add(pos2+31);
  }
}

void check_pending_searches(char *data)
{
  search_t* search;

  if (!strcmp(data, "search failed: too many pending searches")) {
    if (global.limit.max_searches_real > 1) global.limit.max_searches_real--;
    search_update_counter();
    search = search_finish_oldest(1);
    if (search) send_search_again(search);
  }
}

void check_share_limit(char *data)
{
  GList *dlist;

  if (!strncmp(data, "You may only share", 18)) {
    for (dlist = global.share_queue; dlist; dlist = dlist->next)
      l_free(dlist->data);
    g_list_free(global.share_queue);
    global.share_queue = NULL;
  }
}

void send_global(char *text)
{
  char *text2;
  char *pos;

  line_insert_lf(text);
  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    send_command(CMD_CLIENT_ANNOUNCE, text2);
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;
  send_command(CMD_CLIENT_ANNOUNCE, text2);
}

void send_wallop(char *text)
{
  char *text2;
  char *pos;

  line_insert_lf(text);
  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    send_command(CMD_CLIENT_WALLOP, text2);
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;
  send_command(CMD_CLIENT_WALLOP, text2);
}

void send_emote(char *text)
{
  char *text2;
  char *pos;
  char *command;

  if (!in_channel())
    return;

  while ((pos = strchr(text, '\"')))
    *pos = '\'';

  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    command = l_strdup_printf("%s \"%s\"",
			      global.current_page->name, text2);
    send_command(CMD_CLIENT_EMOTE, command);
    l_free(command);
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;

  command = l_strdup_printf("%s \"%s\"", global.current_page->name, text2);
  send_command(CMD_CLIENT_EMOTE, command);
  l_free(command);
}

void send_private(char *nick, char *text, int show)
{
  char *text2;
  char *pos;
  char *command;

  if (global.status.connection < 2)
    return;

  line_insert_lf(text);
  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    command = l_strdup_printf("%s %s", nick, text2);
    send_command(CMD_CLIENT_PRIVMSG, command);
    l_free(command);

    if (show) {
      log(nick, LOG_PRIVATE, "<%s> %s\n", SERVER->nick, text2);
      if (l_strcasecmp(nick, global.current_page->name))
      {
	chat_print_time_stamp(global.current_page, M_PUBLIC);
	chat_print("whisper", _("<to: "));
	chat_print("whisper", nick);
	chat_print("whisper", "> ");
	chat_print_ln("whisper", text2);
      } else {
	command = l_strdup_printf("%s %s %s",
				  global.current_page->name,
				  SERVER->nick, text2);
	cmd_public_message_no_log(command, M_PUBLIC);
	l_free(command);
      }
    }
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;
  command = l_strdup_printf("%s %s", nick, text2);
  send_command(CMD_CLIENT_PRIVMSG, command);
  l_free(command);

  if (show) {
    log(nick, LOG_PRIVATE, "<%s> %s\n", SERVER->nick, text2);
    if (l_strcasecmp(nick, global.current_page->name))
    {
      chat_print_time_stamp(global.current_page, M_PUBLIC);
      chat_print("whisper", _("<to: "));
      chat_print("whisper", nick);
      chat_print("whisper", "> ");
      chat_print_ln("whisper", text2);
    } else {
      command =
	  l_strdup_printf("%s %s %s", global.current_page->name,
			  SERVER->nick, text2);
      cmd_public_message_no_log(command, M_PUBLIC);
      l_free(command);
    }
  }
}

void send_public(char *channel, char *text)
{
  char *text2;
  char *pos;
  char *command;

  line_insert_lf(text);
  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    command = l_strdup_printf("%s %s", channel, text2);
    send_command(CMD_CLIENT_PUBLIC, command);
    l_free(command);
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;
  command = l_strdup_printf("%s %s", channel, text2);
  send_command(CMD_CLIENT_PUBLIC, command);
  l_free(command);
}

void send_chwallop(char *channel, char *text)
{
  char *text2;
  char *pos;
  char *command;

  line_insert_lf(text);
  while ((pos = strchr(text, '\n')) != NULL) {
    *pos = 0;
    if (global.options.parse_color)
      text2 = cparse(text);
    else
      text2 = text;
    command = l_strdup_printf("%s %s", channel, text2);
    send_command(CMD_CLIENT_CHANNEL_WALLOP, command);
    l_free(command);
    text = pos + 1;
  }

  if (global.options.parse_color)
    text2 = cparse(text);
  else
    text2 = text;
  command = l_strdup_printf("%s %s", channel, text2);
  send_command(CMD_CLIENT_CHANNEL_WALLOP, command);
  l_free(command);
}

void mark_user(char *user, GdkPixmap * pix, GdkBitmap * bit)
{
  int i2;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;
  char *text;
  char t[1024];
  GList *dlist;
  chat_page_t *page;
  GtkWidget *temp;

  // mark users in online lists
  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = (chat_page_t *) (dlist->data);
    if (page->type != P_PUBLIC)
      continue;
    temp = page->online;
    if (!temp)
      continue;

    for (i2 = 0; i2 < GTK_CLIST(temp)->rows; i2++) {
      gtk_clist_get_pixtext(GTK_CLIST(temp), i2, 0,
			    &text, &space, &pixmap, &bitmap);
      strcpy(t, text);
      if (!l_strcasecmp(user, t)) {
	gtk_clist_set_pixtext(GTK_CLIST(temp), i2, 0, t, space, pix, bit);
      }
    }
  }
}

void update_user(char *user)
{
  if (string_list_search(LIST_FRIEND, user)) {
    mark_user(user, global.pix.friend, global.pix.friendb);
  } else if (string_list_search(LIST_ENEMY, user)) {
    if (string_list_search(LIST_IGNORE, user))
      mark_user(user, global.pix.enemy2, global.pix.enemy2b);
    else
      mark_user(user, global.pix.enemy, global.pix.enemyb);
  } else if (string_list_search(LIST_IGNORE, user)) {
    mark_user(user, global.pix.ignore, global.pix.ignoreb);
  } else {
    mark_user(user, global.pix.dummy, global.pix.dummyb);
  }
}

void update_users(GList * glist)
{
  int i1;
  char *pos;

  if (!glist)
    return;
  for (i1 = 0; i1 < (int)g_list_length(glist); i1++) {
    pos = (char *) (g_list_nth(glist, i1)->data);
    update_user(pos);
  }
}

int search_user_in_list(GtkCList * list, char *user)
{
  char *text;
  char *t;
  int i1;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;

  if (!user)
    return -1;
  if (!list)
    return -1;

  for (i1 = 0; i1 < list->rows; i1++) {
    gtk_clist_get_pixtext(list, i1, 0, &text, &space, &pixmap, &bitmap);
    if (!text) {
      printf("oops\n");
      continue;
    }
    t = l_strdup(text);
    if (l_strcasecmp(user, t) == 0) {
      l_free(t);
      return i1;
    }
    l_free(t);
  }

  return -1;
}

GList *user_search(char *user)
{
  GtkCList *temp;
  char *text;
  char t[500];
  int i1;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;
  int len = strlen(user);
  GList *result = NULL;

  if (!in_channel())
    return NULL;

  temp = GTK_CLIST(global.current_page->online);
  if (!temp)
    return NULL;
  for (i1 = 0; i1 < temp->rows; i1++) {
    gtk_clist_get_pixtext(temp, i1, 0, &text, &space, &pixmap, &bitmap);
    strcpy(t, text);
    if (strncasecmp(user, t, len) == 0) {
      result = g_list_append(result, l_strdup(t));
    }
  }

  return result;
}

int user_search_full(char *user)
{
  GtkCList *temp;
  char *text;
  char t[500];
  int i1;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;

  if (!in_channel())
    return 0;

  temp = GTK_CLIST(global.current_page->online);
  if (!temp)
    return 0;
  for (i1 = 0; i1 < temp->rows; i1++) {
    gtk_clist_get_pixtext(temp, i1, 0, &text, &space, &pixmap, &bitmap);
    strcpy(t, text);
    if (l_strcasecmp(user, t) == 0)
      return 1;
  }

  return 0;
}

char *search_highlight_string(char *string, int *len)
{
  char *pos;
  char *str;
  char *result = NULL;
  char *temp_str;
  GList *dlist;

  *len = 0;
  temp_str = string;

  for (dlist = global.string_list[LIST_HIGHLIGHT]; dlist;
       dlist = dlist->next) {
    str = (char *) (dlist->data);
    if ((pos = strcasestr(temp_str, str)) != NULL) {
      if ((!result) || (pos < result)) {
	result = pos;
	*len = strlen(str);
      }
    }
  }

  pos = strchr(string, 0x03);
  if (pos) {
    if ((!result) || (pos < result)) {
      result = pos;
      *len = 3;
    }
  }
  return result;
}

// /alias c {/eval say $cparse("$*")}

gint user_compare(GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{

  char *text1 = NULL;
  char *text2 = NULL;
  int u1, u2;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  text1 = GTK_CELL_TEXT(row1->cell[clist->sort_column])->text;
  text2 = GTK_CELL_TEXT(row2->cell[clist->sort_column])->text;

  if (!text2)
    return (text1 != NULL);

  if (!text1)
    return -1;

  if (clist->sort_column == 0) {
    return l_strcasecmp(text1, text2);
  } else if (clist->sort_column == 1) {
    sscanf(text1, "%d", &u1);
    sscanf(text2, "%d", &u2);
    if (u1 < u2)
      return -1;
    if (u1 > u2)
      return 1;
    return 0;
  } else if (clist->sort_column == 2) {
    u1 = speed2int(text1);
    u2 = speed2int(text2);
    if (u1 < u2)
      return -1;
    if (u1 > u2)
      return 1;
    return 0;
  } else {
    return 0;
  }
}

void highlight(chat_page_t * page, int high)
{
  chat_page_t *page2;

  if (page->window)
    return;

  page2 = chat_page_get_current_main();
  if ((page2 == page) && high)
    return;

  gtk_widget_set_style(page->tlabel, global.style[high]);
}

void set_afk(char *message)
{
  GtkWidget *temp;

  printf("set afk [%s]\n", message);
  if (!message) {
    temp = lookup_widget(global.win, "checkbutton9");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(temp), FALSE);
    } else {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(temp), TRUE);
    }
  } else {
    printf("-set afk [%s]\n", message);
    temp = lookup_widget(global.win, "checkbutton9");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(temp), FALSE);
    temp = lookup_widget(global.win, "entry50");
    gtk_entry_set_text(GTK_ENTRY(temp), message);
    temp = lookup_widget(global.win, "checkbutton9");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(temp), TRUE);
  }
}


void chat_page_down(chat_page_t * page)
{
  GtkAdjustment *vertical_adjustment;
  int value;

  if (!page)
    return;
  if (!page->main)
    return;

  vertical_adjustment = GTK_ADJUSTMENT(GTK_TEXT(page->main)->vadj);
  value = vertical_adjustment->value + vertical_adjustment->page_size;
  if (value >
      vertical_adjustment->upper - vertical_adjustment->lower -
      vertical_adjustment->page_size)
    value =
	vertical_adjustment->upper - vertical_adjustment->lower -
	vertical_adjustment->page_size;

  gtk_adjustment_set_value(vertical_adjustment, value);
}

void chat_page_up(chat_page_t * page)
{
  GtkAdjustment *vertical_adjustment;
  int value;

  if (!page)
    return;
  if (!page->main)
    return;

  vertical_adjustment = GTK_ADJUSTMENT(GTK_TEXT(page->main)->vadj);
  value = vertical_adjustment->value - vertical_adjustment->page_size;
  if (value < 0)
    value = 0;

  gtk_adjustment_set_value(vertical_adjustment, value);
}

chat_page_t *chat_page_get_current_main()
{
  GtkNotebook *notebook;
  int page;
  GtkWidget *temp;
  chat_page_t *page1;

  //  return global.current_page;
  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));
  page = gtk_notebook_get_current_page(notebook);
  if (page < 0)
    return NULL;
  temp = gtk_notebook_get_nth_page(notebook, page);
  temp = gtk_notebook_get_tab_label(notebook, temp);

  page1 = (chat_page_t *) gtk_object_get_data(GTK_OBJECT(temp), "page");
  return page1;
}

chat_page_t *chat_page_new(char *name, int type)
{
  chat_page_t *page;

  page = (chat_page_t *) l_malloc(sizeof(chat_page_t));
  page->name = l_strdup(name);
  page->window = NULL;
  page->type = type;
  page->header = NULL;
  page->wallop = NULL;
  page->main = NULL;
  page->mlabel = NULL;
  page->tlabel = NULL;
  page->online = NULL;
  page->no = NULL;
  page->toplevel = NULL;
  page->timeout = -1;
  page->snap = 0;
  page->modified = 7;
  page->active = 1;
  page->lines = 0;
  page->close_on_part = 0;
  page->chmode = 0;
  page->limit = 0;
  page->level = 1;

  global.chat_pages = g_list_append(global.chat_pages, page);
  return page;
}

void chat_page_destroy(chat_page_t * page)
{
  GtkWidget *channel_widget;

  if (page->type == P_FIX)
    return;
  global.chat_pages = g_list_remove(global.chat_pages, page);

  channel_widget = page->toplevel;
  if (!channel_widget)
    return;

  gtk_container_remove(GTK_CONTAINER(channel_widget->parent),
		       channel_widget);

  if (page->window) {
    gtk_object_set_data(GTK_OBJECT(page->window), "channel", NULL);
    gtk_widget_destroy(page->window);
  }

  if (global.current_page == page) {
    global.current_page = chat_page_get_current_main();
  }

  if (page->type == P_PRIVATE)
    hotlist_remove(page->name, 0);
  if (page->type == P_PUBLIC)
    opchannel_remove(page->name);

  if (page->name)
    l_free(page->name);
  l_free(page);
}

void chat_page_attach_to_main(chat_page_t * page)
{
  GtkNotebook *notebook;
  GtkWidget *channel_widget;

  if (page->window == NULL)
    return;

  channel_widget = page->toplevel;
  if (!channel_widget)
    return;

  gtk_widget_ref(channel_widget);
  gtk_container_remove(GTK_CONTAINER(channel_widget->parent),
		       channel_widget);
  
  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));
  gtk_notebook_append_page(notebook, channel_widget, page->mlabel);
  gtk_widget_unref(channel_widget);
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
					      channel_widget));
  page->window = NULL;
}

void chat_page_attach_to_new(chat_page_t * page)
{
  char str[2048];
  GtkWidget *channel;
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *button1;
  GtkWidget *entry;

  if (page->window)
    return;
  channel = page->toplevel;
  if (!channel)
    return;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_default_size(GTK_WINDOW(window),
			      channel->allocation.width,
			      channel->allocation.height + 60);
  if (page->type == P_PRIVATE)
    sprintf(str, _("Private: "));
  else if (page->type == P_PUBLIC)
    sprintf(str, _("Channel: "));
  else
    *str = 0;
  strcat(str, GTK_LABEL(page->tlabel)->label);
  if (!page->active)
    strcat(str, _(" [inactive]"));

  gtk_window_set_title(GTK_WINDOW(window), str);
  gtk_widget_show(window);
  gtk_object_set_data(GTK_OBJECT(window), "channel", page);
  gtk_widget_ref(channel);
  gtk_container_remove(GTK_CONTAINER(channel->parent), channel);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(window), vbox);
  gtk_box_pack_start(GTK_BOX(vbox), channel, TRUE, TRUE, 0);
  gtk_widget_unref(channel);

  hbox = gtk_hbox_new(FALSE, 2);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  entry = gtk_entry_new();
  gtk_widget_show(entry);
  gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(on_chat_entry_activate), NULL);
  gtk_signal_connect(GTK_OBJECT(entry), "key_press_event",
		     GTK_SIGNAL_FUNC(on_chat_entry_key_press_event), NULL);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  button1 = gtk_button_new_with_label(_("Leave Room"));
  gtk_widget_show(button1);
  gtk_container_add(GTK_CONTAINER(frame), button1);
  gtk_signal_connect(GTK_OBJECT(button1), "clicked",
		     GTK_SIGNAL_FUNC(on_chat_leave_clicked), page);
  if (page->type == P_FIX)
    gtk_widget_set_sensitive(button1, FALSE);
  else
    gtk_widget_set_sensitive(button1, TRUE);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  button1 = gtk_button_new_with_label(_("Attach to chat window"));
  gtk_widget_show(button1);
  gtk_container_add(GTK_CONTAINER(frame), button1);
  gtk_signal_connect(GTK_OBJECT(button1), "clicked",
		     GTK_SIGNAL_FUNC(on_chat_attach_clicked), page);

  gtk_signal_connect(GTK_OBJECT(window), "focus_in_event",
		     GTK_SIGNAL_FUNC(on_window_focus_in_event), page);
  gtk_signal_connect(GTK_OBJECT(window), "focus_out_event",
		     GTK_SIGNAL_FUNC(on_window_focus_out_event), page);

  page->window = window;

  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(on_chat_destroy), window);

}

void chat_page_leave(chat_page_t * page)
{
  if (page->type == P_FIX)
    return;

  if (page->type == P_PUBLIC) {
    if ((global.status.connection > 1) &&(page->active)) {
      page->close_on_part = 1;
      send_command(CMD_CLIENT_PART, page->name);
    } else
      chat_page_destroy(page);
  } else {
    chat_page_destroy(page);
  }
}

chat_page_t *chat_page_search(char *name, int type)
{
  GList *dlist;
  chat_page_t *page;

  //  printf("searching [%s]\n", name);
  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = (chat_page_t *) (dlist->data);
    //    printf("s [%s]\n", page->name);
    if (!l_strcasecmp(page->name, name)) {
      if ((type == -1) || (page->type == type))
	return page;
    }
  }
  return NULL;
}

void chat_page_clear(chat_page_t * page, int bottom, int top)
{
  if (top && page->wallop) {
    gtk_text_set_point(GTK_TEXT(page->wallop), 0);
    gtk_text_forward_delete(GTK_TEXT(page->wallop),
			    gtk_text_get_length(GTK_TEXT(page->wallop)));
  }
  if (bottom && page->main) {
    gtk_text_set_point(GTK_TEXT(page->main), 0);
    gtk_text_forward_delete(GTK_TEXT(page->main),
			    gtk_text_get_length(GTK_TEXT(page->main)));
  }
}

int in_private()
{
  return (global.current_page->type == P_PRIVATE);
}

int in_channel()
{
  return (global.current_page->type == P_PUBLIC);
}

void chat_page_set_active(chat_page_t * page, int active)
{
  char str[1024];

  if (!page)
    return;
  if (page->active == active)
    return;
  if (page->type == P_FIX)
    return;

  page->active = active;
  // redraw label

  if (!page->mlabel)
    return;
  if (active)
    gtk_widget_set_sensitive(page->mlabel, TRUE);
  else
    gtk_widget_set_sensitive(page->mlabel, FALSE);

  highlight(page, 0);

  if (!page->window)
    return;

  if (page->active) {
    if (page->type == P_PRIVATE)
      sprintf(str, _("Private: %s"), page->name);
    else if (page->type == P_PUBLIC) {
      sprintf(str, _("Channel: %s"), page->name);
      send_command(CMD_CLIENT_CHANNEL_MODE, page->name);
      send_command(CMD_CLIENT_CHANNEL_LIMIT, page->name);
      send_command(CMD_CLIENT_SET_CHAN_LEVEL, page->name);
    } else
      strcpy(str, _(page->name));
  } else {
    if (page->type == P_PRIVATE)
      sprintf(str, _("Private: %s [inactive]"), page->name);
    else if (page->type == P_PUBLIC)
      sprintf(str, _("Channel: %s [inactive]"), page->name);
    else
      strcpy(str, _(page->name));
  }
  gtk_window_set_title(GTK_WINDOW(page->window), str);
}

chat_page_t *chat_page_find_unfixed()
{
  GList *dlist;
  chat_page_t *page;

  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = (chat_page_t *) (dlist->data);
    if (page->type != P_FIX)
      return page;
  }

  return NULL;
}

user_timestamp_t *timestamp_new(char *user)
{
  user_timestamp_t *result;

  result = (user_timestamp_t *) l_malloc(sizeof(user_timestamp_t *));
  result->user = l_strdup(user);
  timestamp_touch(result);

  return result;
}

user_timestamp_t *timestamp_search(GList * stamps, char *user)
{
  GList *dlist;
  user_timestamp_t *result;

  for (dlist = stamps; dlist; dlist = dlist->next) {
    result = (user_timestamp_t *) (dlist->data);
    if (!l_strcasecmp(result->user, user))
      return result;
  }

  return NULL;
}

void timestamp_touch(user_timestamp_t * stamp)
{
  gettimeofday(&(stamp->tv), 0);
}

int timestamp_difference(user_timestamp_t * stamp)
{
  struct timeval tv;
  int result;

  gettimeofday(&(tv), 0);

  result = (tv.tv_sec - stamp->tv.tv_sec) * 1000 +
      (tv.tv_usec - stamp->tv.tv_usec) / 1000;
  return (result);
}

chat_page_t *create_other_page(char *name, char *title)
{
  GtkWidget *scrolledwindow27;
  GtkWidget *text4;
  GtkWidget *tab_label;
  GtkWidget *entry19 = NULL;
  GtkNotebook *notebook;
  chat_page_t *page;

  scrolledwindow27 = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow27);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow27),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  text4 = gtk_text_new(NULL, NULL);
  gtk_signal_connect_after(GTK_OBJECT(text4), "button_press_event",
			   GTK_SIGNAL_FUNC(on_text_button_press_event),
			   NULL);

  gtk_widget_set_style(text4, global.style[2]);

  gtk_widget_show(text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add(GTK_CONTAINER(scrolledwindow27), text4);

  tab_label = gtk_label_new(title);
  gtk_widget_ref(tab_label);
  gtk_widget_show(tab_label);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));
  gtk_notebook_append_page(notebook, scrolledwindow27, tab_label);
  page = chat_page_new(name, P_OTHER);
  page->header = entry19;
  page->main = text4;
  page->wallop = NULL;
  page->mlabel = tab_label;
  page->tlabel = tab_label;
  page->toplevel = scrolledwindow27;
  gtk_object_set_data(GTK_OBJECT(tab_label), "page", page);

  return page;
}

void channel_set_mode(char* channel, int mode, int set, int send) {
  chat_page_t* page;
  GtkToggleButton* button;
  static char strings[5][3] = {"MT", "MR", "MP", "MI", "MM"};
  int i1;
  char* command;

  page = chat_page_search(channel, P_PUBLIC);
  if (!page) return;
  for (i1 = 0; i1 < 5; i1++) {
    if ((mode & (1<<i1)) == 0) continue;
    button = GTK_TOGGLE_BUTTON(gtk_object_get_data(GTK_OBJECT(page->main), 
						   strings[i1]));
    
    // prevent sending another status if waiting for the servers ack
    if (set) {
      if (page->chmode & (1<<i1)) continue;
    } else {
      if ((page->chmode & (1<<i1)) == 0) continue;
    }
    if (send) {
      if ((global.user.level > L_USER) ||
	  string_list_search(LIST_OPCHANNEL, page->name)) {

	command = l_strdup_printf("%s %c%s", page->name, set?'+':'-', ChannelMode(1<<i1));
	send_command(CMD_CLIENT_CHANNEL_MODE, command);
	l_free(command);
      }
    } else {
      if (set) {
	page->chmode |= (1<<i1);
      } else {
	page->chmode &= 0xff^(1<<i1);
      }
      if (set ^ gtk_toggle_button_get_active(button)) {
	gtk_toggle_button_set_active(button, set);
      }
    }
  }
}

void channel_set_limit(char* channel, int limit, int send) {
  chat_page_t* page;
  GtkEntry* entry;
  char str[1024];

  page = chat_page_search(channel, P_PUBLIC);
  if (!page) return;

  sprintf(str, "%d", limit);
  entry = GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(page->main), "limit"));
  gtk_entry_set_text(entry, str);
  page->limit = limit;
  if (send) {
    sprintf(str, "%s %d", page->name, limit);
    send_command(CMD_CLIENT_CHANNEL_LIMIT, str);
  }
}

void channel_set_level(char* channel, int level, int send) {
  chat_page_t* page;
  GtkEntry* entry;
  GtkWidget* combo;
  char str[1024];

  page = chat_page_search(channel, P_PUBLIC);
  if (!page) return;

  if (page->level == (unsigned)level) return;

  page->level = level;

  if (send) {
    sprintf(str, "%s %s", page->name, LevelShort(level));
    send_command(CMD_CLIENT_SET_CHAN_LEVEL, str);
  } else {
    combo = gtk_object_get_data(GTK_OBJECT(page->main), "level");
    entry = GTK_ENTRY(GTK_COMBO(combo)->entry);
    gtk_entry_set_text(entry, LevelShort(level));
  }
}

void channel_mode_activate(char* channel, int activate) {
  chat_page_t* page;
  GtkWidget* button;
  static char strings[5][3] = {"MT", "MR", "MP", "MI", "MM"};
  int i1;
  GList* dlist;

  if (channel) {
    page = chat_page_search(channel, P_PUBLIC);
    if (!page) return;
    for (i1 = 0; i1 < 5; i1++) {
      button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), 
					      strings[i1]));
      
      if (activate && ((1<<i1) == CMODE_REGISTERED) && 
	  (global.user.level <= L_USER)) continue;
      gtk_widget_set_sensitive(button, activate);
    }
    button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), "limit"));
    gtk_widget_set_sensitive(button, activate);
    button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), "level"));
    gtk_widget_set_sensitive(button, activate);
  } else {
    for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
      page = dlist->data;
      if (page->type != P_PUBLIC) continue;
      for (i1 = 0; i1 < 5; i1++) {
	button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), 
						strings[i1]));
	if (activate && ((1<<i1) == CMODE_REGISTERED) && 
	    (global.user.level <= L_USER)) continue;
	gtk_widget_set_sensitive(button, activate);
      }
      button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), "limit"));
      gtk_widget_set_sensitive(button, activate);
      button = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(page->main), "level"));
      gtk_widget_set_sensitive(button, activate);
    }
  }
}

int channel_refresh(gpointer data ATTR_UNUSED) {
  chat_page_t* page;
  GList* dlist;
  
  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = dlist->data;
    if (page->type != P_PUBLIC) continue;
    if (!page->active) continue;
    send_command(CMD_CLIENT_NAMES_LIST, page->name);
  }
  return 1;
}
