/*
 */

#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>

#include <vector>
#include <string>

#include <lufs/proto.h>
#include <lufs/fs.h>

using namespace std;

#include "tools.h"
#include "file_handle.h"
#include "fs_modules.h"
#include "handlefs.h"


/********************************************************************************
				Inits, mount, umount...
********************************************************************************/

HandleFS::HandleFS(struct list_head *_cfg, struct dir_cache *_cache, struct credentials *_cred)
{
    TRACE("in constructor");

    cfg = _cfg;
    cache = _cache;
    cred = _cred;
	
	// Create modules in parsing order.
	modules.push_back( new FSModule_Flac() );
//	modules.push_back( new FSModule_Ogg() );
//	modules.push_back( new FSModule_MP3() );
	// insert new module here
	
	// this module catches all remaining files.
	modules.push_back( new FSModule_Default() );
}

HandleFS::~HandleFS()
{
    TRACE("in destructor");
}


int	HandleFS::do_mount()
{
    TRACE("do_mount");
    return 1;
}

void HandleFS::do_umount()
{
    TRACE("do_umount");
}

/********************************************************************************
				readdir
********************************************************************************/

int HandleFS::do_readdir(const char *dir_name, struct directory *ddir)
{
    DIR *dir;
    struct lufs_fattr fattr;
    struct dirent *dent;
	bool already_added;

    TRACE("readdir " << dir_name);

    if(chdir(dir_name) < 0)
	{
		WARN("chdir failed: "<<dir_name<<strerror(errno));
		return -1;
    }

    if(!(dir = opendir(dir_name)))
	{
		WARN("could not open directory!");
		return -1;
    }

    while((dent = readdir(dir)))
	{
		TRACE("adding direntry "<<dent->d_name);
		
		if( do_stat(dent->d_name, &fattr) < 0)
		{
			WARN("could not stat file!");
			closedir(dir);
			return -1;
		}
		
		if( S_ISDIR(fattr.f_mode) )
			lu_cache_add2dir(ddir, dent->d_name, NULL, &fattr);	
		else
		{
			// add virtual files
			already_added = false;
		 	for(FSModule::Array::iterator i = modules.begin(); i != modules.end(); i++ )
			{
				if( !(*i)->CanAdd( already_added ))
					continue;
				
				string xlat( (*i)->GetVirtualFilename(dent->d_name) );
				if(!xlat.size())
					continue;
				
				string full_xlat( string(dir_name)+"/"+xlat );
				if(do_stat(full_xlat.c_str(), &fattr)==0)
				{
					TRACE("adding virtual direntry "<<dent->d_name);
					lu_cache_add2dir(ddir, (char*)xlat.c_str(), NULL, &fattr);	
					already_added = true;
				}
			}
		}
    }

    closedir(dir);

    return 0;
}

/********************************************************************************
				stat
********************************************************************************/

int HandleFS::do_stat(const char *file, struct lufs_fattr *fattr)
{
    int res;
	bool created_here;

	// check if it is a directory first...
    TRACE("*** do_stat " << file);
	struct stat stat;
    if(lstat(file, &stat) == 0)
	{
		if( S_ISDIR(stat.st_mode) )
		{
		    fattr->f_mode = stat.st_mode;
			fattr->f_nlink = stat.st_nlink;
			fattr->f_uid = (getuid() == stat.st_uid) ? 1 : 0;
			fattr->f_gid = (getgid() == stat.st_gid) ? 1 : 0;
			fattr->f_size = stat.st_size;
			fattr->f_atime = stat.st_atime;
			fattr->f_mtime = stat.st_mtime;
			fattr->f_ctime = stat.st_ctime;
		    return 0;
		}	
	}

	FileHandle *handle = handles.Find(file);
	if(handle)
	{
		created_here = false;
		handle->KeepAlive();
	}
	else
	{
		handle = Open(file);
		if( !handle )
			return -1;
		created_here = true;
	}

	res = handle->Stat(fattr);
    fattr->f_uid = ((int)fattr->f_uid == (int)cred->uid) ? 1 : 0;
    fattr->f_gid = ((int)fattr->f_gid == (int)cred->gid) ? 1 : 0;

//	deleting will be done by timeout...	
//	if(created_here)
//		handles.Delete(handle->name.c_str());
	
    return res;
}

/********************************************************************************
				open/close
********************************************************************************/

int HandleFS::do_open(const char *file)
{
    TRACE("do_open");
 	FileHandle *handle = Open(file);
	if( !handle )
		return -1;
    return 0;
}

int HandleFS::do_release(const char *file)
{
	int ret=0;
    TRACE("do_release");

	handles.Cleanup();
    TRACE(handles.size() << " files still opened.");
    return ret;
}

FileHandle*	HandleFS::Open( const char* path )
{
	FileHandle *handle = handles.Find(path);
	if(handle)
	{
		handle->KeepAlive();
		return handle;
	}

	for(FSModule::Array::iterator i = modules.begin(); i != modules.end(); i++ )
	{
		handle = (*i)->Open(path);
		if(handle)
		{
			TRACE("Opened with "<<(*i)->Identity());
			handle->KeepAlive();
			handles.Insert( handle );
			return handle;
		}
	}
	
	return 0;
}

/********************************************************************************
				read
********************************************************************************/

int HandleFS::do_read(const char *file, long long offset, unsigned long count, char *b)
{
    TRACE("do_read");
	FileHandle *handle = Open(file);
	if( !handle )
		return -1;
	return handle->Read(offset,count,b);
}
