/*
 *  client/functions-conf.c
 *		The functions to actually execute the commands
 *		for the client.  Commands for handling
 *		conferences.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>
#include "conf.h"

#include <stdio.h>
/* #include <time.h> included from kom-types.h */
/* #include <malloc.h> */
#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#ifdef __vax__
#define __SIZE_T included from string.h
#endif
#include <kom-types.h> /* Before zmalloc.h for those that have size_t in 
			  sys/types.h */
#include <zmalloc.h>
#include <s-string.h>
#include <s-collat.h>

#include <libintl.h>

#include <kom-errno.h>
#include <services.h>

/* Must be included before "read-line.h" on a vax because it includes <sgtty.h>
 * and that does not work after inclusion of <sys/ioctl.h> because of
 * stupid include-file construction.
 */
#include "xoutput.h"

#include "copy.h"
#include "read-line.h"
#include "commands.h"
#include "parser.h"
#include "error.h"
#include "cache.h"
#include "reading.h"
#include "quit.h"

#include "internal.h"
#include "cmds.h"
#include "offline.h"

#include "defaults.h"

/* This is to emphasize that a function is exported. */
#define Export

/* Forward declaration */
static Success
do_leave_conf(Conf_no	conf_to_leave);



/*
 *  The one and only  parameter to all the external functions, is a
 *  String containing the rest of the line, following the command.
 *  E g if the user gave the command "Skapa m|te Inl{gg }t mig",
 *  then the String "Inl{gg }t mig" is passed as the parameter
 *  'ARGUMENT' to the function responsible for the command 'skapa
 *  m|te'.  Leading and trailing spaces are skipped.  EMPTY_STRING
 *  is passed if no arguments to the command was given.
 */




/*  Create a new conference  */
Export  Success
cmd_create_new_conference(String	new_conf_name)
{
#define FUNCTION "create_new_conference()"

    volatile Conf_type	  type_of_conf;
    Bool		  original;
    Bool		  open_conf;
    Bool		  closed;
    Bool		  secret;
    Bool		  write_protect;
    String		  input			= EMPTY_STRING;
    Conf_no		  new_conf_no;
    Parse_token		* words;
    Parse_token *volatile all_words		= NULL;
    Conf_z_info_list	  confs;
    Success		  retval;

#define CLEAN_UP()				\
    {						\
	s_clear(&new_conf_name);		\
	s_clear(&input);			\
	free_tokens(all_words);			\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    if (s_empty(new_conf_name))
    {
	xprintf(gettext("Vad skall det nya mtet heta?\n= "));
	read_line(&new_conf_name);
	new_conf_name = s_strip_trailing(new_conf_name,
					 command_separators);
	if (s_empty(new_conf_name))
	{
	    RETURN (FAILURE);
	}
    }
    else
    {
	s_puts(new_conf_name);
    }
    newline();
    newline();

    /* Get the type of the conference (open, write protect, original, ...) */
    xprintf(gettext("Vilken typ skall mtet ha? "
		    "Vlj en eller flera av fljande:\n"
		    "ppet, slutet, original, skyddat, skrivskyddat\n= "));
    read_line(&input);		/* BUG! Retval not checked! */

    words = tokenize(input, command_separators);
    if (words == NULL)
    {
	fatal1(CLIENT_OUT_OF_MEMORY, "tokenize() failed");
    }

    /* Empty string? Exit then. */
    if (s_empty(words->word))
    {
	RETURN (FAILURE);
    }

    all_words = words;		/* Save the start of the list */
    open_conf = closed = original = secret = write_protect = FALSE;
    while (! s_empty(words->word))
    {
	/* DIRTY!!! */
	if (s_usr_strhead(words->word, s_fcrea_str(gettext("ppet")),
			  DEFAULT_COLLAT_TAB))
	{
	    if (closed)
	    {
		xprintf(gettext("Ett mte kan inte vara bde ppet och slutet.\n"));
		RETURN (FAILURE);
	    }
	    open_conf = TRUE;
	}
	else if (s_usr_strhead(words->word, s_fcrea_str(gettext("slutet")),
			       DEFAULT_COLLAT_TAB))
	{
	    if (open_conf)
	    {
		xprintf(gettext("Ett mte kan inte vara bde ppet och slutet.\n"));
		RETURN (FAILURE);
	    }
	    closed = TRUE;
	}
	else if (s_usr_strhead(words->word, s_fcrea_str(gettext("original")),
			       DEFAULT_COLLAT_TAB))
	{
	    original = TRUE;
	}
	else if (s_usr_strhead(words->word, s_fcrea_str(gettext("skyddat")),
			       DEFAULT_COLLAT_TAB))
	{
	    secret = TRUE;
	}
	else if (s_usr_strhead(words->word,
			       s_fcrea_str(gettext("skrivskyddat")),
			       DEFAULT_COLLAT_TAB))
	{
	    write_protect = TRUE;
	}
	else
	{
	    xprintf(gettext("Kunde ej frsts av LysKOM: "));
	    s_puts(words->word);
	    newline();
	    RETURN (FAILURE);
	}

	words++;
    }	/* While list isn't empty */

    type_of_conf.rd_prot = ! open_conf;
    type_of_conf.original = original;
    type_of_conf.secret = secret;
    type_of_conf.letter_box = FALSE;

    /* Okay, then let's try to create the conference */
    new_conf_no = kom_create_conf(new_conf_name, type_of_conf);
    if (new_conf_no == 0)
    {
	switch (kom_errno)
	{
	case  KOM_PERM:
	    xprintf(gettext("Du fr tyvrr inte lov att skapa mten sjlv\n"));
	    break;

	case  KOM_BAD_NAME:
	case  KOM_LONG_STR:
	    xprintf(gettext("Namnet kan inte godknnas.\n"));
	    break;

	case  KOM_CONF_EXISTS:
	    xprintf(gettext("\nNgon hann fre dig med det namnet.\n"));
	    break;

	default:
	    fatal3(CLIENT_UNEXPECTED_SERVER_ERROR,
		   "create_conf() failed: %d %s",
		   kom_errno, kom_errno_string());
	    break;
	}

	RETURN (FAILURE);
    }	/* If kom_create_conf() failed */
    else 
    {
	/* Join the new conference. */
	do_join_conf(new_conf_no);
    }

    /* Now, let's see if we need to set a super-conference */
    if (original || write_protect)
    {
	xprintf(gettext("Vilket mte ska vara supermte? "));
	read_line(&input);
	input = s_strip_trailing(input, command_separators);
	if (s_empty(input))
	{
	    RETURN (FAILURE);
	}

	confs = find_conf_or_pers_no(input);
	if (confs.no_of_confs != 1)
	{
	    inform_user(ambigous_or_no_conf, &confs);
	    RETURN (FAILURE);
	}

	retval = kom_set_super_conf(new_conf_no, confs.confs[0].conf_no);
	if (retval == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case  KOM_NO_CONNECT:
		fatal2(CLIENT_CONNECTION_LOST, "set_super_conf() failed: %d",
		       confs.confs[0].conf_no);
		break;

		/* Hmm, what reasons are there for this failing?
		 * I can't see any.  If we aren't allowed to set
		 * it as super-conf, then we shouldn't be able to
		 * get its name either...		*/
	    default:
		fatal2(CLIENT_UNEXPECTED_SERVER_ERROR,
		       "set_super_conf() failed: %d", confs.confs[0].conf_no);
		break;
	    }
	}   /* If kom_set_super_conf() failed */

	/* And set the current person as the only one allowed
	 * to submit texts to the new conference.	    */
	retval = kom_set_permitted_submitters(new_conf_no,
					      (Conf_no) cpn);
	if (retval == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case  KOM_NO_CONNECT:
		fatal3(CLIENT_CONNECTION_LOST,
		       "set_permitted_submitters(%d tp be a submitter of %d) failed", 
		       cpn, new_conf_no);
		break;

		/* Hmm, what reasons are there for this failing?
		 * I can't see any.  If we aren't allowed to set
		 * it as super-conf, then we shouldn't be able to
		 * get its name either...		*/
	    default:
		fatal2(CLIENT_UNEXPECTED_SERVER_ERROR,
		       "set_permitted_submitters() in conf %d failed",
		       confs.confs[0].conf_no);
		break;
	    }
	}   /* If 'set_permitted_submitters()' failed */
    }   /* If super-conferece needed */

    RETURN (OK);

#undef	CLEAN_UP
#undef  FUNCTION
}   /* END: create_new_conference */


/* Remove a conference */
Export  Success
cmd_remove_conference(String conference)
{
#define FUNCTION "remove_conference()"

    Conf_z_info      whichconf = EMPTY_CONF_Z_INFO;

#define CLEAN_UP()				\
    {						\
	s_clear(&conference);			\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    whichconf = ask_for_conf(&conference,
			     gettext("Vilket mte vill du ta bort"));
    if (whichconf.conf_no == 0)
    {
	RETURN (FAILURE);
    }

    if (whichconf.conf_no == cpn)
    {
	xprintf(gettext("Du fr inte ta bort dig sjlv"));
	RETURN (FAILURE);
    }

    if (kom_delete_conf(whichconf.conf_no) != FAILURE)
    {
	xprintf(gettext("Har tagit bort "));
	print_conf_name(whichconf.conf_no);
    }
    else
    {
	switch (kom_errno)
	{
	case  KOM_PERM:
	    xprintf(gettext("Du har inte rttigheter att ta bort "));
	    print_conf_name(whichconf.conf_no);
	    break;

	default:
	    fatal4(CLIENT_UNEXPECTED_SERVER_ERROR,
		   "delete_conf(%d) error %d %s",
		   whichconf.conf_no, kom_errno, kom_errno_string());
	    break;
	}
    }

    RETURN (OK);
#undef  CLEAN_UP
#undef  FUNCTION
}

/* Find all names matching the string given as argument */
Export  Success
cmd_show_matching_names(String	name)
{
#define FUNCTION "show_matching_names()"

    Success		  retval;
    Conf_z_info_list	  matches	= EMPTY_CONF_Z_INFO_LIST;
    unsigned long	  i;


#define CLEAN_UP()				\
    {						\
	s_clear(&name);				\
	release_conf_z_info_list(&matches);	\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    if (s_empty(name))
    {
	xprintf(gettext("\nAnge skstrng\n- "));
	read_line(&name);
    }
    else
	s_puts(name);

    newline();

    retval = kom_lookup_z_name(name, TRUE, TRUE, &matches);
    if (retval == FAILURE)
    {
	switch (kom_errno)
	{
	case  KOM_OUT_OF_MEMORY:
	    fatal1(CLIENT_OUT_OF_MEMORY, "lookup_z_name() failed");
	    break;

	case  KOM_NO_CONNECT:
	    fatal1(CLIENT_CONNECTION_LOST, "lookup_z_name() failed");
	    break;

	default:
	    fatal3(CLIENT_UNEXPECTED_SERVER_ERROR, 
		   "lookup_z_name() failed: %d %s",
		   kom_errno, kom_errno_string());
	    break;
	}
    }

    for ( i = 0 ;  i < matches.no_of_confs ;  i++ )
    {
	if (matches.confs[i].type.letter_box)
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("Person %u:\t\t%S\n"),
		    matches.confs[i].conf_no,
		    matches.confs[i].name);
	}
	else
	{
	    /*xgettext:c-format*/
	    xprintf(gettext("Mte %u:\t\t%S\n"),
		    matches.confs[i].conf_no,
		    matches.confs[i].name);
	}
    }

    newline();

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}   /* END: show_matching_names */


/*  Go to the specified conference.  Join it if not a member.  */
Export  Success
cmd_goto_named_conference(String   conf_name_arg)
{
    /* static char	* this_function	= "goto_named_conference()"; */

    Conf_z_info		  go_to_conf = EMPTY_CONF_Z_INFO;
    Conf_no		  conf_to_go_to;
    Conference          * conf_info;
    Membership		* memb_ptr		= NULL;
    volatile Success	  retval		= FAILURE;
    Kom_err               error;

#define CLEAN_UP()				\
    {						\
	s_clear(&conf_name_arg);		\
	zfree(memb_ptr);			\
	release_conf_z_info(&go_to_conf);	\
    }						\


    TOPLOOP_SETJMP();

    OFFLINE();

    go_to_conf = ask_for_conf(&conf_name_arg,
			      gettext("Vilket mte vill du g till"));
    if (go_to_conf.conf_no == 0)
    {
	RETURN (FAILURE);
    }

    conf_to_go_to = go_to_conf.conf_no;
    xprintf("%S", go_to_conf.name);

    /* Well, is the user member of the conference? */
    if (locate_membership(conf_to_go_to, cpn, NULL) == -1)
    {
        conf_info = conf_stat(conf_to_go_to, &error);
            /* Should check error value of conf_stat */
        if (conf_info->presentation)
	{
            show_text_no(conf_info->presentation);
	}
	newline();
	/* Nope. Ask if s/he wants to join */
	xprintf(gettext("Du r inte medlem i mtet. "
			"Vill du bli medlem i mtet?\n"));
	if (yes_or_no_p())
	{
	    retval = do_join_conf(conf_to_go_to);
	}
	else
	{
	    RETURN (FAILURE);
	}

	xprintf("%S", go_to_conf.name);
    }

    review_reset();

    if (conf_to_go_to != 0)
    {
	int 		  unread;

	reading_goto_conf(conf_to_go_to);

	unread = reading_unread_texts_count();

	print_conf_name(conf_to_go_to);
	if (unread > 0)
	{
	    xprintf(" - %d %s.\n", unread,
		    (unread == 1 
		     ? gettext("olst inlgg")
		     : gettext("olsta inlgg")));
	}
	else
	{
	    xprintf(gettext(" - inga olsta.\n"));
	}
    }

    RETURN (retval);

#undef  CLEAN_UP
}   /* END: goto_named_conference */




/*  Join (become a member of) a conference.  */
Export  Success
cmd_join_conference(String   conf_name_arg)
{
/*    char		* this_function	= "join_conference()"; */

    Conf_z_info		  join_conf = EMPTY_CONF_Z_INFO;
    Conf_no		  conf_to_join;


#define CLEAN_UP()				\
    {						\
	s_clear(&conf_name_arg);		\
	release_conf_z_info(&join_conf);	\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    join_conf = ask_for_conf(& conf_name_arg,
			     gettext("Vilket mte vill du g med i"));
    if (join_conf.conf_no == 0)
    {
	RETURN (FAILURE);
    }
    conf_to_join = join_conf.conf_no;

    if (do_join_conf(conf_to_join) == OK)
    {
	reading_goto_conf(conf_to_join);
	RETURN (OK);
    }
    else
    {
	RETURN (FAILURE);
    }

    /*NOTREACHED*/ /* Thanks apogee that found this one! */
    RETURN (OK);
#undef  CLEAN_UP
}   /* END: join_conference */



/*  Leave (resign as a member of) a conference.  */
Export  Success
cmd_leave_conference(String   conf_name_arg)
{
#define FUNCTION "cmd_leave_conference()"

    Conf_z_info		  leave_conf = EMPTY_CONF_Z_INFO;
    Conf_no		  conf_to_leave;

#define CLEAN_UP()				\
    {						\
	s_clear(&conf_name_arg);		\
	release_conf_z_info(&leave_conf);	\
    }


    TOPLOOP_SETJMP();

    OFFLINE();

    leave_conf = ask_for_conf(& conf_name_arg,
			      gettext("Vilket mte vill du g ur"));
    if (leave_conf.conf_no == 0)
    {
	RETURN (FAILURE);
    }
    conf_to_leave = leave_conf.conf_no;

    if (do_leave_conf(conf_to_leave) == OK)
    {
	if (conf_to_leave == ccn)
	    reading_goto_conf(0); /* Leaving conf. Going nowhere. */
	RETURN (OK);
    }
    else
    {
	RETURN (FAILURE);
    }

    /*NOTREACHED*/
    RETURN (OK);
#undef  CLEAN_UP
#undef FUNCTION
}   /* END: leave_conference */



/* Var {r jag
 * Prints some information about the name, conference and conference system. */
Export  Success
cmd_where_am_i(String	argument)

{
#define CLEAN_UP()				\
    {						\
	s_clear(&argument);			\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    newline();
    newline();

    print_conf_name(cpn);
    
    if (ccn == 0)
    {
	xprintf(gettext(" r inte nrvarande i ngot mte"));
    }
    else	
    {
	Membership	  ship = EMPTY_MEMBERSHIP;
	Kom_err		  error;
	int		  unread;

	locate_membership(ccn, cpn, &ship);

	xprintf(gettext(" r nrvarande i "));
	print_conf_name(ccn);
	unread = conf_last(ccn, &error)
	    - ( ship.last_text_read + ship.no_of_read );
	if (unread == 1)
	    xprintf(gettext(" (%d olst)"), unread);
	else
	    xprintf(gettext(" (%d olsta)"), unread);
    }
    xputchar('.');
    newline();

    xprintf(gettext("Detta r %s, version %d.\n"
		    "Programmet heter %s, version %s.\n"),
	    server_name, serv_info.version, 
	    client_name, VERSION);

    RETURN (OK);

#undef CLEAN_UP
}   /* END: where_am_i */


/*
 *  Show the status for a person or a conference.
 */
Export  Success
cmd_status(String	  name)

{
    Conf_z_info		  conf_to_stat = EMPTY_CONF_Z_INFO;
    Conference	*volatile conf		= NULL;
    Member_list	*volatile members	= NULL;
    Person	*volatile pers		= NULL;
    Kom_err		  error_c, error_p;

#define CLEAN_UP()				\
    {						\
	s_clear(&name);				\
	zfree(release_conf(conf));		\
	zfree(release_member_list(members));	\
	zfree(release_person(pers));		\
        release_conf_z_info(&conf_to_stat);	\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    conf_to_stat = ask_for_conf(&name, gettext("Vad vill du veta status fr"));
    if (conf_to_stat.conf_no == 0)
    {
	RETURN (FAILURE);
    }
    newline();

    conf = conf_stat(conf_to_stat.conf_no, &error_c);
    if (conf == NULL)
    {
	inform_user(ambigous_or_no_conf, NULL);
	RETURN (FAILURE);
    }
    members = member_list(conf_to_stat.conf_no, &error_c);
    if (members == NULL)
    {
	inform_user(ambigous_or_no_conf, NULL);
	RETURN (FAILURE);
    }

    if (conf_to_stat.type.letter_box)
    {
	pers = pers_stat(conf_to_stat.conf_no, &error_p);
	if (pers == NULL)
	{
	    inform_user(ambigous_or_no_conf, NULL);
	    RETURN (FAILURE);
	}

	show_short_conf_status(conf_to_stat, conf);
	newline();
	show_pers_status(conf_to_stat, conf, pers);
	newline();
        if (conf->msg_of_day)
        {
	    /*xgettext:c-format*/
            xprintf(gettext("%S har en lapp p drren:\n"), conf->name);
            show_text_no(conf->msg_of_day);
        }
	show_more_conf_status(conf_to_stat, conf);
	newline();
/*	show_membership_list(conf_to_stat, conf, pers, memberships); */
    }
    else
    {
	show_short_conf_status(conf_to_stat, conf);
	newline();
        if (conf->msg_of_day)
        {
	    /*xgettext:c-format*/
            xprintf(gettext("\n%S har en lapp p drren:\n"), conf->name);
            show_text_no(conf->msg_of_day);
            newline();
            newline();
        }
	show_more_conf_status(conf_to_stat, conf);
	newline();
	if (members)
	    show_member_list(conf_to_stat, conf, members);
	else
	    xprintf(gettext("Mtet har inga medlemar\n"));
    }
    
    RETURN (OK);

#undef CLEAN_UP
}   /* END: cmd_status() */



/* ==================================================================== */
/*		Internal functions for 'cmd-*.c'			*/




/*
 *  Find all conferences matching the name NAME.  Also supports the
 *  notation "conference nnn" and "person nnn", where nnn is a number.
 *  If one of the latter notations are used, the 'pers_or_conf' field
 *  of the result will reflect the notation used, if it *is* a person.
 *  If the notation 'person nnn' is used and it *isn't* a person, then
 *  it won't match, i e the EMPTY_CONF_Z_INFO_LIST will be returned.
 */
Export  Conf_z_info_list
find_conf_or_pers_no(String	  name)

{
#define FUNCTION "find_conf_or_pers_no()"

    Conf_no		  conf;
    String_size		  error_check; /* To check for errors */
    String_size		  start_pos;
    String		  first_word;
    Conference	*volatile conf_s	= NULL;
    Conf_z_info_list	  result	= EMPTY_CONF_Z_INFO_LIST;
    Kom_err		  error;
    Success		  retval;

#define CLEAN_UP()				\
    {						\
	zfree(conf_s);				\
    }

    TOPLOOP_SETJMP();

    /* Strip trailing blanks for the ease of parsing "Person nnn"
     * and "Conference nnn"					*/
    name = s_strip_trailing(name, command_separators);

    /* Is it of the format "Person nnn"? */
    /* Well, let's try and see if we have any luck	*/
    start_pos = 0;
    first_word = s_strtok(name, &start_pos, command_separators);
    /* I don't particularly _like_ this, but... */
    if ((start_pos < s_strlen(name))	/* More than one word? */)
    {
	int is_person = FALSE;

	if ((((s_usr_strhead(first_word, s_fcrea_str(gettext("Person")),
			     DEFAULT_COLLAT_TAB))
	      && ((is_person = TRUE)))
	     || (s_usr_strhead(first_word, s_fcrea_str(gettext("Mte")),
			       DEFAULT_COLLAT_TAB)))
	    && ((conf = (Conf_no) s_strtol(s_fsubstr(name, start_pos,
						     END_OF_STRING),
					   &error_check,
					   DEFAULT_NUMBER_BASE)))
	    && (error_check == s_strlen(name) - start_pos)
	    && ((conf_s = conf_stat(conf, &error))))
	{
	    result.confs = zmalloc(sizeof(Conf_z_info));
	    if (result.confs == NULL)
	    {
		/* Seem to be running out of memory */
		fatal1(CLIENT_OUT_OF_MEMORY, "In parsing of conf or person");
	    }
	    result.no_of_confs = 1;
	    result.confs[0].conf_no = conf;
	    result.confs[0].name = conf_s->name;
	    result.confs[0].type = conf_s->type;
	    if (!is_person)
	    {
		result.confs[0].type.letter_box = FALSE;
	    }
	    RETURN (result);
	}
    }

    /* It was neither of the form "Person nnn" nor "Conference nnn" */
    {
	/* Then assume it is a name and find it's number */
	retval = kom_lookup_z_name(name, TRUE, TRUE, &result);
	if (retval == FAILURE)
	{
	    /* Waaahhh!!  I just hate this! */
	    fatal2(CLIENT_ERROR_TOO_boring,
		   "kom_lookup_name() failed: %s", s_crea_c_str(name));
	}

    }

    /* Phew! Finished at last. */
    RETURN (result);

#undef  CLEAN_UP
#undef  FUNCTION
}   /* END: find_conf_or_pers_no */



/*
 * ask_for_conf
 *
 * Ask the user for the name or number of one conference or person.
 * NAME is a pointer to a string containing a name to lookup.
 * If the string is empty, the user is asked to give a name.
 * 
 * PROMPT contains the prompt used in such a case.
 *
 * If the name given is not empty and matches exactly one conference or
 * person a Conf_z_info of that conference is returned otherwise an empty
 * Conf_z_info is returned.
 */
Export  Conf_z_info
ask_for_conf(String			* name,
	     const unsigned char	* prompt)
{
    Conf_z_info_list	  matching	= EMPTY_CONF_Z_INFO_LIST;
    Conf_z_info		  result	= EMPTY_CONF_Z_INFO;

#define CLEAN_UP()				\
    {						\
	release_conf_z_info_list(&matching);	\
    }

    TOPLOOP_SETJMP();

    if (s_empty(*name))
    {
	xprintf("\n%s - ", prompt);
	read_line(name);
	newline();
	*name = s_strip_trailing(*name, command_separators);
	if (s_empty(*name))
	{
	    RETURN (result);
	}
    }

    matching = find_conf_or_pers_no(*name);
    if (matching.no_of_confs != 1)
    {
	inform_user(ambigous_or_no_conf, &matching);
    }
    else
    {
	copy_conf_z_info(&result, &matching.confs[0]);
	xprintf("%S\n", result.name);
    }

    RETURN (result);

#undef CLEAN_UP
}   /* END: ask_for_conf() */


/*
 *  Perform the actions to join a conference.  Inform the
 *  user if he is not allowed to join it.
 */
Export  Success
do_join_conf(Conf_no	conf_to_join)

{
#define FUNCTION "do_join_conf()"

    Success		  retval;
    Conference	*volatile conf			= NULL;
    Conference		* supervisor_conf	= NULL;
    Kom_err		  error;
    Person	*volatile pers_ptr		= NULL;


#define CLEAN_UP()					\
    {							\
	zfree(release_person(pers_ptr));		\
	zfree(release_conf(conf));			\
	zfree(release_conf(supervisor_conf));		\
    }

    TOPLOOP_SETJMP();


    pers_ptr = pers_stat(cpn, &error);
    if (pers_ptr == NULL)
    {
	fatal2(CLIENT_SERVER_ERROR,
	       "Can't get stat for CPN: %d", error);
    }
    /* BUG: Should be cache_add_member */
    retval = kom_add_member(conf_to_join, cpn, DEFAULT_CONF_PRIORITY,
			    pers_ptr->no_of_confs);
    if (retval == FAILURE)
    {
	switch (kom_errno)
	{
	case  KOM_ACCESS:
	    xprintf(gettext("Mtet r slutet. "
			    "Fr medlemskap, kontakta organisatren.\n"
			    "Organisatren r"));
	    conf = conf_stat(conf_to_join, &error);
	    if (conf == NULL)
	    {
		fatal2(CLIENT_ERROR_TOO_boring,
		       "get_conf_stat() failed: %d", conf_to_join);
	    }

	    print_conf_name(conf->supervisor);
	    newline();
	    RETURN (FAILURE);
	    /*NOTREACHED*/
	    break;

	case  KOM_NO_CONNECT:
	    fatal1(CLIENT_CONNECTION_LOST, "add_member() failed");
	    break;

	default:
	    fatal2(CLIENT_UNEXPECTED_SERVER_ERROR,
		   "add_member() failed: %d", conf_to_join);
	}
    }

    /* Invalidate the membership-cache. */
    first_membership(0,NULL);

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}   /* END: do_join_conf */


/*
 * Perform the actions to leave a conference. Inform the user if he was not
 * a member.
 */
static Success
do_leave_conf(Conf_no	conf_to_leave)
{
#define FUNCTION "do_leave_conf()"

    Success		  retval;

#define CLEAN_UP()				\
    {						\
    }

    TOPLOOP_SETJMP();

    if (locate_membership(conf_to_leave, cpn, NULL) == -1) {
	/* We were not members. Can't the user keep track of what conferences
	   he is a member of? Do nothing. */
	xprintf( gettext("Du var inte med i det mtet.\n") );
	RETURN (FAILURE);
    }
    retval = kom_sub_member(conf_to_leave, cpn);

    if (retval == FAILURE)
    { 	/* This is strange, isn't everybody allowed to leave any conference? */
	switch (kom_errno)
	{
	case KOM_NO_CONNECT:
	    fatal1(CLIENT_CONNECTION_LOST, "sub_member() failed");
	    break;
	   
	default:
	    fatal2(CLIENT_UNEXPECTED_SERVER_ERROR,
		    "sub_member() failed: %d", conf_to_leave);
	}
    }

    /* Invalidate the membership-cache. */
    first_membership(0, NULL);

    RETURN (OK);
    
#undef CLEAN_UP
#undef FUNCTION
} /* do_leave_conf */


/*
 *  Print the name of the conference with the number CONF.
 *  Doesn't print anything if it can't find the name.
 */
Export  void
print_conf_name(Conf_no	conf)
{
    String		  name;	/* May be clobbered by longjmp */
    Kom_err		  error;

#define CLEAN_UP()				\
    { 						\
	zfree(name.string); 			\
    }

    TOPLOOP_SETJMP();

    name = conf_name(conf, &error);	/* BUG: Doesn't check for errors */
    if (s_empty(name))
	xprintf(gettext("Nummer %d (skyddat mte)"), conf);
    else
	s_xputs(name);

    RETURNVOID;

#undef CLEAN_UP
}   /* END: print_conf_name */



/*
 *  Print a type for the conference CONF 
 */
extern  void
print_conf_type(Conference	* conf)
{
    Bool started = FALSE;

    if (conf->type.original)
    {
	xprintf(gettext("original "));
	started = TRUE;
    }

    if (conf->permitted_submitters != 0)
    {
	xprintf(gettext("skrivskyddat "));
	started = TRUE;
    }

    if (conf->type.secret)
    {
	xprintf(gettext("skyddat"));
    }
    else
    {
	if (conf->type.rd_prot)
	{
	    xprintf(gettext("slutet"));
	}
	else
	{
	    xprintf(gettext("ppet"));
	}
    }
}   /* END: print_conf_type() */



/*
 *  Show the status of a person.  The conference status and
 *  member-list of the person's letterbox, should already have been
 *  fetched, and be pointed to by CONF and MEMBERS respectively.  They
 *  should be released by the caller.
 */
Export  Success
show_pers_status(Conf_z_info		  conf_to_stat,
		  Conference		* conf_res,
		  Person		* pers		)

{
#define CLEAN_UP()     				\
    {						\
    }


    xprintf(gettext("Senast inne: "));
    print_time(&pers->last_login);
    /*xgettext:c-format*/
    xprintf(gettext(" %32TFrn konto: %S\n"), pers->username);

    /*xgettext:c-format*/
    xprintf(gettext("Antal mten: %d %32TOlsta brev: %ld\n"),
	    pers->no_of_confs,
	    unread_in_conf(conf_to_stat.conf_no, conf_to_stat.conf_no));

    /* sessions, days, hours, minutes, seconds */
    /*xgettext:c-format*/
    xprintf(gettext("Sessioner: %lu %32TNrvarotid: %dd %dh %dm %ds\n"),
	    pers -> sessions,
	    pers->total_time_present / 86400,
	    (pers->total_time_present % 86400) / 3600,
	    (pers->total_time_present % 3600) / 60,
	    (pers->total_time_present % 60));

    /*xgettext:c-format*/
    xprintf(gettext("Skapade texter: %lu %32TSkapade tecken: %lu\n"),
	    pers->first_created_text + pers->no_of_created_texts - 1,
	    pers->created_bytes);

    /*xgettext:c-format*/
    xprintf(gettext("Lsta texter: %lu %32THmtade texter: %lu\n"),
	    pers -> read_texts,
	    pers -> no_of_text_fetches);

    /*xgettext:c-format*/
    xprintf(gettext("Markerade inlgg: %lu\n"),
	    pers->no_of_marks);

    /*xgettext:c-format*/
    xprintf(gettext("Skapade mten: %d %32TSkapade personer: %d\n"),
	    pers->created_confs,
	    pers->created_persons);

    return  OK;
}   /* END: show_pers_status() */



/*
 *  Show some short information about a conference - name, number,
 *  creator, creation time and supervisor.
 */
Export  Success
show_short_conf_status(Conf_z_info	  conf_to_stat,
			Conference	* conf		)

{
    /*xgettext:c-format*/
    xprintf(gettext("Nummer: %s %d %32TNamn: "),
	    conf_to_stat.type.letter_box 
	    ? gettext("Person") 
	    : gettext("Mte"),
	    conf_to_stat.conf_no);
    xprintf("%S\n", conf_to_stat.name);
    xprintf(gettext("Skapelsetid: "));
    print_time(&conf->creation_time);
    xprintf(gettext(" %32TSkapad av: "));
    print_conf_name((Conf_no) conf->creator);
    xprintf(gettext("\nOrganisatr: "));
    print_conf_name(conf->supervisor);
    newline();

    return  OK;
}   /* END: show_short_conf_status() */



/*
 *  Show some more information about a conference - number of texts,
 *  number of members, type of conf, super conf, comment conf,
 *  permitted submitters.
 */
Export  Success
show_more_conf_status(Conf_z_info	  conf_to_stat,
		       Conference	* conf		)

{
    xprintf(gettext("Antal inlgg: %5lu %32TAntal medlemmar: %d\n"),
	    conf->first_local_no + conf->no_of_texts - 1,
	    conf->no_of_members);

    xprintf(gettext("Typ: "));
    print_conf_type(conf);
    if (conf->super_conf != 0)
    {
	xprintf(gettext(" %32TSupermte: "));
	print_conf_name(conf->super_conf);
    }
    newline();

    if (conf->permitted_submitters != 0)
    {
	xprintf(gettext("Redaktr: "));
	print_conf_name(conf->permitted_submitters);
	newline();
    }

    return  OK;
}   /* END: show_more_conf_status() */



/*
 *  Show a list of members in a conference.  The conferences status
 *  and its member-list, should already have been fetched, and be
 *  pointed to by CONF and MEMBERS respectively.  They should be
 *  released by the caller.
 */
Export  Success
show_member_list(Conf_z_info	  conf_to_stat,
		  Conference	* conf_res,
		  Member_list	* members)

{
    unsigned short		  count;


    /*xgettext:c-format*/
    xprintf(gettext("%S har fljande medlemmar:\n"), 
	    conf_to_stat.name);
    xprintf(gettext("Senast inne%18TOsett%26TNamn\n"));
    newline();

    for (count = 0; count < members->no_of_members ; count++)
    {
	Membership	  ship		= EMPTY_MEMBERSHIP;
	long		  unread;

#define CLEAN_UPPER()	release_membership(&ship)

	locate_membership(conf_to_stat.conf_no,
			  members->members[count].member,
			  &ship);
	print_time(&ship.last_time_read);
	unread = unread_in_conf(members->members[count].member,
				conf_to_stat.conf_no);
	if (unread > 0)
	{
	    xprintf("%18T%6ld", unread);
	}
	goto_column(26);
	print_conf_name(members->members[count].member);
	CLEAN_UPPER();
	newline();
    }

    return  OK;
}   /* END: show_member_list() */



/* Return True if MEMBERSHIP is a member of CONF_NO_LIST
 * else return False
 */
Export Bool
membership_in_conf_no_list(Membership membership,
			    Conf_no_list list)
{
    int r;

    for (r=0; r < list.no_of_confs ; r++) {
	if ( membership.conf_no == list.conf_nos[r] ) {
	    return TRUE;
	}
    }
    return FALSE;
}
