pragma License (Modified_GPL);

------------------------------------------------------------------------------
--                                                                          --
--                      CHARLES CONTAINER LIBRARY                           --
--                                                                          --
--              Copyright (C) 2001-2004 Matthew J Heaney                    --
--                                                                          --
-- The Charles Container Library ("Charles") is free software; you can      --
-- redistribute it and/or modify it under terms of the GNU General Public   --
-- License as published by the Free Software Foundation; either version 2,  --
-- or (at your option) any later version.  Charles is distributed in the    --
-- hope that it will be useful, but WITHOUT ANY WARRANTY; without even the  --
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. --
-- See the GNU General Public License for more details.  You should have    --
-- received a copy of the GNU General Public License distributed with       --
-- Charles;  see file COPYING.TXT.  If not, write to the Free Software      --
-- Foundation,  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.    --
--                                                                          --
-- As a special exception, if other files instantiate generics from this    --
-- unit, or you link this unit with other files to produce an executable,   --
-- this unit does not by itself cause the resulting executable to be        --
-- covered by the GNU General Public License.  This exception does not      --
-- however invalidate any other reasons why the executable file might be    --
-- covered by the GNU Public License.                                       --
--                                                                          --
-- Charles is maintained by Matthew J Heaney.                               --
--                                                                          --
-- http://home.earthlink.net/~matthewjheaney/index.html                     --
-- mailto:matthewjheaney@earthlink.net                                      --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Finalization;

generic

   type Element_Type is private;

   with function "=" (Left, Right : Element_Type)
      return Boolean is <>;

package Charles.Lists.Single.Unbounded is
   pragma Preelaborate (Unbounded);

   subtype Element_Subtype is Element_Type;

   function Compare_Elements (Left, Right : Element_Type)
      return Boolean renames "=";


   type Container_Type is private;

   type Iterator_Type is private;

   Null_Iterator : constant Iterator_Type;


   function Empty_Container return Container_Type;

   procedure Assign
     (Target : in out Container_Type;
      Source : in     Container_Type);

   function "=" (Left, Right : Container_Type) return Boolean;

   generic
      with function "<" (Left, Right : Element_Type) return Boolean is <>;
   function Generic_Less (Left, Right : Container_Type) return Boolean;

   function Length (Container : Container_Type) return Natural;

   function Is_Empty (Container : Container_Type) return Boolean;

   procedure Clear (Container : in out Container_Type);

   procedure Swap (Left, Right : in out Container_Type);

   procedure Prepend
     (Container : in out Container_Type;
      New_Item  : in     Element_Type);

   procedure Append
     (Container : in out Container_Type;
      New_Item  : in     Element_Type);

   procedure Insert_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      New_Item  : in     Element_Type;
      Iterator  :    out Iterator_Type);

   procedure Insert_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      New_Item  : in     Element_Type);

   procedure Insert_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      Iterator  :    out Iterator_Type);

   procedure Delete_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type);

   procedure Delete_First (Container : in out Container_Type);

   procedure Delete
     (Container : in out Container_Type;
      Item      : in     Element_Type);

   generic
     with function Predicate (Element : Element_Type)
         return Boolean is <>;
   procedure Generic_Delete (Container : in out Container_Type);

   procedure Delete_Duplicates (Container : in out Container_Type);

   generic
      with function Predicate (Left, Right : Element_Type)
         return Boolean is <>;
   procedure Generic_Delete_Duplicates
     (Container : in out Container_Type);

   generic
      with function "<" (Left, Right : Element_Type)
         return Boolean is <>;
   procedure Generic_Merge
     (Container : in out Container_Type;
      Source    : in out Container_Type);

   procedure Reverse_Container (Container : in out Container_Type);

   generic
      with function "<" (Left, Right : Element_Type)
         return Boolean is <>;
   procedure Generic_Sort (Container : in out Container_Type);

   procedure Splice_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      Pred      : in     Iterator_Type);

   procedure Splice_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      Source    : in out Container_Type);

   procedure Splice_After
     (Container : in out Container_Type;
      Position  : in     Iterator_Type;
      Source    : in out Container_Type;
      Pred      : in     Iterator_Type);

   function First (Container : Container_Type) return Iterator_Type;

   function First_Element (Container : Container_Type) return Element_Type;

   function Last (Container : Container_Type) return Iterator_Type;

   function Last_Element (Container : Container_Type) return Element_Type;

   function Back (Container : Container_Type) return Iterator_Type;

   function Element (Iterator : Iterator_Type) return Element_Type;

   generic
      type Element_Access is access all Element_Type;
   function Generic_Element
     (Iterator : Iterator_Type) return Element_Access;

   procedure Replace_Element
     (Iterator : in Iterator_Type;
      By       : in Element_Type);

   function Is_In
     (Item      : Element_Type;
      Container : Container_Type) return Boolean;

   function Find
     (Container : Container_Type;
      Item      : Element_Type;
      Position  : Iterator_Type := Null_Iterator) return Iterator_Type;

   generic
      with function Predicate (Element : Element_Type)
         return Boolean is <>;
   function Generic_Find
     (Container : Container_Type;
      Position  : Iterator_Type := Null_Iterator) return Iterator_Type;

   function Succ
     (Iterator : Iterator_Type) return Iterator_Type;

   function Pred
     (Container : Container_Type;
      Iterator  : Iterator_Type) return Iterator_Type;

   procedure Increment (Iterator : in out Iterator_Type);

   procedure Decrement
     (Container :        Container_Type;
      Iterator  : in out Iterator_Type);

   generic
      with procedure Process (Iterator : Iterator_Type) is <>;
   procedure Generic_Iteration (Container : in Container_Type);

private

   type Node_Type;
   type Node_Access is access Node_Type;

   type Node_Type is
      record
         Element : aliased Element_Type;
         Next    : Node_Access;
      end record;

   function "=" (L, R : Node_Type) return Boolean is abstract;


   use Ada.Finalization;

   type Container_Type is
      new Controlled with record
         First  : Node_Access;
         Last   : Node_Access;
         Length : Natural := 0;
      end record;

   procedure Adjust (Container : in out Container_Type);

   procedure Finalize (Container : in out Container_Type) renames Clear;


   type Iterator_Type is
      record
         Node : Node_Access;
      end record;

   Null_Iterator : constant Iterator_Type := (Node => null);

end Charles.Lists.Single.Unbounded;
