/** \file
 * Defines helpful macros for factory registration and instantiation.
 *
 * \author Martin F. Krafft <krafft@ailab.ch>
 * \date $Date: 2004/06/03 18:14:03 $
 * \version $Revision: 1.8 $
 */

#ifndef __INCLUDE_FACTORY_MACROS_H__
#define __INCLUDE_FACTORY_MACROS_H__

#ifndef FACTORY_DFLT_KEYTYPE
#  define FACTORY_DFLT_KEYTYPE unsigned int
#endif

/**
Designates Builder as the factory for generating objects of type _Sub
using the given Key. _Sub must be a subtype of _Base. Key must be of
type _Key.

*/
#define FACTORY_REGISTER_CLASS_FULL(_Key, _Base, _Sub, Key, Builder) \
  namespace { \
    ::Factory::Registrar<_Key, _Base, _Sub> \
      Factory::Registrar<_Key, _Base, _Sub>::_S_instance(Key, Builder); \
  }

/**
Designates Builder as the factory for generating objects of type _Sub
using the given Key. _Sub must be a subtype of _Base

*/
#define FACTORY_REGISTER_CLASS_BUILDER(_Base, _Sub, Key, Builder) \
  FACTORY_REGISTER_CLASS_FULL(FACTORY_DFLT_KEYTYPE, _Base, _Sub, Key, Builder)

/**
*/
#define FACTORY_REGISTER_CLASS_KEYTYPE(_Key, _Base, _Sub, Key) \
  namespace { \
    ::Factory::Registrar<_Key, _Base, _Sub> \
      Factory::Registrar<_Key, _Base, _Sub>::_S_instance(Key); \
  }

/**
*/
#define FACTORY_REGISTER_CLASS(_Base, _Sub, Key) \
  FACTORY_REGISTER_CLASS_KEYTYPE(FACTORY_DFLT_KEYTYPE, _Base, _Sub, Key)

/**
*/
#define FACTORY_REGISTER_SINGLE_CLASS(_Type) \
  FACTORY_REGISTER_CLASS(_Type, _Type, FACTORY_DFLT_KEYTYPE())

/**

Returns a _Type using Builder<_Type>. The caller owns the returned
pointer, which may be NULL.
*/
#define FACTORY_BUILD(_Type) \
  ::Factory::Builder<_Type>().build()

/**
Returns a new _Type pointer using Builder<_Type>. Arg is used as a
prototype for the creation, and the returned pointer is a copy of that
type. The caller owns the returned pointer.

(DUNNO: is the copy polymorphic?)
*/
#define FACTORY_BUILD_ARG(_Type, Arg) \
  ::Factory::Builder<_Type>().build(Arg)

/**
Destroys the passed in Obj, which must be a pointer of type _Type.
*/
#define FACTORY_TEARDOWN(_Type, Obj) \
  ::Factory::Builder<_Type>().teardown(Obj)

/**

Returns a new object of type _Sub, which must be a subtype of _Base
(or be the same as _Base).

The caller owns the returned pointer.

*/
#define FACTORY_BUILD_CHILD(_Base, _Sub) \
  ::Factory::Builder<_Base, _Sub>().build()

/**

   Similar to FACTORY_BUILD_CHILD, but returns a pointer of type _Sub,
   and the new object is a copy of Arg, which must be of the type
   _Sub.

(DUNNO: is the copy polymorphic?)
*/
#define FACTORY_BUILD_CHILD_ARG(_Base, _Sub, Arg) \
  ::Factory::Builder<_Base, _Sub>().build(Arg)

/**

Similar to FACTORY_TEARDOWN, but destroys Obj, which must be type _Sub
(which must be a subtype, or the same type as, _Base).

*/
#define FACTORY_TEARDOWN_CHILD(_Base, _Sub, Obj) \
  ::Factory::Builder<_Base, _Sub>().teardown(Obj)

/**
Returns a new object, of type _Type. The exact type is dependent on
the factory registered for the given Key.

The caller owns the returned pointer, which may be NULL.
*/
#define FACTORY_CREATE(_Type, Key) \
  ::Factory::Factory<_Type, FACTORY_DFLT_KEYTYPE>::create(Key)

/**

*/
#define FACTORY_CREATE_ARG(_Type, Key, Arg) \
  ::Factory::Factory<_Type, FACTORY_DFLT_KEYTYPE>::create(Key, Arg)

/**
*/
#define FACTORY_DESTROY(_Type, Key, Obj) \
  ::Factory::Factory<_Type, FACTORY_DFLT_KEYTYPE>::destroy(Key, Obj)

/**
*/
#define FACTORY_CREATE_CHILD(_Base, Key) \
  ::Factory::Factory<_Base, FACTORY_DFLT_KEYTYPE>::create(Key)

/**
*/
#define FACTORY_CREATE_CHILD_ARG(_Base, Key, Arg) \
  ::Factory::Factory<_Base, FACTORY_DFLT_KEYTYPE>::create(Key, Arg)

/**
*/
#define FACTORY_DESTROY_CHILD(_Base, Key, Obj) \
  ::Factory::Factory<_Base, FACTORY_DFLT_KEYTYPE>::destroy(Key, Obj)

/**
*/
#define FACTORY_CREATE_CHILD_KEYTYPE(_Key, _Base, Key) \
  ::Factory::Factory<_Base, _Key>::create(Key)

/**
*/
#define FACTORY_CREATE_CHILD_KEYTYPE_ARG(_Key, _Base, Key, Arg) \
  ::Factory::Factory<_Base, _Key>::create(Key, Arg)

/**
*/
#define FACTORY_DESTROY_CHILD_KEYTYPE(_Key, _Base, Key, Obj) \
  ::Factory::Factory<_Base, _Key>::destroy(Key, Obj)

#endif // include guard

/* COPYRIGHT --
 *
 * This file is part of libfactory++, a C++ template factory framework.
 * libfactory++ is (c) 2004 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
